/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/frame/record.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref diagnostics::Record
 *
 * $Id: record.t.cpp,v 1.17 2005/06/23 09:54:20 esdentem Exp $
 *
 * @author Christian Schallhart
 */
#include <diagnostics/unittest.hpp>

#include <diagnostics/frame/record.hpp>

// used components
#include <diagnostics/util/to_string.hpp>

// test support
#include <diagnostics/util/compilation_mode.ts.hpp>
#include <diagnostics/util/dummy_values_for_tid_pid.ts.hpp>

#define TEST_COMPONENT_NAME Record
#define TEST_COMPONENT_NAMESPACE diagnostics

DIAGNOSTICS_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;

using namespace unittest;
using ::diagnostics::internal::to_string;


/**
 * @brief two records to play with -- assembly_cstr
 */
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
#  define RECORD_1 \
    Record r1(LEVEL_TEST,TYPE_TESTCASE_ENTER,0,"what","base","file",1, \
              Dummy_Values_For_Pid::value2,Dummy_Values_For_Tid::value3,4,5,"host")
#  define RECORD_2 \
    Record r2(LEVEL_AUDIT,TYPE_TESTCASE_EXIT,1,"what1","base1","file1",2, \
              Dummy_Values_For_Pid::value3,Dummy_Values_For_Tid::value2,5,6,"host1")
#else 
#  define RECORD_1 \
    Record r1(LEVEL_TEST,TYPE_TESTCASE_ENTER,0,"what","base","file",1)
#  define RECORD_2 \
    Record r2(LEVEL_AUDIT,TYPE_TESTCASE_EXIT,1,"what1","base1","file1",2)
#endif

/**
 * @brief two different parameter sets into the assembly cstr (all accessors)
 */
void assembly_cstr(Test_Data & test_data)
{
    RECORD_1;
    TEST_ASSERT(r1.level()==LEVEL_TEST);
    TEST_ASSERT(r1.type()==TYPE_TESTCASE_ENTER);
    TEST_ASSERT(r1.nr_what()==0);
    TEST_ASSERT(r1.str_what()=="what");
    TEST_ASSERT(r1.base_file_name()=="base");
    TEST_ASSERT(r1.file_name()=="file");
    TEST_ASSERT(r1.line()==1);
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    TEST_ASSERT(r1.pid()==Dummy_Values_For_Pid::value2);
    TEST_ASSERT(r1.tid()==Dummy_Values_For_Tid::value3);
    TEST_ASSERT(r1.sec()==4);
    TEST_ASSERT(r1.usec()==5);
    TEST_ASSERT(r1.hostname()=="host");
#endif

    ////////////////////////////////////////////////////////////////////////////////
    
    RECORD_2;
    TEST_ASSERT(r2.level()==LEVEL_AUDIT);
    TEST_ASSERT(r2.type()==TYPE_TESTCASE_EXIT);
    TEST_ASSERT(r2.nr_what()==1);
    TEST_ASSERT(r2.str_what()=="what1");
    TEST_ASSERT(r2.base_file_name()=="base1");
    TEST_ASSERT(r2.file_name()=="file1");
    TEST_ASSERT(r2.line()==2);
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    TEST_ASSERT(r2.pid()==Dummy_Values_For_Pid::value3);
    TEST_ASSERT(r2.tid()==Dummy_Values_For_Tid::value2);
    TEST_ASSERT(r2.sec()==5);
    TEST_ASSERT(r2.usec()==6);
    TEST_ASSERT(r2.hostname()=="host1");
#endif
}


/**
 * @brief two different parameter sets -- output is complete, i.e.,
 * all fields are reflected
 */
void stream_operator(Test_Data & test_data)
{
    RECORD_1;
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r1",to_string(r1)));
    
    RECORD_2;
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r2",to_string(r2)));
}


/**
 * @brief using the output as reference -- we check the copy cstr and assignment
 */
void copy_cstr_and_assignment(Test_Data & test_data)
{
    RECORD_1;
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r1",to_string(r1)));
    
    RECORD_2;
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r2",to_string(r2)));

    // copy r1->r3
    Record r3(r1);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r1",to_string(r3)));

    // copy r2->r4
    Record r4(r2);
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r2",to_string(r4)));    

    // assign r2->r3
    r3=r2;
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r2",to_string(r3)));

    // assign r1->r4
    r3=r2;
    TEST_ASSERT(test_data.compare
		(DIAGNOSTICS_COMPILATION_MODE_ID+"r1",to_string(r4)));
}

/**
 * @brief r1 and r2 are different....  use that
 */
void comparison(Test_Data & test_data)
{
    using ::diagnostics::internal::to_string;
    RECORD_1;
    RECORD_2;

    TEST_ASSERT(r1!=r2);

    Record r3(r1);
    TEST_ASSERT(r1==r3);
    TEST_ASSERT(r2!=r3);

    Record r4(r2);
    TEST_ASSERT(r2==r4);
    TEST_ASSERT(r1!=r4);
    TEST_ASSERT(r3!=r4);
}


TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&assembly_cstr,LEVEL_PROD);
TEST_NORMAL_CASE(&stream_operator,LEVEL_PROD);
TEST_NORMAL_CASE(&comparison,LEVEL_PROD);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;

// vim:ts=4:sw=4
