/*
 * Copyright (C) 2009 Florian Weimer <fw@deneb.enyo.de>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <diagnostics/extensions/stacktrace/posix.hpp>
#include <diagnostics/extensions/stacktrace/error.hpp>

#include <algorithm>

#include <cstring>
#include <cstdio>
#include <cerrno>
#include <climits>

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>

//////////////////////////////////////////////////////////////////////
// enyo::posix functions

DIAGNOSTICS_NAMESPACE_BEGIN;
STACKTRACE_NAMESAPCE_BEGIN;

bool
POSIX::IsAbsolute(const std::string& path)
{
  return (!path.empty()) && path[0] == '/';
}

std::string
POSIX::Cwd()
{
  char buf[PATH_MAX];
  Error::Unless(getcwd(buf, sizeof(buf)) == buf);
  return std::string(buf);
}

bool
POSIX::Exists(const std::string& path)
{
  return access(path.c_str(), F_OK) == 0;
}

bool
POSIX::IsExecutable(const std::string& path)
{
  return access(path.c_str(), X_OK) == 0;
}

std::string
POSIX::SearchCommand(const std::string& cmd,
                           const std::string& path)
{
  if (IsAbsolute(cmd)) {
    if (!IsExecutable(cmd)) {
      throw Error(EACCES);
    }
    return cmd;
  }
  if (std::find(cmd.begin(), cmd.end(), '/') != cmd.end()) {
    if (!IsExecutable(cmd)) {
      throw Error(EACCES);
    }
    std::string wd(Cwd());
    std::string result;
    result.reserve(cmd.size() + 1 + wd.size());
    result += wd;
    result += '/';
    result += cmd;
    return result;
  }
  if (path.empty()) {
    throw Error(ENOENT);
  }

  bool found = false;
  std::string wd(Cwd());
  std::string c;
  std::string::const_iterator p = path.begin();
  const std::string::const_iterator end = path.end();
  while (true) {
    std::string::const_iterator colon = std::find(p, end, ':');
    if (p == colon || (std::distance(p, colon) == 1 && *p == '.')) {
      c.assign(wd);
      c += '/';
    } else if (*p == '/') {
      c.assign(p, colon);
      c += '/';
    } else {
      c.assign(wd);
      c += '/';
      c.assign(p, colon);
      c += '/';
    }
    c.append(cmd);
    if (IsExecutable(c)) {
      return c;
    }
    if ((!found) && Exists(c)) {
      found = true;
    }
    if (colon == end) {
      break;
    }
    p = colon + 1;
  }
  if (found) {
    throw Error(EACCES);
  }
  throw Error(ENOENT);
}

void
POSIX::CreateSafely(const std::string& path)
{
  const char* p = path.c_str();
  while (true) {
    int ret = ::open(p, O_CREAT | O_EXCL, 0666);
    if (ret >= 0) {
      ::close(ret);
      return;
    }
    if (errno != EEXIST) {
      throw Error();
    }
    Error::Unless(::unlink(p) == 0);
  }
}

void
POSIX::Rename(const std::string& from, const std::string& to)
{
  Error::Unless(::rename(from.c_str(), to.c_str()) == 0);
}

std::string
POSIX::RealPath(const std::string &path)
{
  char p[PATH_MAX];
  ::realpath(path.c_str(), p);
  std::string result(p);
  return result;
}

STACKTRACE_NAMESAPCE_END;
DIAGNOSTICS_NAMESPACE_END;

