/****************************************************************************
 *                                                                          *
 * U U    6   1            U U   FFF  O   O  TTT                            *
 * U U   6   11   b        U U   F   O O O O  T                             *
 * U U - 66   1   bb  y y  U U - FF  O O O O  T                             *
 * U U   6 6  1   b b  y   U U   F   O O O O  T                             *
 *  U     6   1   bb   y    U    F    O   O   T                             *
 *                                                                          *
 * U61 is another block based game                                          *
 * Copyright (C) 2000 Christian Mauduit (ufoot@ufoot.org / www.ufoot.org)   *
 *                                                                          *
 * This program is free software; you can redistribute it and/or            *
 * modify it under the terms of the GNU General Public License              *
 * as published by the Free Software Foundation; either version 2           *
 * of the License, or (at your option) any later version.                   *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA*
 *                                                                          *
 * This project is also available on SourceForge  (http://sourceforge.net)  *
 ****************************************************************************/

/*
 * file name:   map.cpp
 * author:      U-Foot (ufoot@ufoot.org / www.ufoot.org)
 * description: I called this class map for I can't imagine a game without
 *              a "map" class 8) . seriously, a map is where the information
 *              about where the blocks are is stored. it's basically an array
 *              of squares.
 *              it also provides an API to move the player's falling block
 *              and handle any event.
 *              it's important to note that most of the behaviors of a map
 *              are determined by embedded LUA scripts
 */


/*---------------------------------------------------------------------------
 include
 ---------------------------------------------------------------------------*/

#include "global.h"
#include "map.h"
#include "script.h"
#include "time.h"
#include "sound.h"
#include "utils.h"

/*---------------------------------------------------------------------------
 globals
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * these arrays contain constants used for speeds, this way one can
 * have something more subtle than just a linear behavior
 */ 
const int U61_Map::speeds[MAX_SYSTEM_SPEED+1]=
{
  1,    2,  5, 10, 20, 30, 40, 50, 60, 70, 80,
  84, 88, 92, 96,100,104,108,112,116,120,
  124,128,132,136,140,144,148,152,156,160,
  164,168,172,176,180,184,188,192,196,200,
  210,220,240,260,280,300,350,400,450,500
};
const int U61_Map::accels[MAX_SYSTEM_ACCEL+1]=
{
  0,  5, 10, 15, 20, 25,
  30, 40, 50, 70,100,
  130,160,190,220,250
};

/*---------------------------------------------------------------------------
 functions
 ---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* 
 * creation of a default map
 */ 
U61_Map::U61_Map()
{
  active=false;
}

/*--------------------------------------------------------------------------*/
/* 
 * destruction of a map
 */ 
U61_Map::~U61_Map()
{

}

/*--------------------------------------------------------------------------*/
/*
 * clears a map
 */
void U61_Map::clear()
{
  int i,j;

  for (j=0;j<HEIGHT;++j)
    {
      for (i=0;i<WIDTH;++i)
        {
	  squares[i][j].disable();
	  //squares[i][j].enable();
	  squares[i][j].set_color((i+j)%8);
        }
    }
  curse_state=false;
  curse_x=0;
  curse_y=0;
}


/*--------------------------------------------------------------------------*/
/*
 * begin functino, to call each time the player has lost 
 */
void U61_Map::begin()
{
  int i;
  U61_Event evt;

  auto_events.clear();
  request_events.clear();
  
  clear();

  name[0]=0; 
  block.reset();
  next_block.reset();
  block_requested=false;
  matching=false;
  speed_counter=0;
  accel_counter=0;
  speed=base_speed;
  accel=base_accel; 
  background=0;
  prevision_state=true;
  preview_state=true;
  match_count=0;
  curse_counter=curse_delay;
  nb_antedote=0;
  if (score>best_score)
    {
      best_score=score;
    }
  score=0;
  for (i=0;i<NB_GLOBAL;++i)
    {
      global_val[i]=0;
    }        
  for (i=0;i<MAX_CURSE_ID;++i)
    {
      curse_begin[i]=-1;
    }        
  change_background();

  evt.code=U61_Event::SET_VICTIM;
  evt.par=target_id;
  put_auto_event(evt);
}

/*--------------------------------------------------------------------------*/
/*
 * initialization of a map, must be called before the map is used
 */
void U61_Map::init(int id,int s,int a,int d)
{
  player_id=id;
  base_speed=s;
  base_accel=a;
  curse_delay=d*100;

  reset();
}

/*--------------------------------------------------------------------------*/
/*
 * resets completely a map
 */
void U61_Map::reset()
{
  target_id=player_id;
  map_time=0;
  active=false;
  best_score=0;
  score=0;
  games=0; 
  silent=false;

  begin();
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the map is activated
 * by activated map, we mean a map where blocks are falling and someone
 * is actually playing. a player can have a disactivated map,in this
 * case u61 is waiting for him to press the start key and activate the map
 */
bool U61_Map::is_active()
{
  return active;
}

/*--------------------------------------------------------------------------*/
/*
 * activates a map
 */
void U61_Map::set_active(int time)
{
  reset();
  map_time=time;
  active=true;
}

/*--------------------------------------------------------------------------*/
/*
 * disactivates a map
 */
void U61_Map::set_inactive()
{
  active=false;
}

/*--------------------------------------------------------------------------*/
/*
 * changes the background of the map randomly
 */
void U61_Map::change_background()
{
  //cout<<"Number of backgrounds"<<U61_Global::data.nb_map<<"\n";
  background=U61_Utils::random(U61_Global::data.nb_map);
}

/*--------------------------------------------------------------------------*/
/*
 * returns the background of the map
 */
int U61_Map::get_background()
{
  return background;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the color of a square
 * this function is designed for use in Lua script
 * so it uses a special convention:
 * the -1 colors specifies that there's no square at this location
 */
void U61_Map::set_square_color(int x,int y,int color)
{
  if (x>=0 && y>=0 && x<WIDTH && y<HEIGHT)
    {
      if (color>=0 && color<U61_Square::NB_COLORS)
        {
	  squares[x][y].enable();
	  squares[x][y].set_color(color);
        }
      else
        {
	  squares[x][y].disable();
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the color of a square
 * this function is designed for use in Lua script
 * so it uses a special convention:
 * the -1 colors specifies that there's no square at this location
 */
int U61_Map::get_square_color(int x,int y)
{
  int color=-1;

  if (x>=0 && y>=0 && x<WIDTH && y<HEIGHT)
    {
      if (squares[x][y].is_enabled())
        {
	  color=squares[x][y].get_color();
        }
    }
  return color;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the score of the map, negative scores are forbidden
 */
void U61_Map::set_score(int s)
{
  if (s>=0)
    {
      score=s;
    }
  else
    {
      score=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the score associated to the map
 */
int U61_Map::get_score()
{
  return score;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the best score associated to the map
 */
int U61_Map::get_best_score()
{
  return best_score;
}

/*--------------------------------------------------------------------------*/
/*
 * adds the given value to the current score
 */
void U61_Map::add_score(int s)
{
  score+=s;
  if (score<0)
    {
      score=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the time associated to the map
 */
int U61_Map::get_time()
{
  return map_time;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the number of games associated to the map
 */
int U61_Map::get_nb_games()
{
  return games;
}

/*--------------------------------------------------------------------------*/
/*
 * returns a number which is used by the lua match function
 */
int U61_Map::get_match_count()
{
  return match_count;
}

/*--------------------------------------------------------------------------*/
/*
 * sets a global integer into the map
 * used for lua scripts to store global permanent values
 */
void U61_Map::set_global(int i, int glob)
{
  if (i>=0 && i<NB_GLOBAL)
    {
      global_val[i]=glob;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets a global integer from the map
 * used for lua scripts to store global permanent values
 */
int U61_Map::get_global(int i)
{
  int glob=0;

  if (i>=0 && i<NB_GLOBAL)
    {
      glob=global_val[i];
    }

  return glob;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the age of a persistent curse
 * If negative, it means the curse is not registered or is over
 */
int U61_Map::get_curse_age(int i)
{
  int age=false;

  if (i>=0 && i<MAX_CURSE_ID)
    {
      if (curse_begin[i]<0 || map_time>curse_end[i])
	{
	  age=-1;
	}
      else
	{
          age=((map_time-curse_begin[i])/U61_Time::ONE_SECOND);
	}
    }

  //cout<<"Age="<<age<<"\n";
  return age;
}

/*--------------------------------------------------------------------------*/
/*
 * Registers a persistent curse. After this, calls to get_curse_age
 * will return how long ago the curse has been registered
 */
void U61_Map::register_curse(int i, int length)
{
  if (i>=0 && i<MAX_CURSE_ID)
    {
      /*
       * if length is zero, the default length is 10 hours
       * that's too say infinite...
       */
      if (length<=0)
	{
	  length=36000;
	}
      curse_begin[i]=map_time;
      curse_end[i]=map_time+length*U61_Time::ONE_SECOND;
      //cout<<"Registering "<<i<<" begin="<<curse_begin[i]
      //  <<" end="<<curse_end[i]<<"\n";
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Cancels a registered curse
 */
void U61_Map::cancel_curse(int i)
{
  if (i>=0 && i<MAX_CURSE_ID)
    {
      curse_begin[i]=-1;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the oldest curse available
 * Returns -1 if none existing
 */
int U61_Map::get_oldest_curse()
{
  int oldest;
  int i;
  int result=-1;

  oldest=map_time;
  
  for (i=0;i<MAX_CURSE_ID;++i)
    {
      if (get_curse_age(i)>=0 &&
	  curse_begin[i]<=map_time)
	{ 
	  oldest=map_time;
          result=i;
	}
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * gets the number of persisten curses which are affecting the map
 */
int U61_Map::get_nb_curse()
{
  int nb;
  int i;

  nb=0;
  for (i=0;i<MAX_CURSE_ID;++i)
    {
      if (get_curse_age(i)>=0)
	{
	  nb++;
	}
    }
  return nb;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the number of antedotes
 */
int U61_Map::get_nb_antedote()
{
  return nb_antedote;
}

/*--------------------------------------------------------------------------*/
/*
 * Adds an antedote
 */
void U61_Map::add_antedote()
{
  nb_antedote++;
}

/*--------------------------------------------------------------------------*/
/*
 * Deletes an antedote
 */
void U61_Map::delete_antedote()
{
  if (nb_antedote>0)
    {
      nb_antedote--;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Sets the x coordinate of the curse
 */
void U61_Map::set_curse_x(int x)
{
  if (x>=0 && x<WIDTH)
    {
      curse_x=x;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Sets the y coordinate of the curse
 */
void U61_Map::set_curse_y(int y)
{
  if (y>=0 && y<HEIGHT)
    {
      curse_y=y;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Sets the state of the curse
 */
void U61_Map::set_curse_state(bool state)
{
  curse_state=(state!=false);
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the x coordinate of the curse
 */
int U61_Map::get_curse_x()
{
  return curse_x;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the y coordinate of the curse
 */
int U61_Map::get_curse_y()
{
  return curse_y;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns the state of the curse
 */
bool U61_Map::get_curse_state()
{
  return curse_state!=false;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if the curse state is true and the curse is located in
 * a square which is not empty/disabled
 */
bool U61_Map::is_curse_available()
{
  bool result=false;

  if (curse_x>=0 && curse_y>=0 && curse_x<WIDTH && curse_y<HEIGHT
      && curse_state
      && squares[curse_x][curse_y].is_enabled())
    {
      result=true;
    }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Sends a curse to a remote player
 * In fact it justs puts a REQUEST_CURSE event in the event queue, and
 * the rest is done by dispatchers
 */
void U61_Map::send_curse(int curse_id)
{
  U61_Event evt;

  if (player_id!=target_id && curse_id>=0 && curse_id<MAX_CURSE_ID)
    {
      //cout<<"Sending curse "<<curse_id<<"\n";

      evt.code=U61_Event::REQUEST_CURSE;
      evt.par=curse_id;
      /*
       * We use put_request_event instead of calling put_auto_event
       * since the target is not the player here...
       */
      put_request_event(evt);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the name of the next curse that will be fired
 */
char *U61_Map::get_curse_name()
{
  char *name;

  if (is_curse_available())
    {
      name=U61_Script::get_curse_name(this,curse_id);
    }
  else
    {
      name="";
    }

  return name;
}

/*--------------------------------------------------------------------------*/
/*
 * compute the map to the next instant
 */
void U61_Map::compute_next()
{
  //cout<<"Compute next\n";
  map_time++;
  curse_counter++;
  update_explosions();
  if (!(map_time%TIME_CALLBACK_DELAY))
    {
      time_callback();
    }
  if (!(map_time%CHECK_TARGET_DELAY))
    {
      check_target();
    }
  if (!exists_explosion())
    {
      match_pattern();
    }
    
  update_speed();
}

/*--------------------------------------------------------------------------*/
/* 
 * updates the current speed
 */
void U61_Map::update_speed()
{
  check_speed();
  check_accel();

  speed_counter+=speeds[speed];
  if (speed_counter>SPEED_SCALE)
    {
      speed_counter-=SPEED_SCALE;
      block_move_down(); 
    }    
  accel_counter+=accels[accel];
  if (accel_counter>ACCEL_SCALE)
    {
      accel_counter-=ACCEL_SCALE;
      speed++;
      //cout<<"Next speed ("<<speed<<")\n";
    }    

  check_speed();
  check_accel();
}

/*--------------------------------------------------------------------------*/
/*
 * computes the level up to the givent time, no events are handled
 */
void U61_Map::compute_up_to(int time)
{
  if (active)
    {
      while (map_time<time)
        {
	  compute_next();
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * handles an event, ie does anything that has to be done for this event
 */
void U61_Map::handle_event(U61_Event *event)
{
  if (active)
    {
      if (int(event->time)<map_time)
        {
	  cout<<"Inconsitency in event dates detected event="
	      <<*event<<" map_time="
	      <<map_time<<"\n";
        }  
      else
        {
	  compute_up_to(event->time);
	  switch (event->code)
            {
            case U61_Event::ROTATE_LEFT:
	      block_rotate_left();
	      break;
            case U61_Event::ROTATE_RIGHT:
	      block_rotate_right();
	      break;
            case U61_Event::MOVE_RIGHT:
	      block_move_right();
	      break;
            case U61_Event::MOVE_LEFT:
	      block_move_left();
	      break;
            case U61_Event::MOVE_DOWN:
	      block_move_down();
	      break;
            case U61_Event::DROP:
	      block_drop();
	      break;
            case U61_Event::NEW_BLOCK:
	      new_block(event->par);
	      break;
            case U61_Event::LOOSE:
	      loose();
	      break;
            case U61_Event::PREV_VICTIM:
              prev_victim(event);
              break;
            case U61_Event::NEXT_VICTIM:
              next_victim(event);
              break;
            case U61_Event::SET_VICTIM:
              set_victim(event->par);
              break;
            case U61_Event::NAME_LETTER:
              name_add_letter(event->par);
	      break;
            case U61_Event::NEW_CURSE:
              new_curse(event->par);
	      break;
            case U61_Event::REQUEST_CURSE:
              request_curse(*event);
	      break;
            case U61_Event::DO_CURSE:
              do_curse(*event);
	      break;
            case U61_Event::USE_ANTEDOTE:
              use_antedote();
              break;
            }
        }
    }

  //cout<<"Event low level computed="<<event<<"\n";
}

/*--------------------------------------------------------------------------*/
/*
 * anticipates the level to the given time, it's usefull to display the
 * "last" map. the computer just supposes no events have been generated
 * and anticipates the behavior of the map
 * but careful, since no events are computed, themap may become incoherent
 * it's not a problem for in U61 we keep 2 copies of the map, a sure and
 * an anticipated one, called "last"
 * BTW, this function deletes all events from the auto_events list
 * since they are of no use when the map has been anticipated and/or
 * we don't want the list to grow and explose system memory...
 */
void U61_Map::anticipate(int time)
{
  /*
   * we do not anticipate too much for this can lead to
   * deadlocks when the CPU is overloaded.
   * We limit the anticipation to 10 real seconds, knowing
   * that fake events are sent 10 times/sec on local players
   * and 1 time/sec for remote players
   */
  if (time-map_time>ANTICIPATE_MAX)
    {
      time=map_time+ANTICIPATE_MAX;
    }
  compute_up_to(time);
  auto_events.clear();
  request_events.clear();
}

/*--------------------------------------------------------------------------*/
/*
 * polls some auto generated events, such as requests for new blocks
 */
void U61_Map::poll_auto_events()
{
  U61_Event event;

  if (active)
    {
      if ((!is_block_active()) 
	  && (!block_requested) 
	  && (!matching))
        {
	  block_requested=true;
	  event.code=U61_Event::NEW_BLOCK;
	  event.par=U61_Script::new_shape(this,&block);
	  put_auto_event(event);
	  //cout<<"new block event sent\n";
        }
      if (curse_counter>curse_delay 
	  && (!exists_explosion()))
	{
	  curse_counter=0;
          event.code=U61_Event::NEW_CURSE;
          event.par=U61_Script::new_curse(this,&block);
          put_auto_event(event);
          //cout<<"Auto curse event "<<event.par<<"\n";
	}
    }
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next auto generated event
 */
U61_Event U61_Map::get_auto_event()
{
  U61_Event event;

  event=auto_events.front();
  auto_events.pop_front();

  return event;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if there is an auto event
 */
bool U61_Map::exists_auto_event()
{
  return !auto_events.empty();
}

/*--------------------------------------------------------------------------*/
/*
 * puts an event in the generated event queue
 */
void U61_Map::put_auto_event(U61_Event event)
{
  auto_events.push_back(event);
}

/*--------------------------------------------------------------------------*/
/*
 * returns the next generated event request
 */
U61_Event U61_Map::get_request_event()
{
  U61_Event event;

  event=request_events.front();
  request_events.pop_front();

  return event;
}

/*--------------------------------------------------------------------------*/
/*
 * returns true if there is an auto event request
 */
bool U61_Map::exists_request_event()
{
  return !request_events.empty();
}

/*--------------------------------------------------------------------------*/
/*
 * puts an event in the request event queue
 */
void U61_Map::put_request_event(U61_Event event)
{
  event.target=target_id;
  event.author=player_id;
  event.target=target_id;
  event.time=0;
  request_events.push_back(event);
}

/*--------------------------------------------------------------------------*/
/*
 * returns the name of the player playing on this map
 */
char *U61_Map::get_name()
{
  return name;
}

/*--------------------------------------------------------------------------*/
/*
 * draws the map onto the screen
 */
void U61_Map::draw(int x,int y, int size, bool prev)
{
  int i,j;
  int x_square_offset;
  int y_square_offset;
  int draw_x,draw_y;

  if (active)
    {
      x_square_offset=
	(U61_Global::data.map_w[size]
	 -WIDTH*U61_Global::data.square_w[size])/2;
      y_square_offset=
	(U61_Global::data.map_h[size]
	 -HEIGHT*U61_Global::data.square_h[size])/2;
        
      x_square_offset+=x;
      y_square_offset+=y;

      U61_Global::data.map[background][size]->put_screen(x,y);

      for (j=0;j<HEIGHT;++j)
        {
	  for (i=0;i<WIDTH;++i)
            {
	      draw_x=x_square_offset
		+i*U61_Global::data.square_w[size];

	      draw_y=y_square_offset
		+j*U61_Global::data.square_h[size];

              if (j==curse_y && i==curse_x && 
		  curse_state && squares[i][j].is_enabled())
		{
                  if (U61_Square::should_blink())
		    {
		      squares[i][j].draw_curse(draw_x,draw_y,size);
		    }
		  else
		    {
		      squares[i][j].draw(draw_x,draw_y,size);
		    }
		}
	      else
		{
		  squares[i][j].draw(draw_x,draw_y,size);
		}
            }
        }
      if (prevision_state && prev)
        {
	  draw_prevision(x_square_offset,y_square_offset,size);
        }
      block.draw(x_square_offset,y_square_offset,size);
    }
  //    cout<<"draw square\n";
}

/*--------------------------------------------------------------------------*/
/*
 * sets the map in a mute state (does not generate sounds any more)
 */
void U61_Map::mute()
{
  silent=true;
}

/*--------------------------------------------------------------------------*/
/*
 * sets the prevision state
 */
void U61_Map::set_prevision_state(bool state)
{
  prevision_state=state; 
}

/*--------------------------------------------------------------------------*/
/*
 * gets the prevision state
 */
bool U61_Map::get_prevision_state()
{
  return prevision_state; 
}

/*--------------------------------------------------------------------------*/
/*
 * sets the preview state
 */
void U61_Map::set_preview_state(bool state)
{
  preview_state=state; 
}

/*--------------------------------------------------------------------------*/
/*
 * gets the preview state
 */
bool U61_Map::get_preview_state()
{
  return preview_state; 
}

/*--------------------------------------------------------------------------*/
/*
 * sets the speed 
 */
void U61_Map::set_speed(int s)
{
  speed=s; 
  check_speed();
}

/*--------------------------------------------------------------------------*/
/*
 * gets the speed 
 */
int U61_Map::get_speed()
{
  return speed; 
}

/*--------------------------------------------------------------------------*/
/*
 * checks if the speed is within the correct range
 */
void U61_Map::check_speed()
{
  if (speed>MAX_SYSTEM_SPEED)
    {
      speed=MAX_SYSTEM_SPEED;
    }
  if (speed<0)
    {
      speed=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * sets the accel
 */
void U61_Map::set_accel(int a)
{
  accel=a; 
  check_accel();
}

/*--------------------------------------------------------------------------*/
/*
 * gets the accel
 */
int U61_Map::get_accel()
{
  return accel; 
}

/*--------------------------------------------------------------------------*/
/*
 * checks if the accel is within the correct range
 */
void U61_Map::check_accel()
{
  if (accel>MAX_SYSTEM_ACCEL)
    {
      accel=MAX_SYSTEM_ACCEL;
    }
  if (accel<0)
    {
      accel=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * gets the nest block 
 */
U61_Block U61_Map::get_next_block()
{
  return next_block; 
}

/*--------------------------------------------------------------------------*/
/*
 * puts the square in an exploding state
 */
void U61_Map::blow_up_square(int x,int y)
{
  if (x>=0 && y>=0 && x<WIDTH && y<HEIGHT)
    {
      squares[x][y].begin_explosion();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if the square is exploding
 */
bool U61_Map::is_square_exploding(int x,int y)
{
  bool result=false;

  if (x>=0 && y>=0 && x<WIDTH && y<HEIGHT)
    {
      result= squares[x][y].is_exploding();
    }
    
  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * Returns true if there's at least one square exploding
 */
bool U61_Map::exists_explosion()
{
  bool result=false;
  int x,y;

  for (y=0;y<HEIGHT;++y)
    for (x=0;x<WIDTH;++x)
      {
	if (squares[x][y].is_exploding())
	  {
	    result=true;
	  } 
      }

  return result;
}

/*--------------------------------------------------------------------------*/
/*
 * display the prevision, the prevision is "where the block will land if
 * nothing happens" ie where it will land if I press drop  
 */
void U61_Map::draw_prevision(int x,int y,int size)
{
  U61_Block test;
  U61_Block prev;

  if (block.get_nb_items()>0)
    {
      test=block;
    
      while (is_block_ok(&test))
        {
	  prev=test;
	  U61_Script::move_down(this,&test);
          test.center();
        }

      U61_Script::land(this,&prev);
      prev.draw_prevision(x,y,size); 
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to rotate the block on the left, but cancels the action if
 * it is not possible
 */
void U61_Map::block_rotate_left()
{
  //cout<<"rotate left\n";
  U61_Block test;

  test=block;
  U61_Script::rotate_left(this,&test);
  test.center();

  if (is_block_ok(&test))
    {
      block=test;
    }
  else
    {
      test=block;
      U61_Script::move_left(this,&test);
      U61_Script::rotate_left(this,&test);
      test.center();

      if (is_block_ok(&test))
        {
	  block=test;
        }
      else
        {
	  test=block;
	  U61_Script::move_right(this,&test);
	  U61_Script::rotate_left(this,&test);
          test.center();
	  if (is_block_ok(&test))
            {
	      block=test;
            }
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to rotate the block on the right, but cancels the action if
 * it is not possible
 */
void U61_Map::block_rotate_right()
{
  //cout<<"rotate right\n";
  U61_Block test;

  test=block;
  U61_Script::rotate_right(this,&test);
  test.center();
  
  if (is_block_ok(&test))
    {
      block=test;
    }
  else
    {
      test=block;
      U61_Script::move_right(this,&test);
      U61_Script::rotate_right(this,&test);
      test.center();

      if (is_block_ok(&test))
        {
	  block=test;
        }
      else
        {
	  test=block;
	  U61_Script::move_left(this,&test);
	  U61_Script::rotate_right(this,&test);
          test.center();
	  if (is_block_ok(&test))
            {
	      block=test;
            }
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to move the block on the left, but cancels the action if
 * it is not possible
 */
void U61_Map::block_move_left()
{
  //cout<<"move left\n";
  U61_Block test;

  test=block;
  U61_Script::move_left(this,&test);
  test.center();
  if (is_block_ok(&test))
    {
      block=test;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to move the block on the right, but cancels the action if
 * it is not possible
 */
void U61_Map::block_move_right()
{
  //cout<<"move right\n";
  U61_Block test;

  test=block;
  U61_Script::move_right(this,&test);
  test.center();
  if (is_block_ok(&test))
    {
      block=test;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * tries to move the block down, but cancels the action if
 * it is not possible
 */
void U61_Map::block_move_down()
{
  //cout<<"move down\n";
  U61_Block test;

  test=block;
  U61_Script::move_down(this,&test);
  test.center();
  if (is_block_ok(&test))
    {
      block=test;
    }
  else
    {
      stabilize_block();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * performs calls to move_down as much as possible so that the block lands
 */
void U61_Map::block_drop()
{
  //cout<<"drop\n";
  U61_Block test;

  test=block;
  if (block.get_nb_items()>0)
    {
      while (is_block_ok(&test))
        {
	  block=test;
	  U61_Script::move_down(this,&test);
          test.center();
        }
    
      stabilize_block();
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Adds a letter to the map name
 */
void U61_Map::name_add_letter(char c)
{
  int len;

  len=strlen(name);
  if (len<NAME_SIZE)
    {
      name[len]=c;
      name[len+1]=0;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Uses an antedote
 */
void U61_Map::use_antedote()
{
  //cout<<"Using antedote\n";
  if (nb_antedote>0)
    {
      nb_antedote--;
      U61_Script::use_antedote(this);
    }
}

/*--------------------------------------------------------------------------*/
/*
 * checks if there's an active block (size greater than 0)
 */
bool U61_Map::is_block_active()
{
  return block.get_nb_items()>0;
}

/*--------------------------------------------------------------------------*/
/*
 * checks if a position is possible, if yes, returns true
 * this function is used to check if the result of functions such
 * block_rotate_right or block_move_down are incoherent, and therefore
 * should be cancelled
 */
bool U61_Map::is_block_ok(U61_Block *test)
{
  int i,n,x,y;
  bool ok=true;

  n=test->get_nb_items();
  for (i=0;i<n && ok;++i)
    {
      x=test->get_x()+test->get_item_x(i);
      y=test->get_y()+test->get_item_y(i);
      /*
       * the square must be in the map, or over the map,
       * if the block is brand new and didn't show himself yet
       */
      if (x>=0 && x<WIDTH && y<HEIGHT)
        {
	  /*
	   * if y>=0, there could potentially be a stable square
	   * at this location, so we check that
	   */
	  if (y>=0)
            {
	      if (squares[x][y].is_enabled()) 
                {
		  ok=false;
                }
            }
        }
      else
        {
	  ok=false;
        }
    }

  return ok;
}

/*--------------------------------------------------------------------------*/
/*
 * gets a new block, the shape of the new block is calculated with
 * Lua user defined functions
 */
void U61_Map::new_block(int shape)
{
  block=next_block;
  block.set_x(WIDTH/2);
  block_requested=false;

  next_block.reset();
  U61_Script::do_shape(this,&next_block,shape);
  next_block.center();

  next_block.corner();
  block.center();

  //cout<<"new block given\n";
}

/*--------------------------------------------------------------------------*/
/*
 * this function transfoem the moving block into a stable block,
 * ie it adds it to the current map
 */
void U61_Map::stabilize_block()
{
  U61_Event evt;
  int x,y,i,n;
  bool outside=false;

  U61_Script::land(this,&block);
  n=block.get_nb_items();
  for (i=0;i<n;++i)
    {
      x=block.get_x()+block.get_item_x(i);
      y=block.get_y()+block.get_item_y(i);
      if (x>=0 && y>=0 && x<WIDTH && y<HEIGHT)
        {
	  squares[x][y]=block.get_item(i)->square;
        }
      else
        {
	  outside=true;
        }
    }
  if (outside)
    {
      evt.code=U61_Event::LOOSE;
      put_auto_event(evt);
    }
  else
    {
      match_count=0;
      matching=true;
    }

  block.reset();
}

/*--------------------------------------------------------------------------*/
/*
 * what has to be done when the player looses
 */
void U61_Map::loose()
{
  //cout<<"The game is lost\n";
  begin();
}

/*--------------------------------------------------------------------------*/
/*
 * calls the lua function to match patterns
 * pattern match can be find a line and delete it, or find some colors
 * together, well, it can be pretty much anything...
 * it is usually quite a complex function
 * it also updates the score
 */
void U61_Map::match_pattern()
{
  if (matching)
    {
      if ((U61_Script::match_pattern(this)))
        {
	  if (!silent)
            {
	      U61_Sound::play_block_pattern();
            }
	  match_count++;
        }
      else
        {
	  if (match_count==0 && !silent)
            {
	      U61_Sound::play_block_touch();
            }
	  matching=false;
        }
    }
}

/*--------------------------------------------------------------------------*/
/*
 * calls the lua function time_callback
 * its purpose is to provide control to lua scripts over the game
 * in general. since this function is called on a regular basis
 * it's possible for instance to make the block change now & then
 */
void U61_Map::time_callback()
{
  U61_Script::time_callback(this,&block);
}

/*--------------------------------------------------------------------------*/
/*
 * Calls the update_explosion function for each square,
 * ie each explosion counter will be increased by 1
 */
void U61_Map::update_explosions()
{
  int x,y;
  bool curse_exploded=false;
  U61_Event event;

  for (y=0;y<HEIGHT;++y)
    for (x=0;x<WIDTH;++x)
      { 
	squares[x][y].update_explosion();
	if (squares[x][y].is_explosion_ended())
	  {
	    U61_Script::square_blown_up(this,x,y);
	  }
	if (squares[x][y].is_explosion_new())
	  {
            /*
	     * If we find out that the square that just blew up
             * is the special curse square, then we need to launch the
             * script associated to this curse
             */
            if (curse_state && curse_x==x && curse_y==y)
	      {
                curse_exploded=true;
	      }
	  }
      }

  if (curse_exploded)
    {
      /*
       * Now we could execute the Lua script right away, but we prefer
       * to send an event for 2 reasons:
       * 1 - the curse function can be quite long so it's useless to launch
       *     it too often
       * 2 - this way there's no difference between a curse you got because
       *     someone sent it to you and one you got by yourself matching
       *     a pattern, and I just like this idea
       */
      event.code=U61_Event::DO_CURSE;
      event.par=curse_id;
      put_auto_event(event);

      /*
       * The curse has been used so we disable it and ask for another one
       * by forcing curse_counter to curse_delay
       */
      curse_state=false;
      curse_counter=curse_delay;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * Changes the curse weapon on the map. The weapon is changed on a regular
 * basis. This is done quite randomly, but still we can garantee it will
 * be the same "random" on every machine since the algorithm is based
 * on the map time.
 */
void U61_Map::new_curse(int id)
{
  int nb_squares;
  int x,y;
  int pos,chosen_pos;
 
  //cout<<"New curse\n";

  nb_squares=0;
  for (y=0;y<HEIGHT;++y)
    for (x=0;x<WIDTH;++x)
      {
        if (squares[x][y].is_enabled() && 
	    !squares[x][y].is_exploding())
	  {
	    nb_squares++;
	  }
      }
  //cout<<"nb_squares="<<nb_squares<<"\n";
  if (nb_squares>0)
    {
      /*
       * OK, now that we know there's at least one square, we choose to
       * put the curse square on one of these squares, the random touch
       * being added by dividing the map_time by the number of squares
       * enabled, and taking the rest. It's very important not to use
       * a random-like function since we need to garantee the map
       * will behave the same on every machine.
       */
      chosen_pos=map_time%nb_squares;
      pos=0;
      for (y=0;y<HEIGHT;++y)
        for (x=0;x<WIDTH;++x)
	  {
	    if (squares[x][y].is_enabled() &&
		!squares[x][y].is_exploding())
	      {
		if (pos==chosen_pos)
		  {
		    curse_x=x;
		    curse_y=y;
		    //cout<<"Curse chosen x="<<x<<" y="<<y<<"\n";
		  }
		pos++;
	      }	
	  }
      
      curse_id=id;
      curse_state=true;
    }
}

/*--------------------------------------------------------------------------*/
/*
 * handles REQUEST_CURSE events
 */
void U61_Map::request_curse(U61_Event event)
{
  /*
   * Now we have received a request for a curse so
   * we decide to issue a real curse. It's important to call
   * put_auto_event because of time stamp issues.
   */
  event.code=U61_Event::DO_CURSE;
  event.par+=MAX_CURSE_ID;

  put_auto_event(event);
}

/*--------------------------------------------------------------------------*/
/*
 * Calls the lua function corresponding to the curse
 */
void U61_Map::do_curse(U61_Event evt)
{
  if (evt.par<(unsigned int) MAX_CURSE_ID)
    {
      //cout<<"Do own curse "<<evt<<"\n";
      U61_Script::do_curse(this,evt.par,false);
    }
  else
    {
      evt.par-=MAX_CURSE_ID;
      //cout<<"Do remote curse "<<evt<<"\n";
      U61_Script::do_curse(this,evt.par,true);
    }
}

/*-------------------------------------------------------------------------*/
/*
 * switch to the next victim
 */
void U61_Map::next_victim(U61_Event *event)
{
  target_id=U61_Global::layout.get_next_player_id(target_id,player_id);
  event->code=U61_Event::SET_VICTIM;
  event->par=target_id;
  //cout<<"Next victim "<<target_id<<"\n";
}

/*-------------------------------------------------------------------------*/
/*
 * switch to the prev victim
 */
void U61_Map::prev_victim(U61_Event *event)
{
  target_id=U61_Global::layout.get_prev_player_id(target_id,player_id);
  event->code=U61_Event::SET_VICTIM;
  event->par=target_id;
  //cout<<"Prev victim "<<target_id<<"\n";
}

/*-------------------------------------------------------------------------*/
/*
 * switch to the specified victim
 */
void U61_Map::set_victim(int id)
{
  target_id=id;
  //cout<<"Set victim "<<target_id<<"\n";
}

/*-------------------------------------------------------------------------*/
/*
 * returns the id of the current target
 */
int U61_Map::get_target_id()
{
  return target_id;
}

/*-------------------------------------------------------------------------*/
/*
 * we check that the target we have is not ourself or nobody, and send
 * an auto event if not
 */
void U61_Map::check_target()
{
  int test_id;
  U61_Event event;

  test_id=U61_Global::layout.check_player_id(target_id,player_id);
  if (test_id!=target_id)
    {
      event.code=U61_Event::SET_VICTIM;
      event.par=test_id;
      put_auto_event(event);
      //      cout<<"Correcting victim "<<test_id<<"\n";
    }  
}

