/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
package fr.gouv.culture.sdx.search.lucene.query;

import fr.gouv.culture.sdx.document.IndexableDocument;
import fr.gouv.culture.sdx.documentbase.LuceneDocumentBase;
import fr.gouv.culture.sdx.documentbase.LuceneIndexParameters;
import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.framework.FrameworkImpl;
import fr.gouv.culture.sdx.search.lucene.Field;
import fr.gouv.culture.sdx.search.lucene.FieldsDefinition;
import fr.gouv.culture.sdx.search.lucene.analysis.MetaAnalyzer;
import fr.gouv.culture.sdx.search.lucene.queryparser.QueryParser;
import fr.gouv.culture.sdx.utils.Utilities;
import fr.gouv.culture.sdx.utils.lucene.LuceneDataStore;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.Hits;
import org.apache.lucene.search.Searchable;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.RAMDirectory;

import java.io.File;
import java.io.IOException;
import java.rmi.Naming;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.util.Date;
import java.util.Locale;

/**
 * Information and services related to a LuceneIndex.
 *
 * <p>
 * From here we can get an IndexReader, a Searcher, and know a little more
 * about defined fields.
 */
public class LuceneIndex extends LuceneDataStore implements Configurable, Index {


    /**The container for available search fields. */
    private FieldsDefinition fields = null;

    /**The temporary directory for a batch of documents*/
    private RAMDirectory tempBatch = null;

    /**The MetaAnalyzer*/
    private MetaAnalyzer metaAnalyzer = null;

    /**String representation of the attribute named "class". */
    private final String ATTRIBUTE_CLASS = "class";

    /**The specified query parser*/
    private Class queryParserClass = null;

    /**default query parser class name*/
    private final String DEFAULT_QUERYPARSER = "fr.gouv.culture.sdx.search.lucene.queryparser.DefaultQueryParser";

    /**String representation of the element name "queryParser". */
    private final String ELEMENT_NAME_QUERYPARSER = "queryParser";

    /**The rmi registry name for the remote index*/
    private String remoteIndexName = "";

    /** for remote objectsour rmi port*/
    private int rmiPort;

    /** for remote objectsour rmi port*/
    private String rmiHost = "";

    protected Date creationTimestamp = null;
    protected Date lastModificationTimestamp = null;



    /**Host port combination for rmi registration and serving*/

    /** int representation of indexing action for adding a document. */
    public final static int ACTION_ADD_DOCUMENT = 1;

    /**Defaults for IndexWriter parameters; these are based on lucene defaults*/
    public static final int DEFAULT_MAX_FIELD_LENGTH = 10000;
    /**Defaults for IndexWriter parameters; these are based on lucene defaults*/
    public static final int DEFAULT_MAX_MERGE_DOCS = Integer.MAX_VALUE;
    /**Defaults for IndexWriter parameters; these are based on lucene defaults*/
    public static final int DEFAULT_MERGE_FACTOR = 10;

    /**Lucene IndexWriter parameter*/
    private int maxFieldLength = DEFAULT_MAX_FIELD_LENGTH; //Defaulted
    /**Lucene IndexWriter parameter*/
    private int maxMergeDocs = DEFAULT_MAX_MERGE_DOCS; //Defaulted
    /**Lucene IndexWriter parameter*/
    private int mergeFactor = DEFAULT_MERGE_FACTOR; //Defaulted

    /**String representation of the attribute name "maxFieldLength". */
    private final String ATTRIBUTE_MAX_FIELD_LENGTH = "maxFieldLength";

    /**
     * Builds an index with fields definition and a path to the index files.
     *
     * @param   dir     A directory where the index is kept.
     * TODOJavadoc
     */
    public LuceneIndex(File dir, String host, Integer port, String appId, String dbId) throws SDXException {
        super(dir);
        if (Utilities.checkString(host))
            this.rmiHost = host;

        if (port != null)
            this.rmiPort = port.intValue();
        else
            this.rmiPort = FrameworkImpl.SDX_DEFAULT_RMI_PORT;

        if (Utilities.checkString(appId) && Utilities.checkString(dbId))
            this.remoteIndexName = Utilities.buildRmiName(this.rmiHost, this.rmiPort, appId, dbId);
    }

    public void configure(Configuration configuration) throws ConfigurationException {


        configureQueryParser(configuration);

        //configuring the names for the remote objects
        boolean remoteConf = configuration.getAttributeAsBoolean(LuceneDocumentBase.DBELEM_ATTRIBUTE_REMOTE_ACCESS, false);
        /*TODO:FIXME-if we don't have a remote index configuration we nullify the name so that
        calling methods will not have access to a remote object name*/
        if (!remoteConf) this.remoteIndexName = null;

        if (remoteConf) {
            try {
                bindRemoteObjectToRegistry(this.remoteIndexName, new RemoteIndex(this));
            } catch (SDXException sdxE) {
                throw new ConfigurationException(sdxE.getMessage(), sdxE);
            } catch (RemoteException e) {
                throw new ConfigurationException(e.getMessage(), e);
            }
        }

        //configuring the maxFieldLength for indexation, default is 10000 based on lucene defaults
        this.maxFieldLength = configuration.getAttributeAsInteger(ATTRIBUTE_MAX_FIELD_LENGTH, DEFAULT_MAX_FIELD_LENGTH);

    }

    private void configureQueryParser(Configuration configuration) throws ConfigurationException {
        //configuring the queryparser
        Configuration qpConf = configuration.getChild(ELEMENT_NAME_QUERYPARSER, false);
        String qpClassName = DEFAULT_QUERYPARSER;
        if (qpConf != null)
            qpClassName = qpConf.getAttribute(ATTRIBUTE_CLASS, DEFAULT_QUERYPARSER);
        try {
            this.queryParserClass = Class.forName(qpClassName);
        } catch (ClassNotFoundException e) {
            SDXException sdxE = new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_QUERY_PARSER, null, e);
            throw new ConfigurationException(sdxE.getMessage(), sdxE);
        }

    }


    private void bindRemoteObjectToRegistry(String name, Remote obj) throws SDXException {

        try {
            Naming.rebind(name, obj);
        } catch (IOException e) {
            String[] args = new String[1];
            args[0] = name;
            throw new SDXException(logger, SDXExceptionCode.ERROR_BIND_REMOTE_INDEX, args, e);
        }

    }

    /**
     * Initializes the Lucene database.
     *
     * <p>
     * It the index exists, nothing is done here. If it is doesn't
     * exist, it will be created.
     */
    public void init() throws SDXException {
        super.init(false);
        writeCreationTimestampFile();
        writeLastModificationTimestampFile(false);
        //ensuring we have a metaAnalyzer for fields
        if (this.metaAnalyzer == null)
            throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_INDEX_INIT, null, null);
    }

    protected void writeCreationTimestampFile() {
        try {
            File createFile = new File(super.fsdFile.getParentFile(), "creation-timestamp");
            if (!createFile.exists())
                createFile.createNewFile();

            this.creationTimestamp = new java.util.Date(createFile.lastModified());
        } catch (IOException e) {
            Utilities.logError(logger, e.getMessage(), e);
        }

    }

    public void writeLastModificationTimestampFile(boolean create) {
        try {
            File lastModFile = new File(super.fsdFile.getParentFile(), "last-modification-timestamp");
            if (create) {
                lastModFile.delete();
                lastModFile.createNewFile();
            } else if (!lastModFile.exists())
                lastModFile.createNewFile();
            this.lastModificationTimestamp = new java.util.Date(lastModFile.lastModified());
        } catch (IOException e) {
            Utilities.logError(logger, e.getMessage(), e);
        }
    }

    /** Returns the type of a field.
     *
     * @param name  The name of the field for which the type is desired.
     * @return      The int field code (see Field doc)
     */
    public int getFieldType(String name) {
        return fields.getFieldType(name);
    }

    /** Returns the locale for a field.
     *
     * @param name The name of the field for which the Locale is desired.
     */
    public Locale getLocale(String name) {
        return fields.getLocale(name);
    }

    /**
     * Returns the default field for this index.
     */
    public Field getDefaultField() {
        return fields.getDefaultField();
    }


    /** Returns a field given a name.
     *
     * @param name The name of the field for which the Field is desired.
     */
    public Field getField(String name) {
        return fields.getField(name);
    }

    /** Returns a the FieldsDefinition for this index (basically a Hashtable of all the Fields)
     *
     */
    //TODO?:is this still necessary, as it exists both in LuceneIndex and MetaAnalyzer?-rbp
    public FieldsDefinition getFieldsDefinition() {
        return fields;
    }


    /** Get's an array of all the field names for the search index
     *
     * @return
     */
    /*TODO?: discuss the usefulness of this method and the proper information needed as requested by fred?
	public String[] getFieldNames(){
		if (fields != null){
			fields.
			fields
		}

	}
	 */

    /** Gets a searcher.
     *
     * @return      A lucene Searcher.
     */
    public Searchable getSearcher() {
        return super.getSearcher();
    }

    /**Gets the MetaAnalyzer*/
    public MetaAnalyzer getMetaAnalyzer() {
        return metaAnalyzer;

    }

    /**Sets the MetaAnalyzer
     *
     * @param mAnalyzer A MetaAnalyzer object containing a FieldsDefinition object for this index.
     * @throws SDXException
     */
    public void setMetaAnalyzer(MetaAnalyzer mAnalyzer) throws SDXException {
        //setting the metaAnalyzer
        this.metaAnalyzer = mAnalyzer;
        //assigning the class field
        this.fields = this.metaAnalyzer.getFieldsDefinition();
        super.analyzer = this.metaAnalyzer;

    }

    /**
     * Stores a Lucene document within the database.
     *
     * @param   ldoc     The Lucene document to store.
     * @param   batchIndex     Indicates wheter a tempBatch index is taking place or not. Useful for efficiency of index optimizations
     */
    public synchronized void writeDocument(org.apache.lucene.document.Document ldoc, boolean batchIndex) throws SDXException {
        IndexWriter w = null;
        try {
            //if we have batchIndex and don't already have one, we create the temporary directory for the tempBatch
            if (batchIndex && this.tempBatch == null) {
                this.tempBatch = new RAMDirectory();
                //initializing the directory
                IndexWriter initTempBatch = new IndexWriter(tempBatch, getMetaAnalyzer(), true);
                initTempBatch.close();
            }

            //if we still have a tempBatch  we add to it
            if (this.tempBatch != null) {
                //getting a writer for the document tempBatch directory
                w = this.getWriter(this.tempBatch);
                if (w != null) {
                    //adding the document
                    w.addDocument(ldoc);
                    w.close();
                    w = null;
                }
            }

        } catch (IOException e) {
            String[] args = new String[2];
            args[0] = super.getIndexPath();
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_WRITE, args, e);
        } finally {
            try {
                if (w != null) w.close();
            } catch (IOException e) {
                String[] args = new String[2];
                args[0] = super.getIndexPath();
                args[1] = e.getMessage();
                throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_WRITE, args, e);
            }
        }


        if (!batchIndex && this.tempBatch == null) {
            //we have a single document to dadd
            //initializing an index writer for our main index directory
            super.write(ldoc);
            writeLastModificationTimestampFile(true);
        }


    }


    /**
     * Deletes a document from the index.
     *
     * @param	docId	        The document's id.
     */
    public synchronized void deleteDocument(String docId) throws SDXException {
        super.delete(docId);
    }

    public QueryParser getQueryParser() throws IOException, SDXException {
        try {
            if (this.queryParserClass == null)
                return null;
            else
                return (QueryParser) queryParserClass.newInstance();
        } catch (InstantiationException e) {
            throw new SDXException(logger, SDXExceptionCode.ERROR_GET_QUERY_PARSER, null, e);
        } catch (IllegalAccessException e) {
            throw new SDXException(logger, SDXExceptionCode.ERROR_GET_QUERY_PARSER, null, e);
        }
    }

    public String getRemoteIndexName() {
        return remoteIndexName;
    }

    /**Merge's any batch in memory, if no batch then it optimizes the lucene index*/
    public synchronized void mergeBatch() throws SDXException {

        try {
            if (this.tempBatch != null) {
                //resetting the directory for batches
                //creating the new file container for the original directory
                File tmpDir = new File(this.fsdFile.getParent(), "tmp");
                tmpDir.mkdirs();
                File tmpOldDir = File.createTempFile("old", "Index", tmpDir);
                tmpOldDir.delete();//deleting the temp file
                //a directory for the new index
                File tmpNewDir = File.createTempFile("new", "Index", tmpDir);
                tmpNewDir.delete();//deleting the temp file
                /*we do this in the merge:
                 -write a new index with the two lucene indicies FSDirectory and RAMDirectory in memory and on disk to a new directory
                 -rename the old index (directory)
                 -rename of new index to original name
                 -delete renamed original
                 */
                Directory[] dirs = new Directory[2];
                //adding the original
                dirs[0] = this.fsd;
                //adding the batch
                dirs[1] = this.tempBatch;
                //verifying the directory
                Utilities.checkDirectory(tmpNewDir.getAbsolutePath(), this.logger);
                //creating a new temp lucene directory for the batch merge
                FSDirectory tempNewLuceneDir = FSDirectory.getDirectory(tmpNewDir, true);
                //getting the writer for the new temp dir
                IndexWriter w = new IndexWriter(tempNewLuceneDir, getMetaAnalyzer(), true);
                //adding the two indicies to the new directory
                w.addIndexes(dirs);
                //freeing resources after the merge
                w.close();
                //freeing resources so we can rename the directories
                this.freeResources();
                //renaming the old index
                this.fsdFile.renameTo(tmpOldDir);//TODOException if fails
                //renaming the new index to replace the original index
                tmpNewDir.renameTo(this.fsdFile);//TODOException if fails
                //deleting the tmpOld, containing the preBatch index
                File[] files = tmpOldDir.listFiles();
                for (int i = 0; files != null && i < files.length; i++)
                    files[i].delete();
                tmpOldDir.delete();//TODOException if fails
                tmpOldDir.getParentFile().delete();//TODOException  if fails
            } else
                this.optimize();
        } catch (IOException e) {
            String[] args = new String[2];
            args[0] = super.getIndexPath();
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_WRITE, args, e);
        } finally {
            /*so if something goes wrong the batch will have to be reindexed because
            destroy the batch in memory*/
            this.tempBatch = null;
            //TODO : implement a kind of locking for the searcher ? -pb
            super.init(false);
            writeLastModificationTimestampFile(true);
        }
    }

    /**Free's the resources associated with this index
     * USE WITH CARE!
     *
     * @throws IOException  Lucene IOExceptions
     */
    protected void freeResources() throws IOException {
        super.freeResources();
        if (this.tempBatch != null)
            this.tempBatch.close();
    }

    protected synchronized IndexWriter getWriter(Directory directory) throws IOException {
        IndexWriter w = super.getWriter(directory);
        //setting parameters to the index writer if they differ from the defaults
        if (this.maxFieldLength != DEFAULT_MAX_FIELD_LENGTH)
            w.maxFieldLength = this.maxFieldLength;
        if (this.maxMergeDocs != DEFAULT_MAX_MERGE_DOCS)
            w.maxMergeDocs = this.maxMergeDocs;
        if (this.mergeFactor != DEFAULT_MERGE_FACTOR)
            w.mergeFactor = this.mergeFactor;
        return w;
    }

    /**Set's parameters for this lucene index
     *
     *@param params The parameters relevent to this lucene index.
     */
    public void setParameters(LuceneIndexParameters params) {
        //TODO: should we reset these after indexation or should we leave them set until the server is restarted?
        this.maxMergeDocs = params.getMaxMergeDocs();
        this.mergeFactor = params.getMergeFactor();
    }

    public synchronized void optimize() throws SDXException {
        super.optimize();
    }

    public synchronized IndexReader getReader() throws SDXException {
        return super.getReader();
    }

    public String getFieldValue(IndexableDocument doc, String fieldName) throws SDXException {
        Utilities.checkDocument(logger, doc);
        if (!Utilities.checkString(fieldName)) return null;
        Hits h = search(new TermQuery(new Term(ID_FIELD, doc.getId())));
        // The return it if it exists
        if (h.length() == 1) {
            try {
                org.apache.lucene.document.Field field = h.doc(0).getField(fieldName);
                if (field != null)
                    return field.stringValue();
                else
                    return null;
            } catch (IOException e) {
                String[] args = new String[2];
                args[0] = fsd.toString();
                args[1] = e.getMessage();
                throw new SDXException(logger, SDXExceptionCode.ERROR_LUCENE_RETRIEVE_DOCUMENT, args, e);
            }
        } else
            return null;

    }

    public Date getCreationDate() {
        return creationTimestamp;
    }

    public Date getLastModificationDate() {
        return lastModificationTimestamp;
    }

}
