/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
package fr.gouv.culture.sdx.pipeline;

import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.utils.Utilities;
import org.apache.avalon.framework.component.ComponentException;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.parameters.ParameterException;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.xslt.XSLTProcessor;
import org.apache.cocoon.environment.ModifiableSource;
import org.apache.cocoon.environment.URLFactorySourceResolver;
import org.apache.cocoon.xml.XMLConsumer;
import org.xml.sax.SAXException;

import javax.xml.transform.sax.SAXResult;
import javax.xml.transform.sax.TransformerHandler;
import java.io.File;
import java.io.IOException;

/**A transformation using an XSLTProcessorImpl from Cocoon
 *
 * Basically a wrapper to provide a certain configuration option
 * (stylesheet specification) and forwards events to a consumer
 * after the transformation has been executed.
 *
 */
public class XSLTTransformation extends AbstractTransformation {

    /**XSLT processor to use in the transformation. */
    private XSLTProcessor xsltProcessor;

    /**InputSource for the transformation stylesheet. */
    private ModifiableSource inputSource;

    /**TransformerHandler built from the XSLTProcessor and the stylesheet. */
    private TransformerHandler transformerHandler;

    /**Creates the transformation
     *
     * If logging is desired the logger should be set after creation.
     *
     * This transformation should be composed, properties set, and then configured.
     *
     * @see #enableLogging
     * @see #compose
     * @see #setProperties
     * @see #configure
     */
    public XSLTTransformation() {
    }

    /**Builds this object
     *
     * In addition to the parameters needed in the base configuration handled
     * by the parent class, the "src" parameter is required and can be absolute
     * or relative to the directory containing the application.xconf file
     * or relative to the SDX installation.
     *
     * @param configuration     An Configuration object from the Pipeline
     *
     *<p> Sample configuration entry:
     * <p>&lt;sdx:transformation src = "path to stylesheet, can be absolute or relative to the directory containing this file" sdx:id = "step3" sdx:type = "xslt" keep = "true"/>
     */
    public void configure(Configuration configuration) throws ConfigurationException {

        try {

            super.configure(configuration);
            //getting the "src" attribute from the configuration
            String src = configuration.getAttribute(Transformation.ATTRIBUTE_SRC);
            //verifying we have a good attribute
            Utilities.checkConfAttributeValue(Transformation.ATTRIBUTE_SRC, src, configuration.getLocation());
            File srcFile = null;
            try {
                srcFile = Utilities.resolveFile(null, configuration.getLocation(), props, src, false);
            } catch (SDXException e) {
                throw new ConfigurationException(e.getMessage(), e);
            }
            src = srcFile.toURL().toExternalForm();

            //TODONOw?: is there a problem with keeping this component without releasing it?-rbp
            //getting an xslt processor from cocoon's component manager
            if (this.manager == null) {
                SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_COMPONENT_MANAGER_NULL, null, null);
                throw new ConfigurationException(sdxE.getMessage(), sdxE);
            }
            XSLTProcessor xsltProcessor = (XSLTProcessor) manager.lookup(XSLTProcessor.ROLE);
            //creating a new url resolver for the transformation
            URLFactorySourceResolver resolver = new URLFactorySourceResolver(manager);

            //TODO?:what is the lifecycle of this xsltProcessor, does it actually get re-acquired often for large indexation batches?-rbp
            //setting the class field
            this.xsltProcessor = xsltProcessor;
            //setting the processor's source resolver
            this.xsltProcessor.setSourceResolver(resolver);
            //TODO:is this inputSource cached by cocoon?
            //resolving the stylesheet and setting the class field for the input source
            this.inputSource = (ModifiableSource) resolver.resolve(src);
            // for now, hard-coded to 1 second.
//            this.styleSheetSource = new DelayedRefreshSourceWrapper(resolver.resolve(src), 1000L);

        } catch (ProcessingException e) {
            Utilities.logException(logger, e);
            throw new ConfigurationException(e.getMessage(), e.fillInStackTrace());
        } catch (SAXException e) {
            Utilities.logException(logger, e);
            throw new ConfigurationException(e.getMessage(), e.fillInStackTrace());
        } catch (IOException e) {
            Utilities.logException(logger, e);
            throw new ConfigurationException(e.getMessage(), e.fillInStackTrace());
        } catch (ComponentException e) {
            Utilities.logException(logger, e);
            throw new ConfigurationException(e.getMessage(), e.fillInStackTrace());
        }
    }

    /**Sets the XMLConsumer for this transformation.
     *
     * @param consumer  The consumer for the SAX events of this transformation
     */
    public void setConsumer(XMLConsumer consumer) {
        //verifying the consumer
        try {
            Utilities.checkXmlConsumer(logger, consumer);
        } catch (SDXException e) {
            /*doing nothing here as the exception should have been logged,
            as we are governed by superClasses for the exceceptions signatures*/
        }
//        verifySource();
//refreshing the input source so that any changes in the stylesheet will be reflected
        this.inputSource.refresh();


        try {
            //getting a transform handler based from the processor and stylesheet input source
            //is this necessary every time we set consumer, as the inputSource is cached?-rbp
            transformerHandler = xsltProcessor.getTransformerHandler(inputSource);
            //applying parameters
            if (this.getParameters() != null) {
                //getting the transformer
                javax.xml.transform.Transformer trans = transformerHandler.getTransformer();
                //clearing any existing parameters
                //TODO?:should this be done here, or do we risk destroying something we may need?-rbp
                trans.clearParameters();
                //getting the params
                Parameters params = this.getParameters();
                //getting the param names
                String[] paramNames = params.getNames();
                //setting the params
                for (int i = 0; i < paramNames.length; i++) {
                    try {
                        trans.setParameter(paramNames[i], (String) params.getParameter(paramNames[i]));
                    } catch (ParameterException e) {
                        Utilities.logException(logger, e);
                    }
                }
            }
        } catch (ProcessingException e) {
            Utilities.logException(logger, e);
        }
        //setting the content handler
        super.setContentHandler(transformerHandler);
        //setting the lexical handler
        super.setLexicalHandler(transformerHandler);

        if (transformerHandler instanceof LogEnabled) {
            //TODO?:does transformhandler implement logEnabled?-rbp
            ((LogEnabled) transformerHandler).enableLogging(this.logger);
        }

        // According to TrAX specs, all TransformerHandlers are LexicalHandlers
        //creating a result object for this tansformation
        SAXResult result = new SAXResult(consumer);
        //setting consumer of the result's lexical handler
        result.setLexicalHandler(consumer);
        //passing the  result object to the transformHandler
        if (this.transformerHandler != null)//TODOException: send a message here as we could not get an input source
            transformerHandler.setResult(result);

        //setting the consumer of the transformation
        super.xmlConsumer = consumer;
    }

    /*
     public void verifySource(){

     //if the modification dates are no longer equal, we refresh the style sheet
     if (this.sourceModificationDate != this.inputSource.getLastModified()){
             this.sourceModificationDate = this.inputSource.getLastModified();
             this.inputSource.refresh();
         }

     }
     */

    public void startDocument()
            throws SAXException {
        this.setConsumer(super.xmlConsumer);
        super.startDocument();
    }
}
