/*
 * MicroEngine - A microkernel inspired game engine
 * Copyright (C) 2008 "Slinger" ("Sould Slinger" @ gorcx.net/forum),
 * _unless where otherwise stated_
 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.

 */

/*
 * Functions for interpreting, preprocessing, and executing scripts
 */

#include "scripts.h"

static int get_next_line ()
{
	do {
		argument[0] = '\0'; //in case of no argument
		if (fscanf(file, "%s%[^\n]", command, argument) == EOF)
			return -1;
	} while (command[0]=='\n'||command[0]=='#');
	return 0;
}

static int alloc_script ()
{
	printf("  > Allocating script database storage...");
	int commands = 0;
	int blocks = 0;
	while (get_next_line()==0)
	{
		if (command[0]=='}')
		{
			commands--;
			blocks++;
		}
		else
			commands++;
	}
	rewind(file);

	script[script_num].command = calloc(commands, sizeof(struct command));
	script[script_num].block = calloc (blocks, sizeof (struct block));

	if (script[script_num].command == NULL
			|| script[script_num].block == NULL)
	{
		printf("Allocation failure!\n\n");
		return -1;
	}

	printf("Done!\n");
	return 0;
}

static int command_special ()
{
	if (argument[1] == '{')
	{
		int i;
		for (i=0; i< block_count; i++)
		{
			if (strcmp (block_name[i], command) == 0)
			{
				active_block = i;
				script[script_num].block[active_block].start =
					command_num;
				return 0;
			}
		}
		printf("didn't recognize blockname \"%s\"", command);
		return -1;
	}
	if (command[0] == '}')
	{
		script[script_num].block[active_block].stop = command_num;
		return 0;
	}

	return 1; //not special command
}

static int preprocess_script()
{
	printf("  > Preprocessing...");
	char preproc_name [108];
	int (* preproc_ptr) (char **, char *);

	command_num=0;
        while (get_next_line()==0)
	{
		if (command_special () == 1)
		{
		strcpy (preproc_name, command);
		strcat (preproc_name, "_preproc");

		script[script_num].command[command_num].func
				=plugin_command_lookup(command);
		if (script[script_num].command == NULL)
		{
			printf("can't find command \"%s\"!\n", command);
			return -1;
		}

		preproc_ptr = plugin_command_lookup (preproc_name);
		if (preproc_ptr == NULL)
		{
			printf("can't find preproc: \"%s\"!\n",preproc_name);
			return -1;
		}
		if (preproc_ptr (&script[script_num].command[command_num].args
					,argument))
		{
			printf("preprocessing using \"%s\" failed...\n\n",
					preproc_name);
			return -1;
		}
		command_num++;
		}
	}
	printf("Done!\n");
	return 0;
}

//hopefully a fast execution of "database"
void run_script (int number, int runlevel)
{
	register int i = script[number].block[runlevel].start;
	while (i < script[number].block[runlevel].stop)
	{
		script[number].command[i].func(script[number].command[i].args);
		i++;
	}
}

int load_script (int number, char * PATH)
{
	printf("> Loading script: \"%s\"...\n", PATH);
	script_num = number;
	if ((file = fopen (PATH, "r")) == NULL)
	{
		printf("couldn't open file\n\n");
		return -1;
	}
	if (alloc_script ())
		return -1;
	if (preprocess_script())
		return -1;
	fclose (file);
	run_script (number, load);
	printf("> Done!\n\n");
	return 0;
}

void unload_script (int number)
{
	printf("> Unloading script %i...\n", number);
	run_script (number, unload);
	free (script[number].command);
	free (script[number].block);
	printf("> Done!\n\n");
}

