/*
 * sender/sender.c - simple program for submitting /proc/cpuinfo
 * Copyright (C) 2006 Thomas Cort <linuxgeek@gmail.com>
 *
 * Based on parts of neon and neon examples by Joe Otron
 * Copyright (C) 1999-2005, Joe Orton <joe@manyfish.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <neon/ne_session.h>
#include <neon/ne_string.h>
#include <neon/ne_request.h>
#include <neon/ne_uri.h>
#include <neon/ne_basic.h>
#include <neon/ne_utils.h>

static int dispatch_to_fd(ne_request *req, int fd, const char *range) {
	ne_session *const sess = ne_get_session(req);
	const ne_status *const st = ne_get_status(req);
	int ret;

	do {
		const char *value;

		ret = ne_begin_request(req);
		if (ret != NE_OK) {
			break;
		}

		value = ne_get_response_header(req, "Content-Range");

		/*
		 * For a 206 response, check that a Content-Range header is
		 * given which matches the Range request header.
		 */
		if (range && st->code == 206 && (value == NULL || strncmp(value, "bytes ", 6) != 0 || strcmp(range + 6, value + 6))) {
			ne_set_error(sess, "Response did not include requested range");
			return NE_ERROR;
		}

		if ((range && st->code == 206) || (!range && st->klass == 2)) {
			ret = ne_read_response_to_fd(req, fd);
		} else {
			ret = ne_discard_response(req);
		}

		if (ret == NE_OK) {
			ret = ne_end_request(req);
		}
    } while (ret == NE_RETRY);

    return ret;
}

int read_cpuinfo(char *buf, int len) {
	int fd = open("/proc/cpuinfo",O_RDONLY), rc = 1;
	unsigned int i;

	if (fd < 0)
		return fd;

	for (i = 0; i < len && rc == 1; i++)
		rc = read(fd,(buf+i),1);

	close(fd);
	return rc;
}

int main(int argc, char **argv) {
	ne_session *sess;
	ne_uri uri = {0};
	ne_request *req;
	char *buf, *buffer = (char *) malloc(16384);
	int ret, fd = 1;

	if (read_cpuinfo(buffer,16384) < 0) {
		free(buf);
		free(buffer);
		printf("Error reading /proc/cpuinfo\n");
		return -1;
	}

	buf = ne_path_escape(buffer);
	snprintf(buffer,16383,"cpuinfo=%s",buf);
	free(buf);

	if (ne_uri_parse("http://mediumbagel.org/submit.php", &uri) || uri.host == NULL || uri.path == NULL) {
		printf("Could not parse URL\n");
		free(buffer);
		return -1;
	}

	/* Set defaults. */
	if (uri.scheme == NULL) {
		uri.scheme = "http";
	}

	if (uri.port == 0) {
		uri.port = ne_uri_defaultport(uri.scheme);
	}

	/* Initialize socket libraries */
	if (ne_sock_init()) {
		printf("Failed to initialize socket libraries.\n");
		free(buffer);
		return -1;
	}

	/* Create session. */
	sess = ne_session_create(uri.scheme, uri.host, uri.port);
	req = ne_request_create(sess, "POST", uri.path);

	ne_add_request_header(req, "Content-type",  "application/x-www-form-urlencoded");
	ne_set_request_body_buffer(req, buffer, strlen(buffer));

	ret = dispatch_to_fd(req, fd, NULL);

	if (ret == NE_OK && ne_get_status(req)->klass != 2) {
		ret = NE_ERROR;
	}

	if (ret != NE_OK) {
		fprintf(stderr, "Failed: %s", ne_get_error(sess));
	}

	ne_request_destroy(req);
	ne_session_destroy(sess);
	free(buffer);
	printf("\n");

	return ret;
}
