/*
 * cputype.c - determine cpu type from cpuinfo files
 *
 * Copyright (C) 2006  Thomas Cort <linuxgeek@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <stdlib.h>
#include <sys/types.h>
#include <regex.h>

#include "proccpuinfo.h"

char *__proccpuinfo_get_value(int target_key);

int __proccpuinfo_match(char *string, char *regex) {
	regex_t preg;
	int rc;

	if (string == NULL && regex == NULL)
		return 1;
	else if (string == NULL || regex == NULL)
		return 0;

	if (regcomp(&preg, regex, REG_EXTENDED))
		return 0;

	rc = !regexec(&preg, string, 0, NULL, 0);

	regfree(&preg);
	return rc;
}

char *__proccpuinfo_cpytypes[19][7] = {
	/* i386 */ {
		"unknown","","","","","",""
	},

	/* x86_64 */ {
		"unknown","","","","","",""
	},

	/* ia64 */ {
		"unknown","","","","","",""
	},

	/*alpha*/ {
		"unknown","21064","21164","21164a","21164pc","21264","21264a"
	},

	/* sparc */ {
		"unknown","","","","","",""
	},

	/* hppa */ {
		"unknown","700","7100","7100LC","7200","7300","8000"
	},

	/* mips */ {
		"unknown","","","","","",""
	},

	/* powerpc */ {
		"unknown","","","","","",""
	},

	/* powerpc64 */ {
		"unknown","","","","","",""
	},

	/* arm */ {
		"unknown","","","","","",""
	},

	/* s390 */ {
		"unknown","","","","","",""
	},

	/* s390x */ {
		"unknown","","","","","",""
	},

	/* sh */ {
		"unknown","1","2","3","4","2e","3e"
	},

	/* m68k */ {
		"unknown","68000","68020","68030","68040","68060",""
	},

	/* cris */ {
		"unknown","v3","v10","v8","","",""
	},

	/* frv */ {
		"unknown","simple", "tomcat", "fr500", "fr400", "fr300", "frv"
	},

	/* vax */ {
		"unknown","","","","","",""
	},

	/* m32r */ {
		"unknown","","","","","",""
	},

	/* unknown */ {
		"unknown","","","","","",""
	}

};

struct {
	char *regex;
	unsigned int cputype;
	unsigned int field;
} __proccpuinfo_cputype_regex[19][16] = {
	/* i386 */ {
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0}
	},

	/* x86_64 */ {
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0}
	},

	/* ia64 */ {
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0}
	},

	/* alpha */ {
		{"EV4"   ,1,19}, {"EV45"  ,1,19}, {"LCA4"  ,1,19}, {"LCA45" ,1,19},
		{"EV5"   ,2,19}, {"EV56"  ,3,19}, {"PCA56" ,4,19}, {"PCA57" ,4,19},
		{"EV6"   ,5,19}, {"EV67"  ,6,19}, {"EV68AL",6,19}, {"EV68CB",6,19},
		{"EV68CX",6,19}, {"EV7"   ,6,19}, {"EV7z"  ,6,19}, {"EV79"  ,6,19}
	},

	/* sparc */ {
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0}
	},

	/* hppa */ {
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0}
	},

	/* mips */ {
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0}
	},

	/* powerpc */ {
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0}
	},

	/* powerpc64 */ {
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0}
	},

	/* arm */ {
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0},
		{"xxx",0,0}, {"xxx",0,0},{"xxx",0,0}, {"xxx",0,0}
	},

	/* s390 */ {
		/* s390 has no cpu types used by gcc */
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0}
	},

	/* s390x */ {
		/* s390x has no cpu types used by gcc */
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0}
	},

	/* sh */ {
		{"sh1", 1,18},{"sh2", 2,18},{"sh3" ,3,18},{"sh4", 4,18},
		{"SH1", 1,18},{"SH2", 2,18},{"SH3" ,3,18},{"SH4", 4,18},
		{"SH-1",1,18},{"SH-2",2,18},{"SH-3",3,18},{"SH-4",4,18},
		{"sh5", 4,18},{"SH5", 4,18},{"SH-5",4,18},{"",0,0}
	},

	/* m68k */ {
		{"68000",1,5},{"68020",2,5},  {"68008",1,5},  {"68302",1,5},
		{"68306",1,5},{"68307",1,5},  {"68322",1,5},  {"68328",1,5},
		{"68356",1,5},{"68EZ328",1,5},{"68VZ328",1,5},{"68030",3,5},
		{"68040",4,5},{"68060",5,5},  {"",0,0},       {"",0,0}
	},

	/* cris */ {
		{"ETRAX 4",1,19},{"ETRAX 100LX",2,19},{"ETRAX 100",3,19},{"",0,0},
		{"",0,0},        {"xxx",0,0},         {"",0,0},          {"",0,0},
		{"",0,0},        {"xxx",0,0},         {"",0,0},          {"",0,0},
		{"",0,0},        {"xxx",0,0},         {"",0,0},          {"",0,0}
	},

	/* frv */ {
		{"frv",   6,45},{"fr300", 5,45},{"fr400",4,45},{"fr500",3,0},
		{"simple",1,45},{"tomcat",2,45},{"",0,0},      {"",0,0},
		{"",0,0},       {"",0,0},       {"",0,0},      {"",0,0},
		{"",0,0},       {"",0,0},       {"",0,0},      {"",0,0}
	},

	/* vax */ {
		/* vax has no cpu types used by gcc */
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0}
	},

	/* m32r */ {
		/* m32r has no cpu types used by gcc */
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0}
	},

	/* unknown */ {
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0},
		{"",0,0}, {"",0,0},{"",0,0}, {"",0,0}
	}
};

#define NUM_CPUTYPE_REGEX (16)

char *__proccpuinfo_get_cputype() {
	char *s;
	int i;

	for (i = 0; i < NUM_CPUTYPE_REGEX; i++) {
		s = __proccpuinfo_get_value(__proccpuinfo_cputype_regex[proccpuinfo_get_arch()][i].field);
		if (__proccpuinfo_match(s,__proccpuinfo_cputype_regex[proccpuinfo_get_arch()][i].regex)) {
			free(s);
			return __proccpuinfo_cpytypes[proccpuinfo_get_arch()][__proccpuinfo_cputype_regex[proccpuinfo_get_arch()][i].cputype];
		}
		free(s);
	}

	return 0;
}
