/*
 * pdsneuron.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <time.h>
#include <pds/pdsfuncs.h>
#include <pds/pdsninput.h>
#include <pds/pdsneuron.h>



/** \fn PdsNeuron *pds_neuron_new(PdsSnNatural Nel)
 *  \brief Crea una neurona de tipo PdsNeuron de Nel entradas.
 *
 *  Los valores por defecto son :
 *  \f[ U=1.0 \f]
 *  \f[ \sigma=1.0 \f]
 *  \f[ func(x)=tanh\left(\frac{x}{2}\right) \f]
 *  \param[in] Nel Es el número de entradas de la neurona.
 *  \return Un puntero al vector de tipo PdsNeuron.
 *  \ingroup PdsNeuronGroup
 */
PdsNeuron *pds_neuron_new(PdsSnNatural Nel)
{
	PdsNeuron *Neuron=NULL;
	PdsSnNatural i;

	if(Nel<=0)		return NULL;
	
	Neuron=(PdsNeuron *)calloc(1,sizeof(PdsNeuron));
	if(Neuron==NULL)	return NULL;

	Neuron->Nel=Nel;
	Neuron->X=(PdsSnReal **)calloc(Neuron->Nel,sizeof(PdsSnReal*));
	if(Neuron->X==NULL) 
	{
		free(Neuron);
		return NULL;
	}
	for(i=0;i<Neuron->Nel;i++)	Neuron->X[i]=NULL;

	Neuron->W=(PdsSnReal *)calloc(Neuron->Nel,sizeof(PdsSnReal));
	if(Neuron->W==NULL) 
	{
		free(Neuron);
		return NULL;
	}

	for(i=0;i<Neuron->Nel;i++)	Neuron->W[i]=0.0;
	
	Neuron->U=1.0;
	Neuron->Sigma=1.0;
	Neuron->Theta=0.0;
    Neuron->func=pds_func;
    Neuron->dfunc=pds_dfunc;

	Neuron->F=0.0;
	Neuron->Y[0]=0.0;
	Neuron->Y[1]=0.0;

	return Neuron;
}

/** \fn PdsNeuron *pds_neuron_new_with_params(PdsSnNatural Nel,PdsSnReal W,PdsSnReal U,PdsSnReal Sigma,double (*func)(double),double (*dfunc)(double));
 *  \brief Crea una neurona de tipo PdsNeuron de Nel entradas.
 *
 *  \param[in] Nel Es el número de entradas de la neurona.
 *  \param[in] W Peso inicial de todas las entradas
 *  \param[in] U Umbral de disparo.
 *  \param[in] Sigma Factor de escala de la función de activación.
 *  \param[in] func Función de activación.
 *  \param[in] dfunc Derivada de la función de activación.
 *  \return Un puntero al vector de tipo PdsNeuron.
 *  \ingroup PdsNeuronGroup
 */
PdsNeuron *pds_neuron_new_with_params(PdsSnNatural Nel,PdsSnReal W,PdsSnReal U,PdsSnReal Sigma,double (*func)(double),double (*dfunc)(double))
{
	PdsNeuron *Neuron=NULL;
	PdsSnNatural i;

	if(Nel<=0)		return NULL;
	
	Neuron=(PdsNeuron *)calloc(1,sizeof(PdsNeuron));
	if(Neuron==NULL)	return NULL;

	Neuron->Nel=Nel;
	Neuron->X=(PdsSnReal **)calloc(Neuron->Nel,sizeof(PdsSnReal*));
	if(Neuron->X==NULL) 
	{
		free(Neuron);
		return NULL;
	}
	for(i=0;i<Neuron->Nel;i++)	Neuron->X[i]=NULL;

	Neuron->W=(PdsSnReal *)calloc(Neuron->Nel,sizeof(PdsSnReal));
	if(Neuron->W==NULL) 
	{
		free(Neuron);
		return NULL;
	}

	for(i=0;i<Neuron->Nel;i++)	Neuron->W[i]=W;
	
	Neuron->U=U;
	Neuron->Sigma=Sigma;
	Neuron->Theta=0.0;
    Neuron->func=func;
    Neuron->dfunc=dfunc;

	Neuron->F=0.0;
	Neuron->Y[0]=0.0;
	Neuron->Y[1]=0.0;

	return Neuron;
}


/** \fn int pds_neuron_get_weight(const PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal *m)
 *  \brief Devuelve el valor en la posición (x) del vector de pesos de la Neurona.
 *  (x) inicia con (0).
 *  \param[in] Neuron La neurona en consulta.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[out] m El valor en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces carga 0 en m, en caso de error de vector nulo carga cero en m.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_weight(const PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal *m)
{
	*m=0;

	if(Neuron==NULL)	return FALSE;

	if((x>=0)&&(x<Neuron->Nel))	*m=Neuron->W[x];
	else				*m=0;

	return TRUE;
}


/** \fn int pds_neuron_set_weight(PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal m)
 *  \brief Escribe el valor m en la posición (x) del vector de pesos de la Neurona.
 *  (x) inicia con (0).
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[in] m El valor en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces no hace nada y no se considera como error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_set_weight(PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal m)
{
	if(Neuron==NULL)	return FALSE;

	if((x>=0)&&(x<Neuron->Nel))	Neuron->W[x]=m;

	return TRUE;
}


/** \fn int pds_neuron_init_weight(PdsNeuron *Neuron, PdsSnReal m)
 *  \brief Inicia el vector de pesos de la Neurona.
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] m El valor de los pesos de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_init_weight(PdsNeuron *Neuron, PdsSnReal m)
{
	PdsSnNatural i;

	if(Neuron==NULL)	return FALSE;

	for(i=0;i<Neuron->Nel;i++)	Neuron->W[i]=m;

	return TRUE;
}


/** \fn int pds_neuron_init_weight_rand(PdsNeuron *Neuron, PdsSnReal min, PdsSnReal max)
 *  \brief Inicia el vector de pesos de la Neurona con valores aleatorios
 *  uniformemente distribuido entre min y max.
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] min El minimo valor encontrado en los pesos de la neurona.
 *  \param[in] max El minimo valor encontrado en los pesos de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_init_weight_rand(PdsNeuron *Neuron, PdsSnReal min, PdsSnReal max)
{
	PdsSnNatural i;
    PdsSnReal m;

	if(Neuron==NULL)	return FALSE;

    srand(clock());
	for(i=0;i<Neuron->Nel;i++)
    {
        m = min+(rand()*(max-min))/RAND_MAX;
	    Neuron->W[i]=m;
    }

	return TRUE;
}


/** \fn int pds_neuron_get_number_of_inputs(const PdsNeuron *Neuron,PdsSnNatural *Nelements)
 *  \brief Devuelve el número de entradas de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] Nelements En donde se guardará el número de entradas de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_number_of_inputs(const PdsNeuron *Neuron,PdsSnNatural *Nelements)
{
	*Nelements=0;

	if(Neuron==NULL)	return FALSE;

	*Nelements=Neuron->Nel;


	return TRUE;
}


/** \fn int pds_neuron_get_sigma(const PdsNeuron *Neuron,PdsSnReal *Sigma)
 *  \brief Devuelve el valor de  sigma de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] Sigma En donde se guardará el valor de sigma de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_sigma(const PdsNeuron *Neuron,PdsSnReal *Sigma)
{
	*Sigma=0;

	if(Neuron==NULL)	return FALSE;

	*Sigma=Neuron->Sigma;

	return TRUE;
}


/** \fn int pds_neuron_set_sigma(PdsNeuron *Neuron,PdsSnReal Sigma)
 *  \brief Coloca el valor de  sigma de la Neurona.
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] Sigma El valor de sigma de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_set_sigma(PdsNeuron *Neuron,PdsSnReal Sigma)
{

	if(Neuron==NULL)	return FALSE;

	Neuron->Sigma=Sigma;

	return TRUE;
}


/** \fn int pds_neuron_get_u(const PdsNeuron *Neuron,PdsSnReal *U)
 *  \brief Devuelve el valor de  U de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] U En donde se guardará el valor de U de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_u(const PdsNeuron *Neuron,PdsSnReal *U)
{
	*U=0;

	if(Neuron==NULL)	return FALSE;

	*U=Neuron->U;

	return TRUE;
}


/** \fn int pds_neuron_set_u(PdsNeuron *Neuron,PdsSnReal U)
 *  \brief Coloca el valor de  U de la Neurona.
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] U El valor de U de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_set_u(PdsNeuron *Neuron,PdsSnReal U)
{

	if(Neuron==NULL)	return FALSE;

	Neuron->U=U;

	return TRUE;
}


/** \fn const PdsSnReal *pds_neuron_get_dendrite(const PdsNeuron *Neuron,PdsSnNatural id)
 *  \brief Devuelve la dirección de la entrada X[id] de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[in] id La posición a consultar, id tiene que ser menor que Nel.
 *  \return La dirección de la entrada X[id] o NULL en caso de error (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
const PdsSnReal *pds_neuron_get_dendrite(const PdsNeuron *Neuron,PdsSnNatural id)
{
	if(Neuron==NULL)		return NULL;
	if((id>=Neuron->Nel)||(id<0))	return NULL;

	return Neuron->X[id];
}


/** \fn int pds_neuron_get_input(const PdsNeuron *Neuron,PdsSnNatural id,PdsSnReal *m)
 *  \brief Devuelve el valor de la entrada X[id] de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[in] id La posición a consultar, id tiene que ser menor que Nel.
 *  \param[out] m Donde se cargará el valor de la entrada X[id].
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL, Neuron->X[id]==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_input(const PdsNeuron *Neuron,PdsSnNatural id,PdsSnReal *m)
{
	*m=0;
	if(Neuron==NULL)		return FALSE;
	if((id>=Neuron->Nel)||(id<0))	return FALSE;
	if(Neuron->X[id]==NULL)		return FALSE;

	*m=*(Neuron->X[id]);

	return TRUE;
}


/** \fn const PdsSnReal *pds_neuron_get_axon(const PdsNeuron *Neuron)
 *  \brief Devuelve la dirección del axon (salida) de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \return La dirección del axon (salida) o NULL en caso de error (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
const PdsSnReal *pds_neuron_get_axon(const PdsNeuron *Neuron)
{
	if(Neuron==NULL)		return NULL;

	return Neuron->Y;
}


/** \fn int pds_neuron_get_output(const PdsNeuron *Neuron,PdsSnReal *m)
 *  \brief Devuelve el valor de la salida Y de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] m Donde se cargará el valor de la salida Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_output(const PdsNeuron *Neuron,PdsSnReal *m)
{
	if(Neuron==NULL)		return FALSE;

	*m=*(Neuron->Y);

	return TRUE;
}



/** \fn int pds_neuron_connect_input_with_ninput(PdsNeuron *Neuron,PdsSnNatural id,PdsNInput *NInput)
 *  \brief Conecta el valor de salida Y de la neurona de entrada NInput con la 
 *  entrada X[id] de la neurona Neuron.
 *  \param[in,out] Neuron La neurona a trabajar.
 *  \param[in] id La posición a conectar, id tiene que ser menor que Nel.
 *  \param[in] NInput La neurona de entrada que se conectará a la entrada X[id] de Neuron.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_connect_input_with_ninput(PdsNeuron *Neuron,PdsSnNatural id,PdsNInput *NInput)
{
	if(Neuron==NULL)		return FALSE;
	if(NInput==NULL)		return FALSE;
	if((id>=Neuron->Nel)||(id<0))	return FALSE;

	Neuron->X[id]=NInput->Y;

	return TRUE;
}


/** \fn int pds_neuron_connect_input(PdsNeuron *Neuron,PdsSnNatural id,PdsNeuron *NeuronX)
 *  \brief Conecta el valor de salida Y de la Neurona NeuronX con la entrada X[id]
 *  de la neurona Neuron.
 *  \param[in,out] Neuron La neurona a trabajar.
 *  \param[in] id La posición a conectar, id tiene que ser menor que Nel.
 *  \param[in] NeuronX Neurona que se conectará a la entrada X[id] de Neuron.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_connect_input(PdsNeuron *Neuron,PdsSnNatural id,PdsNeuron *NeuronX)
{
	if(Neuron ==NULL)		return FALSE;
	if(NeuronX==NULL)		return FALSE;
	if((id>=Neuron->Nel)||(id<0))	return FALSE;

	Neuron->X[id]=NeuronX->Y;

	return TRUE;
}


/** \fn int pds_neuron_evaluate_theta(PdsNeuron *Neuron)
 *  \brief Evalua la variable theta de la Neurona Neuron.
 *  \f[ \theta=\sum_{i=0}^{N_{el}-1}{W[i]X[i]} - U \f].
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate_theta(PdsNeuron *Neuron)
{
	PdsSnReal theta;
	PdsSnNatural i;

	if(Neuron==NULL)		return FALSE;

	theta=-Neuron->U;
	for(i=0;i<Neuron->Nel;i++)	
	{
		if(Neuron->X[i]!=NULL)
		theta=theta+(*(Neuron->X[i]))*Neuron->W[i];
	}
	
	Neuron->Theta=theta;

	return TRUE;
}


/** \fn int pds_neuron_evaluate_func(PdsNeuron *Neuron)
 *  \brief Evalua la funcion de activación de la Neurona Neuron.
 *  \f[ func\left(\frac{\theta}{\sigma}\right) \f]
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate_func(PdsNeuron *Neuron)
{
	PdsSnReal x;

	if(Neuron==NULL)		return FALSE;

	x=Neuron->Theta/Neuron->Sigma;

	Neuron->F=Neuron->func(x);

	return TRUE;
}


/** \fn int pds_neuron_evaluate(PdsNeuron *Neuron)
 *  \brief Evalua la Neurona Neuron.
 *  \f[ \theta=\sum_{i=0}^{N_{el}-1}{W[i]X[i]} - U \f]
 *  \f[ func\left(\frac{\theta}{\sigma}\right) \f]
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate(PdsNeuron *Neuron)
{
	int m;
	if(Neuron==NULL)		return FALSE;

	m=pds_neuron_evaluate_theta(Neuron);
	if(m==FALSE)	return FALSE;
	m=pds_neuron_evaluate_func(Neuron);
	if(m==FALSE)	return FALSE;

	return TRUE;
}


/** \fn int pds_neuron_update(PdsNeuron *Neuron)
 *  \brief Actualiza el valor a la salida de la Neurona Neuron.
 *  \f[ Y=func\left(\frac{\theta}{\sigma}\right) \f]
 *  Adicionalmente (Neuron->Y[1]=0)
 *  \f[ e=0 \f]
 *  \param[in,out] Neuron La neurona a trabajar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_update(PdsNeuron *Neuron)
{
	if(Neuron==NULL)		return FALSE;

	*(Neuron->Y)=Neuron->F;
	Neuron->Y[1]=0.0;

	return TRUE;
}


/** \fn int pds_neuron_iterate(PdsNeuron *Neuron)
 *  \brief Itera la Neurona Neuron.
 *  \f[ \theta=\sum_{i=0}^{N_{el}-1}{W[i]X[i]} - U \f]
 *  \f[ func\left(\frac{\theta}{\sigma}\right) \f]
 *  \f[ Y=func\left(\frac{\theta}{\sigma}\right) \f]
 *  Adicionalmente (Neuron->Y[1]=0)
 *  \f[ e=0 \f]
 *  \param[in,out] Neuron La neurona a iterar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_iterate(PdsNeuron *Neuron)
{
	int m;

	if(Neuron==NULL)		return FALSE;

	m=pds_neuron_evaluate_theta(Neuron);
	if(m==FALSE)	return FALSE;
	m=pds_neuron_evaluate_func(Neuron);
	if(m==FALSE)	return FALSE;

	*(Neuron->Y)=Neuron->F;
	Neuron->Y[1]=0.0;

	return TRUE;
}


/** \fn int pds_neuron_evaluate_error(PdsNeuron *Neuron,PdsSnReal y)
 *  \brief Compara el valor de salida \f$Y\f$ de la neurona, con "y", luego lo carga
 *  en la variable Y[1] de la neurona (Neuron->Y[1]=y-Neuron->Y[0]).
 *  \f[ e= y - Y\f]
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \param[in] y Valor a comparar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate_error(PdsNeuron *Neuron,PdsSnReal y)
{

	if(Neuron==NULL)		return FALSE;

	Neuron->Y[1]=y-Neuron->Y[0];

	return TRUE;
}


/** \fn int pds_neuron_backpropagate_error(PdsNeuron *Neuron)
 *  \brief Propaga el error de la salida de la neurona hacia las neuronas conectadas
 *  a sus entradas. <br>
 *  <center>
 *  \image html backpropagation.png "Backpropagation"
 *  </center>
 *  <br>
 *  <center>
 *  \image html backpropagation2.png "Backpropagation"
 *  </center>
 *  <br>
 *  \f[ e = e_{\{Y\}} \f]
 *  \f[ e_i = e_{\{X[i]\}} \f]
 *  \f[ e_i \leftarrow e_i + \frac{func'\left(\frac{\theta}{\sigma}\right)}{\sigma}W[i]e\f]
 *  "e" es el error de la salida de la neurona Neuron :Neuron->Y[1].<br>
 *  "func" es la función de activación da neurona Neuron.<br>
 *  \param[in,out] Neuron La neurona a retropropagar el error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_backpropagate_error(PdsNeuron *Neuron)
{
	PdsSnReal e,Df;
	PdsSnNatural i;
	PdsSnReal *y1;

	if(Neuron==NULL)		return FALSE;

	Df=Neuron->dfunc(Neuron->Theta/Neuron->Sigma)/Neuron->Sigma;

	e=Neuron->Y[1];
	for(i=0;i<Neuron->Nel;i++)	
	{
		if(Neuron->X[i]!=NULL)
        {
            y1=(Neuron->X[i]+1);
		    (*y1)=(*y1)+Neuron->W[i]*Df*e;
        }
	}

	return TRUE;
}


/** \fn int pds_neuron_update_weight(PdsNeuron *Neuron, PdsSnReal Alpha)
 *  \brief Actualiza los pesos W[i] de la neurona (using the Delta rule).
 *  \f[ e = r-Y\f]
 *  \f[ \triangle W[i]= \alpha \frac{func'\left(\frac{\theta}{\sigma}\right)}{\sigma} X[i] e\f]
 *  "e" es el error de la salida de la neurona Neuron :Neuron->Y[1].<br>
 *  "Y" salida de la neurona Neuron :Neuron->Y[0].<br>
 *  "func" es la función de activación da neurona Neuron.<br>
 *  "X" Vector con los valores de todas las entradas de la neurona.<br>
 *  \param[in,out] Neuron La neurona a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_update_weight(PdsNeuron *Neuron, PdsSnReal Alpha)
{
	PdsSnReal e,x,Df;
	PdsSnNatural i;

	if(Neuron==NULL)		return FALSE;

	Df=Neuron->dfunc(Neuron->Theta/Neuron->Sigma)/Neuron->Sigma;
	e=Neuron->Y[1];

	for(i=0;i<Neuron->Nel;i++)	
	{
		if(Neuron->X[i]!=NULL)
		{
			x=(*(Neuron->X[i]));
			Neuron->W[i]=Neuron->W[i]+Alpha*e*Df*x;
		}
	}

	return TRUE;
}


/** \fn int pds_neuron_get_xtx(PdsNeuron *Neuron,PdsSnReal *XtX)
 *  \brief Devuelve la suma cuadrática de los valores de entrada de la neurona.
 *  \f[ X^T X=\sum^{Nel-1}_{i=0} X[i]^2 \f].
 *  \param[in,out] Neuron La neurona en consulta.
 *  \param[in] XtX Donde se cargará la suma cuadrática.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_xtx(PdsNeuron *Neuron,PdsSnReal *XtX)
{
	PdsSnNatural i;

	*XtX=0;
	if(Neuron==NULL)		return FALSE;
	
	for(i=0;i<Neuron->Nel;i++)	
	if(Neuron->X[i]!=NULL)	*XtX=*XtX+(*(Neuron->X[i]))*(*(Neuron->X[i]));

	return TRUE;
}


/** \fn int pds_neuron_update_weight_normalized(PdsNeuron *Neuron, PdsSnReal Alpha)
 *  \brief Actualiza los pesos W[i] de la neurona (using the Delta rule).
 *  \f[ e = r-Y\f]
 *  \f[ \triangle W[i]= \alpha \frac{func'\left(\frac{\theta}{\sigma}\right)}{\sigma} \frac{X[i]}{X^TX} e \f]
 *  "e" es el error de la salida de la neurona Neuron :Neuron->Y[1].<br>
 *  "Y" salida de la neurona Neuron :Neuron->Y[0].<br>
 *  "func" es la función de activación da neurona Neuron.<br>
 *  "X" Vector con los valores de todas las entradas de la neurona.<br>
 *  \param[in,out] Neuron La neurona a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_update_weight_normalized(PdsNeuron *Neuron, PdsSnReal Alpha)
{
	PdsSnReal e,x,Df,XtX;
	PdsSnNatural i;

	if(Neuron==NULL)		return FALSE;

	Df=Neuron->dfunc(Neuron->Theta/Neuron->Sigma)/Neuron->Sigma;
	e=Neuron->Y[1];
	
	for(XtX=0,i=0;i<Neuron->Nel;i++)	
	if(Neuron->X[i]!=NULL)	XtX=XtX+(*(Neuron->X[i]))*(*(Neuron->X[i]));

	for(i=0;i<Neuron->Nel;i++)	
	{
		if((Neuron->X[i]!=NULL)&&(XtX>0))
		{
			x=(*(Neuron->X[i]));
			Neuron->W[i]=Neuron->W[i]+Alpha*e*Df*x/XtX;
		}
	}

	return TRUE;
}


/** \fn int pds_neuron_fprintf(const PdsNeuron *Neuron, FILE *fd)
 *  \brief Guarda en un archivo de texto los pesos {W[i]} y  {U, Sigma}.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[in] Neuron La neurona a leer.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fprintf(const PdsNeuron *Neuron, FILE *fd)
{
	PdsSnNatural i;
	int tmp;

	if(Neuron==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;


	for(i=0;i<Neuron->Nel;i++)	
	{
		if(i!=(Neuron->Nel-1))	tmp=fprintf(fd,"%e\t",Neuron->W[i]);
		else                    tmp=fprintf(fd,"%e\n",Neuron->W[i]);
	}

	tmp=fprintf(fd,"%e\t",Neuron->U);
	tmp=fprintf(fd,"%e\n",Neuron->Sigma);

	return TRUE;
}


/** \fn int pds_neuron_fscanf(PdsNeuron *Neuron, FILE *fd)
 *  \brief Lee de un archivo de texto los pesos {W[i]} y  {U, Sigma}.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[out] Neuron La neurona a escribir.
 *  \param[in,out] fd Manejador del fichero a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fscanf(PdsNeuron *Neuron, FILE *fd)
{
	PdsSnNatural i;
	int tmp;

	if(Neuron==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;


	for(i=0;i<Neuron->Nel;i++)	
	{
		tmp=fscanf(fd,"%e",&(Neuron->W[i]));
		if(tmp==0)	return FALSE;
	}

	tmp=fscanf(fd,"%e",&(Neuron->U));
	if(tmp==0)	return FALSE;

	tmp=fscanf(fd,"%e",&(Neuron->Sigma));
	if(tmp==0)	return FALSE;

	return TRUE;
}


/** \fn int pds_neuron_fwrite(const PdsNeuron *Neuron, FILE *fd)
 *  \brief Guarda en un archivo binario los pesos W[i], el valor de U y Sigma.
 *  \param[in] Neuron La neurona a leer.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fwrite(const PdsNeuron *Neuron, FILE *fd)
{
	PdsSnNatural i;
	size_t tmp;

	if(Neuron==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;


	for(i=0;i<Neuron->Nel;i++)	
	{
		tmp=fwrite(&(Neuron->W[i]),sizeof(Neuron->W[i]),1,fd);
		if(tmp!=1)	return	FALSE;
	}

	tmp=fwrite(&(Neuron->U),sizeof(Neuron->U),1,fd);
	if(tmp!=1)	return	FALSE;

	tmp=fwrite(&(Neuron->Sigma),sizeof(Neuron->Sigma),1,fd);
	if(tmp!=1)	return	FALSE;

	return TRUE;
}


/** \fn int pds_neuron_fread(PdsNeuron *Neuron, FILE *fd)
 *  \brief Lee de un archivo binario los pesos W[i], el valor de U y Sigma.
 *  \param[out] Neuron La neurona a escribir.
 *  \param[in,out] fd Manejador del fichero binario a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fread(PdsNeuron *Neuron, FILE *fd)
{
	PdsSnNatural i;
	int tmp;

	if(Neuron==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;


	for(i=0;i<Neuron->Nel;i++)	
	{
		tmp=fread(&(Neuron->W[i]),sizeof(Neuron->W[i]),1,fd);
		if(tmp!=1)	return FALSE;
	}

	tmp=fread(&(Neuron->U),sizeof(Neuron->U),1,fd);
	if(tmp!=1)	return FALSE;

	tmp=fread(&(Neuron->Sigma),sizeof(Neuron->Sigma),1,fd);
	if(tmp!=1)	return FALSE;

	return TRUE;
}


/** \fn void pds_neuron_free(PdsNeuron *Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuron.
 *  \param[in,out] Neuron La neurona a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronGroup
 */
void pds_neuron_free(PdsNeuron *Neuron)
{
	if(Neuron!=NULL)
	{
		free(Neuron->X);
		free(Neuron->W);
		free(Neuron);
	}
}


/** \fn void pds_neuron_destroy(PdsNeuron **Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuron, y limpia el puntero con NULL.
 *  \param[in,out] Neuron La neurona a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronGroup
 */
void pds_neuron_destroy(PdsNeuron **Neuron)
{
	if((*Neuron)!=NULL)
	{
		free((*Neuron)->X);
		free((*Neuron)->W);
		free(*Neuron);
		(*Neuron)=NULL;
	}
}

