/*
 * pdsneuron.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsneuron.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-04-2011
 *  \brief Funciones que trabajan con vectores.
 *  
 *  <br>Estas funciones trabajan con una neurona de la forma.<br>
 *  \image html PdsNeuron.png "Neurona de n=Nel entradas"
 *  \b Nel es el número de elementos.
 */

#ifndef __PDSNEURON_H__
#define __PDSNEURON_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdssnglobal.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsNeuronGroup Funciones del módulo PdsNeuron.
 *  \brief Funciones que trabajan con neuronas.
 *  
 *  <br>Estas funciones trabajan con una neurona de la forma.<br>
 *  \image html PdsNeuron.png "Neurona de n=Nel entradas."
 * @{
 */

/*! \def _PNSETSW(Neuron,i,m)
 *  Es equivalente a Neuron->W[i]=m.
 *  \ingroup PdsNeuronGroup
*/
#define _PNSETSW(Neuron,i,m) pds_neuron_set_synaptic_weight(Neuron,i,m)


/*! \struct PdsNeuron
 *  \brief La estructura tipo  PdsNeuron .
 *  Esta estructura genera una neurona de Nel entradas.
 *  \image html PdsNeuron.png "Neurona de n=Nel entradas."
 *  Para usar incluir pds/pdssn.h.
 *  \ingroup PdsNeuronGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Un arreglo de punteros con Nel elementos. */
	PdsSnReal **X;
	/*! Un arreglo de Nel elementos. */
	PdsSnReal *W;
	/*! Número de elementos. */
	PdsSnNatural Nel;
	/*! Valor de umbral. */
	PdsSnReal U;
	/*! Valor de Sigma. */
	PdsSnReal Sigma;
	/*! Valor de ponderacion. */
	PdsSnReal Theta;
	/*! Valor de activación. */
	PdsSnReal F;
	/*! Valor de salida *Y, y un espacio extra Y[1]. */
	PdsSnReal Y[2];
}PdsNeuron;


/** \fn PdsNeuron *pds_neuron_new(PdsSnNatural Nel)
 *  \brief Crea un vector de tipo PdsNeuron e inicia con cero todos los elementos.
 *  \param[in] Nel Es el número de elementos del vector.
 *  \return Un puntero al vector de tipo PdsNeuron.
 *  \ingroup PdsNeuronGroup
 */
PdsNeuron *pds_neuron_new(PdsSnNatural Nel);


/** \fn int pds_neuron_get_synaptic_weight(const PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal *m)
 *  \brief Devuelve el valor en la posición (x) del vector de pesos de la Neurona.
 *  (x) inicia con (0).
 *  \param[in] Neuron La neurona en consulta.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[out] m El valor en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces carga 0 en m, en caso de error de vector nulo carga cero en m.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_synaptic_weight(const PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal *m);


/** \fn int pds_neuron_set_synaptic_weight(PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal m)
 *  \brief Escribe el valor m en la posición (x) del vector de pesos de la Neurona.
 *  (x) inicia con (0).
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[in] m El valor en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces no hace nada y no se considera como error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_set_synaptic_weight(PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal m);


/** \fn int pds_neuron_init_synaptic_weight(PdsNeuron *Neuron, PdsSnReal m)
 *  \brief Inicia el vector de pesos de la Neurona.
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] m El valor de los pesos de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_init_synaptic_weight(PdsNeuron *Neuron, PdsSnReal m);


/** \fn int pds_neuron_get_number_of_inputs(const PdsNeuron *Neuron,PdsSnNatural *Nelements)
 *  \brief Devuelve el número de entradas de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] Nelements En donde se guardará el número de entradas de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_number_of_inputs(const PdsNeuron *Neuron,PdsSnNatural *Nelements);


/** \fn int pds_neuron_get_sigma(const PdsNeuron *Neuron,PdsSnReal *Sigma)
 *  \brief Devuelve el valor de  sigma de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] Sigma En donde se guardará el valor de sigma de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_sigma(const PdsNeuron *Neuron,PdsSnReal *Sigma);


/** \fn int pds_neuron_set_sigma(PdsNeuron *Neuron,PdsSnReal Sigma)
 *  \brief Coloca el valor de  sigma de la Neurona.
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] Sigma El valor de sigma de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_set_sigma(PdsNeuron *Neuron,PdsSnReal Sigma);


/** \fn int pds_neuron_get_u(const PdsNeuron *Neuron,PdsSnReal *U)
 *  \brief Devuelve el valor de  U de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] U En donde se guardará el valor de U de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_u(const PdsNeuron *Neuron,PdsSnReal *U);


/** \fn int pds_neuron_set_u(PdsNeuron *Neuron,PdsSnReal U)
 *  \brief Coloca el valor de  U de la Neurona.
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] U El valor de U de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_set_u(PdsNeuron *Neuron,PdsSnReal U);


/** \fn PdsSnReal *pds_neuron_get_dendrite(const PdsNeuron *Neuron,PdsSnNatural id)
 *  \brief Devuelve la dirección de la entrada X[id] de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[in] id La posición a consultar, id tiene que ser menor que Nel.
 *  \return La dirección de la entrada X[id] o NULL en caso de error (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
PdsSnReal *pds_neuron_get_dendrite(const PdsNeuron *Neuron,PdsSnNatural id);


/** \fn int pds_neuron_get_input(const PdsNeuron *Neuron,PdsSnNatural id,PdsSnReal *m)
 *  \brief Devuelve el valor de la entrada X[id] de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[in] id La posición a consultar, id tiene que ser menor que Nel.
 *  \param[out] m Donde se cargará el valor de la entrada X[id].
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_input(const PdsNeuron *Neuron,PdsSnNatural id,PdsSnReal *m);


/** \fn const PdsSnReal *pds_neuron_get_axon(const PdsNeuron *Neuron)
 *  \brief Devuelve la dirección del axon (salida) de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \return La dirección del axon (salida) o NULL en caso de error (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
const PdsSnReal *pds_neuron_get_axon(const PdsNeuron *Neuron);


/** \fn int pds_neuron_get_output(const PdsNeuron *Neuron,PdsSnReal *m)
 *  \brief Devuelve el valor de la salida Y de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] m Donde se cargará el valor de la salida Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_output(const PdsNeuron *Neuron,PdsSnReal *m);


/** \fn int pds_neuron_connect_input(PdsNeuron *Neuron,PdsSnNatural id,PdsNeuron *NeuronX)
 *  \brief Conecta el valor de salida Y de la Neurona NeuronX con la entrada X[id]
 *  de la neurona Neuron.
 *  \param[in,out] Neuron La neurona a trabajar.
 *  \param[in] id La posición a conectar, id tiene que ser menor que Nel.
 *  \param[in] NeuronX Neurona que se conectará a la entrada X[id] de Neuron.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_connect_input(PdsNeuron *Neuron,PdsSnNatural id,PdsNeuron *NeuronX);


/** \fn int pds_neuron_connect_input_data(PdsNeuron *Neuron,PdsSnNatural id,PdsNInput *NInput)
 *  \brief Conecta el valor de salida Y de la neurona de entrada NInput con la 
 *  entrada X[id] de la neurona Neuron.
 *  \param[in,out] Neuron La neurona a trabajar.
 *  \param[in] id La posición a conectar, id tiene que ser menor que Nel.
 *  \param[in] NInput La neurona de entrada que se conectará a la entrada X[id] de Neuron.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_connect_input_data(PdsNeuron *Neuron,PdsSnNatural id,PdsNInput *NInput);


/** \fn int pds_neuron_evaluate_theta(PdsNeuron *Neuron)
 *  \brief Evalua la variable theta de la Neurona Neuron.
 *  \f[ \theta=\sum_{i=0}^{N_{el}-1}{W[i]X[i]} - U \f].
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate_theta(PdsNeuron *Neuron);


/** \fn PdsSnReal pds_neuron_func(PdsSnReal x)
 *  \brief Función de activación de la neurona de tipo PdsNeuron.
 *  \f[ f(x)=\frac{2}{1+e^{-x}} -1 \f].
 *  \param[in] x Parámetro a evaluar.
 *  \return El resultado de la función. 
 *  \ingroup PdsNeuronGroup
 */
PdsSnReal pds_neuron_func(PdsSnReal x);


/** \fn PdsSnReal pds_neuron_dfunc(PdsSnReal x)
 *  \brief Derivada de la función de activación de la neurona de tipo PdsNeuron.
 *  \f[ \frac{\partial  f(x)}{\partial x}=\frac{1-f(x)^2}{2} \f].
 *  \param[in] x Parámetro a evaluar.
 *  \return El resultado de la derivada de la función. 
 *  \ingroup PdsNeuronGroup
 */
PdsSnReal pds_neuron_dfunc(PdsSnReal x);


/** \fn PdsSnReal pds_neuron_invdfunc(PdsSnReal x)
 *  \brief Uno sobre la derivada de la función de activación de la neurona de 
 *  tipo PdsNeuron.
 *  \f[ ( \frac{\partial  f(x)}{\partial x} )^{-1}=\frac{2}{1-f(x)^2} \f].
 *  \param[in] x Parámetro a evaluar.
 *  \return El resultado de uno sobre la derivada de la función. 
 *  \ingroup PdsNeuronGroup
 */
PdsSnReal pds_neuron_invdfunc(PdsSnReal x);


/** \fn int pds_neuron_evaluate_f(PdsNeuron *Neuron)
 *  \brief Evalua la funcion de activación de la Neurona Neuron.
 *  \f[ f(\theta)=\frac{2}{1+e^{-\frac{\theta}{\sigma}}} -1 \f].
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate_f(PdsNeuron *Neuron);


/** \fn int pds_neuron_evaluate(PdsNeuron *Neuron)
 *  \brief Evalua la Neurona Neuron.
 *  \f[ \theta=\sum_{i=0}^{N_{el}-1}{W[i]X[i]} - U \f].
 *  \f[ f(\theta)=\frac{2}{1+e^{-\frac{\theta}{\sigma}}} -1 \f].
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate(PdsNeuron *Neuron);


/** \fn int pds_neuron_update(PdsNeuron *Neuron)
 *  \brief Actualiza el valor a la salida de la Neurona Neuron.
 *  \f[ Y=f(\theta) \f].
 *  \param[in,out] Neuron La neurona a trabajar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_update(PdsNeuron *Neuron);


/** \fn int pds_neuron_iterate(PdsNeuron *Neuron)
 *  \brief Itera la Neurona Neuron.
 *  \f[ \theta=\sum_{i=0}^{N_{el}-1}{W[i]X[i]} - U \f].
 *  \f[ f(\theta)=\frac{2}{1+e^{-\frac{\theta}{\sigma}}} -1 \f].
 *  \f[ Y=f(\theta) \f].
 *  \param[in,out] Neuron La neurona a iterar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_iterate(PdsNeuron *Neuron);



/** \fn int pds_neuron_evaluate_e(PdsNeuron *Neuron,PdsSnReal y)
 *  \brief Compara el valor de salida de la neurona con "y" y luego lo carga en
 *  la variable Y[1] de la neurona. 
 *  Neuron->Y[1]=y-Neuron->Y[0]
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \param[in] y Valor a comparar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate_e(PdsNeuron *Neuron,PdsSnReal y);


/** \fn int pds_neuron_backpropagate_error(PdsNeuron *Neuron)
 *  \brief Propaga el error de la salida de la neurona hacia las neuronas conectadas
 * a sus entradas. E_input (i)=W[i] E_salida
 *  \param[in,out] Neuron La neurona a retropropagar el error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_backpropagate_error(PdsNeuron *Neuron);

/** \fn int pds_neuron_update_weight(PdsNeuron *Neuron, PdsSnReal Alpha)
 *  \brief Actualiza los pesos W[i] de la neurona.
 *  \f[ \triangle W[i]= \alpha e f'(\theta) X[i] \f].
 *  "e" es el error de la salida da neurona Neuron :Neuron->Y[1].
 *  "f" es la función de activación da neurona Neuron.
 *  \param[in,out] Neuron La neurona a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_update_weight(PdsNeuron *Neuron, PdsSnReal Alpha);


/** \fn int pds_neuron_get_xtx(PdsNeuron *Neuron,PdsSnReal *XtX)
 *  \brief Devuelve la suma cuadrática de los valores de entrada de la neurona.
 *  \f[ X^T X=\sum^{Nel-1}_{i=0} X[i]^2 \f].
 *  \param[in,out] Neuron La neurona en consulta.
 *  \param[in] XtX Donde se cargará la suma cuadrática.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_xtx(PdsNeuron *Neuron,PdsSnReal *XtX);


/** \fn int pds_neuron_update_weight_norm(PdsNeuron *Neuron, PdsSnReal Alpha,PdsSnReal XtX)
 *  \brief Actualiza los pesos W[i] de la neurona.
 *  \f[ \triangle W[i]= \alpha e f'(\theta) \frac{X[i]}{X^TX} \f].
 *  "e" es el error de la salida da neurona Neuron :Neuron->Y[1].
 *  "f" es la función de activación da neurona Neuron.
 *  \param[in,out] Neuron La neurona a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \param[in] XtX Suma cuadrática de los valores de entrada X[i].
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_update_weight_norm(PdsNeuron *Neuron, PdsSnReal Alpha,PdsSnReal XtX);


/** \fn int pds_neuron_update_weight_normalized(PdsNeuron *Neuron, PdsSnReal Alpha)
 *  \brief Actualiza los pesos W[i] de la neurona.
 *  \f[ \triangle W[i]= \alpha e f'(\theta) \frac{X[i]}{X^TX} \f].
 *  "e" es el error de la salida da neurona Neuron :Neuron->Y[1].
 *  "f" es la función de activación da neurona Neuron.
 *  \param[in,out] Neuron La neurona a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_update_weight_normalized(PdsNeuron *Neuron, PdsSnReal Alpha);


/** \fn int pds_neuron_fprintf(const PdsNeuron *Neuron, FILE *fd)
 *  \brief Guarda en un archivo de texto los pesos W[i] y el valor de U.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[in] Neuron La neurona a leer.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fprintf(const PdsNeuron *Neuron, FILE *fd);


/** \fn int pds_neuron_fscanf(PdsNeuron *Neuron, FILE *fd)
 *  \brief Lee de un archivo de texto los pesos W[i] y el valor de U.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[out] Neuron La neurona a escribir.
 *  \param[in,out] fd Manejador del fichero a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fscanf(PdsNeuron *Neuron, FILE *fd);


/** \fn int pds_neuron_fwrite(const PdsNeuron *Neuron, FILE *fd)
 *  \brief Guarda en un archivo binario los pesos W[i] y el valor de U.
 *  \param[in] Neuron La neurona a leer.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fwrite(const PdsNeuron *Neuron, FILE *fd);


/** \fn int pds_neuron_fread(PdsNeuron *Neuron, FILE *fd)
 *  \brief Lee de un archivo binario los pesos W[i] y el valor de U.
 *  \param[out] Neuron La neurona a escribir.
 *  \param[in,out] fd Manejador del fichero binario a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fread(PdsNeuron *Neuron, FILE *fd);


/** \fn void pds_neuron_free(PdsNeuron *Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuron.
 *  \param[in,out] Neuron La neurona a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronGroup
 */
void pds_neuron_free(PdsNeuron *Neuron);


/** \fn void pds_neuron_destroy(PdsNeuron **Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuron, y limpia el puntero con NULL.
 *  \param[in,out] Neuron La neurona a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronGroup
 */
void pds_neuron_destroy(PdsNeuron **Neuron);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


