/*
 * pdsregionrect.c
 * 
 * Copyright 2018 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsregionrect.h>
#include <pds/pdsmatrix.h>


/** \fn PdsRegionRect pds_region_rect(PdsRaInteger L0,PdsRaInteger C0,PdsRaNatural Nlin,PdsRaNatural Ncol)
 *  \brief Crea la variable PdsRegionRect.
 *  \param[in] L0 Linea inicial.
 *  \param[in] C0 Columna inicial.
 *  \param[in] Nlin Numero de lineas.
 *  \param[in] Ncol Numero de columnas.
 *  \return La variable PdsRegionRect.
 *  \ingroup PdsRegionRectGroup
 */
PdsRegionRect pds_region_rect(PdsRaInteger L0,PdsRaInteger C0,PdsRaNatural Nlin,PdsRaNatural Ncol)
{
    PdsRegionRect R;

    R.L0=L0;
    R.C0=C0;
    R.Nlin=Nlin;
    R.Ncol=Ncol;

    return R;
}


/** \fn PdsRegionRect pds_region_rect_transfer(PdsRegionRect R,PdsRaInteger L,PdsRaInteger C)
 *  \brief Crea una variable PdsRegionRect a partir de suma R+(L,C).
 *  \param[in] R La region a transferir.
 *  \param[in] L Numero de lineas a sumar.
 *  \param[in] C Numero de columnas a sumar.
 *  \return La variable PdsRegionRect.
 *  \ingroup PdsRegionRectGroup
 */
PdsRegionRect pds_region_rect_transfer(PdsRegionRect R,PdsRaInteger L,PdsRaInteger C)
{
    PdsRegionRect Rout=R;

    Rout.L0=Rout.L0+L;
    Rout.C0=Rout.C0+C;

    return Rout;
}

/** \fn PdsRegionRect pds_region_rect_move(PdsRegionRect R,PdsRaInteger L0,PdsRaInteger C0)
 *  \brief Crea una variable PdsRegionRect, siendo esta una version de R movimentada a (L0,C0).
 *  \param[in] R La region a mover.
 *  \param[in] L0 Nueva linea de origen.
 *  \param[in] C0 Nueva columna de origen
 *  \return La variable PdsRegionRect.
 *  \ingroup PdsRegionRectGroup
 */
PdsRegionRect pds_region_rect_move(PdsRegionRect R,PdsRaInteger L0,PdsRaInteger C0)
{
    PdsRegionRect Rout=R;

    Rout.L0=L0;
    Rout.C0=C0;

    return Rout;
}



/** \fn int pds_region_rect_intersection(PdsRegionRect A,PdsRegionRect B,PdsRegionRect *R)
 *  \brief Intersecta dos regiones.
 *  \param[in] A Primera region.
 *  \param[in] B Segunda region.
 *  \param[out] R Region de interesecion si existe, o ninguna mudanza si  no.
 *  \return PDS_OK si existe la intersecion, o PDS_WRONG si no existe.
 *  \ingroup PdsRegionRectGroup
 */
int pds_region_rect_intersection(PdsRegionRect A,PdsRegionRect B,PdsRegionRect *R)
{
    PdsRaInteger C0;
    PdsRaNatural Ncol;
    PdsRaInteger L0;
    PdsRaNatural Nlin;

    if(R==NULL) return PDS_WRONG;

    if(A.C0<B.C0)
    {
        if( (A.C0+A.Ncol-1)>=B.C0 )
        {
            C0=B.C0;
            if( (A.C0+A.Ncol-1)>(B.C0+B.Ncol-1) )   Ncol=B.Ncol;
            else                                    Ncol=A.C0+A.Ncol-B.C0;
        }
        else
            return PDS_WRONG;
    }
    else
    {
        if( (B.C0+B.Ncol-1)>=A.C0 )
        {
            C0=A.C0;
            if( (B.C0+B.Ncol-1)>(A.C0+A.Ncol-1) )   Ncol=A.Ncol;
            else                                    Ncol=B.C0+B.Ncol-A.C0;

        }
        else
            return PDS_WRONG;
    }


    if(A.L0<B.L0)
    {
        if( (A.L0+A.Nlin-1)>=B.L0 )
        {
            L0=B.L0;
            if( (A.L0+A.Nlin-1)>(B.L0+B.Nlin-1) )   Nlin=B.Nlin;
            else                                    Nlin=A.L0+A.Nlin-B.L0;
        }
        else
            return PDS_WRONG;
    }
    else
    {
        if( (B.L0+B.Nlin-1)>=A.L0 )
        {
            L0=A.L0;
            if( (B.L0+B.Nlin-1)>(A.L0+A.Nlin-1) )   Nlin=A.Nlin;
            else                                    Nlin=B.L0+B.Nlin-A.L0;
        }
        else
            return PDS_WRONG;
    }


    (*R).L0=L0;
    (*R).C0=C0;
    (*R).Nlin=Nlin;
    (*R).Ncol=Ncol;

    return PDS_OK;
}


/** \fn int int pds_region_rect_is_inside(PdsRegionRect R,PdsRegionRect Rin)
 *  \brief Indica si la region Rin está dentro de la region R.
 *  \param[in] Rin Suúesta region interna de R.
 *  \param[in] R Region madre o contenedora.
 *  \return PDS_OK si Rin está adentro de R, o PDS_WRONG si no.
 *  \ingroup PdsRegionRectGroup
 */
int pds_region_rect_is_inside(PdsRegionRect Rin,PdsRegionRect R)
{
    if(Rin.L0<R.L0) return PDS_WRONG;
    if(Rin.C0<R.C0) return PDS_WRONG;

    if( (Rin.L0+Rin.Nlin)>(R.L0+R.Nlin) ) return PDS_WRONG;
    if( (Rin.C0+Rin.Ncol)>(R.C0+R.Ncol) ) return PDS_WRONG;

    return PDS_OK;
}


/** \fn int pds_region_rect_printf(PdsRegionRect R)
 *  \brief Imprime la variable PdsRegionRect.
 *  \param[in] R Estructura de tipo PdsRegionRect.
 *  \ingroup PdsRegionRectGroup
 */
int pds_region_rect_printf(PdsRegionRect R)
{
    return printf("%d\t%d\t%d\t%d\n",R.L0,R.C0,R.Nlin,R.Ncol);
}

/** \fn int pds_region_rect_stylized_printf(PdsRegionRect R)
 *  \brief Imprime la variable PdsRegionRect.
 *  \param[in] R Estructura de tipo PdsRegionRect.
 *  \ingroup PdsRegionRectGroup
 */
int pds_region_rect_stylized_printf(PdsRegionRect R)
{
    return printf("(%d,%d)[%d,%d]\n",R.L0,R.C0,R.Nlin,R.Ncol);
}

/** \fn int pds_region_rect_fprintf(FILE *fd, PdsRegionRect R)
 *  \brief Imprime en un FILE la variable PdsRegionRect.
 *  \param[in] fd File descriptor.
 *  \param[in] R Estructura de tipo PdsRegionRect.
 *  \ingroup PdsRegionRectGroup
 */
int pds_region_rect_fprintf(FILE *fd, PdsRegionRect R)
{
    return fprintf(fd,"%d\t%d\t%d\t%d\n",R.L0,R.C0,R.Nlin,R.Ncol);
}


/** \fn int pds_region_rect_stylized_fprintf(FILE *fd, PdsRegionRect R)
 *  \brief Imprime en un FILE la variable PdsRegionRect.
 *  \param[in] fd File descriptor.
 *  \param[in] R Estructura de tipo PdsRegionRect.
 *  \ingroup PdsRegionRectGroup
 */
int pds_region_rect_stylized_fprintf(FILE *fd, PdsRegionRect R)
{
    return fprintf(fd,"(%d,%d)[%d,%d]\n",R.L0,R.C0,R.Nlin,R.Ncol);
}


