/*
 * pdsmatrix.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsrautils.h>
#include <pds/pdsmatrix.h>
#include <pds/pdsregionrect.h>


////////////////////////////////////////////////////////////////////////////////
////                        Funciones Extras                                ////
////////////////////////////////////////////////////////////////////////////////

int pds_matrix_find_row_in_col(const PdsMatrix *Matrix, PdsRaNatural *row, PdsRaNatural *E, PdsRaNatural col, PdsRaNatural r1);

////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsMatrix                                              ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsMatrix *pds_matrix_new(PdsRaNatural Nlin,PdsRaNatural Ncol)
 *  \brief Crea una matriz de tipo PdsMatrix.
 *  \param[in] Nlin Es el número de lineas de la matriz.
 *  \param[in] Ncol Es el número de columnas de la matriz.
 *  \return Un puntero a la matriz de tipo PdsMatrix.
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_new(PdsRaNatural Nlin,PdsRaNatural Ncol)
{
	PdsMatrix *Matrix=NULL;
	PdsRaNatural i;

	Matrix=(PdsMatrix *)calloc(1,sizeof(PdsMatrix));
	if(Matrix==NULL) {return NULL;}

	Matrix->Nlin=Nlin;
	Matrix->Ncol=Ncol;
	Matrix->M=(PdsRaReal **)calloc(Matrix->Nlin,sizeof(PdsRaReal *));
	if(Matrix->M==NULL) 
	{
		free(Matrix);
		return NULL;
	}
	
	for(i=0;i<Matrix->Nlin;i++)
	{
		Matrix->M[i]=(PdsRaReal *)calloc(Matrix->Ncol,sizeof(PdsRaReal));
		if(Matrix->M[i]==NULL) 
		{
			for(i--;i>=0;i--)	free(Matrix->M[i]);
			free(Matrix);
			return NULL;
		}
	}

	return Matrix;
}


/** \fn PdsMatrix *pds_matrix_new_rand(PdsRaNatural Nlin,PdsRaNatural Ncol)
 *  \brief Crea una matriz de tipo PdsMatrix e inicializa los elementos
 *  con números aleatorios uniformemente distribuidos entre [0.0,1.0>.
 *  \param[in] Nlin Es el número de lineas de la matriz.
 *  \param[in] Ncol Es el número de columnas de la matriz.
 *  \return Un puntero a la matriz de tipo PdsMatrix.
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_new_rand(PdsRaNatural Nlin,PdsRaNatural Ncol)
{
    PdsMatrix *M = pds_matrix_new(Nlin,Ncol);
    if(M==NULL) return NULL;

    pds_matrix_init_rand(M);

    return M;
}
/** \fn PdsMatrix *pds_matrix_new_matrix(const PdsMatrix *MatSrc)
 *  \brief Crea una matriz de tipo PdsMatrix a partir de la matriz MatSrc.
 *  \param[in] MatSrc Matriz que se usará como imagen como fuente.
 *  \return Un puntero a la matriz de tipo PdsMatrix.
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_new_matrix(const PdsMatrix *MatSrc)
{
	PdsMatrix *Matrix=NULL;
	PdsRaNatural i,j;

	if(MatSrc==NULL)	return NULL;

	Matrix=(PdsMatrix *)calloc(1,sizeof(PdsMatrix));
	if(Matrix==NULL) {return NULL;}

	Matrix->Nlin=MatSrc->Nlin;
	Matrix->Ncol=MatSrc->Ncol;
	Matrix->M=(PdsRaReal **)calloc(Matrix->Nlin,sizeof(PdsRaReal *));
	if(Matrix->M==NULL) 
	{
		free(Matrix);
		return NULL;
	}
	
	for(i=0;i<Matrix->Nlin;i++)
	{
		Matrix->M[i]=(PdsRaReal *)calloc(Matrix->Ncol,sizeof(PdsRaReal));
		if(Matrix->M[i]==NULL) 
		{
			for(i--;i>=0;i--)	free(Matrix->M[i]);
			free(Matrix);
			return NULL;
		}
		else
		{
			for(j=0;j<Matrix->Ncol;j++)	Matrix->M[i][j]=MatSrc->M[i][j];
		}
	}

	return Matrix;
}


/** \fn PdsMatrix *pds_matrix_new_load_data(const char* datafile)
 *  \brief Crea una nueva matriz e inicia los datos con los elementos de la linea
 *  line del archivo datafile.
 *  Usa TAB como delimitador de elemento y un salto de linea como delimitador de linea.
 *  Si se pide una linea inexistente la funcion retorna NULL
 *  \param[in] datafile Nombre del archivo de donde se cargará los datos iniciales
 *  de la matriz.
 *  \return Un puntero que apunta a la dirección del nuevo vector, si todo fue bien, 
 *  o NULL si no. (ej. datafile==NULL o line inexistente)
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_new_load_data(const char* datafile)
{
	PdsRaNatural i,j,nc,nl;
	int id;
	int tmp;
	FILE *fd=NULL;
	char x[__MAX_CHAR_DATA_SIZE__];
	PdsMatrix *Matrix=NULL;
	
	fd=fopen(datafile,"r");
	if(fd==NULL)	return NULL;

	id=pds_load_number_of_columns_and_lines(fd,&nc,&nl);
	if(id==PDS_WRONG)	
	{
		fclose(fd);
		return NULL;
	}


	Matrix=pds_matrix_new(nl,nc);
	if(Matrix==NULL)	
	{
		fclose(fd);
		return NULL;
	}

	for(j=0;j<nl;j++)
	for(i=0;i<nc;i++)
	{
		x[0]=0;
		tmp=fscanf(fd,"%s",x);
		if((strlen(x)==0)||(tmp!=1))	break;

		Matrix->M[j][i]=atof(x);
	}
	
	fclose(fd);
	return Matrix;
}


/** \fn int pds_matrix_copy_matrix(PdsMatrix *Matrix,PdsRaNatural X1,PdsRaNatural Y1,const PdsMatrix *MatSrc,PdsRaNatural X2,PdsRaNatural Y2)
 *  \brief Copia en la matriz Matrix en la posicion (X1,Y1) la matriz MatSrc 
 *  desde la posicion (X2,Y2).
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix(X1,Y1)=MatSrc(X2,Y2)</b>.
 *  \param[in,out] Matrix La matriz a copiar.
 *  \param[in] X1 Linea en la que se copiará.
 *  \param[in] Y1 Columna en la que se copiará.
 *  \param[in] MatSrc Matriz que se usara como fuente.
 *  \param[in] X2 Linea que se copiará.
 *  \param[in] Y2 Columna que se copiará.
 *  \return PDS_OK si las matrices existen o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_copy_matrix(PdsMatrix *Matrix,PdsRaNatural X1,PdsRaNatural Y1,const PdsMatrix *MatSrc,PdsRaNatural X2,PdsRaNatural Y2)
{
	PdsRaNatural i,j;
	PdsRaNatural Nlin;
	PdsRaNatural Ncol;

	if(Matrix==NULL)	return PDS_WRONG;
	if(MatSrc==NULL)	return PDS_WRONG;

	if((Matrix->Nlin-X1) < (MatSrc->Nlin-X2))	Nlin=Matrix->Nlin-X1;
	else						Nlin=MatSrc->Nlin-X2;

	if((Matrix->Ncol-Y1) < (MatSrc->Ncol-Y2))	Ncol=Matrix->Ncol-Y1;
	else						Ncol=MatSrc->Ncol-Y2;

	for(i=0;i<Nlin;i++)
	{
		for(j=0;j<Ncol;j++)
		Matrix->M[i+X1][j+Y1]=MatSrc->M[i+X2][j+Y2];
	}

	return PDS_OK;
}


/** \fn int pds_matrix_copy_vector_col(PdsMatrix *Matrix,const PdsVector *VecSrc,PdsRaNatural col)
 *  \brief Copia vector VecSrc en la columna col de la matriz Matrix.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix(:,col)=VecSrc</b>.
 *  \param[in,out] Matrix La matriz a copiar.
 *  \param[in] VecSrc Vector que se usará como fuente.
 *  \param[in] col Columna a escribir.
 *  \return PDS_OK si las matrices existen o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_copy_vector_col(PdsMatrix *Matrix,const PdsVector *VecSrc,PdsRaNatural col)
{
	PdsRaNatural i;
	PdsRaNatural N;

	if(Matrix==NULL)	return PDS_WRONG;
	if(VecSrc==NULL)	return PDS_WRONG;
	if(col>=Matrix->Ncol)	return PDS_WRONG;

	if( Matrix->Nlin < VecSrc->Nel )	N=Matrix->Nlin;
	else					N=VecSrc->Nel;

	for(i=0;i<N;i++)	Matrix->M[i][col]=VecSrc->V[i];

	return PDS_OK;
}


/** \fn int pds_matrix_copy_vector_lin(PdsMatrix *Matrix,const PdsVector *VecSrc,PdsRaNatural lin)
 *  \brief Copia vector VecSrc en la linea lin de la matriz Matrix.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix(:,col)=VecSrc</b>.
 *  \param[in,out] Matrix La matriz a copiar.
 *  \param[in] VecSrc Vector que se usará como fuente.
 *  \param[in] lin Linea a escribir.
 *  \return PDS_OK si las matrices existen o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_copy_vector_lin(PdsMatrix *Matrix,const PdsVector *VecSrc,PdsRaNatural lin)
{
	PdsRaNatural i;
	PdsRaNatural N;

	if(Matrix==NULL)	return PDS_WRONG;
	if(VecSrc==NULL)	return PDS_WRONG;
	if(lin>=Matrix->Nlin)	return PDS_WRONG;

	if( Matrix->Ncol < VecSrc->Nel )	N=Matrix->Ncol;
	else					N=VecSrc->Nel;

	for(i=0;i<N;i++)	Matrix->M[lin][i]=VecSrc->V[i];

	return PDS_OK;
}

/** \fn int pds_matrix_copy_identity(PdsMatrix *Matrix,PdsRaNatural X,PdsRaNatural Y,PdsRaReal Val)
 *  \brief Copia en la matriz Matrix una matriz identidad(Val) en la posicion (X,Y).
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix(NLin,NCol)=MatSrc</b>.
 *  \param[in,out] Matrix La matriz a copiar.
 *  \param[in] X Linea en la que se copiará.
 *  \param[in] Y Columna en la que se copiará.
 *  \param[in] Val El valor que será escrito en la diagonal.
 *  \return PDS_OK si las matrices existen o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_copy_identity(PdsMatrix *Matrix,PdsRaNatural X,PdsRaNatural Y,PdsRaReal Val)
{
	PdsRaNatural i,j;
	PdsRaNatural N;

	if(Matrix==NULL)	return PDS_WRONG;

	if((Matrix->Nlin-X) < (Matrix->Ncol-Y))	N=(Matrix->Nlin-X);
	else					N=(Matrix->Ncol-Y);

	for(i=0;i<N;i++)
	{
		for(j=0;j<N;j++)
			if(i==j)	Matrix->M[i+X][j+Y]=Val;
			else		Matrix->M[i+X][j+Y]=0;
	}

	return PDS_OK;
}


/** \fn int pds_matrix_copy_region_to_region(PdsMatrix *M1,PdsRegionRect R1,const PdsMatrix *M0,PdsRegionRect R0)
 *  \brief Copia la región R0 de la matriz M0 a la región R1 de la matriz M1.
 *  
 *  Ambas regiones deben estar dentro de las matrices y tener el mismo tamaño
 *  para que la copia sea realizada. 
 *
 *  \param[out] M1 La matriz donde se escribirá.
 *  \param[in] R1 Región en donde se escribirá.
 *  \param[in] M0 Matriz que se usará como fuente.
 *  \param[in] R0 Región en donde se leerán los datos.
 *  \return PDS_OK si las matrices existen y son copiables o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_copy_region_to_region(PdsMatrix *M1,PdsRegionRect R1,const PdsMatrix *M0,PdsRegionRect R0)
{
    if(M0==NULL)    return PDS_WRONG;
    if(M1==NULL)    return PDS_WRONG;

    if(R0.Nlin!=R1.Nlin)    return PDS_WRONG;
    if(R0.Ncol!=R1.Ncol)    return PDS_WRONG;

    PdsRaNatural Nlin=R0.Nlin;
    PdsRaNatural Ncol=R0.Ncol;

    PdsRegionRect Rm0=pds_region_rect(0,0,M0->Nlin,M0->Ncol);
    PdsRegionRect Rm1=pds_region_rect(0,0,M1->Nlin,M1->Ncol);

    if(pds_region_rect_is_inside(R0,Rm0)!=PDS_OK)   return  PDS_WRONG;
    if(pds_region_rect_is_inside(R1,Rm1)!=PDS_OK)   return  PDS_WRONG;

    PdsRaNatural lin,col;

    for(lin=0;lin<Nlin;lin++)
    for(col=0;col<Ncol;col++)
    {
        M1->M[lin+R1.L0][col+R1.C0]=M0->M[lin+R0.L0][col+R0.C0];
    }

    return PDS_OK;
}

/** \fn int pds_matrix_init_identity(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Inicia la matriz de tipo puntero PdsMatrix con una diagonal de valor 
 *  Val y llena el resto de ceros. <br><b>Matrix=Val*I</b>.
 *  \param[in,out] Matrix La matriz a iniciar.
 *  \param[in] Val El valor que será escrito en la diagonal.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_init_identity(PdsMatrix *Matrix,PdsRaReal Val)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return PDS_WRONG;

	for(i=0;i<Matrix->Nlin;i++)
	{
		for(j=0;j<Matrix->Ncol;j++)
		{
			if(i==j)	Matrix->M[i][j]=Val;
			else		Matrix->M[i][j]=0;
		}
	}

	return PDS_OK;
}


/** \fn int pds_matrix_init_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
 *  \brief Inicia la matriz de tipo puntero PdsMatrix con una matriz.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix=MatSrc</b>.
 *  \param[in,out] Matrix La matriz a iniciar.
 *  \param[in] MatSrc Matriz que se usara como fuente.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_init_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
{
	PdsRaNatural i,j;
	PdsRaNatural Nlin;
	PdsRaNatural Ncol;

	if(Matrix==NULL)	return PDS_WRONG;
	if(MatSrc==NULL)	return PDS_WRONG;

	if(Matrix->Nlin < MatSrc->Nlin)	Nlin=Matrix->Nlin;
	else				Nlin=MatSrc->Nlin;

	if(Matrix->Ncol < MatSrc->Ncol)	Ncol=Matrix->Ncol;
	else				Ncol=MatSrc->Ncol;

	for(i=0;i<Nlin;i++)
	{
		for(j=0;j<Ncol;j++)
		Matrix->M[i][j]=MatSrc->M[i][j];
	}

	return PDS_OK;
}


/** \fn int pds_matrix_init_value(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Inicia la matriz de tipo puntero PdsMatrix con un valor. <br><b>Matrix=Val</b>.
 *  \param[in,out] Matrix La matriz a iniciar con un valor.
 *  \param[in] Val Es el valor inicial de los elementos.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_init_value(PdsMatrix *Matrix,PdsRaReal Val)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return PDS_WRONG;

	for(i=0;i<Matrix->Nlin;i++)
	{
		for(j=0;j<Matrix->Ncol;j++)
		Matrix->M[i][j]=Val;
	}

	return PDS_OK;
}

/** \fn int pds_matrix_init_rand(PdsMatrix *Matrix)
 *  \brief Inicia la matriz de tipo puntero PdsMatrix con valores aleatorios entre
 *  0.0 e 1.0. La función usa internamente la función rand().
 *  \param[in,out] Matrix La matriz a iniciar con un valor.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_init_rand(PdsMatrix *Matrix)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return PDS_WRONG;

	for(i=0;i<Matrix->Nlin;i++)
	for(j=0;j<Matrix->Ncol;j++)
    {
		Matrix->M[i][j]=rand()/(RAND_MAX+1.0);
	}

	return PDS_OK;
}

/** \fn int pds_matrix_init_region(PdsMatrix *Matrix,PdsRegionRect R,PdsRaReal val)
 *  \brief Inicializa la región R de matriz Matrix con el valor val. 
 *  \param[in,out] Matrix La matriz a iniciar con un valor.
 *  \param[in] R La region a inicializar.
 *  \param[in] val valor a ser usado en la inicialización.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_init_region(PdsMatrix *Matrix,PdsRegionRect R,PdsRaReal val)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return PDS_WRONG;

    PdsRegionRect Rm=pds_region_rect(0,0,Matrix->Nlin,Matrix->Ncol);
    if(pds_region_rect_is_inside(R,Rm)!=PDS_OK) return PDS_WRONG;

	for(i=0;i<R.Nlin;i++)
	for(j=0;j<R.Ncol;j++)
    {
		Matrix->M[R.L0+i][R.C0+j]=val;
	}

	return PDS_OK;
}


/** \fn int pds_matrix_add_identity(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Suma a la matriz de tipo puntero PdsMatrix con una diagonal de valor 
 *  Val y suma cero al resto. <br><b>Matrix=Matrix+Val*I</b>.
 *  \param[in,out] Matrix La matriz destino y operando de la suma.
 *  \param[in] Val El valor que será sumado a la diagonal.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_add_identity(PdsMatrix *Matrix,PdsRaReal Val)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return PDS_WRONG;

	for(i=0;i<Matrix->Nlin;i++)
	{
		for(j=0;j<Matrix->Ncol;j++)
		{
			if(i==j)	Matrix->M[i][j]=Matrix->M[i][j]+Val;
		}
	}

	return PDS_OK;
}



/** \fn int pds_matrix_add_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
 *  \brief Suma la matriz de tipo puntero PdsMatrix con una matriz MatSrc y lo 
 *  carga en Matrix. Si los tamaños son diferentes intersecta los tamaños y hace 
 *  la copia en la  intersección solamente. <br><b>Matrix=Matrix+MatSrc</b>.
 *  \param[in,out] Matrix La matriz que se usará como destino y operando.
 *  \param[in] MatSrc Matriz que se sumará a Matrix.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_add_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
{
	PdsRaNatural i,j;
	PdsRaNatural Nlin;
	PdsRaNatural Ncol;

	if(Matrix==NULL)	return PDS_WRONG;
	if(MatSrc==NULL)	return PDS_WRONG;

	if(Matrix->Nlin < MatSrc->Nlin)	Nlin=Matrix->Nlin;
	else				Nlin=MatSrc->Nlin;

	if(Matrix->Ncol < MatSrc->Ncol)	Ncol=Matrix->Ncol;
	else				Ncol=MatSrc->Ncol;

	for(i=0;i<Nlin;i++)
	{
		for(j=0;j<Ncol;j++)
		Matrix->M[i][j]=Matrix->M[i][j]+MatSrc->M[i][j];
	}

	return PDS_OK;
}


/** \fn int pds_matrix_add_value(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Suma a la matriz de tipo puntero PdsMatrix un valor. <br><b>Matrix=Matrix+Val</b>.
 *  \param[in,out] Matrix La matriz a la cual se le sumará un valor.
 *  \param[in] Val Es el valor que se sumará a los elementos.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_add_value(PdsMatrix *Matrix,PdsRaReal Val)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return PDS_WRONG;

	for(i=0;i<Matrix->Nlin;i++)
	{
		for(j=0;j<Matrix->Ncol;j++)
		Matrix->M[i][j]=Matrix->M[i][j]+Val;
	}

	return PDS_OK;
}


/** \fn int pds_matrix_sub_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
 *  \brief Resta la matriz de tipo puntero PdsMatrix con una matriz MatSrc y lo 
 *  carga en Matrix. Si los tamaños son diferentes intersecta los tamaños y hace 
 *  la copia en la  intersección solamente. <br><b>Matrix=Matrix-MatSrc</b>.
 *  \param[in,out] Matrix La matriz que se usará como destino y operando.
 *  \param[in] MatSrc Matriz que se sumará a Matrix.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_sub_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
{
	PdsRaNatural i,j;
	PdsRaNatural Nlin;
	PdsRaNatural Ncol;

	if(Matrix==NULL)	return PDS_WRONG;
	if(MatSrc==NULL)	return PDS_WRONG;

	if(Matrix->Nlin < MatSrc->Nlin)	Nlin=Matrix->Nlin;
	else				Nlin=MatSrc->Nlin;

	if(Matrix->Ncol < MatSrc->Ncol)	Ncol=Matrix->Ncol;
	else				Ncol=MatSrc->Ncol;

	for(i=0;i<Nlin;i++)
	{
		for(j=0;j<Ncol;j++)
		Matrix->M[i][j]=Matrix->M[i][j]-MatSrc->M[i][j];
	}

	return PDS_OK;
}


/** \fn int pds_matrix_mul_matrix_elements(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
 *  \brief Multiplica la matriz de tipo puntero PdsMatrix con una matriz MatSrc 
 *  elemento a elemento y lo carga en Matrix. Si los tamaños son diferentes 
 *  intersecta los tamaños y hace la copia en la  intersección solamente. <br><b>Matrix=Matrix.MatSrc</b>.
 *  \param[in,out] Matrix La matriz que se usará como destino y operando.
 *  \param[in] MatSrc Matriz que se multiplicará a Matrix.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_mul_matrix_elements(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
{
	PdsRaNatural i,j;
	PdsRaNatural Nlin;
	PdsRaNatural Ncol;

	if(Matrix==NULL)	return PDS_WRONG;
	if(MatSrc==NULL)	return PDS_WRONG;

	if(Matrix->Nlin < MatSrc->Nlin)	Nlin=Matrix->Nlin;
	else				Nlin=MatSrc->Nlin;

	if(Matrix->Ncol < MatSrc->Ncol)	Ncol=Matrix->Ncol;
	else				Ncol=MatSrc->Ncol;

	for(i=0;i<Nlin;i++)
	{
		for(j=0;j<Ncol;j++)
		Matrix->M[i][j]=Matrix->M[i][j]*MatSrc->M[i][j];
	}

	return PDS_OK;
}


/** \fn int pds_matrix_mul_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc1,const PdsMatrix *MatSrc2)
 *  \brief Multiplica la matriz MatSrc1 (Nlin1xNcol1) con la matriz MatSrc2 (Nlin2xNcol2)
 *  y lo carga en Matrix (Nlin1xNcol2). <br><b>Matrix=MatSrc1*MatSrc2</b>.
 *  \param[in,out] Matrix La matriz que se usará como destino, esta matriz ya debe 
 *  estar creada, todo su contenido será sobrescrito.
 *  \param[in] MatSrc1 Matriz que se multiplicará a MatSrc2.
 *  \param[in] MatSrc2 Matriz que se multiplicará a MatSrc1.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  Puede retornar PDS_WRONG si alguna de las matrices es NULL, o si el número de 
 *  lineas (Nlin1) de Matrix no corresponde al número de lineas (Nlin1) de MatSrc1, 
 *  o si el número de columnas (Ncol2) de Matrix no corresponde al número de 
 *  columnas (Ncol2) de MatSrc2, o si Ncol1!=Nlin2, e cualquiera de esos errores
 *  la matriz Matrix no es alterada.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_mul_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc1,const PdsMatrix *MatSrc2)
{
	PdsRaNatural i,j,k;
	PdsRaNatural Nlin;
	PdsRaNatural Ncol;
	PdsRaNatural N;
	PdsRaReal S;

	if(Matrix==NULL)	return PDS_WRONG;
	if(MatSrc1==NULL)	return PDS_WRONG;
	if(MatSrc2==NULL)	return PDS_WRONG;

	if(Matrix->Nlin  != MatSrc1->Nlin)	return PDS_WRONG;
	if(Matrix->Ncol  != MatSrc2->Ncol)	return PDS_WRONG;

	if(MatSrc1->Ncol != MatSrc2->Nlin)	return PDS_WRONG;

	Nlin=Matrix->Nlin;
	Ncol=Matrix->Ncol;

	N=MatSrc1->Ncol;

	for(i=0;i<Nlin;i++)
	{
		for(j=0;j<Ncol;j++)
		{
			S=0;
			for(k=0;k<N;k++)
			{
				S=S+MatSrc1->M[i][k]*MatSrc2->M[k][j];
			}
			Matrix->M[i][j]=S;
		}
	}

	return PDS_OK;
}


/** \fn PdsMatrix *pds_matrix_mul_matrix_new(const PdsMatrix *MatSrc1,const PdsMatrix *MatSrc2)
 *  \brief Multiplica la matriz MatSrc1 (Nlin1xNcol1) con la matriz MatSrc2 (Nlin2xNcol2)
 *  y devuelve una matriz Matrix (Nlin1xNcol2). <br><b>Matrix=MatSrc1*MatSrc2</b>.
 *  \param[in] MatSrc1 Matriz que se multiplicará a MatSrc2.
 *  \param[in] MatSrc2 Matriz que se multiplicará a MatSrc1.
 *  \return La multiplicación de las matrices si todo fue bien o NULL si no.
 *  Puede retornar NULL si alguna de las matrices es NULL, o si Ncol1!=Nlin2.
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_mul_matrix_new(const PdsMatrix *MatSrc1,const PdsMatrix *MatSrc2)
{
	PdsRaNatural i,j,k;
	PdsRaNatural Nlin;
	PdsRaNatural Ncol;
	PdsRaNatural N;
	PdsRaReal S;

	PdsMatrix *Matrix=NULL;

	if(MatSrc1==NULL)	return NULL;
	if(MatSrc2==NULL)	return NULL;

	if(MatSrc1->Ncol != MatSrc2->Nlin)	return NULL;

	Nlin=MatSrc1->Nlin;
	Ncol=MatSrc2->Ncol;

	N   =MatSrc1->Ncol;

	Matrix=pds_matrix_new(Nlin,Ncol);
	if(Matrix==NULL)	return NULL;

	for(i=0;i<Nlin;i++)
	{
		for(j=0;j<Ncol;j++)
		{
			S=0;
			for(k=0;k<N;k++)
			{
				S=S+MatSrc1->M[i][k]*MatSrc2->M[k][j];
			}
			Matrix->M[i][j]=S;
		}
	}

	return Matrix;
}


/** \fn int pds_matrix_mul_vector(PdsVector *A,const PdsMatrix *M,const PdsVector *V)
 *  \brief Multiplica la matriz M con el vector V y devuelve un vector A. 
 *  <br><b>A=M*V</b>.
 *  \param[out] A Vector donde se cargará la multiplicación.
 *  \param[in] M Matriz que se multiplicará.
 *  \param[in] V Vector que se multiplicará.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  Puede retornar PDS_WRONG si la matriz no tiene el mismo número de columnas que elementos
 *  tiene V. O si el numero de elementos de A es distinto al número de lineas de M.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_mul_vector(PdsVector *A,const PdsMatrix *M,const PdsVector *V)
{
	PdsRaNatural i,j;

	if(M==NULL)	return PDS_WRONG;
	if(V==NULL)	return PDS_WRONG;
	if(A==NULL)	return PDS_WRONG;

	if(M->Ncol!=V->Nel)	return PDS_WRONG;
	if(M->Nlin!=A->Nel)	return PDS_WRONG;

	for(i=0;i<M->Nlin;i++)
	{
		A->V[i]=0;
		for(j=0;j<M->Ncol;j++)
		{
			A->V[i]=A->V[i]+M->M[i][j]*V->V[j];
		}
	}
	return PDS_OK;
}


/** \fn PdsVector *pds_matrix_mul_vector_new(const PdsMatrix *M,const PdsVector *V)
 *  \brief Multiplica la matriz M con el vector V y devuelve un vector A. 
 *  <br><b>A=M*V</b>.
 *  \param[in] M Matriz que se multiplicará.
 *  \param[in] V Vector que se multiplicará.
 *  \return La multiplicación de la matriz por el vector si todo fue bien o NULL si no.
 *  Puede retornar NULL si la matriz no tiene el mismo número de columnas que elementos
 *  tiene V. O si el numero de elementos de A es distinto al número de lineas de M.
 *  \ingroup PdsMatrixGroup
 */
PdsVector *pds_matrix_mul_vector_new(const PdsMatrix *M,const PdsVector *V)
{
	PdsRaNatural i,j;
	PdsVector *A=NULL;

	if(M==NULL)	return NULL;
	if(V==NULL)	return NULL;

	if(M->Ncol!=V->Nel)	return NULL;

	A=pds_vector_new(M->Nlin);
	if(A==NULL)	return NULL;

	for(i=0;i<M->Nlin;i++)
	{
		A->V[i]=0;
		for(j=0;j<M->Ncol;j++)
		{
			A->V[i]=A->V[i]+M->M[i][j]*V->V[j];
		}
	}
	return A;
}


/** \fn int pds_matrix_mul_value(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Multiplica a la matriz de tipo puntero PdsMatrix por un valor.  <br><b>Matrix=Val*Matrix</b>.
 *  \param[in,out] Matrix La matriz a la cual se le multiplicará un valor.
 *  \param[in] Val Es el valor que se sumará a los elementos.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_mul_value(PdsMatrix *Matrix,PdsRaReal Val)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return PDS_WRONG;

	for(i=0;i<Matrix->Nlin;i++)
	{
		for(j=0;j<Matrix->Ncol;j++)
		Matrix->M[i][j]=Matrix->M[i][j]*Val;
	}

	return PDS_OK;
}


/** \fn PdsMatrix *pds_matrix_transpose_new(const PdsMatrix *M)
 *  \brief Genera la transpuesta de la matriz.  <br><b>Mt=M^T</b>.
 *  \param[in] M La matriz a la cual se le aplicará la transpuesta.
 *  \return La transpuesta de la matriz o NULL en caso de error.(ej. M==NULL).
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_transpose_new(const PdsMatrix *M)
{
	PdsRaNatural i,j;
	PdsMatrix *Mt=NULL;

	if(M==NULL)	return NULL;

	Mt=pds_matrix_new(M->Ncol,M->Nlin);
	if(Mt==NULL)	return NULL;

	for(i=0;i<M->Ncol;i++)
	{
		for(j=0;j<M->Nlin;j++)
		{
			Mt->M[i][j]=M->M[j][i];
		}
	}
	return Mt;
}


/** \fn int pds_matrix_transpose(PdsMatrix *Mt,const PdsMatrix *M)
 *  \brief Genera la transpuesta de la matriz.  <br><b>Mt=M^T</b>.
 *  \param[in] Mt La transpuesta de la matriz M.
 *  \param[in] M La matriz a la cual se le aplicará la transpuesta.
 *  \return RUE si todo fue bien o PDS_WRONG si no.(ej. M==NULL Mt==NULL).
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_transpose(PdsMatrix *Mt,const PdsMatrix *M)
{
	PdsRaNatural i,j;

	if(M==NULL)	return PDS_WRONG;
	if(Mt==NULL)	return PDS_WRONG;

	if(M->Nlin!=Mt->Ncol)	return PDS_WRONG;
	if(M->Ncol!=Mt->Nlin)	return PDS_WRONG;

	for(i=0;i<M->Ncol;i++)
	{
		for(j=0;j<M->Nlin;j++)
		{
			Mt->M[i][j]=M->M[j][i];
		}
	}
	return PDS_OK;
}


/** \fn PdsMatrix *pds_matrix_mtm_new(const PdsMatrix *M)
 *  \brief Genera  <br><b>M^T*M</b>.
 *
 *  \f[ M^T M \f]
 *  \param[in] M La matriz a la cual se le aplicará la operación.
 *  \return La <br><b>M^T * M</b> de la matriz o NULL en caso de error.(ej. M==NULL).
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_mtm_new(const PdsMatrix *M)
{
	PdsRaNatural i,j,k;
	PdsMatrix *MtM=NULL;
	PdsRaReal S;

	if(M==NULL)	return NULL;

	MtM=pds_matrix_new(M->Ncol,M->Ncol);
	if(MtM==NULL)	return NULL;

	for(i=0;i<M->Ncol;i++)
	{
		for(j=0;j<=i;j++)
		{
			S=0;
			for(k=0;k<M->Nlin;k++)
			{
				S=S+M->M[k][i]*M->M[k][j];
			}
			MtM->M[i][j]=S;
		}
	}

	for(i=0;i<M->Ncol;i++)
	{
		for(j=i+1;j<M->Ncol;j++)
		{
			MtM->M[i][j]=MtM->M[j][i];
		}
	}

	return MtM;
}


/** \fn PdsMatrix *pds_matrix_mtm_ai_new(const PdsMatrix *M,PdsRaReal Alpha)
 *  \brief Genera  <br><b>M^T*M + Alpha*I</b>.
 *
 *  \f[ M^T M + \alpha I \f]
 *  \param[in] M La matriz a la cual se le aplicará la operación.
 *  \param[in] Alpha Factor que se multiplicará a la identidad.
 *  \return La <br><b>M^T * M + Alpha * I</b> de la matriz o NULL en caso de 
 *  error.(ej. M==NULL).
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_mtm_ai_new(const PdsMatrix *M,PdsRaReal Alpha)
{
	PdsRaNatural i,j,k;
	PdsMatrix *MtMaI=NULL;
	PdsRaReal S;

	if(M==NULL)	return NULL;

	MtMaI=pds_matrix_new(M->Ncol,M->Ncol);
	if(MtMaI==NULL)	return NULL;

	for(i=0;i<M->Ncol;i++)
	{
		for(j=0;j<=i;j++)
		{
			S=0;
			for(k=0;k<M->Nlin;k++)
			{
				S=S+M->M[k][i]*M->M[k][j];
			}
			MtMaI->M[i][j]=S;
		}
	}

	for(i=0;i<M->Ncol;i++)
	{
		for(j=i+1;j<M->Ncol;j++)
		{
			MtMaI->M[i][j]=MtMaI->M[j][i];
		}
	}

	for(i=0;i<M->Ncol;i++)
	{
			MtMaI->M[i][i]=MtMaI->M[i][i]+Alpha;
	}


	return MtMaI;
}


/** \fn int pds_matrix_linear_least_square(const PdsVector *U,const PdsMatrix *H,PdsVector *Z)
 *  \brief Encuentra Z de modo que intenta satifazer <b>U=HZ</b>.
 *  Para conseguir esto usa la siguiente asignación :
 *  <br><b>Z<=(H'H)^(-1)H'U</b>
 *
 *  \f[ e^2=||U - H Z||^2 \f]
 *  \f[ \frac{\partial e^2}{\partial Z}=0 \f]
 *  \f[ Z\leftarrow ( H^T H )^{-1} H^T U \f]
 *  \param[in] U Valor de salida del sistema de ecuaciones <b>U=HZ</b>.
 *  \param[in] H Matriz del sistema.
 *  \param[out] Z Incognita del sistema, aquí es cargado el resultado.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. (ej U==NULL,H==NULL o Z==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_linear_least_square(const PdsVector *U,const PdsMatrix *H,PdsVector *Z)
{
	PdsRaNatural i,j;
	PdsMatrix *HtH=NULL;
	PdsMatrix *HtH1=NULL;
	PdsVector *HtU=NULL;

	if(U->Nel!=H->Nlin)	return PDS_WRONG;
	if(Z->Nel!=H->Ncol)	return PDS_WRONG;

	HtH=pds_matrix_mtm_new(H);
	if(HtH==NULL)	return PDS_WRONG;

	HtH1=pds_matrix_inverse_matrix_new(HtH);
	if(HtH1==NULL)	
	{
		pds_matrix_free(HtH);
		return PDS_WRONG;
	}

	pds_matrix_free(HtH);

	HtU=pds_vector_new (H->Ncol);
	if(HtU==NULL)	
	{
		pds_matrix_free(HtH);
		pds_matrix_free(HtH1);
		return PDS_WRONG;
	}

	for(i=0;i<H->Ncol;i++)
	{
		HtU->V[i]=0;
		for(j=0;j<H->Nlin;j++)
		HtU->V[i]=HtU->V[i]+H->M[j][i]*U->V[j];
	}

	pds_matrix_mul_vector(Z,HtH1,HtU);

	pds_matrix_free(HtH1);
	pds_vector_free(HtU);

	return PDS_OK;
}


/** \fn int pds_matrix_tikhonov_nolinear_least_square(const PdsVector *U,const PdsMatrix *J,const PdsVector *F,PdsRaReal Alpha,PdsVector *Z)
 *  \brief Encuentra Z iterativamente de modo que intenta satifazer <b>U=F(Z)</b>.
 *  Usa la siguiente equacion iterativa:
 *  <br><b>Z <= Z + (J'J + Alpha I)^(-1)(J'(U-F(Z))-Alpha Z)</b>.
 *
 *  \f[ e^2={||U-F(Z)||}^2+\alpha {||Z-Z_{old}||}^2 \f]
 *  \f[ \frac{\partial e^2}{\partial Z}=0 \f]
 *  \f[ Z \leftarrow Z+ {\left[ {J(Z)}^T {J(Z)} + \alpha I\right]}^{-1} \left[{J(Z)}^T (U-F(Z))-\alpha Z\right] \f]
 *  \param[in] U Valor de salida del sistema de ecuaciones <b>U=HZ</b>.
 *  \param[in] J matriz jacobiano de F con respecto a Z.
 *  \param[in] F Vector que depende de Z.
 *  \param[in] Alpha Factor de regularización.
 *  \param[out] Z Incognita del sistema, aquí es cargado el resultado.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. (ej U==NULL,H==NULL o Z==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_tikhonov_nolinear_least_square(const PdsVector *U,const PdsMatrix *J,const PdsVector *F,PdsRaReal Alpha,PdsVector *Z)
{
	PdsRaNatural i,j;

	PdsMatrix *JtJaI=NULL;
	PdsMatrix *JtJaI1=NULL;
	////////////////////////////////////////////////////////////////////////////
	PdsVector *JtUFaZ=NULL;
	////////////////////////////////////////////////////////////////////////////
	PdsVector *V=NULL;
	PdsRaReal S;
	
	if(U==NULL)	return PDS_WRONG;
	if(J==NULL)	return PDS_WRONG;
	if(F==NULL)	return PDS_WRONG;
	if(Z==NULL)	return PDS_WRONG;

	if(U->Nel!=F->Nel)	return PDS_WRONG;
	if(U->Nel!=J->Nlin)	return PDS_WRONG;
	if(Z->Nel!=J->Ncol)	return PDS_WRONG;

	JtJaI=pds_matrix_mtm_ai_new(J,Alpha);
	if(JtJaI==NULL)	return PDS_WRONG;

	JtJaI1=pds_matrix_inverse_matrix_new(JtJaI);
	pds_matrix_free(JtJaI);
	if(JtJaI1==NULL)	return PDS_WRONG;
	
	////////////////////////////////////////////////////////////////////////////

	JtUFaZ=pds_vector_new(J->Ncol);
	if(JtUFaZ==NULL)	
	{
		pds_matrix_free(JtJaI1);            
		return PDS_WRONG;
	}

	for(i=0;i<J->Ncol;i++)
	{
		S=-Alpha*Z->V[i];
		for(j=0;j<J->Nlin;j++)
		{
			S = S + J->M[j][i] * (U->V[j]-F->V[j]);
		}

		JtUFaZ->V[i]=S;
	}
	
	////////////////////////////////////////////////////////////////////////////
    	
	V=pds_matrix_mul_vector_new(JtJaI1,JtUFaZ);
	pds_matrix_free(JtJaI1);	
	pds_vector_free(JtUFaZ);
	if(V==NULL)	return PDS_WRONG;
	
	pds_vector_add_vector(Z,V);  

	return PDS_OK;
}


/** \fn int pds_matrix_print(const PdsMatrix *Matrix)
 *  \brief Imprime en pantalla una matriz de tipo puntero PdsMatrix.
 *  \param[in] Matrix La matriz a imprimir en pantalla.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_printf(const PdsMatrix *Matrix)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return PDS_WRONG;
	//printf("\n");
	for(i=0;i<Matrix->Nlin;i++)
	{
		for(j=0;j<Matrix->Ncol;j++)
		{
			printf("%e",Matrix->M[i][j]);
			
			if(j==(Matrix->Ncol-1))	printf("\n");
			else			printf("\t");
		}
	}
	return PDS_OK;
}


/** \fn int pds_matrix_fprintf(const PdsMatrix *Matrix,FILE *fd)
 *  \brief Imprime en el archivo que apunta fd una matriz de tipo puntero PdsMatrix.
 *  \param[in] Matrix La matriz a imprimir en fd.
 *  \param[in,out] fd El puntero al archivo.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_fprintf(const PdsMatrix *Matrix,FILE *fd)
{
	PdsRaNatural i,j;

	if(Matrix==NULL)	return PDS_WRONG;
	if(fd==NULL)		return PDS_WRONG;

	//fprintf(fd,"\n");
	for(i=0;i<Matrix->Nlin;i++)
	{
		for(j=0;j<Matrix->Ncol;j++)
		{
			fprintf(fd,"%e",Matrix->M[i][j]);
			
			if(j==(Matrix->Ncol-1))	fprintf(fd,"\n");
			else			fprintf(fd,"\t");
		}
	}
	return PDS_OK;
}


/** \fn int pds_matrix_fscanf(PdsMatrix *Matrix, FILE *fd)
 *  \brief Inicializa una matriz con los datos del archivo apuntado por fd.
 *  Usa TAB o un salto de linea como delimitador de elemento.
 *  \param[in] Matrix Matriz en donde se cargaran los datos.
 *  \param[in] fd Apuntador del archivo de donde se cargará los datos iniciales
 *  de la matriz.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. (ej. Matrix==NULL o fd==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_fscanf(PdsMatrix *Matrix, FILE *fd)
{
	PdsRaNatural i=0,j=0;
	int id;
	int tmp;
	char x[__MAX_CHAR_DATA_SIZE__];
	
	if(Matrix==NULL)	return PDS_WRONG;
	if(fd==NULL)		return PDS_WRONG;

	id=PDS_OK;

	for(j=0;(j<Matrix->Nlin)&&(feof(fd)==0);j++)
	{
		for(i=0;(i<Matrix->Ncol)&&(feof(fd)==0);i++)
		{
			do{
				x[0]=0;
				tmp=fscanf(fd,"%s",x);
			}while( ((strlen(x)==0)||(tmp!=1)) && (feof(fd)==0) );

			if(feof(fd)!=0)	{id=PDS_WRONG;	break;}

			Matrix->M[j][i]=atof(x);
		}

		if(feof(fd)!=0)	{id=PDS_WRONG;	break;}
	}

    if(i==Matrix->Ncol) i=0;

	for(;j<Matrix->Nlin;j++)
	{
		for(;i<Matrix->Ncol;i++)
		{
			Matrix->M[j][i]=0.0;
		}
	}

	return id;
}


/** \fn int pds_matrix_fwrite(PdsMatrix *Matrix, FILE *fd)
 *  \brief Escribe los datos de una matriz en el archivo binario apuntado por fd.
 *  \param[in] Matrix Matrix de donde se leeran los datos.
 *  \param[in] fd Apuntador del archivo binario de donde se escribiran los datos 
 *  de la matriz.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. (ej. Matrix==NULL o fd==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_fwrite(PdsMatrix *Matrix, FILE *fd)
{
	size_t n;
	PdsRaNatural i;
	
	if(Matrix==NULL)	return PDS_WRONG;
	if(fd==NULL)		return PDS_WRONG;

	for(i=0;i<Matrix->Nlin;i++)
	{
		n=fwrite(Matrix->M[i],sizeof(Matrix->M[i][0]),Matrix->Ncol,fd);
	
		if(n!=Matrix->Ncol)	return PDS_WRONG;
	}
	return PDS_OK;
}


/** \fn int pds_matrix_fread(PdsMatrix *Matrix, FILE *fd)
 *  \brief Inicializa una matriz con los datos del archivo binario apuntado por fd.
 *  \param[in] Matrix Matriz en donde se cargaran los datos.
 *  \param[in] fd Apuntador del archivo binario de donde se cargará los datos 
 *  iniciales de la matriz.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. (ej. Matrix==NULL o fd==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_fread(PdsMatrix *Matrix, FILE *fd)
{
	PdsRaNatural i=0,j=0;
	size_t n;
	
	if(Matrix==NULL)	return PDS_WRONG;
	if(fd==NULL)		return PDS_WRONG;

	for(j=0;(j<Matrix->Nlin)&&(feof(fd)==0);j++)
	{
		n=0;
		for(i=0;(i<Matrix->Ncol)&&(feof(fd)==0);i++)
		{
			n=fread(&(Matrix->M[j][i]),sizeof(Matrix->M[j][0]),1,fd);
			if(n!=1)	break;
		}
		if(n!=1)	break;
	}

    if(i==Matrix->Ncol) i=0;

	for(;j<Matrix->Nlin;j++)
	{
		for(;i<Matrix->Ncol;i++)
		{
			Matrix->M[j][i]=0.0;
			
		}
	}

	return PDS_OK;
}


/** \fn int pds_matrix_save_rgb_bmp(const PdsMatrix *R,const PdsMatrix *G,const PdsMatrix *B,const char *bmpfilename)
 *  \brief Escribe los datos de una matriz en un archivo de en formato BMP.
 *  \param[in] R Matriz donde se leerán los datos de color rojo. 0<= r<= 255
 *  \param[in] G Matriz donde se leerán los datos de color verde. 0<= g<= 255
 *  \param[in] B Matriz donde se leerán los datos de color azul. 0<= b<= 255
 *  \param[in] bmpfilename Nombre del archivo donde se escribirán los datos 
 *  de la matriz.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. (ej. R,G,B,bmpfilename==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_save_rgb_bmp(const PdsMatrix *R,const PdsMatrix *G,const PdsMatrix *B,const char *bmpfilename)
{
    FILE *bmpfd=NULL;

    unsigned char *img = NULL;
    double r,g,b;

    int x,y,i,j;
    int WIDTH  = R->Ncol;
    int HEIGHT = R->Nlin;
    int FILESIZE = 54 + 3*WIDTH*HEIGHT; 

    if(R==NULL) return PDS_WRONG;
    if(G==NULL) return PDS_WRONG;
    if(B==NULL) return PDS_WRONG;
    if(bmpfilename==NULL) return PDS_WRONG;

    img = (unsigned char *)calloc(3*WIDTH*HEIGHT,sizeof(unsigned char));
    if(img==NULL)   return PDS_WRONG;

    for(j=0; j<HEIGHT; j++)
    for(i=0; i<WIDTH ; i++)
    {
        x=i; 
        y=j;

        r = R->M[j][i];    if (r > 255) r=255;
        g = G->M[j][i];    if (g > 255) g=255;
        b = B->M[j][i];    if (b > 255) b=255;
        
        img[(x+y*WIDTH)*3+2] = (unsigned char)(r);
        img[(x+y*WIDTH)*3+1] = (unsigned char)(g);
        img[(x+y*WIDTH)*3+0] = (unsigned char)(b);
    }

    ///////////////////////////////// Cabecera //////////////////////////////////
    // [ 2, 6> tamaño total del archivo (calculalo luego)
    // [ 6,10> puros cero
    // [10,14> Tamanho total de la cabecera
    unsigned char bmpfileheader[14] = {'B','M',   0,0,0,0,  0,0,0,0,  54,0,0,0};

    bmpfileheader[ 2] = (unsigned char)(FILESIZE    );
    bmpfileheader[ 3] = (unsigned char)(FILESIZE>> 8);
    bmpfileheader[ 4] = (unsigned char)(FILESIZE>>16);
    bmpfileheader[ 5] = (unsigned char)(FILESIZE>>24);

    /////////////////////////// Cabecera informativa ////////////////////////////
    // [ 0, 4> Tamanho total de informacion de la cabecera
    // [ 4, 8> Ancho en pixels de la imagen
    // [ 8,12> Alto en pixels de la imagen
    unsigned char bmpinfoheader[40] = {40,0,0,0,  0,0,0,0,  0,0,0,0,  1,0, 24,0};

    bmpinfoheader[ 4] = (unsigned char)(WIDTH    );
    bmpinfoheader[ 5] = (unsigned char)(WIDTH>> 8);
    bmpinfoheader[ 6] = (unsigned char)(WIDTH>>16);
    bmpinfoheader[ 7] = (unsigned char)(WIDTH>>24);

    bmpinfoheader[ 8] = (unsigned char)(HEIGHT    );
    bmpinfoheader[ 9] = (unsigned char)(HEIGHT>> 8);
    bmpinfoheader[10] = (unsigned char)(HEIGHT>>16);
    bmpinfoheader[11] = (unsigned char)(HEIGHT>>24);

    unsigned char bmppad[3] = {0,0,0};

    bmpfd = fopen("img.bmp","wb");
    if(bmpfd==NULL)
    {
        free(img);
        return PDS_WRONG;
    }
    fwrite(bmpfileheader,1,14,bmpfd);
    fwrite(bmpinfoheader,1,40,bmpfd);
    for(i=0; i<HEIGHT; i++)
    {
        fwrite(img+(WIDTH*i*3),3,WIDTH            ,bmpfd);
        fwrite(bmppad         ,1,(4-(WIDTH*3)%4)%4,bmpfd);
    }
    fclose(bmpfd);
    return PDS_OK;
}


/** \fn int pds_matrix_save_grayscale_bmp(const PdsMatrix *Mat,const char *bmpfilename)
 *  \brief Escribe los datos de una matriz en un archivo de en formato BMP.
 *
 *  <center>
 *  \image html pds_matrix_save_grayscale_bmp.bmp "grafico usando escala de grises"
 *  </center>
 *  \param[in] Mat Matriz donde se leerán los datos de escala de gris. 0<= r<= 255
 *  \param[in] bmpfilename Nombre del archivo donde se escribirán los datos 
 *  de la matriz.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. (ej. Mat,bmpfilename==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_save_grayscale_bmp(const PdsMatrix *Mat,const char *bmpfilename)
{
    FILE * bmpfd=NULL;

    unsigned char *img = NULL;
    unsigned char UCHAR;

    int i,j;

    if(Mat==NULL)   return PDS_WRONG;
    if(bmpfilename==NULL)   return PDS_WRONG;

    //float Maximo=0;
    //float Minimo=0;
    //maximo y minimo de la matriz
    /*
    for(i=0;i<HEIGHT;i++)
    for(j=0;j<WIDTH;j++)
    {
        if(Maximo<Data[i][j]) Maximo=Data[i][j];
        if(Minimo>Data[i][j]) Minimo=Data[i][j];
    }
    */
    int WIDTH  = Mat->Ncol;
    int HEIGHT = Mat->Nlin;

    int FILESIZE=54+(int)WIDTH*HEIGHT+1024;

   

    img = (unsigned char *)calloc(WIDTH*HEIGHT,sizeof(unsigned char));
    if(img==NULL)
    {
        return PDS_WRONG;
    }

    for(j=0; j<HEIGHT; j++)
    for(i=0; i<WIDTH ; i++)
    {

        if      (Mat->M[j][i] > 255) UCHAR = 255;
        else if (Mat->M[j][i] <   0) UCHAR = 0;
        else                         UCHAR = (unsigned char)Mat->M[j][i];
        
        img[i+j*WIDTH] = UCHAR;
    }

    ///////////////////////////////// Paleta ///////////////////////////////////
    unsigned char paleta[1024];
    for(i=0;i<256;i++)
    {
        paleta[i*4+0]=i;
        paleta[i*4+1]=i;
        paleta[i*4+2]=i;
        paleta[i*4+3]=0;
    }
    ////////////////////////////////////////////////////////////////////////////


    ///////////////////////////////// Cabecera //////////////////////////////////
    // [ 2, 6> tamaño total del archivo (calculalo luego)
    // [ 6,10> puros cero
    // [10,14> Tamanho total de la cabecera (1024+54)
    unsigned char bmpfileheader[14] = {'B','M',  0,0,0,0,  0,0,0,0,  54,4,0,0};
   
    bmpfileheader[ 2] = (unsigned char)(FILESIZE    );
    bmpfileheader[ 3] = (unsigned char)(FILESIZE>> 8);
    bmpfileheader[ 4] = (unsigned char)(FILESIZE>>16);
    bmpfileheader[ 5] = (unsigned char)(FILESIZE>>24);
    ////////////////////////////////////////////////////////////////////////////


    /////////////////////////// Cabecera informativa ////////////////////////////
    // [ 0, 4> Tamanho total de informacion de la cabecera
    // [ 4, 8> Ancho en pixels de la imagen
    // [ 8,12> Alto en pixels de la imagen
    // [12,14> Numero de Planos ¿?
    // [14,16> Numero de bits por pixel (1, 4, 8, 16, 24 o 32.)
    // [16,20> Sin compresion
    unsigned char bmpinfoheader[40] = {40,0,0,0,  0,0,0,0,  0,0,0,0,  1,0,  8,0,  0,0,0,0};
    // [32,36> 100h=256 numero de colores
    // [36,40> 100h=256 numero de colores

    bmpinfoheader[ 4] = (unsigned char)(WIDTH    );
    bmpinfoheader[ 5] = (unsigned char)(WIDTH>> 8);
    bmpinfoheader[ 6] = (unsigned char)(WIDTH>>16);
    bmpinfoheader[ 7] = (unsigned char)(WIDTH>>24);

    bmpinfoheader[ 8] = (unsigned char)(HEIGHT    );
    bmpinfoheader[ 9] = (unsigned char)(HEIGHT>> 8);
    bmpinfoheader[10] = (unsigned char)(HEIGHT>>16);
    bmpinfoheader[11] = (unsigned char)(HEIGHT>>24);

    bmpinfoheader[32] = (unsigned char)(0);
    bmpinfoheader[33] = (unsigned char)(1);

    bmpinfoheader[36] = (unsigned char)(0);
    bmpinfoheader[37] = (unsigned char)(1);
    ////////////////////////////////////////////////////////////////////////////

    unsigned char bmppad[3] = {0,0,0};

    bmpfd = fopen(bmpfilename, "wb");
    if (bmpfd == NULL) 
    {
        free(img);
        return PDS_WRONG;
    }
    
    fwrite(bmpfileheader, 1,  14, bmpfd);
    fwrite(bmpinfoheader, 1,  40, bmpfd);
    fwrite(paleta       , 1,1024, bmpfd);


    for(i=0;i<HEIGHT;i++)
    {
        fwrite(img+(WIDTH*i),1,WIDTH          ,bmpfd);
        fwrite(bmppad       ,1,(4-(WIDTH%4))%4,bmpfd);
    }

    fclose(bmpfd);

    return 0;
}




/** \fn int pds_matrix_save_bmp_with_colormap(const PdsMatrix *Mat,const char *bmpfilename,const unsigned char colormap[256][3])
 *  \brief Escribe los datos de una matriz en un archivo de en formato BMP.
 *
 *  <center>
 *  \image html pds_matrix_save_bmp_with_colormap_jet.bmp "grafico usando el colormap PDS_COLORMAP_JET."
 *  </center>
 *  \param[in] Mat Matriz donde se leerán los datos de escala de gris. 0<= r<= 255
 *  \param[in] bmpfilename Nombre del archivo donde se escribirán los datos 
 *  \param[in] colormap Mapa de colores. Ejemplo: PDS_COLORMAP_JET, PDS_COLORMAP_BONE,
 *  PDS_COLORMAP_HOT,PDS_COLORMAP_JOLLY.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. (ej. Mat,bmpfilename==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_save_bmp_with_colormap(const PdsMatrix *Mat,const char *bmpfilename,const unsigned char colormap[256][3])
{
    FILE * bmpfd=NULL;

    unsigned char *img = NULL;
    unsigned char UCHAR;

    int i,j;

    if(Mat==NULL)   return PDS_WRONG;
    if(bmpfilename==NULL)   return PDS_WRONG;

    //float Maximo=0;
    //float Minimo=0;
    //maximo y minimo de la matriz
    /*
    for(i=0;i<HEIGHT;i++)
    for(j=0;j<WIDTH;j++)
    {
        if(Maximo<Data[i][j]) Maximo=Data[i][j];
        if(Minimo>Data[i][j]) Minimo=Data[i][j];
    }
    */
    int WIDTH  = Mat->Ncol;
    int HEIGHT = Mat->Nlin;

    int FILESIZE=54+(int)WIDTH*HEIGHT+1024;

   

    img = (unsigned char *)calloc(WIDTH*HEIGHT,sizeof(unsigned char));
    if(img==NULL)
    {
        return PDS_WRONG;
    }

    for(j=0; j<HEIGHT; j++)
    for(i=0; i<WIDTH ; i++)
    {

        if      (Mat->M[j][i] > 255) UCHAR = 255;
        else if (Mat->M[j][i] <   0) UCHAR = 0;
        else                         UCHAR = (unsigned char)Mat->M[j][i];
        
        img[i+j*WIDTH] = UCHAR;
    }

    ///////////////////////////////// Paleta ///////////////////////////////////
    unsigned char paleta[1024];
    for(i=0;i<256;i++)
    {
        paleta[i*4+0]=colormap[i][2];
        paleta[i*4+1]=colormap[i][1];
        paleta[i*4+2]=colormap[i][0];
        paleta[i*4+3]=0;
    }
    ////////////////////////////////////////////////////////////////////////////


    ///////////////////////////////// Cabecera //////////////////////////////////
    // [ 2, 6> tamaño total del archivo (calculalo luego)
    // [ 6,10> puros cero
    // [10,14> Tamanho total de la cabecera (1024+54)
    unsigned char bmpfileheader[14] = {'B','M',  0,0,0,0,  0,0,0,0,  54,4,0,0};
   
    bmpfileheader[ 2] = (unsigned char)(FILESIZE    );
    bmpfileheader[ 3] = (unsigned char)(FILESIZE>> 8);
    bmpfileheader[ 4] = (unsigned char)(FILESIZE>>16);
    bmpfileheader[ 5] = (unsigned char)(FILESIZE>>24);
    ////////////////////////////////////////////////////////////////////////////


    /////////////////////////// Cabecera informativa ////////////////////////////
    // [ 0, 4> Tamanho total de informacion de la cabecera
    // [ 4, 8> Ancho en pixels de la imagen
    // [ 8,12> Alto en pixels de la imagen
    // [12,14> Numero de Planos ¿?
    // [14,16> Numero de bits por pixel (1, 4, 8, 16, 24 o 32.)
    // [16,20> Sin compresion
    unsigned char bmpinfoheader[40] = {40,0,0,0,  0,0,0,0,  0,0,0,0,  1,0,  8,0,  0,0,0,0};
    // [32,36> 100h=256 numero de colores
    // [36,40> 100h=256 numero de colores

    bmpinfoheader[ 4] = (unsigned char)(WIDTH    );
    bmpinfoheader[ 5] = (unsigned char)(WIDTH>> 8);
    bmpinfoheader[ 6] = (unsigned char)(WIDTH>>16);
    bmpinfoheader[ 7] = (unsigned char)(WIDTH>>24);

    bmpinfoheader[ 8] = (unsigned char)(HEIGHT    );
    bmpinfoheader[ 9] = (unsigned char)(HEIGHT>> 8);
    bmpinfoheader[10] = (unsigned char)(HEIGHT>>16);
    bmpinfoheader[11] = (unsigned char)(HEIGHT>>24);

    bmpinfoheader[32] = (unsigned char)(0);
    bmpinfoheader[33] = (unsigned char)(1);

    bmpinfoheader[36] = (unsigned char)(0);
    bmpinfoheader[37] = (unsigned char)(1);
    ////////////////////////////////////////////////////////////////////////////

    unsigned char bmppad[3] = {0,0,0};

    bmpfd = fopen(bmpfilename, "wb");
    if (bmpfd == NULL) 
    {
        free(img);
        return PDS_WRONG;
    }
    
    fwrite(bmpfileheader, 1,  14, bmpfd);
    fwrite(bmpinfoheader, 1,  40, bmpfd);
    fwrite(paleta       , 1,1024, bmpfd);


    for(i=0;i<HEIGHT;i++)
    {
        fwrite(img+(WIDTH*i),1,WIDTH          ,bmpfd);
        fwrite(bmppad       ,1,(4-(WIDTH%4))%4,bmpfd);
    }

    fclose(bmpfd);

    return 0;
}


/** \fn pds_matrix_octave_surf(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *imgfile)
 *  \brief Imprime una matriz M en un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave. Usando el comando surf.
 *
 *  <center>
 *  \image html pds_matrix_octave_surf.png "Usando la función surf() de Octave."
 *  </center>
 *  \param[in] M Matriz a imprimir.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] labelz Etiqueta del eje Z: Altura.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si octfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_octave_surf(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *imgfile)
{
	int i,j;
	FILE *fd=NULL;
	char *orden=NULL;
    int FONTSIZE=24;

	if(M==NULL)		return PDS_WRONG;
	if(labelx==NULL)	return PDS_WRONG;
	if(labely==NULL)	return PDS_WRONG;
	if(labelz==NULL)	return PDS_WRONG;
	if(imgfile==NULL)	return PDS_WRONG;

    char *OCTAVEFILE=NULL;
    if(octfile==NULL)
    {
        OCTAVEFILE=(char*)calloc(64,sizeof(char));
        if(OCTAVEFILE==NULL)    return PDS_WRONG;
        sprintf(OCTAVEFILE,"temp%d.m",rand());
    }
    else
    {
        OCTAVEFILE=strdup(octfile);
        if(OCTAVEFILE==NULL)    return PDS_WRONG;
    }



	fd=fopen(OCTAVEFILE,"w");
	if(fd==NULL)		return PDS_WRONG;
	fprintf(fd,"M=[");
	for(i=0 ; i<M->Nlin ; i++)	
	{
		for(j=0 ; j<M->Ncol ; j++)
		{
			if(j!=M->Ncol-1)	fprintf(fd,"%e\t",M->M[i][j]);
			else			fprintf(fd,"%e\n",M->M[i][j]);
		}
	}
	fprintf(fd,"];\n");
	fprintf(fd,"h=figure(1);\n");// h=figure('Visible', 'off');
	fprintf(fd,"surf(M);\n");
	fprintf(fd,"shading interp;\n");
	fprintf(fd,"grid on;\n");

	fprintf(fd,"hx=xlabel(\'%s\');\n",labelx);
	fprintf(fd,"hy=ylabel(\'%s\');\n",labely);
	fprintf(fd,"hz=zlabel(\'%s\');\n",labelz);
    fprintf(fd,"ha = gca(); \n");
    fprintf(fd,"FONTSIZE=%d;\n",FONTSIZE);
    fprintf(fd,"set(hx,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(hy,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(hz,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(ha,'fontsize',FONTSIZE);\n");

    fprintf(fd,"refresh();\n");
	fprintf(fd,"saveas (h,\'%s\');\n",imgfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,48+strlen(OCTAVEFILE));
	if(orden==NULL)	return PDS_WRONG;
	sprintf(orden,"octave --no-gui -q %s",OCTAVEFILE);
	i=system(orden);

    if(octfile==NULL)
    {
        remove(OCTAVEFILE); 
    }
    free(OCTAVEFILE);
	return PDS_OK;
}


/** \fn int pds_matrix_get_value(const PdsMatrix *Matrix, PdsRaNatural x,PdsRaNatural y, PdsRaReal *m)
 *  \brief Devuelve el valor en la posición (x,y) de la matriz  Matrix.
 *  (x,y) inicia con (0,0).
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[in] x Linea de posición x, el primer valor es cero.
 *  \param[in] y Columna de posición y, el primer valor es cero.
 *  \param[out] m El valor en la posición (x,y), en caso de error por fuera de rango 
 *  (x,y) entonces carga 0, en caso de error de matriz nula carga cero.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_value(const PdsMatrix *Matrix, PdsRaNatural x,PdsRaNatural y, PdsRaReal *m)
{
	*m=0;

	if(Matrix==NULL)	return PDS_WRONG;

	if((x>=0)&&(y>=0)&&(x<Matrix->Nlin)&&(y<Matrix->Ncol))	*m=Matrix->M[x][y];
	else							*m=0;

	return PDS_OK;
}


/** \fn int pds_matrix_set_value(PdsMatrix *Matrix, PdsRaNatural x,PdsRaNatural y,PdsRaReal m)
 *  \brief Escribe el valor en la posición (x,y) de la matriz  Matrix.
 *  En caso de que (x,y) estuvieran fuera de rango, no se considera como error
 *  simplemente no se escribe nada. (x,y) inicia con (0,0).
 *  \param[in,out] Matrix La matriz a modificar.
 *  \param[in] x Linea de posición x, el primer valor es cero.
 *  \param[in] y Columna de posición y, el primer valor es cero.
 *  \param[in] m Valor real que se colocara en (x,y).
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_set_value(PdsMatrix *Matrix, PdsRaNatural x,PdsRaNatural y,PdsRaReal m)
{
	//PdsRaReal z=0;

	if(Matrix==NULL)	return PDS_WRONG;

	if((x>=0)&&(y>=0)&&(x<Matrix->Nlin)&&(y<Matrix->Ncol))	Matrix->M[x][y]=m;

	return PDS_OK;
}


/** \fn int pds_matrix_get_nlines(const PdsMatrix *Matrix, PdsRaNatural *Nlines)
 *  \brief Devuelve el número de lineas de la matriz  Matrix.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] Nlines En donde se guardará el número de lineas de la matriz  Matrix.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_nlines(const PdsMatrix *Matrix, PdsRaNatural *Nlines)
{
	*Nlines=0;

	if(Matrix==NULL)	return PDS_WRONG;

	*Nlines=Matrix->Nlin;

	return PDS_OK;
}


/** \fn int pds_matrix_get_ncolumns(const PdsMatrix *Matrix, PdsRaNatural *Ncolumns)
 *  \brief Devuelve el número de columnas de la matriz  Matrix.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] Ncolumns En donde se guardará el número de columnas de la matriz  Matrix.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_ncolumns(const PdsMatrix *Matrix, PdsRaNatural *Ncolumns)
{
	*Ncolumns=0;

	if(Matrix==NULL)	return PDS_WRONG;

	*Ncolumns=Matrix->Ncol;

	return PDS_OK;
}


/** \fn int pds_matrix_get_region_rect(const PdsMatrix *Mat,PdsRegionRect *R)
 *  \brief Carga la variable PdsRegionRect desde la posicion (0,0), con ancho y alto (Mat->Nlin,Mat->Ncol).
 *  \param[in] Mat matriz de donde se generará la region.
 *  \param[in] R La region que representa la matriz Mat.
 *  \return PDS_WRONG Si Mat==NULL o R==NULL, sino retorna PDS_OK.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_region_rect(const PdsMatrix *Mat,PdsRegionRect *R)
{

	if(Mat==NULL)	return PDS_WRONG;
	if(  R==NULL)	return PDS_WRONG;

    (*R).L0=0;
    (*R).C0=0;

    (*R).Nlin=Mat->Nlin;
    (*R).Ncol=Mat->Ncol;


    return PDS_OK;
}

/** \fn int pds_matrix_get_min_value(const PdsMatrix *Matrix, PdsRaReal *m)
 *  \brief Devuelve el valor mínimo de los elementos de la matriz  Matrix.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] m El valor mínimo de los elementos de la matriz  Matrix.
 *  Si este parámetro es NULL la función retorna PDS_WRONG.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_min_value(const PdsMatrix *Matrix, PdsRaReal *m)
{
	return pds_matrix_get_min_value_and_pos(Matrix,m,NULL,NULL);
}


/** \fn int pds_matrix_get_min_value_and_pos(const PdsMatrix *Matrix, PdsRaReal *m,PdsRaNatural *lin,PdsRaNatural *col)
 *  \brief Devuelve el valor mínimo de los elementos de la matriz  Matrix.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] m El valor mínimo de los elementos de la matriz  Matrix.
 *  Si este parámetro es NULL la función retorna PDS_WRONG.
 *  \param[out] lin La linea del valor mínimo de los elementos de la matriz  Matrix.
 *  Si este parámetro es NULL la función retorna PDS_OK.
 *  \param[out] col La columna del valor mínimo de los elementos de la matriz  Matrix.
 *  Si este parámetro es NULL la función retorna PDS_OK.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL o m==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_min_value_and_pos(const PdsMatrix *Matrix, PdsRaReal *m,PdsRaNatural *lin,PdsRaNatural *col)
{
	PdsRaNatural i,j;
	if(     m==NULL)	return PDS_WRONG;
    *m=0;
	if(Matrix==NULL)	return PDS_WRONG;

	*m=Matrix->M[0][0];
    if(lin!=NULL)   *lin=0;
    if(col!=NULL)   *col=0;

	for(i=0;i<Matrix->Nlin;i++)
		for(j=0;j<Matrix->Ncol;j++)
			if(*m > Matrix->M[i][j])
            {
            	*m=Matrix->M[i][j];
                if(lin!=NULL)   *lin=i;
                if(col!=NULL)   *col=j;
            }

	return PDS_OK;
}


/** \fn int pds_matrix_get_max_value(const PdsMatrix *Matrix, PdsRaReal *m)
 *  \brief Devuelve el valor máximo de los elementos de la matriz  Matrix.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] m El valor máximo de los elementos de la matriz  Matrix.
 *  Si este parámetro es NULL la función retorna PDS_WRONG.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_max_value(const PdsMatrix *Matrix, PdsRaReal *m)
{
	return pds_matrix_get_max_value_and_pos(Matrix,m,NULL,NULL);
}


/** \fn int pds_matrix_get_max_value_and_pos(const PdsMatrix *Matrix, PdsRaReal *m,PdsRaNatural *lin,PdsRaNatural *col)
 *  \brief Devuelve el valor máximo de los elementos de la matriz  Matrix.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] m El valor máximo de los elementos de la matriz Matrix.
 *  Si este parámetro es NULL la función retorna PDS_WRONG.
 *  \param[out] lin La linea del valor máximo de los elementos de la matriz Matrix.
 *  Si este parámetro es NULL la función retorna PDS_OK.
 *  \param[out] col La columna del valor máximo de los elementos de la matriz Matrix.
 *  Si este parámetro es NULL la función retorna PDS_OK.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL o m==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_max_value_and_pos(const PdsMatrix *Matrix, PdsRaReal *m,PdsRaNatural *lin,PdsRaNatural *col)
{
	PdsRaNatural i,j;
	if(     m==NULL)	return PDS_WRONG;
    *m=0;
	if(Matrix==NULL)	return PDS_WRONG;

	*m=Matrix->M[0][0];
    if(lin!=NULL)   *lin=0;
    if(col!=NULL)   *col=0;

	for(i=0;i<Matrix->Nlin;i++)
		for(j=0;j<Matrix->Ncol;j++)
			if(*m < Matrix->M[i][j])
            {
            	*m=Matrix->M[i][j];
                if(lin!=NULL)   *lin=i;
                if(col!=NULL)   *col=j;
            }

	return PDS_OK;
}

/** \fn int pds_matrix_swap_rows(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaNatural r2)
 *  \brief Intercambia dos lineas de la matriz  Matrix.
 *  \param[in,out] Matrix La matriz de en consulta.
 *  \param[in] r1 La linea a intercambiar con r2.
 *  \param[in] r2 La linea a intercambiar con r1.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL). Puede retornar
 *  PDS_WRONG por causa de r1>=Nlin, r2>=Nlin o Matrix==NULL. En todos los casos de 
 *  error Matrix no es alterado.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_swap_rows(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaNatural r2)
{
	PdsRaNatural j;
	PdsRaReal tmp;

	if(Matrix==NULL)	return PDS_WRONG;
	if(r1>=Matrix->Nlin)	return PDS_WRONG;
	if(r2>=Matrix->Nlin)	return PDS_WRONG;

	for(j=0;j<Matrix->Ncol;j++)
	{
		tmp=Matrix->M[r1][j];
		Matrix->M[r1][j]=Matrix->M[r2][j];
		Matrix->M[r2][j]=tmp;
	}

	return PDS_OK;
}


/** \fn int pds_matrix_row_add_row_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaNatural r2, PdsRaReal factor)
 *  \brief Multiplica la linea r2 por un factor y el resulltado lo resta a linea r1.
 *  <br><b>Matrix{r1}=Matrix{r1}+factor*Matrix{r2}</b>.
 *  \param[in,out] Matrix La matriz de en consulta.
 *  \param[in] r1 La linea r1.
 *  \param[in] r2 La linea r2.
 *  \param[in] factor Factor que multiplicará a la linea r2.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL). Puede retornar
 *  PDS_WRONG por causa de r1>=Nlin, r2>=Nlin o Matrix==NULL. En todos los casos de 
 *  error Matrix no es alterado.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_row_add_row_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaNatural r2, PdsRaReal factor)
{
	PdsRaNatural j;

	if(Matrix==NULL)	return PDS_WRONG;
	if(r1>=Matrix->Nlin)	return PDS_WRONG;
	if(r2>=Matrix->Nlin)	return PDS_WRONG;

	for(j=0;j<Matrix->Ncol;j++)
	{
		Matrix->M[r1][j]=Matrix->M[r1][j]+factor*Matrix->M[r2][j];
	}

	return PDS_OK;
}


/** \fn int pds_matrix_row_mul_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaReal factor)
 *  \brief Multiplica la linea r1 por un factor y el resulltado lo carga a linea r1.
 *  <br><b>Matrix{r1}=Matrix{r1}*factor</b>.
 *  \param[in,out] Matrix La matriz de en consulta.
 *  \param[in] r1 La linea r1.
 *  \param[in] factor Factor que multiplicará a la linea r1.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL). Puede retornar
 *  PDS_WRONG por causa de r1>=Nlin o Matrix==NULL. En todos los casos de 
 *  error Matrix no es alterado.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_row_mul_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaReal factor)
{
	PdsRaNatural j;

	if(Matrix==NULL)	return PDS_WRONG;
	if(r1>=Matrix->Nlin)	return PDS_WRONG;

	for(j=0;j<Matrix->Ncol;j++)
	{
		Matrix->M[r1][j]=Matrix->M[r1][j]*factor;
	}

	return PDS_OK;
}


/** \fn int pds_matrix_row_div_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaReal factor)
 *  \brief Divide la linea r1 por un factor y el resulltado lo carga a linea r1.
 *  <br><b>Matrix{r1}=Matrix{r1}/factor</b>.
 *  \param[in,out] Matrix La matriz de en consulta.
 *  \param[in] r1 La linea r1.
 *  \param[in] factor Factor que dividirá a la linea r1.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL). Puede retornar
 *  PDS_WRONG por causa de r1>=Nlin o Matrix==NULL. En todos los casos de 
 *  error Matrix no es alterado.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_row_div_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaReal factor)
{
	PdsRaNatural j;

	if(Matrix==NULL)	return PDS_WRONG;
	if(r1>=Matrix->Nlin)	return PDS_WRONG;

	for(j=0;j<Matrix->Ncol;j++)
	{
		Matrix->M[r1][j]=Matrix->M[r1][j]/factor;
	}

	return PDS_OK;
}


/** \fn int pds_matrix_reduced_matrix(const PdsMatrix *Matrix, PdsMatrix *MatReduced)
 *  \brief Reduce la matriz Matrix y la guarda en MatReduced. 
 *  <br><b>MatReduced=Reduced{Matrix}</b>.
 *
 *  Una matriz es reducida cuando:
 *  Es una matriz escalonada.
 *  Sus pivotes son todos iguales a 1
 *  En cada fila el pivote es el único elemento no nulo de su columna
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] MatReduced Matriz reducida.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL). Puede retornar
 *  PDS_WRONG por causa de MatReduced tenga distinto tamaño que Matrix, 
 *  MatReduced==NULL o Matrix==NULL. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_reduced_matrix(const PdsMatrix *Matrix, PdsMatrix *MatReduced)
{
	PdsRaNatural i,j,k;
	PdsRaNatural r,E,N;
	//PdsRaReal factor;
	int id;

	if(Matrix==NULL)	return PDS_WRONG;
	if(MatReduced==NULL)	return PDS_WRONG;

	if(MatReduced->Nlin!=Matrix->Nlin)	return PDS_WRONG;
	if(MatReduced->Ncol!=Matrix->Ncol)	return PDS_WRONG;

	if(MatReduced->Ncol<MatReduced->Nlin)	N=MatReduced->Ncol;
	else					N=MatReduced->Nlin;

	id=pds_matrix_init_matrix(MatReduced,Matrix);
	if(id==PDS_WRONG)	return PDS_WRONG;

	for(i=0,j=0;(i<MatReduced->Nlin)&&(j<MatReduced->Ncol);j++)
	{
		r=0;E=1;
		// Como el primer elemento de la linea es cero busco otra que no 
		// lo sea, si no existe paso a la siguiente columna.
		if(MatReduced->M[i][j]==0)
		{
			// Esta función carga con E=1 encontró una linea para 
			// intercambiar y carga en r la linea.
			// Devuelve E=0 si no encontró ninguna linea
			id=pds_matrix_find_row_in_col(MatReduced,&r,&E,j,i+1);
			if(id==PDS_WRONG)	return PDS_WRONG;

			// Intercambia la linea con r.
			if(E==1) 
			{
				id=pds_matrix_swap_rows(MatReduced,r,i);
				if(id==PDS_WRONG)	return PDS_WRONG;
			}
		}
		// E==1 Indica que M[i][j]!=0, da igual si fue con ayuda de swap.
		// E==0 Indica que M[k][j]==0     i<= k <Nlin.
		if(E==1)
		{
			// Divido la linea en estudio para que inicie con 1.
			id=pds_matrix_row_div_factor(MatReduced,i,MatReduced->M[i][j]);
			if(id==PDS_WRONG)	return PDS_WRONG;

			for(k=i+1;k<MatReduced->Nlin;k++)
			{
				if(MatReduced->M[k][j]!=0)
				{
					id=pds_matrix_row_add_row_factor(MatReduced,k,i,-MatReduced->M[k][j]);
					if(id==PDS_WRONG)	return PDS_WRONG;
					//MatReduced->M[k][j]=0;//Esto porque no tiene buena precisión.
				}
			}
			i++;
		}
	}

	// Verifica si el ultimo elemento de la diagonal es distinto de cero
	// sino no tiene sentido seguir.
	if(MatReduced->M[N-1][N-1]==0)	return PDS_WRONG;

	// Comienza a restar desde la última fila, de manera que quede una diagonal
	// iniciando con 1 y seguido de ceros, hasta que termine la mayor matriz 
	// cuadrada que puede ser contenida por MatReduced.
	for(i=0;i<N;i++)
	{
		// Solo por si acaso verifico si cada elemento de la diagonal es
		// distinto de cero.
		if(MatReduced->M[(N-1)-i][(N-1)-i]!=0)
		{
			// Opero con las lineas para generar ceros en todas
			// las lineas arriba de la linea i.
			for(k=0;k<(N-1-i);k++)
			{
				id=pds_matrix_row_add_row_factor(MatReduced,k,(N-1)-i,-MatReduced->M[k][(N-1)-i]);
				if(id==PDS_WRONG)	return PDS_WRONG;
			}
		}
		else return PDS_WRONG;
	}

	return PDS_OK;
}


/** \fn int pds_matrix_mean(const PdsMatrix *Mat, PdsRaReal *mean)
 *  \brief Calcula la media de los elementos de la matriz.
 *  <br><b>mean=MEAN{Mat}</b>.
 *
 *  \f[  mean=\mu_{Mat}=\frac{\sum \limits_{i=0}^{N-1} Mat\{i\}}{N-1}\f]
 *
 *  \param[in] Mat La matriz a calcular.
 *  \param[out] mean Media.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Mat==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_mean(const PdsMatrix *Mat, PdsRaReal *mean)
{
	PdsRaNatural i,j;
	PdsRaReal S=0;

	if(Mat==NULL)   return PDS_WRONG;
	if(mean==NULL)	return PDS_WRONG;

	for(i=0;i<Mat->Nlin;i++)
	for(j=0;j<Mat->Ncol;j++)
    {
        S=S+Mat->M[i][j];
    }

    *mean=S/(Mat->Nlin*Mat->Ncol);

    return PDS_OK;
}

/** \fn int pds_matrix_std(const PdsMatrix *Mat, PdsRaReal *std)
 *  \brief Calcula el desvío padrón de los elementos de la matriz.
 *  <br><b>std=STD{Mat}</b>.
 *
 *  \f[  std=\sigma_{Mat}=\sqrt{\frac{\sum \limits_{i=0}^{N-1} (Mat\{i\}-\mu_{Mat})^2}{N-1}}\f]
 *
 *  \param[in] Mat La matriz a calcular.
 *  \param[out] std Desvío padrón.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Mat==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_std(const PdsMatrix *Mat, PdsRaReal *std)
{
	int ID;
    PdsRaReal mean;

    ID=pds_matrix_std_and_mean(Mat,std,&mean);

    return ID;
}


/** \fn int pds_matrix_std_and_mean(const PdsMatrix *Mat, PdsRaReal *std, PdsRaReal *mean)
 *  \brief Calcula el desvío padrón y la media de los elementos de la matriz.
 *  <br><b>std=STD{Mat}</b>, <br><b>mean=MEAN{Mat}</b>.
 *
 *  \f[  std=\sigma_{Mat}=\sqrt{\frac{\sum \limits_{i=0}^{N-1} (Mat\{i\}-\mu_{Mat})^2}{N-1}}\f]
 *
 *  \f[  mean=\mu_{Mat}=\frac{\sum \limits_{i=0}^{N-1} Mat\{i\}}{N-1}\f]
 *
 *  \param[in] Mat La matriz a calcular.
 *  \param[out] std Desvío padrón.
 *  \param[out] mean Media.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Mat==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_std_and_mean(const PdsMatrix *Mat, PdsRaReal *std, PdsRaReal *mean)
{
	PdsRaNatural i,j;
	PdsRaReal S=0;

	if(Mat ==NULL)  return PDS_WRONG;
	if(std ==NULL)	return PDS_WRONG;
	if(mean==NULL)	return PDS_WRONG;

    pds_matrix_mean(Mat,mean);

	for(i=0;i<Mat->Nlin;i++)
	for(j=0;j<Mat->Ncol;j++)
    {
        S=S+(Mat->M[i][j]-(*mean))*(Mat->M[i][j]-(*mean));
    }

    *std=sqrt(S/(Mat->Nlin*Mat->Ncol));

    return PDS_OK;
}

/** \fn int pds_matrix_corr_matrix(const PdsMatrix *M0,const PdsMatrix *M1, PdsRaReal *pcc)
 *  \brief Calcula el coeficiente de correlación de Pearson (PCC) entre dos matrices.
 *  <br><b>PCC=corr{M0,M1}</b>.
 *
 *  \f[  PCC=\frac{E[(M_0-\mu_{M_0})(M_1-\mu_{M_1})]}{\sigma_{M_0}\sigma_{M_1}} \f]
 *
 *  \param[in] M0 Una matriz de la correlación.
 *  \param[in] M1 Una matriz de la correlación.
 *  \param[out] pcc Coeficiente de correlación de Pearson.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: M0==NULL). Retorna
 *  PDS_WRONG por causa de que M0 y M1 tengan tamaños distintos.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_corr_matrix(const PdsMatrix *M0,const PdsMatrix *M1, PdsRaReal *pcc)
{
	PdsRaNatural i,j;
	PdsRaReal S=0;

    PdsRaReal mean0;
    PdsRaReal mean1;

    PdsRaReal std0;
    PdsRaReal std1;

	if(M0==NULL)    return PDS_WRONG;
	if(M1==NULL)    return PDS_WRONG;
	if(pcc==NULL)	return PDS_WRONG;

    if(M0->Nlin!=M1->Nlin)  return PDS_WRONG;
    if(M0->Ncol!=M1->Ncol)  return PDS_WRONG;

    int Nlin=M0->Nlin;
    int Ncol=M0->Ncol;

    pds_matrix_mean(M0,&mean0);
    pds_matrix_mean(M1,&mean1);

    pds_matrix_std(M0,&std0);
    pds_matrix_std(M1,&std1);


	for(i=0;i<Nlin;i++)
	for(j=0;j<Ncol;j++)
    {
        S=S+(M0->M[i][j]-mean0)*(M1->M[i][j]-mean1);
    }


    *pcc=S/(Nlin*Ncol*std0*std1);

    return PDS_OK;
}



/** \fn int pds_matrix_mean_of_region(const PdsMatrix *Mat,PdsRegionRect Rin, PdsRaReal *mean)
 *  \brief Calcula la media de los elementos de la intersecion de la region con la matriz.
 *  <br><b>mean=MEAN{Mat II Rin}</b>.
 *
 *  \f[  mean=\mu_{Mat}=\frac{\sum \limits_{i\in Rin} Mat\{i\}}{N-1}\f]
 *
 *  \param[in] Mat La matriz usar.
 *  \param[in] Rin La region a calcular.
 *  \param[out] mean Media.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Mat==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_mean_of_region(const PdsMatrix *Mat,PdsRegionRect Rin, PdsRaReal *mean)
{
	PdsRaNatural i,j;
	PdsRaReal S=0;

	if(Mat==NULL)   return PDS_WRONG;
	if(mean==NULL)	return PDS_WRONG;

    PdsRegionRect Rmat=pds_region_rect(0,0,Mat->Nlin,Mat->Ncol);
    PdsRegionRect R;   
    int ID=pds_region_rect_intersection(Rmat,Rin,&R);
    if(ID==PDS_WRONG)   return PDS_WRONG;

	for(i=0;i<R.Nlin;i++)
	for(j=0;j<R.Ncol;j++)
    {
        S=S+Mat->M[R.L0+i][R.C0+j];
    }

    *mean=S/(R.Nlin*R.Ncol);

    return PDS_OK;
}


/** \fn int pds_matrix_std_of_region(const PdsMatrix *Mat,PdsRegionRect Rin, PdsRaReal *std)
 *  \brief Calcula el desvío padrón de los elementos de la intersección de la región con la matriz.
 *  <br><b>std=STD{Mat II Rin}</b>.
 *
 *  \f[  std=\sigma_{Mat}=\sqrt{\frac{\sum \limits_{i\in Rin} (Mat\{i\}-\mu_{Mat})^2}{N-1}}\f]
 *
 *  \param[in] Mat La matriz usar.
 *  \param[in] Rin La región a calcular.
 *  \param[out] std Desvío padrón.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Mat==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_std_of_region(const PdsMatrix *Mat,PdsRegionRect Rin, PdsRaReal *std)
{
    PdsRaReal mean;
	int ID=pds_matrix_std_and_mean_of_region(Mat,Rin,std,&mean);

    return ID;
}

/** \fn int pds_matrix_std_and_mean_of_region(const PdsMatrix *Mat,PdsRegionRect Rin, PdsRaReal *std, PdsRaReal *mean)
 *  \brief Calcula el desvío padrón y la media de los elementos de la intersección de la región con la matriz.
 *  <br><b>std=STD{Mat II Rin}</b>, <br><b>mean=MEAN{Mat II Rin}</b>.
 *
 *  \f[  std=\sigma_{Mat}=\sqrt{\frac{\sum \limits_{i\in Rin} (Mat\{i\}-\mu_{Mat})^2}{N-1}}\f]
 *
 *  \f[  mean=\mu_{Mat}=\frac{\sum \limits_{i\in Rin} Mat\{i\}}{N-1}\f]
 *
 *  \param[in] Mat La matriz usar.
 *  \param[in] Rin La región a calcular.
 *  \param[out] std Desvío padrón.
 *  \param[out] mean Media.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Mat==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_std_and_mean_of_region(const PdsMatrix *Mat,PdsRegionRect Rin, PdsRaReal *std, PdsRaReal *mean)
{
	PdsRaNatural i,j;
	PdsRaReal S=0;

	if(Mat==NULL)   return PDS_WRONG;
	if(std==NULL)	return PDS_WRONG;
	if(mean==NULL)	return PDS_WRONG;

    PdsRegionRect R;
    PdsRegionRect Rmat=pds_region_rect(0,0,Mat->Nlin,Mat->Ncol);
    int ID=pds_region_rect_intersection(Rmat,Rin,&R);
    if(ID==PDS_WRONG)   return PDS_WRONG;

    pds_matrix_mean_of_region(Mat,R,mean);

	for(i=0;i<R.Nlin;i++)
	for(j=0;j<R.Ncol;j++)
    {
        S=S+(Mat->M[R.L0+i][R.C0+j]-(*mean))*(Mat->M[R.L0+i][R.C0+j]-(*mean));
    }

    *std=sqrt(S/(R.Nlin*R.Ncol));

    return PDS_OK;
}



/** \fn int pds_matrix_corr_matrix_of_region(const PdsMatrix *M0,const PdsMatrix *M1,PdsRegionRect R0,PdsRegionRect R1, PdsRaReal *pcc)
 *  \brief Calcula el coeficiente de correlación de Pearson (PCC) entre los
 *  elementos de la intersección de la regiones con sus matrices. 
 *
 *  El tamaño de las regiones son re-dimensionadas al ancho y alto de menor 
 *  longitud en las regiones; solo se retorna PDS_WRONG cuando el ancho o el alto de las
 *  regiones sea cero. 
 *
 *  Las regiones rectangulares son definidas como un punto en la esquina superior
 *  izquierda mas el alto y ancho.
 *
 *  \f[ S_0= M_0 \cap R_0\f]
 *
 *  \f[ S_1= M_1 \cap R_1\f]
 *
 *  \f[ WIDTH\{S_0,S_1\} \leftarrow  MIN(WIDTH\{S_0\},WIDTH\{S_1\}) \f]
 *
 *  \f[ HEIGHT\{S_0,S_1\} \leftarrow  MIN(HEIGHT\{S_0\},HEIGHT\{S_1\}) \f]
 *
 *  \f[  PCC=\frac{E[(S_0-\mu_{S_0})(S_1-\mu_{S_1})]}{\sigma_{S_0}\sigma_{S_1}} \f]
 *
 *  Si solo un desvío padrón es cero entonces la correlación se considera 0.0.
 *  Si ambos desvío padrón son cero entonces la correlación se considera 
 *  1.0 si los signos de las medias son iguales y son diferentes de cero, 
 *  -1.0 si los signos de las medias son diferentes y son diferentes de cero, 
 *  0.0 se alguna media es cero y
 *  1.0 se ambas medias son cero.
 *
 *  \param[in] M0 Una matriz de la correlación.
 *  \param[in] M1 Una matriz de la correlación.
 *  \param[in] R0 La región a analizar en la matriz M0.
 *  \param[in] R1 La región a analizar en la matriz M1.
 *  \param[out] pcc Coeficiente de correlación de Pearson.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: M0==NULL, pcc==NULL).
 *  Retorna PDS_WRONG cuando algún lado de las regiones finales a analizar son cero.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_corr_matrix_of_region(const PdsMatrix *M0,const PdsMatrix *M1,PdsRegionRect R0,PdsRegionRect R1, PdsRaReal *pcc)
{
	PdsRaNatural i,j;
	PdsRaReal S=0;

    int ID;

    PdsRaReal mean0;
    PdsRaReal mean1;
    PdsRaReal std0;
    PdsRaReal std1;

	if(M0==NULL)    return PDS_WRONG;
	if(M1==NULL)    return PDS_WRONG;
	if(pcc==NULL)	return PDS_WRONG;

    PdsRegionRect Rout0;
    PdsRegionRect Rout1;

    PdsRegionRect Rm0=pds_region_rect(0,0,M0->Nlin,M0->Ncol);
    PdsRegionRect Rm1=pds_region_rect(0,0,M1->Nlin,M1->Ncol);  

    ID=pds_region_rect_intersection(R0,Rm0,&Rout0);
    if(ID==PDS_WRONG)   return PDS_WRONG;
  
    ID=pds_region_rect_intersection(R1,Rm1,&Rout1);
    if(ID==PDS_WRONG)   return PDS_WRONG;


    if(Rout0.Nlin<Rout1.Nlin)   Rout1.Nlin=Rout0.Nlin;
    else                        Rout0.Nlin=Rout1.Nlin;

    if(Rout0.Ncol<Rout1.Ncol)   Rout1.Ncol=Rout0.Ncol;
    else                        Rout0.Ncol=Rout1.Ncol;


    if(Rout0.Nlin==0)   return PDS_WRONG;
    if(Rout0.Ncol==0)   return PDS_WRONG;

    pds_matrix_std_and_mean_of_region(M0,Rout0,&std0,&mean0);
    pds_matrix_std_and_mean_of_region(M1,Rout1,&std1,&mean1);
    
    if( ((std0==0.0)&&(std1!=0.0))||((std0!=0.0)&&(std1==0.0)) )
    {
        *pcc=0.0;
        return PDS_OK;
    }

    if( (std0==0.0)&&(std1==0.0) )
    {
        if((mean0*mean1)>0.0)         *pcc=+1.0;
        else if((mean0*mean1)==0.0)   *pcc=0.0;
        else                          *pcc=-1.0;

        if((mean0==0.0)&&(mean1==0.0))   *pcc=1.0;
        return PDS_OK;
    }

	for(i=0;i<Rout0.Nlin;i++)
	for(j=0;j<Rout0.Ncol;j++)
    {
        S=S+(M0->M[Rout0.L0+i][Rout0.C0+j]-mean0)*(M1->M[Rout1.L0+i][Rout1.C0+j]-mean1);
    }

    *pcc=S/(Rout0.Nlin*Rout0.Ncol*std0*std1);

    return PDS_OK;
}


/** \fn int pds_matrix_inverse_matrix(const PdsMatrix *Matrix, PdsMatrix *MatInv)
 *  \brief Invierte la matriz Matrix y la guarda en MatInv. Ambas matrices deben
 *  existir y tener el mismo tamanho. <br><b>MatInv=Matrix^{-1}</b>.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] MatInv Matriz inversa.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL). Puede retornar
 *  PDS_WRONG por causa de MatInv tenga distinto tamaño que Matrix, 
 *  MatInv==NULL o Matrix==NULL. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_inverse_matrix(const PdsMatrix *Matrix, PdsMatrix *MatInv)
{
	PdsRaNatural i,j,k;
	PdsRaNatural r,E,N;
	PdsRaReal factor;
	PdsMatrix *Mat=NULL;
	int id;

	if(Matrix==NULL)	return PDS_WRONG;
	if(MatInv==NULL)	return PDS_WRONG;

	if(MatInv->Nlin!=Matrix->Nlin)	return PDS_WRONG;
	if(MatInv->Nlin!=Matrix->Ncol)	return PDS_WRONG;
	if(MatInv->Ncol!=Matrix->Nlin)	return PDS_WRONG;
	if(MatInv->Ncol!=Matrix->Ncol)	return PDS_WRONG;

	N=MatInv->Nlin;

	Mat=pds_matrix_new_matrix(Matrix);
	if(Mat==NULL)	{return PDS_WRONG;}

	id=pds_matrix_init_identity(MatInv,1.0);
	if(id==PDS_WRONG)	{pds_matrix_free(Mat); return PDS_WRONG;}

	for(i=0,j=0;(i<N)&&(j<N);j++)
	{
		r=0;E=1;
		// Como el primer elemento de la linea es cero busco otra que no 
		// lo sea, si no existe paso a la siguiente columna.
		if(Mat->M[i][j]==0)
		{
			// Esta función carga con E=1 encontró una linea para 
			// intercambiar y carga en r la linea.
			// Devuelve E=0 si no encontró ninguna linea
			id=pds_matrix_find_row_in_col(Mat,&r,&E,j,i+1);
			if(id==PDS_WRONG)	{pds_matrix_free(Mat); return PDS_WRONG;}

			// Intercambia la linea con r.
			if(E==1) 
			{
				id=pds_matrix_swap_rows(Mat,r,i);
				if(id==PDS_WRONG)	{pds_matrix_free(Mat); return PDS_WRONG;}

				id=pds_matrix_swap_rows(MatInv,r,i);
				if(id==PDS_WRONG)	{pds_matrix_free(Mat); return PDS_WRONG;}
			}
		}
		// E==1 Indica que M[i][j]!=0, da igual si fue con ayuda de swap.
		// E==0 Indica que M[k][j]==0     i<= k <Nlin.
		if(E==1)
		{

			factor=Mat->M[i][j];
			// Divido la linea en estudio para que inicie con 1.
			id=pds_matrix_row_div_factor(Mat,i,factor);
			if(id==PDS_WRONG)	{pds_matrix_free(Mat); return PDS_WRONG;}

			id=pds_matrix_row_div_factor(MatInv,i,factor);
			if(id==PDS_WRONG)	{pds_matrix_free(Mat); return PDS_WRONG;}

			for(k=i+1;k<Mat->Nlin;k++)
			{
				if(Mat->M[k][j]!=0)
				{
					factor=-Mat->M[k][j];
					id=pds_matrix_row_add_row_factor(Mat,k,i,factor);
					if(id==PDS_WRONG)	{pds_matrix_free(Mat); return PDS_WRONG;}
					//MatInv->M[k][j]=0;//Esto porque no tiene buena precisión.

					id=pds_matrix_row_add_row_factor(MatInv,k,i,factor);
					if(id==PDS_WRONG)	{pds_matrix_free(Mat); return PDS_WRONG;}
				}
			}
			i++;
		}
	}


	// Verifica si el ultimo elemento de la diagonal es distinto de cero
	// sino no tiene sentido seguir.
	if(Mat->M[N-1][N-1]==0)	{pds_matrix_free(Mat); return PDS_WRONG;}

	// Comienza a restar desde la última fila, de manera que quede una diagonal
	// iniciando con 1 y seguido de ceros, hasta que termine la mayor matriz 
	// cuadrada que puede ser contenida por MatInv.
	for(i=0;i<N;i++)
	{
		// Solo por si acaso verifico si cada elemento de la diagonal es
		// distinto de cero.
		if(Mat->M[(N-1)-i][(N-1)-i]!=0)
		{
			// Opero con las lineas para generar ceros en todas
			// las lineas arriba de la linea i.
			for(k=0;k<(N-1-i);k++)
			{
				factor=-Mat->M[k][(N-1)-i];
				id=pds_matrix_row_add_row_factor(Mat,k,(N-1)-i,factor);
				if(id==PDS_WRONG)	{pds_matrix_free(Mat); return PDS_WRONG;}

				id=pds_matrix_row_add_row_factor(MatInv,k,(N-1)-i,factor);
				if(id==PDS_WRONG)	{pds_matrix_free(Mat); return PDS_WRONG;}
			}
		}
		else return PDS_WRONG;
	}

	pds_matrix_free(Mat);
	return PDS_OK;
}


/** \fn PdsMatrix *pds_matrix_inverse_matrix_new(const PdsMatrix *Matrix)
 *  \brief Invierte la matriz Matrix y la guarda en MatInv. Ambas matrices deben
 *  existir y tener el mismo tamanho. <br><b>MatInv=Matrix^{-1}</b>.
 *  \param[in] Matrix La matriz de en consulta.
 *  \return Un puntero que apunta a una nueva estructura con la multiplicación.
 *  Retorna NULL si fallo en la inversión, o si Matrix==NULL. 
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_inverse_matrix_new(const PdsMatrix *Matrix)
{
	PdsRaNatural i,j,k;
	PdsRaNatural r,E,N;
	PdsRaReal factor;
	PdsMatrix *Mat=NULL;
	PdsMatrix *MatInv=NULL;
	int id;

	if(Matrix==NULL)	return NULL;

	if(Matrix->Ncol!=Matrix->Nlin)	return NULL;

	N=Matrix->Nlin;

	MatInv=pds_matrix_new(N,N);
	if(MatInv==NULL)	{return NULL;}

	Mat=pds_matrix_new_matrix(Matrix);
	if(Mat==NULL)	{pds_matrix_free(MatInv); return NULL;}

	id=pds_matrix_init_identity(MatInv,1.0);
	if(id==PDS_WRONG)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}

	for(i=0,j=0;(i<N)&&(j<N);j++)
	{
		r=0;E=1;
		// Como el primer elemento de la linea es cero busco otra que no 
		// lo sea, si no existe paso a la siguiente columna.
		if(Mat->M[i][j]==0)
		{
			// Esta función carga con E=1 encontró una linea para 
			// intercambiar y carga en r la linea.
			// Devuelve E=0 si no encontró ninguna linea
			id=pds_matrix_find_row_in_col(Mat,&r,&E,j,i+1);
			if(id==PDS_WRONG)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}

			// Intercambia la linea con r.
			if(E==1) 
			{
				id=pds_matrix_swap_rows(Mat,r,i);
				if(id==PDS_WRONG)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}

				id=pds_matrix_swap_rows(MatInv,r,i);
				if(id==PDS_WRONG)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}
			}
		}
		// E==1 Indica que M[i][j]!=0, da igual si fue con ayuda de swap.
		// E==0 Indica que M[k][j]==0     i<= k <Nlin.
		if(E==1)
		{

			factor=Mat->M[i][j];
			// Divido la linea en estudio para que inicie con 1.
			id=pds_matrix_row_div_factor(Mat,i,factor);
			if(id==PDS_WRONG)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}

			id=pds_matrix_row_div_factor(MatInv,i,factor);
			if(id==PDS_WRONG)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}

			for(k=i+1;k<Mat->Nlin;k++)
			{
				if(Mat->M[k][j]!=0)
				{
					factor=-Mat->M[k][j];
					id=pds_matrix_row_add_row_factor(Mat,k,i,factor);
					if(id==PDS_WRONG)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}
					//MatInv->M[k][j]=0;//Esto porque no tiene buena precisión.

					id=pds_matrix_row_add_row_factor(MatInv,k,i,factor);
					if(id==PDS_WRONG)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}
				}
			}
			i++;
		}
	}


	// Verifica si el ultimo elemento de la diagonal es distinto de cero
	// sino no tiene sentido seguir.
	if(Mat->M[N-1][N-1]==0)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}

	// Comienza a restar desde la última fila, de manera que quede una diagonal
	// iniciando con 1 y seguido de ceros, hasta que termine la mayor matriz 
	// cuadrada que puede ser contenida por MatInv.
	for(i=0;i<N;i++)
	{
		// Solo por si acaso verifico si cada elemento de la diagonal es
		// distinto de cero.
		if(Mat->M[(N-1)-i][(N-1)-i]!=0)
		{
			// Opero con las lineas para generar ceros en todas
			// las lineas arriba de la linea i.
			for(k=0;k<(N-1-i);k++)
			{
				factor=-Mat->M[k][(N-1)-i];
				id=pds_matrix_row_add_row_factor(Mat,k,(N-1)-i,factor);
				if(id==PDS_WRONG)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}

				id=pds_matrix_row_add_row_factor(MatInv,k,(N-1)-i,factor);
				if(id==PDS_WRONG)	{pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}
			}
		}
		else {pds_matrix_free(MatInv); pds_matrix_free(Mat); return NULL;}
	}

	pds_matrix_free(Mat);
	return MatInv;
}


/** \fn int pds_matrix_is_equal (PdsMatrix *M1,PdsMatrix *M2)
 *  \brief Compara dos matrices.
 *  \param[in] M1 La matriz a comparar com M2.
 *  \param[in] M2 La matriz a comparar com M1.
 *  \return Retorna PDS_OK si las matrices son iguales, caso contrario retorna PDS_WRONG. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_equal (PdsMatrix *M1,PdsMatrix *M2)
{
	int i,j;
	if ((M1==NULL)|| (M2==NULL))	return PDS_WRONG;
	if ((M1->Nlin!=M2->Nlin)|| (M1->Ncol!=M2->Ncol))	return PDS_WRONG;
	
	for(i=0;i<M1->Nlin;i++)
	for(j=0;j<M1->Ncol;j++)
	{
		if(M1->M[i][j]!=M2->M[i][j])	return PDS_WRONG;
	}
	
	return PDS_OK;
}

/** \fn int pds_matrix_is_quasiequal (PdsMatrix *M1,PdsMatrix *M2,double u)
 *  \brief Compara dos matrices, y verifica si la diferencia absoluta de cada 
 *  elemento es menor que un umbral u.
 *  \param[in] M1 La matriz a comparar com M2.
 *  \param[in] M2 La matriz a comparar com M1.
 *  \param[in] u Es el umbrarl a testar.
 *  \return Retorna PDS_OK si las matrices son quasi iguales 
 *  (diferencia de elementos menoro igual a u), caso contrario retorna PDS_WRONG. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_quasiequal (PdsMatrix *M1,PdsMatrix *M2,double u)
{
	int i,j;
	if ((M1==NULL)|| (M2==NULL))	return PDS_WRONG;
	if ((M1->Nlin!=M2->Nlin)|| (M1->Ncol!=M2->Ncol))	return PDS_WRONG;
	
	for(i=0;i<M1->Nlin;i++)
	for(j=0;j<M1->Ncol;j++)
	{
		if (fabs(M1->M[i][j]-M2->M[i][j])>u)	return PDS_WRONG;
	}
	
	return PDS_OK;
}

/** \fn int pds_matrix_is_diagonal (PdsMatrix *M1)
 *  \brief Verifica si una matriz es diagonal.
 *
 *  Una matriz es diagonal cuando sus elementos exteriores a la diagonal 
 *  principal son nulos. Una matriz nula (llena de ceros) tambien es diagonal.
 *
 *  \param[in] M1 La matriz a verificar su diagonalidad.
 *  \return Retorna PDS_OK si las matriz es diagonal, caso contrario retorna PDS_WRONG. 
 *  Tambien retorna PDS_WRONG si la matriz no es cuadrada.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_diagonal (PdsMatrix *M1)
{
	int i,j;
	if (M1==NULL)	return PDS_WRONG;
	if (M1->Nlin!=M1->Ncol)	return PDS_WRONG;	
	
	for(i=0;i<M1->Nlin;i++)
	for(j=0;j<M1->Ncol;j++)
	{
		if(i!=j)
		if(M1->M[i][j]!=0)	return PDS_WRONG;
	}
	
	return PDS_OK;
}

/** \fn int pds_matrix_is_quasidiagonal (PdsMatrix *M1,double u)
 *  \brief Verifica si una matriz es casi diagonal. Usa un umbral u, qualquier 
 *  valor menor o igual que u es considerado cero.
 *
 *  Una matriz es diagonal cuando sus elementos exteriores a la diagonal 
 *  principal son nulos. Una matriz nula (llena de ceros) tambien es diagonal.
 *
 *  \param[in] M1 La matriz a verificar su diagonalidad.
 *  \param[in] u Es el umbral bajo el cual todo lo demas se considera cero.
 *  \return Retorna PDS_OK si las matriz es diagonal, caso contrario retorna PDS_WRONG.
 *  Tambien retorna PDS_WRONG si la matriz no es cuadrada. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_quasidiagonal (PdsMatrix *M1,double u)
{
	int i,j;
	if (M1==NULL)	return PDS_WRONG;
	if (M1->Nlin!=M1->Ncol)	return PDS_WRONG;	
	
	for(i=0;i<M1->Nlin;i++)
	for(j=0;j<M1->Ncol;j++)
	{
		if(i!=j)
		if(M1->M[i][j]>u)	return PDS_WRONG;
	}
	
	return PDS_OK;
}


/** \fn int pds_matrix_is_symetric (PdsMatrix *M1)
 *  \brief Verifica si una matriz es simétrica.
 *
 *  Una matriz es simétrica si coincide con su transpuesta.
 *
 *  \param[in] M1 La matriz a verificar su simetria.
 *  \return Retorna PDS_OK si las matriz es simetrica, caso contrario retorna PDS_WRONG. 
 *  Tambien retorna PDS_WRONG si la matriz no es cuadrada.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_symetric (PdsMatrix *M1)
{
	int i,j;
	if (M1==NULL)	return PDS_WRONG;
	if (M1->Nlin!=M1->Ncol)	return PDS_WRONG;
	
	for(i=0;i<M1->Nlin;i++)
	for(j=i+1;j<M1->Ncol;j++)
	{
		if(M1->M[i][j]!=M1->M[j][i])	return PDS_WRONG;
	}
	
	return PDS_OK;
}


/** \fn int pds_matrix_is_antisymetric (PdsMatrix *M1)
 *  \brief Verifica si una matriz es anti simétrica.
 *
 *  Una matriz es anti simétrica cuando el opuesto de esta  coincide con su transpuesta.
 *  De esto se deduce que los elementos de la diagonal son nulos.
 *  <br><b>M1^t=-M1</b>.
 *
 *  \param[in] M1 La matriz a verificar su anti simetria.
 *  \return Retorna PDS_OK si las matriz es anti simetrica, caso contrario retorna PDS_WRONG. 
 *  Tambien retorna PDS_WRONG si la matriz no es cuadrada.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_antisymetric (PdsMatrix *M1)
{
	int i,j;
	if (M1==NULL)	return PDS_WRONG;
	if (M1->Nlin!=M1->Ncol)	return PDS_WRONG;
	
	for(i=0;i<M1->Nlin;i++)
	for(j=i;j<M1->Ncol;j++)
	{
		if(M1->M[i][j]!=-M1->M[j][i])	return PDS_WRONG;
	}
	
	return PDS_OK;
}

/** \fn void pds_matrix_free(PdsMatrix *Matrix)
 *  \brief Libera una matriz de tipo puntero PdsMatrix.
 *  \param[in,out] Matrix la matriz a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsMatrixGroup
 */
void pds_matrix_free(PdsMatrix *Matrix)
{
	PdsRaNatural i;

	if(Matrix!=NULL)
	{
		for(i=0;i<Matrix->Nlin;i++)
		{
			free(Matrix->M[i]);
		}
		free(Matrix);
	}
}


/** \fn void pds_matrix_destroy(PdsMatrix **Matrix)
 *  \brief Libera una matriz de tipo puntero PdsMatrix, y limpia el puntero con NULL.
 *  \param[in,out] Matrix la matriz a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsMatrixGroup
 */
void pds_matrix_destroy(PdsMatrix **Matrix)
{
	PdsRaNatural i;

	if((*Matrix)!=NULL)
	{
		for(i=0;i<(*Matrix)->Nlin;i++)
		{
			free((*Matrix)->M[i]);
		}
		free(*Matrix);
		*Matrix=NULL;
	}
}

////////////////////////////////////////////////////////////////////////////////
////                        Funciones Extras                                ////
////////////////////////////////////////////////////////////////////////////////

/** \fn int pds_matrix_find_row_in_col(const PdsMatrix *Matrix, PdsRaNatural *row, PdsRaNatural *E, PdsRaNatural col, PdsRaNatural r1)
 *  \brief Encuentra la primera linea row que sea mayor o igual a r1 y menor que 
 *  Nlin (r1<=row<Nlin), de tal forma que en la columna col  el elemento sea 
 *  distinto de cero.
 *  Osea busca en la columna "col" el primer elemento distinto de cero, pero lo
 *  busca en el intervalo de lineas r1<=row<Nlin.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] row La linea con el elemento distinto de cero entre [r1,Nlin).
 *  \param[out] E Si se encontró un elemento distinto de cero en la columna col, E=1,
 *  caso contrario E=1.
 *  \param[in] col Columna en la que buscaremos.
 *  \param[in] r1 Límite inferior de búsqueda de linea.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: Matrix==NULL). Puede retornar
 *  PDS_WRONG por causa de r1>=Nlin, col>=Ncol o Matrix==NULL. En todos los casos de 
 *  error Matrix no es alterado.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_find_row_in_col(const PdsMatrix *Matrix, PdsRaNatural *row, PdsRaNatural *E, PdsRaNatural col, PdsRaNatural r1)
{
	PdsRaNatural i;

	if(Matrix==NULL)	return PDS_WRONG;
	if(r1 >=Matrix->Nlin)	return PDS_WRONG;
	if(col>=Matrix->Ncol)	return PDS_WRONG;

	for(i=r1;i<Matrix->Nlin;i++)
	{
		if(Matrix->M[i][col]!=0.0)
		{
			*E=1;
			*row=i;
			return PDS_OK;
		}
	}

	*E=0;
	*row=0;
	return PDS_OK;
}


