/*
 * pdsvector.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsvector.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-04-2011
 *  \brief Funciones que trabajan con vectores.
 *  
 *  <br>Estas funciones trabajan con un vector de la forma.<br>
 *  \image html vector.png "Vector de Nel elementos"
 *  \b Nel es el número de elementos.
 */

#ifndef __PDSVECTOR_H__
#define __PDSVECTOR_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdsraglobal.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsVectorGroup Funciones del módulo PdsVector.
 *  \brief Funciones que trabajan con vectores.
 *  
 *  <br>Estas funciones trabajan con un vector de la forma.<br>
 *  \image html vector.png "Vector de Nel elementos"
 *  \b Nel es el número de elementos.
 * @{
 */

/*! \def _VECSETVAL(Vector,X,m)
 *  Es equivalente a Vector(X)=m.
 *  \ingroup PdsVectorGroup
*/
#define _VECSETVAL(Vector,X,m) pds_vector_set_value(Vector,X,m)

/*! \struct PdsVector
 *  \brief La estructura tipo  PdsVector .
 *  Esta estructura genera una matriz de Nel elementos.
 *  Para usar incluir pds/pdsvector.h.
 *  \ingroup PdsVectorGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Un arreglo de Nel elementos. */
	PdsRaReal *V;
	/*! Número de elementos. */
	PdsRaNatural Nel;
}PdsVector;

/** \fn PdsVector *pds_vector_new(PdsRaNatural Nel)
 *  \brief Crea un vector de tipo PdsVector e inicia con cero todos los elementos.
 *  \param[in] Nel Es el número de elementos del vector.
 *  \return Un puntero al vector de tipo PdsVector.
 *  \ingroup PdsVectorGroup
 */
PdsVector *pds_vector_new(PdsRaNatural Nel);


/** \fn PdsVector *pds_vector_new_vector(const PdsVector *VecSrc)
 *  \brief Crea un vector de tipo PdsVector usando como fuente el vector VecSrc.
 *  \param[in] VecSrc Es el vector imagen que se usará como fuente.
 *  \return Un puntero al vector de tipo PdsVector.
 *  \ingroup PdsVectorGroup
 */
PdsVector *pds_vector_new_vector(const PdsVector *VecSrc);


/** \fn PdsVector *pds_vector_new_load_data(const char* datafile)
 *  \brief Crea un nuevo vector e inicia los datos con los elementos del archivo
 *  datafile.
 *  Usa TAB o un salto de linea como delimitador de elemento.
 *  \param[in] datafile Nombre del archivo de donde se cargará los datos iniciales
 *  del vector.
 *  \return Un puntero que apunta a la dirección del nuevo vector, si todo fue bien, 
 *  o NULL si no. (ej. datafile==NULL)
 *  \ingroup PdsVectorGroup
 */
PdsVector *pds_vector_new_load_data(const char* datafile);


/** \fn PdsVector *pds_vector_new_load_data_line(const char* datafile,PdsRaNatural line)
 *  \brief Crea un nuevo vector e inicia los datos con los elementos de la linea
 *  line del archivo datafile.
 *  Usa TAB como delimitador de elemento y un salto de linea como delimitador de linea.
 *  Si se pide una linea inexistente la funcion retorna NULL
 *  \param[in] datafile Nombre del archivo de donde se cargará los datos iniciales
 *  del vector.
 *  \param[in] line Número de linea que se cargará, line inicia en cero. 
 *  \return Un puntero que apunta a la dirección del nuevo vector, si todo fue bien, 
 *  o NULL si no. (ej. datafile==NULL o line inexistente)
 *  \ingroup PdsVectorGroup
 */
PdsVector *pds_vector_new_load_data_line(const char* datafile,PdsRaNatural line);


/** \fn PdsVector *pds_vector_new_load_data_col(const char* datafile,PdsRaNatural col)
 *  \brief Crea un nuevo vector e inicia los datos con los elementos de la columna
 *  col del archivo datafile.
 *  Usa TAB como delimitador de columna y un salto de linea como delimitador de linea.
 *  Si se pide una linea inexistente la función retorna NULL
 *  \param[in] datafile Nombre del archivo de donde se cargará los datos iniciales
 *  del vector.
 *  \param[in] col Número de columna que se cargará, col inicia en cero. 
 *  \return Un puntero que apunta a la dirección del nuevo vector, si todo fue bien, 
 *  o NULL si no. (ej. datafile==NULL o col inexistente)
 *  \ingroup PdsVectorGroup
 */
PdsVector *pds_vector_new_load_data_col(const char* datafile,PdsRaNatural col);


/** \fn int pds_vector_init_value(PdsVector *Vector,PdsRaReal Val)
 *  \brief Inicia el vector de tipo puntero PdsVector. <br><b>Vector=Val</b>.
 *  \param[in,out] Vector El vector a iniciar.
 *  \param[in] Val Es el valor inicial de los elementos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_init_value(PdsVector *Vector,PdsRaReal Val);


/** \fn int pds_vector_init_vector(PdsVector *Vector,const PdsVector *VecSrc)
 *  \brief Inicia el vector de tipo puntero PdsVector con otro vector.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Vector=VecSrc</b>.
 *  \param[in,out] Vector Es el vector a iniciar.
 *  \param[in] VecSrc Es el vector fuente de la copia.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_init_vector(PdsVector *Vector,const PdsVector *VecSrc);


/** \fn int pds_vector_add_vector(PdsVector *Vector,const PdsVector *VecSrc)
 *  \brief Suma al vector de tipo puntero PdsVector con otro vector y lo copia en Vector.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Vector=Vector+VecSrc</b>.
 *  \param[in,out] Vector Es el vector de destino y operando.
 *  \param[in] VecSrc Es el Vector fuente del operando.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_add_vector(PdsVector *Vector,const PdsVector *VecSrc);


/** \fn int pds_vector_add_factor_vector(PdsVector *Vector,PdsRaReal Alpha,const PdsVector *VecSrc)
 *  \brief Suma al vector de tipo puntero PdsVector con otro vector con unf actor 
 *  y lo copia en Vector. Si los tamaños son diferentes intersecta los tamaños y 
 *  hace la copia en la intersección solamente. <br><b>Vector=Vector+Alpha*VecSrc</b>.
 *  \param[in,out] Vector Es el vector de destino y operando.
 *  \param[in] Alpha Factor que multiplica al operando.
 *  \param[in] VecSrc Es el Vector fuente del operando.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_add_factor_vector(PdsVector *Vector,PdsRaReal Alpha,const PdsVector *VecSrc);


/** \fn int pds_vector_add_value(PdsVector *Vector,PdsRaReal Val)
 *  \brief Suma al vector de tipo puntero PdsVector el valor Val. <br><b>Vector=Vector+Val</b>.
 *  \param[in,out] Vector El vector a sumar.
 *  \param[in] Val Es el valor inicial de los elementos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_add_value(PdsVector *Vector,PdsRaReal Val);


/** \fn int pds_vector_sub_vector(PdsVector *Vector,const PdsVector *VecSrc)
 *  \brief Resta al vector de tipo puntero PdsVector con otro vector y lo copia en Vector.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Vector=Vector-VecSrc</b>.
 *  \param[in,out] Vector Es el vector de destino y operando.
 *  \param[in] VecSrc Es el Vector fuente del operando.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_sub_vector(PdsVector *Vector,const PdsVector *VecSrc);


/** \fn int pds_vector_mul_vector_elements(PdsVector *Vector,const PdsVector *VecSrc)
 *  \brief Multiplica al vector de tipo puntero PdsVector con otro vector y lo copia en Vector.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Vector=Vector.VecSrc</b>.
 *  \param[in,out] Vector Es el vector de destino y operando.
 *  \param[in] VecSrc Es el Vector fuente del operando.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_mul_vector_elements(PdsVector *Vector,const PdsVector *VecSrc);


/** \fn int pds_vector_mul_value(PdsVector *Vector,PdsRaReal Val)
 *  \brief Multiplica al vector de tipo puntero PdsVector por un escalar. <br><b>Vector=Val*Vector</b>.
 *  \param[in,out] Vector El vector a multiplicar.
 *  \param[in] Val Es el valor que multiplicará a los elementos del vector.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_mul_value(PdsVector *Vector,PdsRaReal Val);


/** \fn int pds_vector_div_value(PdsVector *Vector,PdsRaReal Val)
 *  \brief Divide al vector de tipo puntero PdsVector por un escalar. <br><b>Vector=Vector/Val</b>.
 *  \param[in,out] Vector El vector a dividir.
 *  \param[in] Val Es el valor que dividirá a los elementos del vector.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_div_value(PdsVector *Vector,PdsRaReal Val);


/** \fn int pds_vector_dot_vector(const PdsVector *Vector1,const PdsVector *Vector2,PdsRaReal *Val)
 *  \brief Ejecuta el producto punto en el vector Vector1 y Vector2, y lo carga en Val.
 *  Los tamaños deben ser iguales. <br><b>Val=(Vector1,Vector2) </b>.
 *  \param[in] Vector1 Es el vector operando.
 *  \param[in] Vector2 Es el Vector operando.
 *  \param[out] Val El producto punto entre Vector1 y Vector2.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_dot_vector(const PdsVector *Vector1,const PdsVector *Vector2,PdsRaReal *Val);


/** \fn int pds_vector_printf(const PdsVector *Vector)
 *  \brief Imprime en pantalla un vector de tipo puntero PdsVector.
 *  \param[in] Vector El vector a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_printf(const PdsVector *Vector);


/** \fn int pds_vector_fprintf(const PdsVector *Vector,FILE *fd)
 *  \brief Imprime en el archivo que apunta fd un vector de tipo puntero PdsVector.
 *  \param[in] Vector El vector a imprimir en pantalla.
 *  \param[in,out] fd El puntero al archivo.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_fprintf(const PdsVector *Vector,FILE *fd);


/** \fn int pds_vector_fscanf(PdsVector *Vector, FILE *fd)
 *  \brief Inicializa un vector con los datos del archivo apuntado por fd.
 *  Usa TAB o un salto de linea como delimitador de elemento.
 *  \param[in] Vector Vector en donde se cargaran los datos.
 *  \param[in] fd Apuntador del archivo de donde se cargará los datos iniciales
 *  del vector.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. Vector==NULL o fd==NULL)
 *  \ingroup PdsVectorGroup
 */
int pds_vector_fscanf(PdsVector *Vector, FILE *fd);


/** \fn int pds_vector_fwrite(PdsVector *Vector, FILE *fd)
 *  \brief Escribe los datos de un vector en el archivo binario apuntado por fd.
 *  \param[in] Vector Vector de donde se leeran los datos.
 *  \param[in] fd Apuntador del archivo binario de donde se escribiran los datos 
 *  del vector.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. Vector==NULL o fd==NULL)
 *  \ingroup PdsVectorGroup
 */
int pds_vector_fwrite(PdsVector *Vector, FILE *fd);


/** \fn int pds_vector_fread(PdsVector *Vector, FILE *fd)
 *  \brief Inicializa un vector con los datos del archivo binario apuntado por fd.
 *  \param[in] Vector Vector en donde se cargaran los datos.
 *  \param[in] fd Apuntador del archivo binario de donde se cargará los datos 
 *  iniciales del vector.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. Vector==NULL o fd==NULL)
 *  \ingroup PdsVectorGroup
 */
int pds_vector_fread(PdsVector *Vector, FILE *fd);


/** \fn int pds_vector_norm_vector(const PdsVector *Vector, PdsRaReal *m)
 *  \brief Devuelve la norma del vector Vector.
 *  \param[in] Vector El vector en consulta.
 *  \param[out] m El valor de la norma del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_norm_vector(const PdsVector *Vector, PdsRaReal *m);


/** \fn int pds_vector_norm2_vector(const PdsVector *Vector, PdsRaReal *m)
 *  \brief Devuelve la norma^2 del vector Vector.
 *  \param[in] Vector El vector en consulta.
 *  \param[out] m El valor de la norma^2 del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_norm2_vector(const PdsVector *Vector, PdsRaReal *m);


/** \fn int pds_vector_rms_vector(const PdsVector *Vector, PdsRaReal *m)
 *  \brief Devuelve el valor raíz cuadrático medio del vector Vector.
 *  \param[in] Vector El vector en consulta.
 *  \param[out] m El valor rms del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_rms_vector(const PdsVector *Vector, PdsRaReal *m);


/** \fn int pds_vector_rms_vectors(const PdsVector *VectorA, const PdsVector *VectorB, PdsRaReal *m)
 *  \brief Devuelve el valor raíz cuadrático medio de la diferencia de los vectores.
 *  \param[in] VectorA El vector en consulta.
 *  \param[in] VectorB El vector en consulta.
 *  \param[out] m El valor rms de la diferencias de vectores.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL, o diferentes tamaños). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_rms_vectors(const PdsVector *VectorA, const PdsVector *VectorB, PdsRaReal *m);


/** \fn int pds_vector_rms_factor_vectors(const PdsVector *VectorA, const PdsVector *VectorB, PdsRaReal *m)
 *  \brief Devuelve el valor raíz cuadrático medio de la diferencia de los vectores
 *  entre el valor raíz cuadrático medio de VectorB, si este valor es cero, se 
 *  usa entonces el valor raíz cuadrático medio de VectorA.
 *
 *  \f[ 
	\frac{\sqrt{\frac{\sum_{i=0}^{Nel-1}{(VectorA[i]-VectorB[i])^2}}{Nel}}}
	{\sqrt{\frac{\sum_{i=0}^{Nel-1}{VectorB[i]^2}}{Nel}}} 
    \f]
 *  \param[in] VectorA El vector en consulta.
 *  \param[in] VectorB El vector en consulta.
 *  \param[out] m El valor factor rms de la diferencias de vectores.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL, o diferentes tamaños). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_rms_factor_vectors(const PdsVector *VectorA, const PdsVector *VectorB, PdsRaReal *m);

/** \fn int pds_vector_mean_vector(const PdsVector *Vector, PdsRaReal *m)
 *  \brief Devuelve el valor medio del vector (X) Vector.
 *
 *  \f[ \eta=\frac{\sum_{i=0}^{Nel-1} {X_i}}{Nel} \f]
 *  \param[in] Vector El vector en consulta.
 *  \param[out] m El valor medio del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_mean_vector(const PdsVector *Vector, PdsRaReal *m);


/** \fn int pds_vector_var_vector(const PdsVector *Vector, PdsRaReal *v)
 *  \brief Devuelve el valor de la varianza muestral \f$\sigma^2\f$ del vector Vector (X).
 *
 *  \f[ \eta=\frac{\sum_{i=0}^{Nel-1} {X_i}}{Nel} \f]
 *  \f[ \sigma^2=\frac{\sum_{i=0}^{Nel-1} {(X_i-\eta)^2}}{Nel-1} \f]
 *  \param[in] Vector El vector en consulta.
 *  \param[out] v El valor de la varianza del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_var_vector(const PdsVector *Vector, PdsRaReal *v);


/** \fn int pds_vector_cov_vector (const PdsVector *VectorX,const PdsVector *VectorY, PdsRaReal *c)
 *  \brief Devuelve el valor de la covarianza muestral de los vectores VectorX y VectorY.
 *
 *  \f[ \eta_x=\frac{\sum_{i=0}^{Nel-1} {X_i}}{Nel} \f]
 *  \f[ \eta_y=\frac{\sum_{i=0}^{Nel-1} {Y_i}}{Nel} \f]
 *  \f[ cov(X,Y)=\frac{\sum_{i=0}^{Nel-1} {(X_i -{\eta}_x)(Y_i -{\eta}_y)}}{Nel-1} \f]
 *  \param[in] VectorX El vector en consulta.
 *  \param[in] VectorY El vector en consulta.
 *  \param[out] c El valor de la covarianza de los vectores VectorX y VectorY.
 *  \return TRUE si todo fue bien o FALSE si no (ej: VectorX==NULL, VectorY==NULL o
 *  longitudes distintas). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_cov_vector (const PdsVector *VectorX,const PdsVector *VectorY, PdsRaReal *c);


/** \fn int pds_vector_cor_vector (const PdsVector *VectorX,const PdsVector *VectorY, PdsRaReal *c)
 *  \brief Devuelve el valor del coeficiente de correlación muestral de los vectores VectorX y VectorY.
 *
 *  \f[ \eta_x=\frac{\sum_{i=0}^{Nel-1} {X_i}}{Nel} \f]
 *  \f[ \eta_y=\frac{\sum_{i=0}^{Nel-1} {Y_i}}{Nel} \f]
 *  \f[ cor(X,Y)=\frac{\sum_{i=0}^{Nel-1} {(X_i -{\eta}_x)(Y_i -{\eta}_y)}}{\sqrt{\sum_{i=0}^{Nel-1} {(X_i -{\eta}_x)^2}}\sqrt{\sum_{i=0}^{Nel-1} {(Y_i -{\eta}_y)^2}}} \f]
 *  \param[in] VectorX El vector en consulta.
 *  \param[in] VectorY El vector en consulta.
 *  \param[out] c El valor de la correlación de los vectores VectorX y VectorY.
 *  \return TRUE si todo fue bien o FALSE si no (ej: VectorX==NULL, VectorY==NULL o
 *  longitudes distintas). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_cor_vector (const PdsVector *VectorX,const PdsVector *VectorY, PdsRaReal *c);


/** \fn int pds_vector_get_value(const PdsVector *Vector, PdsRaNatural x, PdsRaReal *m)
 *  \brief Devuelve el valor en la posición (x) del vector Vector.
 *  (x) inicia con (0).
 *  \param[in] Vector El vector en consulta.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[out] m El valor en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces carga 0 en m, en caso de error de vector nulo carga cero en m.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_get_value(const PdsVector *Vector, PdsRaNatural x, PdsRaReal *m);


/** \fn int pds_vector_set_value(PdsVector *Vector, PdsRaNatural x, PdsRaReal m)
 *  \brief Escribe el valor m en la posición (x) del vector Vector.
 *  (x) inicia con (0).
 *  \param[in,out] Vector El vector a escribir.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[in] m El valor en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces no hace nada y no se considera como error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_set_value(PdsVector *Vector, PdsRaNatural x, PdsRaReal m);


/** \fn int pds_vector_get_nelements(const PdsVector *Vector,PdsRaReal *Nelements)
 *  \brief Devuelve el número de elementos del vector Vector.
 *  \param[in] Vector El vector en consulta.
 *  \param[out] Nelements En donde se guardará el número de elementos del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_get_nelements(const PdsVector *Vector,PdsRaReal *Nelements);

/** \fn int pds_vector_get_min_value(const PdsVector *Vector, PdsRaReal *m)
 *  \brief Devuelve el valor mínimo de todos los elementos del vector.
 *  \param[in] Vector El vector en consulta.
 *  \param[out] m El valor mínimo de todos los elementos del vector Vector. En 
 *  caso de error por vector nulo carga cero en m.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_get_min_value(const PdsVector *Vector, PdsRaReal *m);


/** \fn int pds_vector_get_max_value(const PdsVector *Vector, PdsRaReal *m)
 *  \brief Devuelve el valor máximo de todos los elementos del vector.
 *  \param[in] Vector El vector en consulta.
 *  \param[out] m El valor máximo de todos los elementos del vector Vector. En 
 *  caso de error por vector nulo carga cero en m.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_get_max_value(const PdsVector *Vector, PdsRaReal *m);


/** \fn int pds_vector_swap_elements(PdsVector *Vector, PdsRaNatural X, PdsRaNatural Y)
 *  \brief Intercambia los elementos de la posición X e Y del vector.
 *  \param[in] Vector El vector en consulta.
 *  \param[out] X Posicion del vector. X in [0,Nel-1].
 *  \param[out] Y Posicion del vector. Y in [0,Nel-1].
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL o X,Y fuera de rango). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_swap_elements(PdsVector *Vector, PdsRaNatural X, PdsRaNatural Y);


/** \fn int pds_vector_asc_order(PdsVector *Vector)
 *  \brief Ordena ascendentemente todos los elementos del vector.
 *  \param[in,out] Vector El vector en consulta.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_asc_order(PdsVector *Vector);


/** \fn int pds_vector_desc_order(PdsVector *Vector)
 *  \brief Ordena descendentemente todos los elementos del vector.
 *  \param[in,out] Vector El vector en consulta.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_desc_order(PdsVector *Vector);


/** \fn int pds_vector_set_bcd(PdsVector *Vector,PdsRaNatural n,PdsRaNatural b0,PdsRaNatural b1)
 *  \brief Escribe en formato BCD el numero n en el vector Vector.
 *  \param[in,out] Vector El vector a escribir.
 *  \param[in] n EL número decimal e escribir en BCD en el vector Vector
 *  \param[in] b0 El valor que será escrito cuando se representa a un bit 0.
 *  \param[in] b1 El valor que será escrito cuando se representa a un bit 1.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_set_bcd(PdsVector *Vector,PdsRaNatural n,PdsRaNatural b0,PdsRaNatural b1);


/** \fn int pds_vector_set_gray(PdsVector *Vector,PdsRaNatural n,PdsRaNatural b0,PdsRaNatural b1)
 *  \brief Escribe en formato Gray el numero n en el vector Vector.
 *  \param[in,out] Vector El vector a escribir.
 *  \param[in] n EL número decimal e escribir en Gray en el vector Vector
 *  \param[in] b0 El valor que será escrito cuando se representa a un bit 0.
 *  \param[in] b1 El valor que será escrito cuando se representa a un bit 1.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_set_gray(PdsVector *Vector,PdsRaNatural n,PdsRaNatural b0,PdsRaNatural b1);


/** \fn int pds_vector_cmp(const PdsVector *Vector1,const PdsVector *Vector2,PdsRaNatural *n)
 *  \brief Compara los vectores Vector1 y Vector2 y retorna la cantidad de elementos distintos.
 *  \param[in] Vector1 Un vector a comparar.
 *  \param[in] Vector2 Un vector a comparar.
 *  \param[out] n EL número de elementos diferentes entre los vectores.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector1==NULL o Vector2==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_cmp(const PdsVector *Vector1,const PdsVector *Vector2,PdsRaNatural *n);


/** \fn void pds_vector_free(PdsVector *Vector)
 *  \brief Libera un vector de tipo puntero PdsVector.
 *  \param[in,out] Vector El vector a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsVectorGroup
 */
void pds_vector_free(PdsVector *Vector);


/** \fn void pds_vector_destroy(PdsVector **Vector)
 *  \brief Libera un vector de tipo puntero PdsVector, y limpia el puntero con NULL.
 *  \param[in,out] Vector El vector a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsVectorGroup
 */
void pds_vector_destroy(PdsVector **Vector);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


