/*
 * pdslvector.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdslvector.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-02-2012
 *  \brief Funciones que trabajan con vectores.
 *  
 *  <br>Estas funciones trabajan con una lista de vectores de la forma.<br>
 *  \image html listvector.png "Lista de vectores"
 */

#ifndef __PDSLVECTOR_H__
#define __PDSLVECTOR_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdsraglobal.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsLVectorGroup Funciones del módulo PdsListVector.
 *  \brief Funciones que trabajan con listas vectores.
 *  
 *  <br>Estas funciones trabajan con una lista de vectores de la forma.<br>
 *  \image html listvector.png "Lista de vectores"
 * @{
 */

/*! \struct PdsListVector
 *  \brief La estructura tipo  PdsListVector .
 *  Esta estructura genera una doblemente enlazada lista de vectores.
 *  Para usar incluir pds/pdslvector.h.
 *  \ingroup PdsListVectorGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct PdsListVector
{
	// Dirección del nodo anterios de la lista.
	struct PdsListVector *AddPrev;
	// Vector real asociado; dentro;del nodo.
	PdsVector *V;
	// Direccion del nodo siguiente en la lista.
	struct PdsListVector *AddNext;
}PdsListVector;

/** \fn PdsListVector *pds_list_vector_new(void)
 *  \brief Crea una lista de tipo PdsListVector vacia.
 *  \return Un puntero al vector de tipo PdsListVector.
 *  \ingroup PdsListVectorGroup
 */
PdsListVector *pds_list_vector_new(void);


/** \fn int pds_list_vector_push(PdsListVector **Lprev,PdsRaNatural N)
 *  \brief Agrega un elemento a la cima de la lista.
 *  \param[in,out] Lprev El nodo de la cima de la lista.
 *  \param[in] N La cantidad de elementos del vector.
 *  \return TRUE si todo fue bien o FALSE si no. Si no le das el último nodo da error
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_push(PdsListVector **Lprev,PdsRaNatural N);


/** \fn int pds_list_vector_pop(PdsListVector **L)
 *  \brief Quita un elemento de la lista. Si no hay elementos retorna FALSE.
 *  \param[in,out] L El nodo de la cima de la lista.
 *  \return TRUE si todo fue bien o FALSE si no. Si no le das el último nodo da error
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_pop(PdsListVector **L);


/** \fn int pds_list_vector_shift(PdsListVector **L)
 *  \brief Quita el elemento inicial;mas antiguo; de la lista, si no hay elementos retorna FALSE.
 *  \param[in,out] L El primer nodo de la lista.
 *  \return TRUE si todo fue bien o FALSE si no. Si no le das el primer nodo da error.
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_shift(PdsListVector **L);


/** \fn int pds_list_vector_unshift(PdsListVector **Lnext,PdsRaNatural N)
 *  \brief Agrega un elemento al inicio de la lista.
 *  \param[in,out] Lnext El primer nodo de la lista.
 *  \param[in] N La cantidad de elementos del vector.
 *  \return TRUE si todo fue bien o FALSE si no. Si no le das el primer nodo da error.
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_unshift(PdsListVector **Lnext,PdsRaNatural N);


/** \fn int pds_list_vector_top(PdsListVector **L)
 *  \brief Busca el elemento final; superior; de la lista.
 *  \param[in,out] L Un nodo de la lista, en donde se cargará el último nodo.
 *  \return TRUE si todo fue bien o FALSE si no. 
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_top(PdsListVector **L);


/** \fn int pds_list_vector_bottom(PdsListVector **L)
 *  \brief Busca el elemento inicial; inferior; de la lista.
 *  \param[in,out] L Un nodo de la lista, en donde se cargará el primer nodo.
 *  \return TRUE si todo fue bien o FALSE si no. 
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_bottom(PdsListVector **L);


/** \fn int pds_list_vector_printf(const PdsListVector *L)
 *  \brief Imprime en pantalla los datos de un nodo de tipo puntero PdsListVector.
 *  \param[in] L El nodo a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_printf(const PdsListVector *L);


/** \fn int pds_list_vector_all_printf(const PdsListVector *L)
 *  \brief Imprime en pantalla todos los datos de la lista.
 *  \param[in] L Un nodo de la lista a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_all_printf(const PdsListVector *L);


/** \fn void pds_list_vector_free(PdsListVector *L)
 *  \brief Libera una lista entera de tipo puntero PdsListVector. 
 *  \param[in,out] L La lista a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsListVectorGroup
 */
void pds_list_vector_free(PdsListVector *L);


/** \fn void pds_list_vector_destroy(PdsListVector **L)
 *  \brief Libera una lista de tipo puntero PdsListVector, y limpia el puntero con NULL.
 *  \param[in,out] L La lista a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsListVectorGroup
 */
void pds_list_vector_destroy(PdsListVector **L);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


