/*
 * pdsneuronml.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsneuronml.h
 *  \author Fernando Pujaico Rivera
 *  \date 05-01-2012
 *  \brief Funciones que trabajan con redes neuronales multi capas.
 *  
 *  <br>Estas funciones trabajan con redes neuronales multi capas de la forma.<br>
 *  \image html PdsNeuronML.png "Red neuronal multi capa."
 */

#ifndef __PDSNEURONML_H__
#define __PDSNEURONML_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdsnnglobal.h>
#include <pds/pdssn.h>
#include <pds/pdsnivector.h>
#include <pds/pdsnvector.h>


#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif


/** \defgroup PdsNeuronMLGroup Módulo PdsNeuronML.
 *  \brief Funciones que trabajan con redes neuronales multi capas.
 *  
 *  <br>Estas funciones trabajan con redes neuronales multi capas de la forma.<br>
 *  \image html PdsNeuronML.png "Red neuronal multi capa."
 * @{
 */


/*! \struct PdsNeuronML
 *  \brief La estructura tipo  PdsNeuronML .
 *  Esta estructura genera una red neuronal multi capa.
 *  Para usar incluir pds/pdsnn.h.
 *
 *  \image html PdsNeuronML.png "Red neuronal multi capa."
 *  \ingroup PdsNeuronMLGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Un vector que indica el número de neuronas por capa. */
	PdsNnNatural *L;
	/*! Número de capas. */
	PdsNnNatural N;

	/*! Vector de neuronas de entrada Nel elementos. */
	PdsNIVector *LayerInput;
	/*! Arreglo de Nel elementos, cada elemento es un arreglo de neuronas. */
	PdsNVector **Layer;

	/*! Valor de error, un valor menor a este detiene el aprendizaje. */
	PdsNnReal MaxError;
	/*! Factor de aprendizaje. */
	PdsNnReal Alpha;
	/*! Número máximo de iteraciones. */
	PdsNnNatural MaxIter;
	/*! Número de iteraciones del ultimo entrenamiento. */
	PdsNnNatural Iter;
	/*! Habilitador de printf de avance. 1 habilitado 0 deshabilitado*/
	PdsNnBool ShowPrintf;

}PdsNeuronML;

/** @name pds_neuronml_<new>
 *  Creando nuevas neuronas
 *
 *  <center>
 *  \image html PdsNeuronML.png "Red neuronal multi capa."
 *  </center>
 *  <br>
 * @{
 */

/** \fn PdsNeuronML *pds_neuronml_new(const PdsVector *L)
 *  \brief Crea una estructura de tipo PdsNeuronML.
 *
 *  <ul>
 *  <li>Inicia con cero todos los Pesos. </li>
 *  <li>El máximo margen de error RMS aceptado en la salida de la estructura 
    PdsNeuronML es PDS_DEFAULT_MAX_RMS_ERROR.</li>
 *  <li>El valor de de Alpha se inicia con PDS_DEFAULT_ALPHA.</li>
 *  <li>El valor U y W son iniciados aleatoriamente entre -1.0 1.0.</li>
 *  <li>El valor func(x) se inicia con tanh(x/2).</li>
 *  </ul> 
 *  
 *  \param[in] L Es un vector con el numero de elementos por capas.
 *  \return Un puntero al vector de tipo PdsNeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNeuronML *pds_neuronml_new(const PdsVector *L);

/** \fn PdsNeuronML *pds_neuronml_new_from_array(const PdsNnNatural *L,PdsNnNatural Nel)
 *  \brief Crea una estructura de tipo PdsNeuronML.
 *
 *  <ul>
 *  <li>Inicia con cero todos los Pesos. </li>
 *  <li>El máximo margen de error RMS aceptado en la salida de la estructura 
    PdsNeuronML es PDS_DEFAULT_MAX_RMS_ERROR.</li>
 *  <li>El valor de de Alpha se inicia con PDS_DEFAULT_ALPHA.</li>
 *  <li>El valor U y W son iniciados aleatoriamente entre -1.0 1.0.</li>
 *  <li>El valor func(x) se inicia con tanh(x/2).</li>
 *  </ul> 
 *  
 *  \param[in] L Es un arreglo con el numero de elementos por capas.
 *  \param[in] Nel Número de elementos del arreglo.
 *  \return Un puntero al vector de tipo PdsNeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNeuronML *pds_neuronml_new_from_array(const PdsNnNatural *L,PdsNnNatural Nel);

/** \fn PdsNeuronML *pds_neuronml_new_load_data(const char *mlayer_filename)
 *  \brief Crea una estructura de tipo PdsNeuronML e inicia todos los elementos.
 *  desde un archivo de texto.
 *
 *  Los datos cargados son :
 *  <ul>
 *  <li>N, el numero de capas. </li>
 *  <li>MaxError, El máximo margen de error RMS aceptado en la salida.</li>
 *  <li>Alpha, El factor de aprendizaje.</li>
 *  <li>MaxIter, El máximo número de iteraciones.</li>
 *  <li>Iter, El número de iteraciones del último entrenamiento</li>.
 *  <li>ShowPrintf, Si la estructura muestra información adicional.</li>
 *  <li>L, Vector de N+1 elementos con el número de neuronas por capa.</li>
 *  <li>Vector de pesos de neurona.</li>
 *  <li>...</li>
 *  </ul> 
 *  
 *  \param[in] mlayer_filename Es el archivo de texto de donde se leen los datos.
 *  \return Un puntero al vector de tipo PdsNeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNeuronML *pds_neuronml_new_load_data(const char *mlayer_filename);


/** \fn int pds_neuronml_save_data(const PdsNeuronML *NeuronML, const char *mlayer_filename)
 *  \brief Salva una estructura de tipo PdsNeuronML en un archivo de texto.
 *
 *  Los datos salvados son :
 *  <ul>
 *  <li>N, el numero de capas. </li>
 *  <li>MaxError, El máximo margen de error RMS aceptado en la salida.</li>
 *  <li>Alpha, El factor de aprendizaje.</li>
 *  <li>MaxIter, El máximo número de iteraciones.</li>
 *  <li>Iter, El número de iteraciones del último entrenamiento</li>.
 *  <li>ShowPrintf, Si la estructura muestra información adicional.</li>
 *  <li>L, Vector de N+1 elementos con el número de neuronas por capa.</li>
 *  <li>Vector de pesos de neurona.</li>
 *  <li>...</li>
 *  </ul> 
 *  
 *  \param[in] NeuronML Es la red neuronal multicapa a salvar.
 *  \param[in] mlayer_filename Es el archivo de texto donde se guardará la red.
 *  \return TRUE si todo fue bien o FALSE si no. (ejem: NeuronML==NULL o mlayer_filename=NULL).
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_save_data(const PdsNeuronML *NeuronML, const char *mlayer_filename);


//@}

/** @name pds_neuronml_<get_seT_configuration_methods>
 *  Métodos que alteran o describen el comportamiento de la estructura.
 *
 * @{
 */


/** \fn int pds_neuronml_set_max_error(PdsNeuronML *NeuronML,PdsNnReal MaxError)
 *  \brief Coloca el maximo margen error RMS aceptado en la salida de la estructura
 *  PdsNeuronML.
 *
 *  Cualquier valor de error que es menor
 *  provoca que se detenga el algoritmo de aprendizaje.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] MaxError Máximo error aceptado.
 *  El valor cero no es aceptado, en su defecto es escrito PDS_DEFAULT_MAX_RMS_ERROR.
 *  Valores negativos no son aceptados, en su defecto es escrito |MaxError|.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_max_error(PdsNeuronML *NeuronML,PdsNnReal MaxError);


/** \fn int pds_neuronml_set_max_iterations(PdsNeuronML *NeuronML,PdsNnNatural MaxIter)
 *  \brief Coloca el máximo numero de iteraciones aceptado, cualquier valor mayor
 *  provoca que se detenga el algoritmo de aprendizaje pds_neuronml_training(). si este número es cero
 *  entonces el algoritmo continua hasta que el error en la salida sea menor que
 *  NeuronML->Error.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] MaxIter Número máximo de iteraciones.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_max_iterations(PdsNeuronML *NeuronML,PdsNnNatural MaxIter);


/** \fn int pds_neuronml_enable_printf(PdsNeuronML *NeuronML)
 *  \brief Habilita la muestra en pantalla del avance del aprendizaje.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_enable_printf(PdsNeuronML *NeuronML);

/** \fn int pds_neuronml_get_number_of_input_neurons(PdsNeuronML *NeuronML,PdsNnNatural *Number)
 *  \brief Retorna o numero de neuronas de entrada.
 *  \param[in] NeuronML Red neuronal Multi capa a cosultar.
 *  \param[out] Number Número de neuronas de entrada.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_number_of_input_neurons(PdsNeuronML *NeuronML,PdsNnNatural *Number);

/** \fn int pds_neuronml_get_number_of_output_neurons(PdsNeuronML *NeuronML,PdsNnNatural *Number)
 *  \brief Retorna o numero de neuronas de salida.
 *  \param[in] NeuronML Red neuronal Multi capa a cosultar.
 *  \param[out] Number Número de neuronas de salida.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_number_of_output_neurons(PdsNeuronML *NeuronML,PdsNnNatural *Number);

/** \fn int pds_neuronml_get_maximum_output(PdsNeuronML *NeuronML,PdsNnReal *Max)
 *  \brief Retorna o valor maximo de las neuronas de salida
 *  \param[in] NeuronML Red neuronal Multi capa a cosultar.
 *  \param[out] Max Valor máximo de las neuronas de salida.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_maximum_output(PdsNeuronML *NeuronML,PdsNnReal *Max);

/** \fn int pds_neuronml_get_minimum_output(PdsNeuronML *NeuronML,PdsNnReal *Min)
 *  \brief Retorna o valor maximo de las neuronas de salida
 *  \param[in] NeuronML Red neuronal Multi capa a cosultar.
 *  \param[out] Min Valor mínimo de las neuronas de salida.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_minimum_output(PdsNeuronML *NeuronML,PdsNnReal *Min);

//@}

/** @name pds_neuronml_<u_weight_methods>
 *  Métodos que alteran las variables de la estructura.
 *
 * @{
 */


/** \fn int pds_neuronml_init_weight_uniform(PdsNeuronML *NeuronML,PdsUniform *RV)
 *  \brief Inicializa todos los pesos usando una variable aleatoria uniforme.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, RV==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_init_weight_uniform(PdsNeuronML *NeuronML,PdsUniform *RV);

/** \fn int pds_neuronml_init_weight_rand(PdsNeuronML *NeuronML,PdsSnReal min, PdsSnReal max)
 *  \brief Inicializa todos los pesos entre los valores min y max, 
 *  usando internamente la funcion rand.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] min Valor mínimo.
 *  \param[in] max Valor máximo.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_init_weight_rand(PdsNeuronML *NeuronML,PdsSnReal min, PdsSnReal max);


/** \fn int pds_neuronml_init_u_uniform(PdsNeuronML *NeuronML,PdsUniform *RV)
 *  \brief Inicializa todos los umbrales U usando una variable aleatoria uniforme.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, RV==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_init_u_uniform(PdsNeuronML *NeuronML,PdsUniform *RV);


/** \fn int pds_neuronml_init_u_rand(PdsNeuronML *NeuronML,PdsSnReal min, PdsSnReal max)
 *  \brief Inicializa todos los umbrales U de forma aleatoria entre min y max.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] min Valor mínimo.
 *  \param[in] max Valor máximo.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_init_u_rand(PdsNeuronML *NeuronML,PdsSnReal min, PdsSnReal max);

//@}

/** @name pds_neuronml_<iterate_methods>
 *  Métodos sobre iteración de la estructura.
 *
 * @{
 */


/** \fn int pds_neuronml_iterate(PdsNeuronML *NeuronML,const PdsVector *Input,PdsVector *Output)
 *  \brief Itera la red neuronal multicapa NeuronML.
 *  \f[ \theta_j=\sum_{k=0}^{N_{d}-1}{W_{kj}X_k} - U_j \f]
 *  \f[ y_j \leftarrow func\left(\theta_j\right) \f]
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Input Datos de entrada de la red neuronal multicapa.
 *  \param[out] Output Datos de salida de la red neuronal multicapa.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, o tamaños de
 *  vectores incompatibles). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_iterate(PdsNeuronML *NeuronML,const PdsVector *Input,PdsVector *Output);

/** \fn int pds_neuronml_get_number_iterations(const PdsNeuronML *NeuronML,PdsNnNatural *Iter)
 *  \brief Devuelve el número de iteraciones del último entrenamiento hecho con
 *  la función pds_neuronml_training().
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Iter Número de iteraciones del último entrenamiento.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_number_iterations(const PdsNeuronML *NeuronML,PdsNnNatural *Iter);


/** \fn PdsNnBool pds_neuronml_iterate_results_printf(const PdsNeuronML *MLayer,const char* Type)
 *  \brief Muestra los resultados de la iteración de la red neuronal multicapa NeuronML.
 *
 *  \param[in] MLayer Red neuronal Multi capa a trabajar.
 *  \param[in] Type Etiqueda da iteracion.
 *  \return El valor medio de las salidas de la red neuronal multicapa NeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNnBool pds_neuronml_iterate_results_printf(const PdsNeuronML *MLayer,const char* Type);

/** \fn int pds_neuronml_get_output(const PdsNeuronML *NeuronML,PdsVector *Y)
 *  \brief Devuelve la salida Y de la red neuronal.
 *  \param[in] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[out] Y Vector donde se cargará los valores de la salida Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o Y==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_output(const PdsNeuronML *NeuronML,PdsVector *Y);


/** \fn int pds_neuronml_get_output_soft_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote)
 *  \brief Evalúa una votación con desición suave de los valores de salida
 *  de la red neuronal multicapa. 
 *
 *  Se realiza una suma simple de todos los valores de salida.
 *  \param[in] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[out] Vote Valor de la votación suave.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_output_soft_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote);



/** \fn int pds_neuronml_get_output_rms_error(const PdsNeuronML *NeuronML,PdsNnReal *RmsError)
 *  \brief Evalúa el error cuadrático medio de todos los errores Y[1] de las
 *  neuronas de salida de la red neuronal.
 *  \param[in] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] RmsError Valor raíz cuadrático medio.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_output_rms_error(const PdsNeuronML *NeuronML,PdsNnReal *RmsError);

//@}

/** @name pds_neuronml_<training_methods>
 *  Métodos sobre entrenamiento de la estructura.
 *
 * @{
 */

/** \fn int pds_neuronml_training(PdsNeuronML *NeuronML,const PdsVector *Input,PdsVector *Output)
 *  \brief Entrena la red neuronal multicapa NeuronML. Usa el valor de Output como 
 *  entrenamiento, para finalizar carga el nuevo valor de salida en Output.
 *
 *  Bucle de retropropagación e iteración :
 *  <ul>
 *  <li>pds_nvector_evaluate_diff_error()</li>
 *  <li>pds_nvector_backpropagate_error() : \f$ e_k \leftarrow e_k + \frac{\alpha}{N_j} func'\left(\theta_j\right) W_{kj} e_j \f$ </li>
 *  <li>pds_nvector_update_u_and_weight() : \f$ W_{kj} = W_{kj} + \alpha e_j func'\left(\theta_j\right) \frac{X_{k}}{X^TX} \f$</li>
 *  <li>pds_nvector_update_u_and_weight() : \f$ U_{k} = U_{k} - \alpha e_j func'\left(\theta_j\right) \frac{1}{X^TX} \f$</li>
 *  <li>pds_neuronml_iterate()</li>
 *  </ul>
 *  El bucle es repetido hasta que el error cuadratico medio en la salida sea menor
 *  NeuronML->MaxError o que el numero de iteraciones sea mayor que NeuronML->MaxIter
 *  (Solo si NeuronML->MaxIter es diferente de cero).
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Input Datos de entrada de la red neuronal multicapa.
 *  \param[in,out] Output Datos esperados de salida de la red neuronal multicapa.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, o tamaños de
 *  vectores incompatibles). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_training(PdsNeuronML *NeuronML,const PdsVector *Input,PdsVector *Output);

/** \fn PdsNnNatural pds_neuronml_training_results_printf(PdsNeuronML *MLayer,const char* Type)
 *  \brief Muestra los resultados del entrenamiento de la red neuronal multicapa NeuronML.
 *
 *  \param[in] MLayer Red neuronal Multi capa a trabajar.
 *  \param[in] Type Etiqueda del entrenamiento.
 *  \return El valor medio de las salidas de la red neuronal multicapa NeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNnNatural pds_neuronml_training_results_printf(PdsNeuronML *MLayer,const char* Type);

/** \fn int pds_neuronml_set_alpha(PdsNeuronML *NeuronML,PdsNnReal Alpha)
 *  \brief Coloca el valor del factor de aprendizaje Alpha.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Alpha Factor de aprendizaje.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_alpha(PdsNeuronML *NeuronML,PdsNnReal Alpha);


//@}


/** @name pds_neuronml_<fprintf>
 *  Funciones para leer y escribir datos de {Pesos, U} de todas las neuronas.
 *
 * @{
 */


/** \fn int pds_neuronml_head_printf(const PdsNeuronML *NeuronML)
 *  \brief Mostra em pantalla los datos de cabecera de la red neuronal multicapa
 *  \param[in] NeuronML La estructura multicapa a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_head_printf(const PdsNeuronML *NeuronML);


/** \fn int pds_neuronml_head_fprintf(const PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Escribe los datos de cabecera de la red neuronal.
 *  En una linea separando los elementos por un TAB.
 *  {NeuronML->N,NeuronML->MaxError,NeuronML->Alpha,NeuronML->MaxIter,NeuronML->Iter,NeuronML->ShowPrintf}
 *  En la siguiente linea separando los elementos por un TAB, el numero de neuronas por capa.
 *  \param[in] NeuronML La estructura multicapa a leer.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_head_fprintf(const PdsNeuronML *NeuronML, FILE *fd);

/** \fn PdsNeuronML *pds_neuronml_new_head_fscanf(FILE *fd)
 *  \brief Lee los datos de cabecera de una red neuronal desde un archivo
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return Un puntero al vector de tipo PdsNeuronML o NULL en caso de error.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNeuronML *pds_neuronml_new_head_fscanf(FILE *fd);

/** \fn int pds_neuronml_body_fprintf(const PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Guarda en un archivo de texto los pesos W[i], el valor de U.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  (neurona por neurona)
 *  \param[in] NeuronML La estructura multicapa a leer.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_body_fprintf(const PdsNeuronML *NeuronML, FILE *fd);


/** \fn int pds_neuronml_body_fscanf(PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Lee de un archivo de texto los pesos W[i], el valor de U.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  (neurona por neurona)
 *  \param[out] NeuronML La estructura multicapa a escribir.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_body_fscanf(PdsNeuronML *NeuronML, FILE *fd);


/** \fn int pds_neuronml_body_fwrite(const PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Guarda en un archivo binario los pesos W[i], el valor de U.
 *  (neurona por neurona)
 *  \param[in] NeuronML La estructura multicapa a leer.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_body_fwrite(const PdsNeuronML *NeuronML, FILE *fd);


/** \fn int pds_neuronml_body_fread(PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Lee de un archivo binario los pesos W[i], el valor de U.
 *  (neurona por neurona)
 *  \param[out] NeuronML La estructura multicapa a escribir.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_body_fread(PdsNeuronML *NeuronML, FILE *fd);

//@}

/** @name pds_neuronml_<free>
 *  Liberar memoria.
 *
 * @{
 */

/** \fn void pds_neuronml_free(PdsNeuronML *Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuronML.
 *  \param[in,out] Neuron La neurona a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronMLGroup
 */
void pds_neuronml_free(PdsNeuronML *NeuronML);


/** \fn void pds_neuronml_destroy(PdsNeuronML **Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuronML, y limpia el puntero con NULL.
 *  \param[in,out] Neuron La neurona a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronMLGroup
 */
void pds_neuronml_destroy(PdsNeuronML **NeuronML);

//@}

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


