/*
 * pdsgeneratedata.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsgeneratedata.h
 *  \author Fernando Pujaico Rivera
 *  \date 05-06-2020
 *  \brief Funciones que trabajan con datos.
 *  
 */

#ifndef __PDSGENERATEDATA_H__
#define __PDSGENERATEDATA_H__

#ifdef __cplusplus
extern "C" {
#endif 


#include <pds/pdsnnglobal.h>
#include <pds/pdsra.h>
#include <pds/pdsrv.h>


#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif


/** \defgroup PdsGenerateDataGroup Módulo para creacion de datos.
 *  \brief Funciones que trabajan con datos.
 *  
 * @{
 */


/** \fn double pds_generate_data_func1(double x)
 *  \brief Genera una funcion de periodo 2*pi y dominio entre -1 y 1.
 *
 *  \param[out] x Valor de entrada.
 *  \return Retorna la funcion avaliada.
 *  \ingroup PdsGenerateDataGroup
 */
double pds_generate_data_func1(double x);

/** \fn int pds_generate_data_input_func2pi(PdsVector *X,double (*func)(double),PdsNnReal C,PdsNnReal OffsetError)
 *  \brief Genera un vector con C ciclos de la función func() 
 *  con un error de offset de valor OffsetError.
 *
 *  \param[out] X Vector onde se colocaran los datos.
 *  \param[in] func funcion de periodo 2 pi.
 *  \param[in] C Número de veces que se repite la funcion func() en el vector X.
 *  \param[in] OffsetError Error no offset de cada muestra del vector.
 *  \return TRUE si todo fue bien o falso si no (ex: X==NULL ).
 *  \ingroup PdsGenerateDataGroup
 */
int pds_generate_data_input_func2pi(PdsVector *X,double (*func)(double),PdsNnReal C,PdsNnReal OffsetError);


/** \fn int pds_generate_data_input_rand(PdsVector *X)
 *  \brief Genera un vector con valores uniformemente distribuidos 
 *  entre -|A| y +|A| para los elementos del vector X.
 *
 *  \param[out] X Vector onde se colocaran los datos.
 *  \param[in] A Valor limite de los números aleatorios.
 *  \return TRUE si todo fue bien o falso si no (ex: X==NULL ).
 *  \ingroup PdsGenerateDataGroup
 */
int pds_generate_data_input_rand(PdsVector *X,PdsNnReal A);


/** \fn int pds_generate_data_output_negative(PdsVector *Y)
 *  \brief Genera un vector con -1 en todos los elementos del vector Y.
 *
 *  \param[out] Y Vector onde se colocaran los datos.
 *  \return TRUE si todo fue bien o falso si no (ex: Y==NULL ).
 *  \ingroup PdsGenerateDataGroup
 */
int pds_generate_data_output_negative(PdsVector *Y);


/** \fn int pds_generate_data_output_positive(PdsVector *Y)
 *  \brief Genera un vector con +1 en todos los elementos del vector Y.
 *
 *  \param[out] Y Vector onde se colocaran los datos.
 *  \return TRUE si todo fue bien o falso si no (ex: Y==NULL ).
 *  \ingroup PdsGenerateDataGroup
 */
int pds_generate_data_output_positive(PdsVector *Y);


/** \fn int pds_generate_data_output_mspace(PdsVector *Y,PdsRaNatural M)
 *  \brief Genera un vector con +1 en la posición M del vector Y,
 *  los otros valores son -1.
 *
 *  \param[out] Y Vector onde se colocaran los datos.
 *  \param[in] M Posición del +1.
 *  \return TRUE si todo fue bien o falso si no (ex: Y==NULL o Y->Nel<M ).
 *  \ingroup PdsGenerateDataGroup
 */
int pds_generate_data_output_mspace(PdsVector *Y,PdsRaNatural M);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


