/*
 * pdsnvector.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsra.h>
#include <pds/pdsrv.h>
#include <pds/pdssn.h>
#include <pds/pdsnivector.h>
#include <pds/pdsnvector.h>


/** \fn PdsNVector *pds_nvector_new(PdsNnNatural Nel,PdsNnNatural Nd)
 *  \brief Crea un vector de neuronas de tipo PdsNVector e inicia con cero todos 
 *  los pesos,  U=1.0.
 *  \param[in] Nel Es el número de elementos del vector.
 *  \param[in] Nd Es el número de entradas de cada neurona del vector.
 *  \return Un puntero al vector de tipo PdsNVector.
 *  \ingroup PdsNVectorGroup
 */
PdsNVector *pds_nvector_new(PdsNnNatural Nel,PdsNnNatural Nd)
{
	PdsNVector *NVector=NULL;
	PdsNnNatural i,j;

	if(Nel<=0)		return NULL;
	
	// Reservo memoria de la estructura.
	NVector=(PdsNVector *)calloc(1,sizeof(PdsNVector));
	if(NVector==NULL)	return NULL;

	// Numero de elementos del vector.
	NVector->Nel=Nel;

	NVector->V=(PdsNeuron **)calloc(NVector->Nel,sizeof(PdsNeuron *));
	if(NVector->V==NULL) 
	{
		free(NVector);
		return NULL;
	}

	for(i=0;i<NVector->Nel;i++)
	{	
		NVector->V[i]=pds_neuron_new(Nd);
		if(NVector->V[i]==NULL)
		{
			for(j=0;j<i;j++)	pds_neuron_free(NVector->V[j]);
			free(NVector->V);
			free(NVector);
		}
	}

	return NVector;
}


/** \fn int pds_nvector_init_weight_uniform(PdsNVector *NVector,PdsUniform *RV)
 *  \brief Inicia aleatoriamente siguiendo una distribución uniforme, los pesos
 *  de todas las neuronas del vector.
 *  \param[in,out] NVector El vector a escribir.
 *  \param[in,out] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_init_weight_uniform(PdsNVector *NVector,PdsUniform *RV)
{
	PdsNnNatural i,j;
	PdsNnReal m;

	if(NVector==NULL)	return FALSE;
	if(RV     ==NULL)	return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		for(j=0;j<NVector->V[i]->Nel;j++)
		{
			pds_uniform_get_value(RV,&m);
			pds_neuron_set_weight(NVector->V[i],j,m);
		}
	}

	return TRUE;
}

/** \fn int pds_nvector_init_weight_rand(PdsNVector *NVector,PdsSnReal min, PdsSnReal max)
 *  \brief Inicia aleatoriamente siguiendo una distribución uniforme entre min e max
 *  en todas las neuronas del vector, usando la funcion rand.
 *  \param[in,out] NVector El vector a escribir.
 *  \param[in] min valor mínimo.
 *  \param[in] max valor máximo.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_init_weight_rand(PdsNVector *NVector,PdsSnReal min, PdsSnReal max)
{
	PdsNnNatural i;

	if(NVector==NULL)	return FALSE;

	for(i=0;i<NVector->Nel;i++)
        pds_neuron_init_weight_rand(NVector->V[i],min,max);

	return TRUE;
}

/** \fn int pds_nvector_init_u_uniform(PdsNVector *NVector,PdsUniform *RV)
 *  \brief Coloca de forma aleatoria siguiendo una distribucion uniforme, el 
 *  valor de  U de las Neuronas.
 *  \param[in,out] NVector El vector de neurons a escribir.
 *  \param[in] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_init_u_uniform(PdsNVector *NVector,PdsUniform *RV)
{
	PdsNnNatural i;
	PdsNnReal m;

	if(NVector==NULL)	return FALSE;
	if(RV     ==NULL)	return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		pds_uniform_get_value(RV,&m);
		NVector->V[i]->U=m;
	}

	return TRUE;
}

/** \fn int pds_nvector_init_u_rand(PdsNVector *NVector,PdsSnReal min, PdsSnReal max)
 *  \brief Coloca de forma aleatoria valores entre min y max, en el 
 *  valor de  U de todas las Neuronas, es usada la funcion rand.
 *  \param[in,out] NVector El vector de neurons a escribir.
 *  \param[in] min valor mínimo.
 *  \param[in] max valor máximo.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_init_u_rand(PdsNVector *NVector,PdsSnReal min, PdsSnReal max)
{
	PdsNnNatural i;

	if(NVector==NULL)	return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		NVector->V[i]->U=min+(rand()*(max-min))/RAND_MAX;
	}

	return TRUE;
}


/** \fn int pds_nvector_get_output(const PdsNVector *NVector,PdsVector *Y)
 *  \brief Devuelve el valor de la salida Y de las Neuronas.
 *  \param[in] NVector El vector de neuronas en consulta.
 *  \param[out] Y Vector donde se cargará los valores de la salida Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL, Y==NULL o
 *  tamaños distintos). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_get_output(const PdsNVector *NVector,PdsVector *Y)
{
	PdsNnNatural i;

	if(NVector==NULL)		return FALSE;
	if(Y      ==NULL)		return FALSE;
	if(NVector->Nel!=Y->Nel)	return FALSE;

	for(i=0;i<NVector->Nel;i++)	Y->V[i]=*(NVector->V[i]->Y);

	return TRUE;
}


/** \fn int pds_nvector_connect_input(PdsNVector *NVector,PdsNVector *NVectorBefore)
 *  \brief Conecta el valor de salida Y de las Neuronas del vector NVectorBefore con 
 *  las entradas X[id] de las neuronas del vector NVector.
 *
 *  Si existen mas entradas de NVector que salidas de NVectorBefore las entradas quedan
 *  desconectadas cargadas con NULL.
 *  Si existen menos entradas de NVector que salidas de NVectorBefore las salidas quedan
 *  desconectadas.
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \param[in,out] NVectorBefore El vector de neuronas que se conectará a las entradas 
 *  X[id] de NVector.
 *  \return TRUE si todo fue bien o FALSE si no(ej: NVector==NULL, NVectorBefore==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_connect_input(PdsNVector *NVector,PdsNVector *NVectorBefore)
{
	PdsNnNatural i,j;

	if(NVector ==NULL)		return FALSE;
	if(NVectorBefore==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		for(j=0;(j<NVector->V[i]->Nel)&&(j<NVectorBefore->Nel);j++)
		NVector->V[i]->X[j]=NVectorBefore->V[j]->Y;

		for(;j<NVector->V[i]->Nel;j++)
		NVector->V[i]->X[j]=NULL;
	}
	return TRUE;
}


/** \fn int pds_nvector_connect_input_with_nivector(PdsNVector *NVector,PdsNIVector *X)
 *  \brief Conecta los elementos del vector X de neuronas de entradas, con las 
 *  entradas X[id] de las neuronas del vector NVector.
 *
 *  Si existen mas entradas de NVector que elementos de X las entradas quedan
 *  desconectadas cargadas con NULL.
 *  Si existen menos entradas de NVector que elementos de X los elementos quedan
 *  desconectadas.
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \param[in,out] X El vector que se conectará a las entradas X[id] de NVector.
 *  \return TRUE si todo fue bien o FALSE si no(ej: NVector==NULL, X==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_connect_input_with_nivector(PdsNVector *NVector,PdsNIVector *X)
{
	PdsNnNatural i,j;

	if(NVector==NULL)		return FALSE;
	if(X      ==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		for(j=0;(j<NVector->V[i]->Nel)&&(j<X->Nel);j++)
		NVector->V[i]->X[j]=X->V[j]->Y;

		for(;j<NVector->V[i]->Nel;j++)
		NVector->V[i]->X[j]=NULL;
	}
	return TRUE;
}


/** \fn int pds_nvector_evaluate_theta(PdsNVector *NVector)
 *  \brief Evalua la variable theta de la Neurona Neuron.
 *  \f[ \theta_j=\sum_{k=0}^{N_{d}-1}{W_{kj}X_k} - U_j \f]
 *  \param[in,out] NVector El vector de neuronas a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_evaluate_theta(PdsNVector *NVector)
{
	PdsNnNatural j;
    int dat;

	if(NVector==NULL)		return FALSE;

	for(j=0;j<NVector->Nel;j++)
	{
        if((NVector->V[j])==NULL)  return FALSE;
        dat=pds_neuron_evaluate_theta(NVector->V[j]);
        if(dat==FALSE)  return FALSE;
	}
	return TRUE;
}


/** \fn int pds_nvector_evaluate_func(PdsNVector *NVector)
 *  \brief Evalua la funcion de activación de las Neuronas de NVector.
 *  \f[ func\left(\theta_j\right) \f]
 *  \param[in,out] NVector El vector de neuronas a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_evaluate_func(PdsNVector *NVector)
{
	int dat;
	PdsNnNatural i;

	if(NVector==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
        if((NVector->V[i])==NULL)  return FALSE;
        dat=pds_neuron_evaluate_func(NVector->V[i]);
        if(dat==FALSE)  return FALSE;
	}
	return TRUE;
}



/** \fn int pds_nvector_iterate(PdsNVector *NVector)
 *  \brief Itera las neuronas del vector NVector.
 *  \f[ \theta_j=\sum_{k=0}^{N_{d}-1}{W_{kj}X_k} - U_j \f]
 *  \f[ y_j \leftarrow func\left(\theta_j\right) \f]
 *  \param[in,out] NVector El vector de neuronas a iterar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_iterate(PdsNVector *NVector)
{
	int m;

	if(NVector==NULL)		return FALSE;

	m=pds_nvector_evaluate_theta(NVector);
	if(m==FALSE)	return FALSE;
	m=pds_nvector_evaluate_func(NVector);
	if(m==FALSE)	return FALSE;

	return TRUE;
}


/** \fn int pds_nvector_evaluate_diff_error(PdsNVector *NVector,const PdsVector * R)
 *  \brief Compara la salida de las neuronas del vector NVector con el vector de 
 *  referencia R, el resultado es cargado en la salida Y[1] de cada neurona del 
 *  vector NVector en  NVector->V[i]->Y[1].
 *
 *  Los tamaños de los vectores deben ser iguales.
 *  \param[in,out] NVector El vector de neuronas a comparar.
 *  \param[in]  R El vector a comparar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_evaluate_diff_error(PdsNVector *NVector,const PdsVector * R)
{
	PdsNnNatural i;

	if(NVector==NULL)		return FALSE;
	if(NVector->Nel!= R->Nel)	return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
        if((NVector->V[i])==NULL)  return FALSE;

		NVector->V[i]->Y[1]=R->V[i]-NVector->V[i]->Y[0];
		//pds_neuron_evaluate_error(NVector->V[i],R->V[i]);
	}

	return TRUE;
}


/** \fn int pds_nvector_get_rms_error(const PdsNVector *NVector,PdsNnReal *RmsError)
 *  \brief Evalúa el error cuadrático medio de todos los errores Y[1] de las.
 *  neuronas del vector.
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \param[in] RmsError Valor raíz cuadrático medio.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_get_rms_error(const PdsNVector *NVector,PdsNnReal *RmsError)
{
	PdsNnNatural i;

	*RmsError=0;

	if(NVector==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
        if((NVector->V[i])==NULL)  return FALSE;

		(*RmsError)=(*RmsError)+NVector->V[i]->Y[1]*NVector->V[i]->Y[1];
	}
	(*RmsError)=sqrt((*RmsError)/NVector->Nel);

	return TRUE;
}


/** \fn int pds_nvector_get_soft_vote(const PdsNVector *NVector,PdsNnReal *Vote)
 *  \brief Evalúa una votación de todos los valores de salida Y[0] de las
 *  neuronas, se realiza una suma simple de todos los valores de salida.
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \param[out] Vote Valor promedio de la votación suave.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_get_soft_vote(const PdsNVector *NVector,PdsNnReal *Vote)
{
	PdsNnNatural i;

	*Vote=0.0;

	if(NVector==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
        if((NVector->V[i])==NULL)  return FALSE;

		(*Vote)=(*Vote)+NVector->V[i]->Y[0];
	}

    (*Vote)=(*Vote)/NVector->Nel;
	return TRUE;
}

/** \fn int pds_nvector_backpropagate_error(PdsNVector *NVector,PdsNnReal Alpha)
 *  \brief Retro propaga el error de las neuronas del vector NVector
 *  el resultado es cargado en la salida Y[1] de cada neurona del vector NVector
 *  conectada a las entradas.
 *  \f[ e_k \leftarrow e_k + \frac{\alpha}{N_j} func'\left(\theta_j\right) W_{kj} e_j \f]
 *
 *  \image html backpropagation.png "Retropropagación del error en las neuronas." 
 *  \image html nvector_error_prop.png "Retropropagación del error en las neuronas." 
 *  \param[in,out] NVector El vector de neuronas a retropropagar el error.
 *  \param[in] Alpha Factor de aprendizaje de los errores.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_backpropagate_error(PdsNVector *NVector,PdsNnReal Alpha)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
        if((NVector->V[i])==NULL)  return FALSE;

		m=pds_neuron_backpropagate_error(NVector->V[i],Alpha);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_nvector_update_u_and_weight(PdsNVector *NVector, PdsNnReal Alpha)
 *  \brief Actualiza los pesos W[k]j de las neuronas.
 *  \f[ \triangle W_{kj}= \alpha e_j func'\left(\theta_j\right) X_k \f]
 *  \f[ \triangle U_{j}= -\alpha e_j func'\left(\theta_j\right) \f]
 *  "ej" es el error de la salida de la j-essima neurona.
 *  "func" es la función de activación da neurona.
 *  \param[in,out] NVector El vector de neuronas a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_update_u_and_weight(PdsNVector *NVector, PdsNnReal Alpha)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		m=pds_neuron_update_u_and_weight(NVector->V[i],Alpha);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}




/** \fn int pds_nvector_fprintf(const PdsNVector *NVector, FILE *fd)
 *  \brief Guarda en un archivo de texto los pesos \f$W_{kj}\f$ y los valores \f$\{U_j\}\f$.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[in] NVector El vector de neuronas a leer.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_fprintf(const PdsNVector *NVector, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		m=pds_neuron_fprintf(NVector->V[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_nvector_fscanf(PdsNVector *NVector, FILE *fd)
 *  \brief Lee de un archivo de texto los pesos \f$W_{kj}\f$ y los valores \f$\{U_j\}\f$.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[out] NVector El vector a escribir.
 *  \param[in,out] fd Manejador del fichero a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_fscanf(PdsNVector *NVector, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		if(feof(fd)!=0)	return FALSE;
		m=pds_neuron_fscanf(NVector->V[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_nvector_fwrite(const PdsNVector *NVector, FILE *fd)
 *  \brief Guarda en un archivo de texto binario los pesos \f$W_{kj}\f$ y los valores \f$\{U_j\}\f$.
 *  \param[in] NVector El vector de neuronas a leer.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_fwrite(const PdsNVector *NVector, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		m=pds_neuron_fwrite(NVector->V[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_nvector_fread(PdsNVector *NVector, FILE *fd)
 *  \brief Lee de un archivo de texto binario los pesos \f$W_{kj}\f$ y los valores \f$\{U_j\}\f$.
 *  \param[out] NVector El vector a escribir.
 *  \param[in,out] fd Manejador del fichero binario a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_fread(PdsNVector *NVector, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		if(feof(fd)!=0)	return FALSE;
		m=pds_neuron_fread(NVector->V[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn void pds_nvector_free(PdsNVector *NVector)
 *  \brief Libera un vector de neuronas de tipo puntero PdsNVector.
 *  \param[in,out] NVector el vector de neuronas a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNVectorGroup
 */
void pds_nvector_free(PdsNVector *NVector)
{
	PdsNnNatural i;
	if(NVector!=NULL)
	{
		for(i=0;i<NVector->Nel;i++)	pds_neuron_free(NVector->V[i]);
		free(NVector);
	}
}


/** \fn void pds_nvector_destroy(PdsNVector **NVector)
 *  \brief Libera un vector de neuronas de tipo puntero PdsNVector, y limpia el 
 *   puntero con NULL.
 *  \param[in,out] NVector El vector de neuronas a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNVectorGroup
 */
void pds_nvector_destroy(PdsNVector **NVector)
{
	PdsNnNatural i;
	if((*NVector)!=NULL)
	{
		for(i=0;i<(*NVector)->Nel;i++)	pds_neuron_destroy(&((*NVector)->V[i]));
		free(*NVector);
		(*NVector)=NULL;
	}
}

