/*
 * pdsneuronml.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <float.h>

#include <pds/pdsra.h>
#include <pds/pdsrv.h>
#include <pds/pdssn.h>
#include <pds/pdsnn.h>


/** \fn PdsNeuronML *pds_neuronml_new(const PdsVector *L)
 *  \brief Crea una estructura de tipo PdsNeuronML.
 *
 *  <ul>
 *  <li>Inicia con cero todos los Pesos. </li>
 *  <li>El máximo margen de error RMS aceptado en la salida de la estructura 
    PdsNeuronML es PDS_DEFAULT_MAX_RMS_ERROR.</li>
 *  <li>El valor de de Alpha se inicia con PDS_DEFAULT_ALPHA.</li>
 *  <li>El valor U y W son iniciados aleatoriamente entre -1.0 1.0.</li>
 *  <li>El valor func(x) se inicia con tanh(x/2).</li>
 *  </ul> 
 *  
 *  \param[in] L Es un vector con el numero de elementos por capas.
 *  \return Un puntero al vector de tipo PdsNeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNeuronML *pds_neuronml_new(const PdsVector *L)
{
	PdsNeuronML *NeuronML=NULL;
	PdsNnNatural i;

	if(L==NULL)		return NULL;
	if(L->Nel<2)	return NULL;
	
    PdsNnNatural *Larray=(PdsNnNatural *)calloc(L->Nel,sizeof(PdsNnNatural));

    for(i=0;i<L->Nel;i++)
    {
        Larray[i]=(PdsNnNatural)fabs(L->V[i]);
    }

	// Reservo memoria de la estructura.
	NeuronML=pds_neuronml_new_from_array(Larray,L->Nel);
	if(NeuronML==NULL)	return NULL;

	return NeuronML;
}

/** \fn PdsNeuronML *pds_neuronml_new_from_array(const PdsNnNatural *L,PdsNnNatural Nel)
 *  \brief Crea una estructura de tipo PdsNeuronML.
 *
 *  <ul>
 *  <li>Inicia con cero todos los Pesos. </li>
 *  <li>El máximo margen de error RMS aceptado en la salida de la estructura 
    PdsNeuronML es PDS_DEFAULT_MAX_RMS_ERROR.</li>
 *  <li>El valor de de Alpha se inicia con PDS_DEFAULT_ALPHA.</li>
 *  <li>El valor U y W son iniciados aleatoriamente entre -1.0 1.0.</li>
 *  <li>El valor func(x) se inicia con tanh(x/2).</li>
 *  </ul> 
 *  
 *  \param[in] L Es un arreglo con el numero de elementos por capas.
 *  \param[in] Nel Número de elementos del arreglo.
 *  \return Un puntero al vector de tipo PdsNeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNeuronML *pds_neuronml_new_from_array(const PdsNnNatural *L,PdsNnNatural Nel)
{
	PdsNeuronML *NeuronML=NULL;
	PdsNnNatural i,j;

	if(L==NULL)		return NULL;
	if(Nel<2)		return NULL;
	
    for(i=0;i<Nel;i++)  if(L[i]==0) return NULL;

	// Reservo memoria de la estructura.
	NeuronML=(PdsNeuronML *)calloc(1,sizeof(PdsNeuronML));
	if(NeuronML==NULL)	return NULL;

	// Reservo vector de capas.
	NeuronML->L=(PdsNnNatural*)calloc(Nel,sizeof(PdsNnNatural));

	if(NeuronML->L==NULL)	
	{
		free(NeuronML);
		return NULL;
	}

	for(i=0;i<Nel;i++)
	{
		NeuronML->L[i]=(PdsNnNatural)L[i];
	}

	// Numero de capas.
	NeuronML->N=Nel-1;

	// Creo vector de neuronas de entrada.
	NeuronML->LayerInput=pds_nivector_new(NeuronML->L[0]);
	if(NeuronML->LayerInput==NULL)	
	{
		free(NeuronML->L);
		free(NeuronML);
		return NULL;
	}



	NeuronML->Layer=(PdsNVector **)calloc(Nel-1,sizeof(PdsNVector *));
	if(NeuronML->Layer==NULL) 
	{
		free(NeuronML->L);
		pds_nivector_free(NeuronML->LayerInput);
		free(NeuronML);
		return NULL;
	}

	for(i=0;i<NeuronML->N;i++)
	{	
		NeuronML->Layer[i]=pds_nvector_new(NeuronML->L[i+1],NeuronML->L[i]);
		if(NeuronML->Layer[i]==NULL)
		{
			for(j=0;j<i;j++)	pds_nvector_free(NeuronML->Layer[i]);
			free(NeuronML->L);
			pds_nivector_free(NeuronML->LayerInput);
			free(NeuronML);
			return NULL;
		}
		
	}

	pds_nvector_connect_input_with_nivector(NeuronML->Layer[0],NeuronML->LayerInput);
	for(i=1;i<NeuronML->N;i++)
	{
		pds_nvector_connect_input(NeuronML->Layer[i],NeuronML->Layer[i-1]);
	}

	NeuronML->MaxError=PDS_DEFAULT_MAX_RMS_ERROR;
	NeuronML->Alpha=PDS_DEFAULT_ALPHA;
	NeuronML->MaxIter=PDS_DEFAULT_MAX_ITER;

	NeuronML->Iter=0;
	NeuronML->ShowPrintf=0;

    pds_neuronml_init_u_rand(NeuronML,-1,1);
	pds_neuronml_init_weight_rand(NeuronML,-1,1);

	return NeuronML;
}

/** \fn PdsNeuronML *pds_neuronml_new_load_data(const char *mlayer_filename)
 *  \brief Crea una estructura de tipo PdsNeuronML e inicia todos los elementos.
 *  desde un archivo de texto.
 *
 *  Los datos cargados son :
 *  <ul>
 *  <li>N, el numero de capas. </li>
 *  <li>MaxError, El máximo margen de error RMS aceptado en la salida.</li>
 *  <li>Alpha, El factor de aprendizaje.</li>
 *  <li>MaxIter, El máximo número de iteraciones.</li>
 *  <li>Iter, El número de iteraciones del último entrenamiento</li>.
 *  <li>ShowPrintf, Si la estructura muestra información adicional.</li>
 *  <li>L, Vector de N+1 elementos con el número de neuronas por capa.</li>
 *  <li>Vector de pesos de neurona.</li>
 *  <li>...</li>
 *  </ul> 
 *  
 *  \param[in] mlayer_filename Es el archivo de texto de donde se leen los datos.
 *  \return Un puntero al vector de tipo PdsNeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNeuronML *pds_neuronml_new_load_data(const char *mlayer_filename)
{
	PdsNeuronML *NeuronML=NULL;
	FILE *fd=NULL;

	if(mlayer_filename==NULL)	return NULL;

	fd=fopen(mlayer_filename,"r");
	if(fd==NULL)	return NULL;

    NeuronML=pds_neuronml_new_head_fscanf(fd);

	pds_neuronml_body_fscanf(NeuronML,fd);
	
	fclose(fd);

	return NeuronML;
}


/** \fn int pds_neuronml_save_data(const PdsNeuronML *NeuronML, const char *mlayer_filename)
 *  \brief Salva una estructura de tipo PdsNeuronML en un archivo de texto.
 *
 *  Los datos salvados son :
 *  <ul>
 *  <li>N, el numero de capas. </li>
 *  <li>MaxError, El máximo margen de error RMS aceptado en la salida.</li>
 *  <li>Alpha, El factor de aprendizaje.</li>
 *  <li>MaxIter, El máximo número de iteraciones.</li>
 *  <li>Iter, El número de iteraciones del último entrenamiento</li>.
 *  <li>ShowPrintf, Si la estructura muestra información adicional.</li>
 *  <li>L, Vector de N+1 elementos con el número de neuronas por capa.</li>
 *  <li>Vector de pesos de neurona.</li>
 *  <li>...</li>
 *  </ul> 
 *  
 *  \param[in] NeuronML Es la red neuronal multicapa a salvar.
 *  \param[in] mlayer_filename Es el archivo de texto donde se guardará la red.
 *  \return TRUE si todo fue bien o FALSE si no. (ejem: NeuronML==NULL o mlayer_filename=NULL).
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_save_data(const PdsNeuronML *NeuronML, const char *mlayer_filename)
{
	FILE *fd=NULL;
	//PdsNnNatural i;

	if(NeuronML==NULL)	return FALSE;
	if(mlayer_filename==NULL)	return FALSE;

	fd=fopen(mlayer_filename,"w");
	if(fd==NULL)	return FALSE;

    pds_neuronml_head_fprintf(NeuronML,fd);

	pds_neuronml_body_fprintf(NeuronML,fd);

	fclose(fd);

	return TRUE;
}


/** \fn int pds_neuronml_get_number_of_input_neurons(PdsNeuronML *NeuronML,PdsNnNatural *Number)
 *  \brief Retorna o numero de neuronas de entrada.
 *  \param[in] NeuronML Red neuronal Multi capa a cosultar.
 *  \param[out] Number Número de neuronas de entrada.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_number_of_input_neurons(PdsNeuronML *NeuronML,PdsNnNatural *Number)
{
	*Number=0;

	if(NeuronML==NULL)	return FALSE;

	*Number=NeuronML->L[0];

	return TRUE;
}

/** \fn int pds_neuronml_get_number_of_output_neurons(PdsNeuronML *NeuronML,PdsNnNatural *Number)
 *  \brief Retorna o numero de neuronas de salida.
 *  \param[in] NeuronML Red neuronal Multi capa a cosultar.
 *  \param[out] Number Número de neuronas de salida.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_number_of_output_neurons(PdsNeuronML *NeuronML,PdsNnNatural *Number)
{
	*Number=0;

	if(NeuronML==NULL)	return FALSE;

	*Number=NeuronML->L[NeuronML->N];

	return TRUE;
}

/** \fn int pds_neuronml_get_maximum_output(PdsNeuronML *NeuronML,PdsNnReal *Max)
 *  \brief Retorna o valor maximo de las neuronas de salida
 *  \param[in] NeuronML Red neuronal Multi capa a cosultar.
 *  \param[out] Max Valor máximo de las neuronas de salida.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_maximum_output(PdsNeuronML *NeuronML,PdsNnReal *Max)
{
	*Max=0;

	if(NeuronML==NULL)	return FALSE;

	*Max=NeuronML->Layer[NeuronML->N-1]->V[0]->func(FLT_MAX);

	return TRUE;
}

/** \fn int pds_neuronml_get_minimum_output(PdsNeuronML *NeuronML,PdsNnReal *Min)
 *  \brief Retorna o valor maximo de las neuronas de salida
 *  \param[in] NeuronML Red neuronal Multi capa a cosultar.
 *  \param[out] Min Valor mínimo de las neuronas de salida.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_minimum_output(PdsNeuronML *NeuronML,PdsNnReal *Min)
{
	*Min=0;

	if(NeuronML==NULL)	return FALSE;

	*Min=NeuronML->Layer[NeuronML->N-1]->V[0]->func(-FLT_MAX);

	return TRUE;
}

/** \fn int pds_neuronml_set_max_error(PdsNeuronML *NeuronML,PdsNnReal MaxError)
 *  \brief Coloca el maximo margen error RMS aceptado en la salida de la estructura
 *  PdsNeuronML.
 *
 *  Cualquier valor de error que es menor
 *  provoca que se detenga el algoritmo de aprendizaje.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] MaxError Máximo error aceptado.
 *  El valor cero no es aceptado, en su defecto es escrito PDS_DEFAULT_MAX_RMS_ERROR.
 *  Valores negativos no son aceptados, en su defecto es escrito |MaxError|.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_max_error(PdsNeuronML *NeuronML,PdsNnReal MaxError)
{
	PdsNnReal e;
	if(NeuronML==NULL)	return FALSE;

	e=fabs(MaxError);
	if(e!=0.0)	NeuronML->MaxError=e;
	else		NeuronML->MaxError=PDS_DEFAULT_MAX_RMS_ERROR;
	
	return TRUE;
}


/** \fn int pds_neuronml_set_alpha(PdsNeuronML *NeuronML,PdsNnReal Alpha)
 *  \brief Coloca el valor del factor de aprendizaje Alpha.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Alpha Factor de aprendizaje.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_alpha(PdsNeuronML *NeuronML,PdsNnReal Alpha)
{
	if(NeuronML==NULL)	return FALSE;

	NeuronML->Alpha=Alpha;
	
	return TRUE;
}


/** \fn int pds_neuronml_set_max_iterations(PdsNeuronML *NeuronML,PdsNnNatural MaxIter)
 *  \brief Coloca el máximo numero de iteraciones aceptado, cualquier valor mayor
 *  provoca que se detenga el algoritmo de aprendizaje pds_neuronml_training(). si este número es cero
 *  entonces el algoritmo continua hasta que el error en la salida sea menor que
 *  NeuronML->MaxError.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] MaxIter Número máximo de iteraciones.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_max_iterations(PdsNeuronML *NeuronML,PdsNnNatural MaxIter)
{

	if(NeuronML==NULL)	return FALSE;

	NeuronML->MaxIter=MaxIter;

	return TRUE;
}


/** \fn int pds_neuronml_get_number_iterations(const PdsNeuronML *NeuronML,PdsNnNatural *Iter)
 *  \brief Devuelve el número de iteraciones del último entrenamiento hecho con
 *  la función pds_neuronml_training().
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Iter Número de iteraciones del último entrenamiento.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_number_iterations(const PdsNeuronML *NeuronML,PdsNnNatural *Iter)
{
	*Iter=0;

	if(NeuronML==NULL)	return FALSE;

	*Iter=NeuronML->Iter;

	return TRUE;
}


/** \fn int pds_neuronml_enable_printf(PdsNeuronML *NeuronML)
 *  \brief Habilita la muestra en pantalla del avance del aprendizaje.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_enable_printf(PdsNeuronML *NeuronML)
{

	if(NeuronML==NULL)	return FALSE;

	NeuronML->ShowPrintf=1;

	return TRUE;
}


/** \fn int pds_neuronml_init_weight_uniform(PdsNeuronML *NeuronML,PdsUniform *RV)
 *  \brief Inicializa todos los pesos usando una variable aleatoria uniforme.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, RV==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_init_weight_uniform(PdsNeuronML *NeuronML,PdsUniform *RV)
{
	PdsNnNatural i;
	
	if(NeuronML==NULL)	return FALSE;
	if(RV      ==NULL)	return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		pds_nvector_init_weight_uniform(NeuronML->Layer[i],RV);
	}

	return TRUE;
}


/** \fn int pds_neuronml_init_weight_rand(PdsNeuronML *NeuronML,PdsSnReal min, PdsSnReal max)
 *  \brief Inicializa todos los pesos entre los valores min y max, 
 *  usando internamente la funcion rand.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] min Valor mínimo.
 *  \param[in] max Valor máximo.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_init_weight_rand(PdsNeuronML *NeuronML,PdsSnReal min, PdsSnReal max)
{
	PdsNnNatural i;
	
	if(NeuronML==NULL)	return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		pds_nvector_init_weight_rand(NeuronML->Layer[i],min,max);
	}

	return TRUE;
}


/** \fn int pds_neuronml_init_u_uniform(PdsNeuronML *NeuronML,PdsUniform *RV)
 *  \brief Inicializa todos los umbrales U usando una variable aleatoria uniforme.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, RV==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_init_u_uniform(PdsNeuronML *NeuronML,PdsUniform *RV)
{
	PdsNnNatural i;
	
	if(NeuronML==NULL)	return FALSE;
	if(RV      ==NULL)	return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		pds_nvector_init_u_uniform(NeuronML->Layer[i],RV);
	}

	return TRUE;
}

/** \fn int pds_neuronml_init_u_rand(PdsNeuronML *NeuronML,PdsSnReal min, PdsSnReal max)
 *  \brief Inicializa todos los umbrales U de forma aleatoria entre min y max.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] min Valor mínimo.
 *  \param[in] max Valor máximo.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_init_u_rand(PdsNeuronML *NeuronML,PdsSnReal min, PdsSnReal max)
{
	PdsNnNatural i;
	
	if(NeuronML==NULL)	return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		pds_nvector_init_u_rand(NeuronML->Layer[i],min,max);
	}

	return TRUE;
}


/** \fn int pds_neuronml_iterate(PdsNeuronML *NeuronML,const PdsVector *Input,PdsVector *Output)
 *  \brief Itera la red neuronal multicapa NeuronML.
 *  \f[ \theta_j=\sum_{k=0}^{N_{d}-1}{W_{kj}X_k} - U_j \f]
 *  \f[ y_j \leftarrow func\left(\theta_j\right) \f]
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Input Datos de entrada de la red neuronal multicapa.
 *  \param[out] Output Datos de salida de la red neuronal multicapa.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, o tamaños de
 *  vectores incompatibles). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_iterate(PdsNeuronML *NeuronML,const PdsVector *Input,PdsVector *Output)
{
	PdsNnNatural N,i;

	if(NeuronML==NULL)	return FALSE;

	N=NeuronML->N;
	if(NeuronML->LayerInput->Nel!=Input->Nel)	return FALSE;
	if(NeuronML->Layer[N-1]->Nel!=Output->Nel)	return FALSE;

	pds_nivector_set_input_from_array(NeuronML->LayerInput,Input->V,Input->Nel);

	for(i=0;i<N;i++)	pds_nvector_iterate(NeuronML->Layer[i]);

	pds_nvector_get_output(NeuronML->Layer[N-1],Output);

	return TRUE;
}


/** \fn PdsNnBool pds_neuronml_iterate_results_printf(const PdsNeuronML *MLayer,const char* Type)
 *  \brief Muestra los resultados de la iteración de la red neuronal multicapa NeuronML.
 *
 *  \param[in] MLayer Red neuronal Multi capa a trabajar.
 *  \param[in] Type Etiqueda da iteracion.
 *  \return El valor medio de las salidas de la red neuronal multicapa NeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNnBool pds_neuronml_iterate_results_printf(const PdsNeuronML *MLayer,const char* Type)
{
    if(MLayer==NULL)    return FALSE;
    
    printf("[%s]\t",Type);
    
    PdsVector *Y=NULL;
    Y=pds_vector_new(MLayer->L[MLayer->N]);
    pds_neuronml_get_output(MLayer,Y);
    printf("Y= {\t");
    for (int i=0;i<Y->Nel;i++)
    {
        printf("%5.3f\t",Y->V[i]);
    }
    printf("}\n");
    pds_vector_destroy(&Y);
    
    return TRUE;
}

/** \fn int pds_neuronml_training(PdsNeuronML *NeuronML,const PdsVector *Input,PdsVector *Output)
 *  \brief Entrena la red neuronal multicapa NeuronML. Usa el valor de Output como 
 *  entrenamiento, para finalizar carga el nuevo valor de salida en Output.
 *
 *  Bucle de retropropagación e iteración :
 *  <ul>
 *  <li>pds_nvector_evaluate_diff_error()</li>
 *  <li>pds_nvector_backpropagate_error() : \f$ e_k \leftarrow e_k + \frac{\alpha}{N_j} func'\left(\theta_j\right) W_{kj} e_j \f$ </li>
 *  <li>pds_nvector_update_u_and_weight() : \f$ W_{kj} = W_{kj} + \alpha e_j func'\left(\theta_j\right) \frac{X_{k}}{X^TX} \f$</li>
 *  <li>pds_nvector_update_u_and_weight() : \f$ U_{k} = U_{k} - \alpha e_j func'\left(\theta_j\right) \frac{1}{X^TX} \f$</li>
 *  <li>pds_neuronml_iterate()</li>
 *  </ul>
 *  El bucle es repetido hasta que el error cuadratico medio en la salida sea menor
 *  NeuronML->MaxError o que el numero de iteraciones sea mayor que NeuronML->MaxIter
 *  (Solo si NeuronML->MaxIter es diferente de cero).
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Input Datos de entrada de la red neuronal multicapa.
 *  \param[in,out] Output Datos esperados de salida de la red neuronal multicapa.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, o tamaños de
 *  vectores incompatibles). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_training(PdsNeuronML *NeuronML,const PdsVector *Input,PdsVector *Output)
{
	PdsNnNatural N,i,j;
	PdsNnReal e;
	PdsSnReal XtX=0.0;

	if(NeuronML==NULL)	return FALSE;

	N=NeuronML->N;
	if(NeuronML->LayerInput->Nel!=Input->Nel)	return FALSE;
	if(NeuronML->Layer[N-1]->Nel!=Output->Nel)	return FALSE;

	pds_nivector_set_input_from_array(NeuronML->LayerInput,Input->V,Input->Nel);

	for(i=0;i<N;i++)	pds_nvector_iterate(NeuronML->Layer[i]);

	pds_nvector_evaluate_diff_error(NeuronML->Layer[N-1],Output);

	pds_nvector_get_rms_error(NeuronML->Layer[N-1],&e);


	j=0;
	while(e>(NeuronML->MaxError))
	{
		for(i=0;i<N;i++)
		pds_nvector_backpropagate_error(NeuronML->Layer[N-1-i],NeuronML->Alpha/NeuronML->L[N-1-i]);

		for(i=0;i<N;i++)
		{
			pds_neuron_get_xtx(NeuronML->Layer[i]->V[0],&XtX);
			pds_nvector_update_u_and_weight(NeuronML->Layer[i],(NeuronML->Alpha)/(XtX+1));
		}

		pds_nivector_iterate(NeuronML->LayerInput);
		for(i=0;i<N;i++)	
		pds_nvector_iterate(NeuronML->Layer[i]);

		pds_nvector_evaluate_diff_error(NeuronML->Layer[N-1],Output);

		pds_nvector_get_rms_error(NeuronML->Layer[N-1],&e);

		j++;
		if((j >= NeuronML->MaxIter)&&(NeuronML->MaxIter!=0))	break;
	}

	pds_nvector_get_output(NeuronML->Layer[N-1],Output);

	NeuronML->Iter=j;

	return TRUE;
}

/** \fn PdsNnNatural pds_neuronml_training_results_printf(PdsNeuronML *MLayer,const char* Type)
 *  \brief Muestra los resultados del entrenamiento de la red neuronal multicapa NeuronML.
 *
 *  \param[in] MLayer Red neuronal Multi capa a trabajar.
 *  \param[in] Type Etiqueda del entrenamiento.
 *  \return El valor medio de las salidas de la red neuronal multicapa NeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNnNatural pds_neuronml_training_results_printf(PdsNeuronML *MLayer,const char* Type)
{
    PdsNnNatural IterN;
    PdsNnReal Err;
    if(MLayer==NULL)    return 0;

    pds_neuronml_get_number_iterations(MLayer,&IterN);
    pds_neuronml_get_output_rms_error(MLayer,&Err);


    printf("[%s] %6dIters RMSE=%6.3f\t",Type,IterN,Err);

    PdsVector *Y=NULL;
    Y=pds_vector_new(MLayer->L[MLayer->N]);
    pds_neuronml_get_output(MLayer,Y);
    printf("Y= {\t");
    for (int i=0;i<Y->Nel;i++)
    {
        printf("%5.3f\t",Y->V[i]);
    }
    printf("}\n");
    pds_vector_destroy(&Y);

    return IterN;
}

/** \fn int pds_neuronml_get_output(const PdsNeuronML *NeuronML,PdsVector *Y)
 *  \brief Devuelve la salida Y de la red neuronal.
 *  \param[in] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[out] Y Vector donde se cargará los valores de la salida Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o Y==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_output(const PdsNeuronML *NeuronML,PdsVector *Y)
{
	if(NeuronML==NULL)	return FALSE;
	if(Y==NULL)		    return FALSE;

	pds_nvector_get_output(NeuronML->Layer[NeuronML->N-1],Y);

	return TRUE;
}

/** \fn int pds_neuronml_get_output_soft_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote)
 *  \brief Evalúa una votación con desición suave de los valores de salida
 *  de la red neuronal multicapa. 
 *
 *  Se realiza una suma simple de todos los valores de salida.
 *  \param[in] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[out] Vote Valor de la votación suave.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_output_soft_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote)
{
	*Vote=0;

	if(NeuronML==NULL)		return FALSE;

	pds_nvector_get_soft_vote(NeuronML->Layer[NeuronML->N-1],Vote);

	return TRUE;
}



/** \fn int pds_neuronml_get_output_rms_error(const PdsNeuronML *NeuronML,PdsNnReal *RmsError)
 *  \brief Evalúa el error cuadrático medio de todos los errores Y[1] de las
 *  neuronas de salida de la red neuronal.
 *  \param[in] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] RmsError Valor raíz cuadrático medio.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_output_rms_error(const PdsNeuronML *NeuronML,PdsNnReal *RmsError)
{
	*RmsError=0;

	if(NeuronML==NULL)		return FALSE;

	pds_nvector_get_rms_error(NeuronML->Layer[NeuronML->N-1],RmsError);

	return TRUE;
}


/** \fn int pds_neuronml_head_printf(const PdsNeuronML *NeuronML)
 *  \brief Mostra em pantalla los datos de cabecera de la red neuronal multicapa
 *  \param[in] NeuronML La estructura multicapa a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_head_printf(const PdsNeuronML *NeuronML)
{
	if(NeuronML==NULL)		return FALSE;

    printf("Number of layers:\t%d\n",NeuronML->N);

    printf("Number of neurons by layer L:\n");
    for(int i=0;i<=NeuronML->N;i++)
    {
        printf("L[%d]=%d",i,NeuronML->L[i]);
        if(i==NeuronML->N)  printf("\n");
        else                printf("\t");
    }

    printf("Maximum accepted RMS of output error:\t%f\n",NeuronML->MaxError);

    printf("Trainning factor alpha:\t%f\n",NeuronML->Alpha);

    printf("Maximum number of iterations:\t%d\n",NeuronML->MaxIter);

	/*! Habilitador de printf de avance. 1 habilitado 0 deshabilitado*/
	//PdsNnBool ShowPrintf;

    return TRUE;
}


/** \fn int pds_neuronml_head_fprintf(const PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Escribe los datos de cabecera de la red neuronal.
 *  En una linea separando los elementos por un TAB.
 *  {NeuronML->N,NeuronML->MaxError,NeuronML->Alpha,NeuronML->MaxIter,NeuronML->Iter,NeuronML->ShowPrintf}
 *  En la siguiente linea separando los elementos por un TAB, el numero de neuronas por capa.
 *  \param[in] NeuronML La estructura multicapa a leer.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_head_fprintf(const PdsNeuronML *NeuronML, FILE *fd)
{
    if(NeuronML==NULL)  return FALSE;
    if(fd==NULL)        return FALSE;

	fprintf(fd,"%u\t",NeuronML->N);

	fprintf(fd,"%e\t",NeuronML->MaxError);

	fprintf(fd,"%e\t",NeuronML->Alpha);

	fprintf(fd,"%u\t",NeuronML->MaxIter);

	fprintf(fd,"%u\t",NeuronML->Iter);

	fprintf(fd,"%u\n",NeuronML->ShowPrintf);

	for(PdsNnNatural i=0;i<=NeuronML->N;i++)
	{	
		if(i!=NeuronML->N)	fprintf(fd,"%u\t",NeuronML->L[i]);
		else			    fprintf(fd,"%u\n",NeuronML->L[i]);
	}

    return TRUE;
}


/** \fn PdsNeuronML *pds_neuronml_new_head_fscanf(FILE *fd)
 *  \brief Lee los datos de cabecera de una red neuronal desde un archivo
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return Un puntero al vector de tipo PdsNeuronML o NULL en caso de error.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNeuronML *pds_neuronml_new_head_fscanf(FILE *fd)
{
	PdsNeuronML *NeuronML=NULL;
	PdsNnNatural i,j;
	PdsNnReal x;
	int tmp;

	if(fd==NULL)	return NULL;

	// Reservo memoria de la estructura.
	NeuronML=(PdsNeuronML *)calloc(1,sizeof(PdsNeuronML));
	if(NeuronML==NULL)	return NULL;

	if(feof(fd)==0)	
	{
		tmp=fscanf(fd,"%e",&x);
		if(tmp==0)	{free(NeuronML);	return NULL;}
		NeuronML->N=(PdsNnNatural)x;
	}
	else		{free(NeuronML);	return NULL;}

	if(feof(fd)==0)	
	{
		tmp=fscanf(fd,"%e",&(NeuronML->MaxError));
		if(tmp==0)	{free(NeuronML);	return NULL;}
	}
	else		{free(NeuronML);	return NULL;}

	if(feof(fd)==0)	
	{
		tmp=fscanf(fd,"%e",&(NeuronML->Alpha));
		if(tmp==0)	{free(NeuronML);	return NULL;}
	}
	else		{free(NeuronML);	return NULL;}

	if(feof(fd)==0)	
	{
		tmp=fscanf(fd,"%e",&x);
		if(tmp==0)	{free(NeuronML);	return NULL;}
		NeuronML->MaxIter=(PdsNnNatural)x;
	}
	else		{free(NeuronML);	return NULL;}

	if(feof(fd)==0)	
	{
		tmp=fscanf(fd,"%e",&x);
		if(tmp==0)	{free(NeuronML);	return NULL;}
		NeuronML->Iter=(PdsNnNatural)x;
	}
	else		{free(NeuronML);	return NULL;}

	if(feof(fd)==0)	
	{
		tmp=fscanf(fd,"%e",&x);
		if(tmp==0)	{free(NeuronML);	return NULL;}
		NeuronML->ShowPrintf=(PdsNnBool)x;
	}
	else		{free(NeuronML);	return NULL;}

	// Reservo vector de capas.
	NeuronML->L=(PdsNnNatural*)calloc(NeuronML->N+1,sizeof(PdsNnNatural));

	if(NeuronML->L==NULL)	{free(NeuronML);	return NULL;}


	for(i=0;i<=NeuronML->N;i++)
	{
		x=0;
		if(feof(fd)==0)	
		{
			tmp=fscanf(fd,"%e",&x);
			if(tmp==0)	
			{free(NeuronML->L);	free(NeuronML);	return NULL;}
		}
		else	{free(NeuronML->L);	free(NeuronML);	return NULL;}


		if(fabs(x)>1.0) NeuronML->L[i]=(PdsNnNatural)fabs(x);
		else		    NeuronML->L[i]=1;

	}

	// Creo vector de neuronas de entrada.
	NeuronML->LayerInput=pds_nivector_new(NeuronML->L[0]);
	if(NeuronML->LayerInput==NULL)	
	{
		free(NeuronML->L);
		free(NeuronML);
		return NULL;
	}


	// Creo vector de neuronas de capas oculta y de salida.
	NeuronML->Layer=(PdsNVector **)calloc(NeuronML->N,sizeof(PdsNVector *));
	if(NeuronML->Layer==NULL) 
	{
		free(NeuronML->L);
		pds_nivector_free(NeuronML->LayerInput);
		free(NeuronML);
		return NULL;
	}

	for(i=0;i<NeuronML->N;i++)
	{	
		NeuronML->Layer[i]=pds_nvector_new(NeuronML->L[i+1],NeuronML->L[i]);
		if(NeuronML->Layer[i]==NULL)
		{
			for(j=0;j<i;j++)	pds_nvector_free(NeuronML->Layer[i]);
			free(NeuronML->L);
			pds_nivector_free(NeuronML->LayerInput);
			free(NeuronML);
			return NULL;
		}
		
	}

	pds_nvector_connect_input_with_nivector(NeuronML->Layer[0],NeuronML->LayerInput);
	for(i=1;i<NeuronML->N;i++)
	{
		pds_nvector_connect_input(NeuronML->Layer[i],NeuronML->Layer[i-1]);
	}

    return NeuronML;
}



/** \fn int pds_neuronml_body_fprintf(const PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Guarda en un archivo de texto los pesos W[i], el valor de U.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  (neurona por neurona)
 *  \param[in] NeuronML La estructura multicapa a leer.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_body_fprintf(const PdsNeuronML *NeuronML, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NeuronML==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		m=pds_nvector_fprintf(NeuronML->Layer[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_neuronml_body_fscanf(PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Lee de un archivo de texto los pesos W[i], el valor de U.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  (neurona por neurona)
 *  \param[out] NeuronML La estructura multicapa a escribir.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_body_fscanf(PdsNeuronML *NeuronML, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NeuronML==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		if(feof(fd)!=0)	return FALSE;
		m=pds_nvector_fscanf(NeuronML->Layer[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_neuronml_body_fwrite(const PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Guarda en un archivo binario los pesos W[i], el valor de U.
 *  (neurona por neurona)
 *  \param[in] NeuronML La estructura multicapa a leer.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_body_fwrite(const PdsNeuronML *NeuronML, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NeuronML==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		m=pds_nvector_fwrite(NeuronML->Layer[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_neuronml_body_fread(PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Lee de un archivo binario los pesos W[i], el valor de U.
 *  (neurona por neurona)
 *  \param[out] NeuronML La estructura multicapa a escribir.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_body_fread(PdsNeuronML *NeuronML, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NeuronML==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		if(feof(fd)!=0)	return FALSE;
		m=pds_nvector_fscanf(NeuronML->Layer[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}

/** \fn void pds_neuronml_free(PdsNeuronML *Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuronML.
 *  \param[in,out] Neuron La neurona a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronMLGroup
 */
void pds_neuronml_free(PdsNeuronML *NeuronML)
{
	PdsNnNatural i;
	if(NeuronML!=NULL)
	{
		free(NeuronML->L);
		pds_nivector_free(NeuronML->LayerInput);
		for(i=0;i<NeuronML->N;i++)	pds_nvector_free(NeuronML->Layer[i]);
		free(NeuronML);
	}
}


/** \fn void pds_neuronml_destroy(PdsNeuronML **Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuronML, y limpia el puntero con NULL.
 *  \param[in,out] Neuron La neurona a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronMLGroup
 */
void pds_neuronml_destroy(PdsNeuronML **NeuronML)
{
	PdsNnNatural i;
	if((*NeuronML)!=NULL)
	{
		free((*NeuronML)->L);	
		pds_nivector_destroy(&((*NeuronML)->LayerInput));
		for(i=0;i<(*NeuronML)->N;i++)	pds_nvector_destroy(&((*NeuronML)->Layer[i]));
		free((*NeuronML));
		(*NeuronML)=NULL;
	}
}

