/*
 * pdsnvector.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsnvector.h
 *  \author Fernando Pujaico Rivera
 *  \date 05-01-2012
 *  \brief Funciones que trabajan con vectores de neuronas.
 *  
 *  <br>Estas funciones trabajan con neuronas de la forma.<br>
 *  \image html PdsNeuron.png "Neurona de tipo PdsNeuron"
 */

#ifndef __PDSNVECTOR_H__
#define __PDSNVECTOR_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdsnnglobal.h>
#include <pds/pdsra.h>
#include <pds/pdsrv.h>
#include <pds/pdssn.h>
#include <pds/pdsnivector.h>


#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif


/** \defgroup PdsNVectorGroup Módulo PdsNVector.
 *  \brief Funciones que trabajan con vectores de neuronas.
 *  
 *  <br>Estas funciones trabajan con vectores de neuronas de la forma.<br>
 *  \image html nvector.png "Vector de Neuronas."
 * @{
 */


/*! \struct PdsNVector
 *  \brief La estructura tipo  PdsNVector .
 *  Esta estructura genera un arreglo de neuronas.
 *  Para usar incluir pds/pdsnn.h.
 *  \image html nvector.png "Vector de Neuronas."
 *  \ingroup PdsNVectorGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Arreglo de neuronas. */
	PdsNeuron **V;
	/*! Número de neuronas en el vector. */
	PdsNnNatural Nel;
}PdsNVector;



/** @name pds_nvector_<new>
 *  Creando un vector de neuronas.
 *  \image html nvector.png "Vector de Neuronas."
 *
 * @{
 */

/** \fn PdsNVector *pds_nvector_new(PdsNnNatural Nel,PdsNnNatural Nd)
 *  \brief Crea un vector de neuronas de tipo PdsNVector e inicia con cero todos 
 *  los pesos,  U=1.0 e Sigma=1.0.
 *  \param[in] Nel Es el número de elementos del vector.
 *  \param[in] Nd Es el número de entradas de cada neurona del vector.
 *  \return Un puntero al vector de tipo PdsNVector.
 *  \ingroup PdsNVectorGroup
 */
PdsNVector *pds_nvector_new(PdsNnNatural Nel,PdsNnNatural Nd);

//@}

/** @name pds_nvector_<weight_methods>
 *  Trabajando con el synaptic weight 
 *
 * @{
 */

/** \fn int pds_nvector_init_weight_uniform(PdsNVector *NVector,PdsUniform *RV)
 *  \brief Inicia aleatoriamente siguiendo una distribución uniforme, los pesos
 *  de todas las neuronas del vector.
 *  \param[in,out] NVector El vector a escribir.
 *  \param[in,out] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_init_weight_uniform(PdsNVector *NVector,PdsUniform *RV);


//@}


/** @name pds_nvector_<output_methods>
 *  Trabajando con funciones de saida
 *
 * @{
 */


/** \fn int pds_nvector_get_output(const PdsNVector *NVector,PdsVector *Y)
 *  \brief Devuelve el valor de la salida Y de las Neuronas.
 *  \param[in] NVector El vector de neuronas en consulta.
 *  \param[out] Y Vector donde se cargará los valores de la salida Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL, Y==NULL o
 *  tamaños distintos). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_get_output(const PdsNVector *NVector,PdsVector *Y);


//@}

/** @name pds_nvector_<connect_methods>
 *  Trabajando con el metodos para conectar vectores de neuronas.
 *
 * @{
 */


/** \fn int pds_nvector_connect_input(PdsNVector *NVector,PdsNVector *NVectorBefore)
 *  \brief Conecta el valor de salida Y de las Neuronas del vector NVectorBefore con 
 *  las entradas X[id] de las neuronas del vector NVector.
 *
 *  Si existen mas entradas de NVector que salidas de NVectorBefore las entradas quedan
 *  desconectadas cargadas con NULL.
 *  Si existen menos entradas de NVector que salidas de NVectorBefore las salidas quedan
 *  desconectadas.
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \param[in,out] NVectorBefore El vector de neuronas que se conectará a las entradas 
 *  X[id] de NVector.
 *  \return TRUE si todo fue bien o FALSE si no(ej: NVector==NULL, NVectorBefore==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_connect_input(PdsNVector *NVector,PdsNVector *NVectorBefore);


/** \fn int pds_nvector_connect_input_with_nivector(PdsNVector *NVector,PdsNIVector *X)
 *  \brief Conecta los elementos del vector X de neuronas de entradas, con las 
 *  entradas X[id] de las neuronas del vector NVector.
 * 
 *  Si existen mas entradas de NVector que elementos de X las entradas quedan
 *  desconectadas cargadas con NULL.
 *  Si existen menos entradas de NVector que elementos de X los elementos quedan
 *  desconectadas.
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \param[in,out] X El vector que se conectará a las entradas X[id] de NVector.
 *  \return TRUE si todo fue bien o FALSE si no(ej: NVector==NULL, X==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_connect_input_with_nivector(PdsNVector *NVector,PdsNIVector *X);


//@}

/** @name pds_nvector_<theta_methods>
 *  Funciones que trabajan con theta
 *
 * @{
 */


/** \fn int pds_nvector_evaluate_theta(PdsNVector *NVector)
 *  \brief Evalua la variable theta de la Neurona Neuron.
 *  \f[ \theta_j=\sum_{k=0}^{N_{d}-1}{W_{kj}X_k} - U_j \f]
 *  \param[in,out] NVector El vector de neuronas a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_evaluate_theta(PdsNVector *NVector);


/** \fn int pds_nvector_init_u_uniform(PdsNVector *NVector,PdsUniform *RV)
 *  \brief Coloca de forma aleatoria siguiendo una distribucion uniforme, el 
 *  valor de  U de las Neuronas.
 *  \param[in,out] NVector El vector de neuronas a escribir.
 *  \param[in] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_init_u_uniform(PdsNVector *NVector,PdsUniform *RV);


//@}

/** @name pds_nvector_<func_methods>
 *  Funciones que trabajan con func
 *
 * @{
 */

/** \fn int pds_nvector_evaluate_func(PdsNVector *NVector)
 *  \brief Evalua la funcion de activación de las Neuronas de NVector.
 *  \f[ func\left(\frac{\theta_j}{\sigma_j}\right) \f]
 *  \param[in,out] NVector El vector de neuronas a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_evaluate_func(PdsNVector *NVector);


/** \fn int pds_nvector_set_sigma(PdsNVector *NVector,PdsNnReal Sigma)
 *  \brief Coloca el valor de  sigma de las Neuronas.
 *  \param[in,out] NVector El vector de neuronas a escribir.
 *  \param[in] Sigma El valor de sigma de las neuronas.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_set_sigma(PdsNVector *NVector,PdsNnReal Sigma);


//@}

/** @name pds_nvector_<evaluate_methods>
 *  Funciones que trabajan con theta
 *
 * @{
 */

/** \fn int pds_nvector_evaluate(PdsNVector *NVector)
 *  \brief Evalua las neuronas del vector NVector.
 *  \f[ \theta_j=\sum_{k=0}^{N_{d}-1}{W_{kj}X_k} - U_j \f]
 *  \f[ func\left(\frac{\theta_j}{\sigma_j}\right) \f]
 *  \param[in,out] NVector El vector de neuronas a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_evaluate(PdsNVector *NVector);


/** \fn int pds_nvector_update(PdsNVector *NVector)
 *  \brief Actualiza el valor a la salida de las neuronas NVector
 *  \f[ y_j \leftarrow func\left(\frac{\theta_j}{\sigma_j}\right) \f]
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_update(PdsNVector *NVector);


/** \fn int pds_nvector_iterate(PdsNVector *NVector)
 *  \brief Itera las neuronas del vector NVector.
 *  \f[ \theta_j=\sum_{k=0}^{N_{d}-1}{W_{kj}X_k} - U_j \f]
 *  \f[ y_j \leftarrow func\left(\frac{\theta_j}{\sigma_j}\right) \f]
 *  \param[in,out] NVector El vector de neuronas a iterar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_iterate(PdsNVector *NVector);



//@}

/** @name pds_nvector_<others_methods>
 *  Metodos adicionales
 *
 * @{
 */


/** \fn int pds_nvector_evaluate_error(PdsNVector *NVector,const PdsVector * R)
 *  \brief Compara la salida de las neuronas del vector NVector con el vector de 
 *  referencia R, el resultado es cargado en la salida Y[1] de cada neurona del 
 *  vector NVector en  NVector->V[i]->Y[1].
 *
 *  Los tamaños de los vectores deben ser iguales.
 *  \param[in,out] NVector El vector de neuronas a comparar.
 *  \param[in]  R El vector a comparar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_evaluate_error(PdsNVector *NVector,const PdsVector * R);


/** \fn int pds_nvector_get_rms_error(const PdsNVector *NVector,PdsNnReal *RmsError)
 *  \brief Evalúa el error cuadrático medio de todos los errores Y[1] de las.
 *  neuronas del vector.
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \param[in] RmsError Valor raíz cuadrático medio.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_get_rms_error(const PdsNVector *NVector,PdsNnReal *RmsError);


/** \fn int pds_nvector_get_hard_vote(const PdsNVector *NVector,PdsNnReal *Vote)
 *  \brief Evalúa una votación de todos los valores de salida Y[0] de las
 *  neuronas.
 *
 *  Se considera como un voto positivo "+1" cualquier valor mayor o 
 *  igual a "0". Se considera como un voto negativo "-1" cualquier valor menor 
 *  a "0". Luego se realiza la suma con signo de las votaciones.
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \param[out] Vote Valor de la votación abrupta.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_get_hard_vote(const PdsNVector *NVector,PdsNnReal *Vote);


/** \fn int pds_nvector_get_soft_vote(const PdsNVector *NVector,PdsNnReal *Vote)
 *  \brief Evalúa una votación de todos los valores de salida Y[0] de las
 *  neuronas, se realiza una suma simple de todos los valores de salida.
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \param[out] Vote Valor de la votación suave.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_get_soft_vote(const PdsNVector *NVector,PdsNnReal *Vote);



/** \fn int pds_nvector_backpropagate_error(PdsNVector *NVector)
 *  \brief Retro propaga el error de las neuronas del vector NVector
 *  el resultado es cargado en la salida Y[1] de cada neurona del vector NVector
 *  conectada a las entradas.
 *  \f[ e_k \leftarrow e_k + \frac{func'\left(\frac{\theta_j}{\sigma_j}\right)}{\sigma_j} W_{kj} e_j \f]
 *
 *  \image html backpropagation.png "Retropropagación del error en las neuronas." 
 *  \image html nvector_error_prop.png "Retropropagación del error en las neuronas." 
 *  \param[in,out] NVector El vector de neuronas a retropropagar el error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_backpropagate_error(PdsNVector *NVector);



//@}

/** @name pds_nvector_<update_weight>
 *  Atualizando peso de las neuronas. 
 *
 * @{
 */


/** \fn int pds_nvector_update_weight(PdsNVector *NVector, PdsNnReal Alpha)
 *  \brief Actualiza los pesos W[k]j de las neuronas.
 *  \f[ \triangle W_{kj}= \alpha e_j \frac{func'\left(\frac{\theta_j}{\sigma_j}\right)}{\sigma_j} X_k \f]
 *  "ej" es el error de la salida de la j-essima neurona.
 *  "func" es la función de activación da neurona.
 *  \param[in,out] NVector El vector de neuronas a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_update_weight(PdsNVector *NVector, PdsNnReal Alpha);



/** \fn int pds_nvector_update_weight_normalized(PdsNVector *NVector, PdsNnReal Alpha)
 *  \brief Actualiza los pesos W[k]j de las neuronas.
 *  \f[ \triangle W_{kj}= \alpha e_j \frac{func'\left(\frac{\theta_j}{\sigma_j}\right)}{\sigma_j} \frac{X_k}{X^TX} \f]
 *  "ej" es el error de la salida de la j-essima neurona.
 *  "func" es la función de activación da neurona.
 *  \param[in,out] NVector El vector de neuronas a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_update_weight_normalized(PdsNVector *NVector, PdsNnReal Alpha);


//@}

/** @name pds_nvector_<printf>
 *  Imprimeindo y leyendo datos. 
 *
 * @{
 */


/** \fn int pds_nvector_fprintf(const PdsNVector *NVector, FILE *fd)
 *  \brief Guarda en un archivo de texto los pesos \f$W_{kj}\f$ y los valores \f$\{U_j,\sigma_j\}\f$.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[in] NVector El vector de neuronas a leer.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_fprintf(const PdsNVector *NVector, FILE *fd);


/** \fn int pds_nvector_fscanf(PdsNVector *NVector, FILE *fd)
 *  \brief Lee de un archivo de texto los pesos \f$W_{kj}\f$ y los valores \f$\{U_j,\sigma_j\}\f$.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[out] NVector El vector a escribir.
 *  \param[in,out] fd Manejador del fichero a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_fscanf(PdsNVector *NVector, FILE *fd);


/** \fn int pds_nvector_fwrite(const PdsNVector *NVector, FILE *fd)
 *  \brief Guarda en un archivo de texto binario los pesos \f$W_{kj}\f$ y los valores \f$\{U_j,\sigma_j\}\f$.
 *  \param[in] NVector El vector de neuronas a leer.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_fwrite(const PdsNVector *NVector, FILE *fd);


/** \fn int pds_nvector_fread(PdsNVector *NVector, FILE *fd)
 *  \brief Lee de un archivo de texto binario los pesos \f$W_{kj}\f$ y los valores \f$\{U_j,\sigma_j\}\f$.
 *  \param[out] NVector El vector a escribir.
 *  \param[in,out] fd Manejador del fichero binario a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_fread(PdsNVector *NVector, FILE *fd);


//@}

/** @name pds_nvector_<free>
 *  Liberando memoria.
 *
 * @{
 */

/** \fn void pds_nvector_free(PdsNVector *NVector)
 *  \brief Libera un vector de neuronas de tipo puntero PdsNVector.
 *  \param[in,out] NVector el vector de neuronas a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNVectorGroup
 */
void pds_nvector_free(PdsNVector *NVector);


/** \fn void pds_nvector_destroy(PdsNVector **NVector)
 *  \brief Libera un vector de neuronas de tipo puntero PdsNVector, y limpia el 
 *   puntero con NULL.
 *  \param[in,out] NVector El vector de neuronas a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNVectorGroup
 */
void pds_nvector_destroy(PdsNVector **NVector);


//@}

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


