/*
 * pdsnivector.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsra.h>
#include <pds/pdssn.h>
#include <pds/pdsnivector.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsNIVector                                             ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsNIVector *pds_nivector_new(PdsNnNatural Nel)
 *  \brief Crea un vector de neuronas de entrada de tipo PdsNIVector e inicia 
 *  con cero todos los valores.
 *  \param[in] Nel Es el número de elementos del vector.
 *  \return Un puntero al vector de tipo PdsNIVector.
 *  \ingroup PdsNIVectorGroup
 */
PdsNIVector *pds_nivector_new(PdsNnNatural Nel)
{
	PdsNIVector *NIVector=NULL;
	PdsNnNatural i,j;

	if(Nel<=0)		return NULL;
	
	// Reservo memoria de la estructura.
	NIVector=(PdsNIVector *)calloc(1,sizeof(PdsNIVector));
	if(NIVector==NULL)	return NULL;

	// Numero de elementos del vector.
	NIVector->Nel=Nel;

	NIVector->V=(PdsNInput **)calloc(NIVector->Nel,sizeof(PdsNInput *));
	if(NIVector->V==NULL) 
	{
		free(NIVector);
		return NULL;
	}

	for(i=0;i<NIVector->Nel;i++)
	{	
		NIVector->V[i]=pds_ninput_new();
		if(NIVector->V[i]==NULL)
		{
			for(j=0;j<i;j++)	pds_ninput_free(NIVector->V[j]);
			free(NIVector->V);
			free(NIVector);
		}
	}

	return NIVector;
}


/** \fn int pds_nivector_get_output(const PdsNIVector *NIVector,PdsVector *Y)
 *  \brief Devuelve el valor de la salida Y de las Neuronas.
 *  \param[in] NIVector La neurona en consulta.
 *  \param[out] Y Vector donde se cargará los valores de la salida Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NIVector==NULL, Y==NULL o
 *  tamaños distintos). 
 *  \ingroup PdsNIVectorGroup
 */
int pds_nivector_get_output(const PdsNIVector *NIVector,PdsVector *Y)
{
	PdsNnNatural i;

	if(NIVector==NULL)		return FALSE;
	if(Y      ==NULL)		return FALSE;
	if(NIVector->Nel!=Y->Nel)	return FALSE;

	for(i=0;i<NIVector->Nel;i++)	Y->V[i]=*(NIVector->V[i]->Y);

	return TRUE;
}


/** \fn int pds_nivector_set_input(PdsNIVector *NIVector,const PdsVector *Y)
 *  \brief Escribe el valor de la entrada de las Neuronas.
 *  \param[in,out] NIVector La neurona en consulta.
 *  \param[in] Y Vector donde se cargará los valores de la entrada Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NIVector==NULL, Y==NULL o
 *  tamaños distintos). 
 *  \ingroup PdsNIVectorGroup
 */
int pds_nivector_set_input(PdsNIVector *NIVector,const PdsVector *Y)
{
	PdsNnNatural i;

	if(NIVector==NULL)		return FALSE;
	if(Y       ==NULL)		return FALSE;
	if(NIVector->Nel!=Y->Nel)	return FALSE;

	for(i=0;i<NIVector->Nel;i++)	
	{
		NIVector->V[i]->Y[0]=Y->V[i];
		NIVector->V[i]->Y[1]=0.0;
	}

	return TRUE;
}


/** \fn int pds_nivector_iterate(PdsNIVector *NIVector)
 *  \brief Escribe el valor cero en todas las segundas salidas (error) de cada elemento 
 *  del vector. \f$e_k=0.0\f$.
 *  \param[in,out] NIVector La neurona en consulta.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NIVector==NULL). 
 *  \ingroup PdsNIVectorGroup
 */
int pds_nivector_iterate(PdsNIVector *NIVector)
{
	PdsNnNatural i;

	if(NIVector==NULL)		return FALSE;

	for(i=0;i<NIVector->Nel;i++)	NIVector->V[i]->Y[1]=0.0;

	return TRUE;
}

/** \fn void pds_nivector_free(PdsNIVector *NIVector)
 *  \brief Libera un vector de neuronas de tipo puntero PdsNIVector.
 *  \param[in,out] NIVector el vector de neuronas a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNIVectorGroup
 */
void pds_nivector_free(PdsNIVector *NIVector)
{
	PdsNnNatural i;
	if(NIVector!=NULL)
	{
		for(i=0;i<NIVector->Nel;i++)	pds_ninput_free(NIVector->V[i]);
		free(NIVector);
	}
}


/** \fn void pds_nivector_destroy(PdsNIVector **NIVector)
 *  \brief Libera un vector de neuronas de tipo puntero PdsNIVector, y limpia el 
 *   puntero con NULL.
 *  \param[in,out] NIVector El vector de neuronas a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNIVectorGroup
 */
void pds_nivector_destroy(PdsNIVector **NIVector)
{
	PdsNnNatural i;
	if((*NIVector)!=NULL)
	{
		for(i=0;i<(*NIVector)->Nel;i++)	
		pds_ninput_destroy(&((*NIVector)->V[i]));
		free(*NIVector);
		(*NIVector)=NULL;
	}
}

