/*
 * pdsnivector.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsnivector.h
 *  \author Fernando Pujaico Rivera
 *  \date 05-01-2012
 *  \brief Funciones que trabajan con vectores de neuronas de entrada PdsNInput.
 *
 *  <br>Estas funciones trabajan con vectores de neuronas de la forma.<br>
 *  \image html PdsNIVector.png "Vector de neuronas de entrada."
 *  \f[ NInput=\{Y[0],Y[1]\} \f].
 */

#ifndef __PDSNIVECTOR_H__
#define __PDSNIVECTOR_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdsnnglobal.h>
#include <pds/pdsra.h>
#include <pds/pdsrv.h>
#include <pds/pdssn.h>


/** \defgroup PdsNIVectorGroup Funciones del módulo PdsNIVector.
 *  \brief Funciones que trabajan con vectores de neuronas de entrada PdsNInput.
 *
 *  \f[ NInput=\{Y[0],Y[1]\} \f].
 *  <br>Estas funciones trabajan con vectores de neuronas de la forma.<br>
 *  \image html PdsNIVector.png "Vector de neuronas de entrada."
 * @{
 */


#ifndef TRUE
/*! \def TRUE
 *  Define la palabra TRUE equivalente con 1.
 *  \ingroup PdsNIVectorGroup
*/
	#define TRUE 1
#endif

#ifndef FALSE
/*! \def FALSE
 *  Define la palabra FALSE equivalente con 0.
 *  \ingroup PdsNIVectorGroup
*/
	#define FALSE 0
#endif


/*! \struct PdsNIVector
 *  \brief La estructura tipo  PdsNIVector .
 *  Esta estructura genera un arreglo de neuronas.
 *  Para usar incluir pds/pdsnn.h.
 *  \brief Funciones que trabajan con vectores de neuronas de entrada PdsNInput.
 *  \f[ NInput=\{Y[0],Y[1]\} \f].
 *  <br>Estas funciones trabajan con vectores de neuronas de la forma.<br>
 *  \image html PdsNIVector.png "Vector de neuronas de entrada."
 *  \ingroup PdsNIVectorGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Arreglo de neuronas de entrada. */
	PdsNInput **V;
	/*! Número de neuronas de entrada del vector. */
	PdsNnNatural Nel;
	/*! ID del arreglo de neuronas. */
	PdsNnNatural ID;
}PdsNIVector;


/** \fn PdsNIVector *pds_nivector_new(PdsNnNatural Nel)
 *  \brief Crea un vector de neuronas de entrada de tipo PdsNIVector e inicia 
 *  con cero todos los valores.
 *  \param[in] Nel Es el número de elementos del vector.
 *  \return Un puntero al vector de tipo PdsNIVector.
 *  \ingroup PdsNIVectorGroup
 */
PdsNIVector *pds_nivector_new(PdsNnNatural Nel);


/** \fn int pds_nivector_get_output(const PdsNIVector *NIVector,PdsVector *Y)
 *  \brief Devuelve el valor de la salida Y de las Neuronas.
 *  \param[in] NIVector La neurona en consulta.
 *  \param[out] Y Vector donde se cargará los valores de la salida Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NIVector==NULL, Y==NULL o
 *  tamaños distintos). 
 *  \ingroup PdsNIVectorGroup
 */
int pds_nivector_get_output(const PdsNIVector *NIVector,PdsVector *Y);


/** \fn int pds_nivector_get_input(const PdsNIVector *NIVector,PdsVector *Y)
 *  \brief Devuelve el valor de la entrada de las Neuronas.
 *  \param[in] NIVector La neurona en consulta.
 *  \param[out] Y Vector donde se cargará los valores de la entrada Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NIVector==NULL, Y==NULL o
 *  tamaños distintos). 
 *  \ingroup PdsNIVectorGroup
 */
int pds_nivector_get_input(const PdsNIVector *NIVector,PdsVector *Y);


/** \fn int pds_nivector_set_input(PdsNIVector *NIVector,const PdsVector *Y)
 *  \brief Escribe el valor de la entrada de las Neuronas.
 *  \param[in,out] NIVector La neurona en consulta.
 *  \param[in] Y Vector donde se cargará los valores de la entrada Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NIVector==NULL, Y==NULL o
 *  tamaños distintos). 
 *  \ingroup PdsNIVectorGroup
 */
int pds_nivector_set_input(PdsNIVector *NIVector,const PdsVector *Y);


/** \fn int pds_nivector_iterate(PdsNIVector *NIVector)
 *  \brief Escribe el valor cero en todas las segundas salidas de cada elemento 
 *  del vector: NIVector->V[i]->Y[1]=0.0.
 *  \param[in,out] NIVector La neurona en consulta.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NIVector==NULL). 
 *  \ingroup PdsNIVectorGroup
 */
int pds_nivector_iterate(PdsNIVector *NIVector);


/** \fn void pds_nivector_free(PdsNIVector *NIVector)
 *  \brief Libera un vector de neuronas de tipo puntero PdsNIVector.
 *  \param[in,out] NIVector el vector de neuronas a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNIVectorGroup
 */
void pds_nivector_free(PdsNIVector *NIVector);


/** \fn void pds_nivector_destroy(PdsNIVector **NIVector)
 *  \brief Libera un vector de neuronas de tipo puntero PdsNIVector, y limpia el 
 *   puntero con NULL.
 *  \param[in,out] NIVector El vector de neuronas a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNIVectorGroup
 */
void pds_nivector_destroy(PdsNIVector **NIVector);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


