/*
 * pdsnvector.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsra.h>
#include <pds/pdsrv.h>
#include <pds/pdssn.h>
#include <pds/pdsnivector.h>
#include <pds/pdsnvector.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsNVector                                              ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsNVector *pds_nvector_new(PdsNnNatural Nel,PdsNnNatural Ninputs)
 *  \brief Crea un vector de neuronas de tipo PdsNVector e inicia con cero todos 
 *  los pesos y umbrales.
 *  \param[in] Nel Es el número de elementos del vector.
 *  \param[in] Ninputs Es el número de entradas de cada neurona del vector.
 *  \return Un puntero al vector de tipo PdsNVector.
 *  \ingroup PdsNVectorGroup
 */
PdsNVector *pds_nvector_new(PdsNnNatural Nel,PdsNnNatural Ninputs)
{
	PdsNVector *NVector=NULL;
	PdsNnNatural i,j;

	if(Nel<=0)		return NULL;
	
	// Reservo memoria de la estructura.
	NVector=(PdsNVector *)calloc(1,sizeof(PdsNVector));
	if(NVector==NULL)	return NULL;

	// Numero de elementos del vector.
	NVector->Nel=Nel;

	NVector->V=(PdsNeuron **)calloc(NVector->Nel,sizeof(PdsNeuron *));
	if(NVector->V==NULL) 
	{
		free(NVector);
		return NULL;
	}

	for(i=0;i<NVector->Nel;i++)
	{	
		NVector->V[i]=pds_neuron_new(Ninputs);
		if(NVector->V[i]==NULL)
		{
			for(j=0;j<i;j++)	pds_neuron_free(NVector->V[j]);
			free(NVector->V);
			free(NVector);
		}
	}

	NVector->ID=0;

	return NVector;
}


/** \fn int pds_nvector_init_uniform_synaptic_weight(PdsNVector *NVector,PdsUniform *RV)
 *  \brief Inicia aleatoriamente siguiendo una distribución uniforme, los pesos
 *  de todas las neuronas del vector.
 *  \param[in,out] NVector El vector a escribir.
 *  \param[in,out] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_init_uniform_synaptic_weight(PdsNVector *NVector,PdsUniform *RV)
{
	PdsNnNatural i,j;
	PdsNnReal m;

	if(NVector==NULL)	return FALSE;
	if(RV     ==NULL)	return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		for(j=0;j<NVector->V[i]->Nel;j++)
		{
			pds_uniform_get_value(RV,&m);
			pds_neuron_set_synaptic_weight(NVector->V[i],j,m);
		}
	}

	return TRUE;
}


/** \fn int pds_nvector_set_sigma(PdsNVector *NVector,PdsNnReal Sigma)
 *  \brief Coloca el valor de  sigma de las Neuronas.
 *  \param[in,out] NVector El vector de neuronas a escribir.
 *  \param[in] Sigma El valor de sigma de las neuronas.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_set_sigma(PdsNVector *NVector,PdsNnReal Sigma)
{
	PdsNnNatural i;

	if(NVector==NULL)	return FALSE;

	for(i=0;i<NVector->Nel;i++)
	NVector->V[i]->Sigma=Sigma;

	return TRUE;
}




/** \fn int pds_nvector_init_uniform_u(PdsNVector *NVector,PdsUniform *RV)
 *  \brief Coloca de forma aleatoria siguiendo una distribucion uniforme, el 
 *  valor de  U de las Neuronas.
 *  \param[in,out] NVector La neurona a escribir.
 *  \param[in] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_init_uniform_u(PdsNVector *NVector,PdsUniform *RV)
{
	PdsNnNatural i;
	PdsNnReal m;

	if(NVector==NULL)	return FALSE;
	if(RV     ==NULL)	return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		pds_uniform_get_value(RV,&m);
		NVector->V[i]->U=m;
	}

	return TRUE;
}


/** \fn int pds_nvector_get_output(const PdsNVector *NVector,PdsVector *Y)
 *  \brief Devuelve el valor de la salida Y de las Neuronas.
 *  \param[in] NVector La neurona en consulta.
 *  \param[out] Y Vector donde se cargará los valores de la salida Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL, Y==NULL o
 *  tamaños distintos). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_get_output(const PdsNVector *NVector,PdsVector *Y)
{
	PdsNnNatural i;

	if(NVector==NULL)		return FALSE;
	if(Y      ==NULL)		return FALSE;
	if(NVector->Nel!=Y->Nel)	return FALSE;

	for(i=0;i<NVector->Nel;i++)	Y->V[i]=*(NVector->V[i]->Y);

	return TRUE;
}


/** \fn int pds_nvector_connect_input(PdsNVector *NVector,PdsNVector *NVectorX)
 *  \brief Conecta el valor de salida Y de las Neuronas del vector NVectorX con 
 *  las entradas X[id] de las neuronas del vector NVector.
 *  Si existen mas entradas de NVector que salidas de NVectorX las entradas quedan
 *  desconectadas cargadas con NULL.
 *  Si existen menos entradas de NVector que salidas de NVectorX las salidas quedan
 *  desconectadas.
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \param[in,out] NVectorX El vector de neuronas que se conectará a las entradas 
 *  X[id] de NVector.
 *  \return TRUE si todo fue bien o FALSE si no(ej: NVector==NULL, NVectorX==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_connect_input(PdsNVector *NVector,PdsNVector *NVectorX)
{
	PdsNnNatural i,j;

	if(NVector ==NULL)		return FALSE;
	if(NVectorX==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		for(j=0;(j<NVector->V[i]->Nel)&&(j<NVectorX->Nel);j++)
		NVector->V[i]->X[j]=NVectorX->V[j]->Y;

		for(;j<NVector->V[i]->Nel;j++)
		NVector->V[i]->X[j]=NULL;
	}
	return TRUE;
}


/** \fn int pds_nvector_connect_input_data(PdsNVector *NVector,PdsNIVector *X)
 *  \brief Conecta los elementos del vector X de neuronas de entradas, con las 
 *  entradas X[id] de las neuronas del vector NVector.
 *  Si existen mas entradas de NVector que elementos de X las entradas quedan
 *  desconectadas cargadas con NULL.
 *  Si existen menos entradas de NVector que elementos de X los elementos quedan
 *  desconectadas.
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \param[in,out] X El vector que se conectará a las entradas X[id] de NVector.
 *  \return TRUE si todo fue bien o FALSE si no(ej: NVector==NULL, X==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_connect_input_data(PdsNVector *NVector,PdsNIVector *X)
{
	PdsNnNatural i,j;

	if(NVector==NULL)		return FALSE;
	if(X      ==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		for(j=0;(j<NVector->V[i]->Nel)&&(j<X->Nel);j++)
		NVector->V[i]->X[j]=X->V[j]->Y;

		for(;j<NVector->V[i]->Nel;j++)
		NVector->V[i]->X[j]=NULL;
	}
	return TRUE;
}


/** \fn int pds_nvector_evaluate_theta(PdsNVector *NVector)
 *  \brief Evalua la variable theta de la Neurona Neuron.
 *  \f[ \theta=\sum_{i=0}^{N_{el}-1}{W[i]X[i]} - U \f].
 *  \param[in,out] NVector La neurona a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_evaluate_theta(PdsNVector *NVector)
{
	PdsNnReal theta;
	PdsNnNatural i,j;

	if(NVector==NULL)		return FALSE;

	for(j=0;j<NVector->Nel;j++)
	{
		theta=-NVector->V[j]->U;
		for(i=0;i<NVector->V[j]->Nel;i++)	
		{
			if(NVector->V[j]->X[i]!=NULL)
			theta=theta+(*(NVector->V[j]->X[i]))*NVector->V[j]->W[i];
		}
		NVector->V[j]->Theta=theta;
	}
	return TRUE;
}


/** \fn int pds_nvector_evaluate_f(PdsNVector *NVector)
 *  \brief Evalua la funcion de activación de las Neuronas de NVector.
 *  \f[ f(\theta)=\frac{2}{1+e^{-\frac{\theta}{\sigma}}} -1 \f].
 *  \param[in,out] NVector El vector de neuronas a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_evaluate_f(PdsNVector *NVector)
{
	PdsNnReal x;
	PdsNnNatural i;

	if(NVector==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		x=NVector->V[i]->Theta/NVector->V[i]->Sigma;

		NVector->V[i]->F=(2.0/(1.0+exp(-x)))-1.0;
	}
	return TRUE;
}


/** \fn int pds_nvector_evaluate(PdsNVector *NVector)
 *  \brief Evalua las neuronas del vector NVector.
 *  \f[ \theta=\sum_{i=0}^{N_{el}-1}{W[i]X[i]} - U \f].
 *  \f[ f(\theta)=\frac{2}{1+e^{-\frac{\theta}{\sigma}}} -1 \f].
 *  \param[in,out] NVector El vector de neuronas a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_evaluate(PdsNVector *NVector)
{
	int m;
	if(NVector==NULL)		return FALSE;

	m=pds_nvector_evaluate_theta(NVector);
	if(m==FALSE)	return FALSE;
	m=pds_nvector_evaluate_f(NVector);
	if(m==FALSE)	return FALSE;

	return TRUE;
}


/** \fn int pds_nvector_update(PdsNVector *NVector)
 *  \brief Actualiza el valor a la salida de las neuronas NVector
 *  \f[ Y=f(\theta) \f].
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_update(PdsNVector *NVector)
{
	PdsNnNatural i;

	if(NVector==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		*(NVector->V[i]->Y)=NVector->V[i]->F;
		NVector->V[i]->Y[1]=0.0;
	}

	return TRUE;
}


/** \fn int pds_nvector_iterate(PdsNVector *NVector)
 *  \brief Itera las neuronas del vector NVector.
 *  \f[ \theta=\sum_{i=0}^{N_{el}-1}{W[i]X[i]} - U \f].
 *  \f[ f(\theta)=\frac{2}{1+e^{-\frac{\theta}{\sigma}}} -1 \f].
 *  \f[ Y=f(\theta) \f].
 *  \param[in,out] NVector El vector de neuronas a iterar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_iterate(PdsNVector *NVector)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)		return FALSE;

	m=pds_nvector_evaluate_theta(NVector);
	if(m==FALSE)	return FALSE;
	m=pds_nvector_evaluate_f(NVector);
	if(m==FALSE)	return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		*(NVector->V[i]->Y)=NVector->V[i]->F;
		NVector->V[i]->Y[1]=0.0;
	}

	return TRUE;
}


/** \fn int pds_nvector_evaluate_e(PdsNVector *NVector,const PdsVector *Y)
 *  \brief Compara la salida de las neuronas del vector NVector con el vector Y
 *  el resultado es cargado en la salida Y[1] de cada neurona del vector NVector:
 *  NVector->V[i]->Y[1].
 *  Los tamaños de los vectores deben ser iguales.
 *  \param[in,out] NVector El vector de neuronas a comparar.
 *  \param[in] Y El vector a comparar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_evaluate_e(PdsNVector *NVector,const PdsVector *Y)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)		return FALSE;
	if(NVector->Nel!=Y->Nel)	return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		NVector->V[i]->Y[1]=Y->V[i]-NVector->V[i]->Y[0];
		//pds_neuron_evaluate_e(NVector->V[i],Y->V[i]);
	}

	return TRUE;
}


/** \fn int pds_nvector_get_rms_error(const PdsNVector *NVector,PdsNnReal *RmsError)
 *  \brief Evalúa el error cuadrático medio de todos los errores Y[1] de las.
 *  neuronas del vector.
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \param[in] RmsError Valor raíz cuadrático medio.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_get_rms_error(const PdsNVector *NVector,PdsNnReal *RmsError)
{
	PdsNnNatural i;

	*RmsError=0;

	if(NVector==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		(*RmsError)=(*RmsError)+NVector->V[i]->Y[1]*NVector->V[i]->Y[1];
	}
	(*RmsError)=sqrt((*RmsError)/NVector->Nel);

	return TRUE;
}


/** \fn int pds_nvector_get_hard_vote(const PdsNVector *NVector,PdsNnReal *Vote)
 *  \brief Evalúa una votación de todos los valores de salida Y[0] de las
 *  neuronas, se considera como un voto positivo "+1" cualquier valor mayor o 
 *  igual a "0". Se considera como un voto negativo "-1" cualquier valor menor 
 *  a "0". Luego se realiza la suma con signo de las votaciones.
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \param[out] Vote Valor de la votación abrupta.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_get_hard_vote(const PdsNVector *NVector,PdsNnReal *Vote)
{
	PdsNnNatural i;

	*Vote=0.0;

	if(NVector==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		if(NVector->V[i]->Y[0]>=0)	(*Vote)=(*Vote)+1.0;
		else				(*Vote)=(*Vote)-1.0;
	}

	return TRUE;
}


/** \fn int pds_nvector_get_soft_vote(const PdsNVector *NVector,PdsNnReal *Vote)
 *  \brief Evalúa una votación de todos los valores de salida Y[0] de las
 *  neuronas, se realiza una suma simple de todos los valores de salida.
 *  \param[in,out] NVector El vector de neuronas a trabajar.
 *  \param[out] Vote Valor de la votación suave.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_get_soft_vote(const PdsNVector *NVector,PdsNnReal *Vote)
{
	PdsNnNatural i;

	*Vote=0.0;

	if(NVector==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		(*Vote)=(*Vote)+NVector->V[i]->Y[0];
	}

	return TRUE;
}

/** \fn int pds_nvector_backpropagate_error(PdsNVector *NVector)
 *  \brief Retro propaga el error de las neuronas del vector NVector
 *  el resultado es cargado en la salida Y[1] de cada neurona del vector NVector
 *  conectada a las entradas.
 *  \param[in,out] NVector El vector de neuronas retropropagar el error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_backpropagate_error(PdsNVector *NVector)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		m=pds_neuron_backpropagate_error(NVector->V[i]);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_nvector_update_weight(PdsNVector *NVector, PdsNnReal Alpha)
 *  \brief Actualiza los pesos W[i] de las neuronas.
 *  \f[ \triangle W[i]= \alpha e f'(\theta) X[i] \f].
 *  "e" es el error de la salida da neurona Neuron :Neuron->Y[1].
 *  "f" es la función de activación da neurona Neuron.
 *  \param[in,out] NVector El vector de neuronas a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_update_weight(PdsNVector *NVector, PdsNnReal Alpha)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		m=pds_neuron_update_weight(NVector->V[i],Alpha);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_nvector_update_weight_norm(PdsNVector *NVector, PdsNnReal Alpha,PdsSnReal XtX)
 *  \brief Actualiza los pesos W[i] de las neuronas.
 *  \f[ \triangle W[i]= \alpha e f'(\theta) \frac{X[i]}{X^TX} \f].
 *  "e" es el error de la salida da neurona Neuron :Neuron->Y[1].
 *  "f" es la función de activación da neurona Neuron.
 *  \param[in,out] NVector El vector de neuronas a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \param[in] XtX Suma cuadrática de los valores de entrada X[i].
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_update_weight_norm(PdsNVector *NVector, PdsNnReal Alpha,PdsSnReal XtX)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		m=pds_neuron_update_weight_norm(NVector->V[i],Alpha,XtX);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_nvector_update_weight_normalized(PdsNVector *NVector, PdsNnReal Alpha)
 *  \brief Actualiza los pesos W[i] de las neuronas.
 *  \f[ \triangle W[i]= \alpha e f'(\theta) \frac{X[i]}{X^TX} \f].
 *  "e" es el error de la salida da neurona Neuron :Neuron->Y[1].
 *  "f" es la función de activación da neurona Neuron.
 *  \param[in,out] NVector El vector de neuronas a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_update_weight_normalized(PdsNVector *NVector, PdsNnReal Alpha)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		m=pds_neuron_update_weight_normalized(NVector->V[i],Alpha);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_nvector_fprintf(const PdsNVector *NVector, FILE *fd)
 *  \brief Guarda en un archivo de texto los pesos W[i] y el valor de U.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[in] NVector El vector de neuronas a leer.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_fprintf(const PdsNVector *NVector, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		m=pds_neuron_fprintf(NVector->V[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_nvector_fscanf(PdsNVector *NVector, FILE *fd)
 *  \brief Lee de un archivo de texto los pesos W[i] y el valor de U.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[out] NVector El vector a escribir.
 *  \param[in,out] fd Manejador del fichero a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_fscanf(PdsNVector *NVector, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		if(feof(fd)!=0)	return FALSE;
		m=pds_neuron_fscanf(NVector->V[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_nvector_fwrite(const PdsNVector *NVector, FILE *fd)
 *  \brief Guarda en un archivo de texto binario los pesos W[i] y el valor de U.
 *  \param[in] NVector El vector de neuronas a leer.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_fwrite(const PdsNVector *NVector, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		m=pds_neuron_fwrite(NVector->V[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_nvector_fread(PdsNVector *NVector, FILE *fd)
 *  \brief Lee de un archivo de texto binario los pesos W[i] y el valor de U.
 *  \param[out] NVector El vector a escribir.
 *  \param[in,out] fd Manejador del fichero binario a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNVectorGroup
 */
int pds_nvector_fread(PdsNVector *NVector, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NVector==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NVector->Nel;i++)
	{
		if(feof(fd)!=0)	return FALSE;
		m=pds_neuron_fread(NVector->V[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn void pds_nvector_free(PdsNVector *NVector)
 *  \brief Libera un vector de neuronas de tipo puntero PdsNVector.
 *  \param[in,out] NVector el vector de neuronas a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNVectorGroup
 */
void pds_nvector_free(PdsNVector *NVector)
{
	PdsNnNatural i;
	if(NVector!=NULL)
	{
		for(i=0;i<NVector->Nel;i++)	pds_neuron_free(NVector->V[i]);
		free(NVector);
	}
}


/** \fn void pds_nvector_destroy(PdsNVector **NVector)
 *  \brief Libera un vector de neuronas de tipo puntero PdsNVector, y limpia el 
 *   puntero con NULL.
 *  \param[in,out] NVector El vector de neuronas a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNVectorGroup
 */
void pds_nvector_destroy(PdsNVector **NVector)
{
	PdsNnNatural i;
	if((*NVector)!=NULL)
	{
		for(i=0;i<(*NVector)->Nel;i++)	pds_neuron_destroy(&((*NVector)->V[i]));
		free(*NVector);
		(*NVector)=NULL;
	}
}

