/*
 * pdsneuronml.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsra.h>
#include <pds/pdsrv.h>
#include <pds/pdssn.h>
#include <pds/pdsnivector.h>
#include <pds/pdsnvector.h>
#include <pds/pdsneuronml.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsNeuronML                                              ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsNeuronML *pds_neuronml_new(const PdsVector *L)
 *  \brief Crea una estructura de tipo PdsNeuronML e inicia con cero todos los 
 *  elementos. Pero con un error mínimo de 1/sqrt(2), que indica una incertidumbre 
 *  en la mitad de los datos de salida y los demás datos son aciertos. El valor de
 *  de Alpha se inicia con 0.5.
 *  \param[in] L Es un vector con el numero de elementos por capas.
 *  \return Un puntero al vector de tipo PdsNeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNeuronML *pds_neuronml_new(const PdsVector *L)
{
	PdsNeuronML *NeuronML=NULL;
	PdsNnNatural i,j;

	if(L==NULL)		return NULL;
	if(L->Nel<2)		return NULL;
	
	// Reservo memoria de la estructura.
	NeuronML=(PdsNeuronML *)calloc(1,sizeof(PdsNeuronML));
	if(NeuronML==NULL)	return NULL;

	// Reservo vector de capas.
	NeuronML->L=(PdsNnNatural*)calloc(L->Nel,sizeof(PdsNnNatural));

	if(NeuronML->L==NULL)	
	{
		free(NeuronML);
		return NULL;
	}

	for(i=0;i<L->Nel;i++)
	{
		if(L->V[i]>0)
		{
			if(L->V[i]>1)	NeuronML->L[i]=(PdsNnNatural)L->V[i];
			else		NeuronML->L[i]=1;
		}
		else
		{
			if(L->V[i]<-1)	NeuronML->L[i]=(PdsNnNatural)(-L->V[i]);
			else		NeuronML->L[i]=1;
		}
	}

	// Numero de capas.
	NeuronML->N=L->Nel-1;

	// Creo vector de neuronas de entrada.
	NeuronML->LayerInput=pds_nivector_new(NeuronML->L[0]);
	if(NeuronML->LayerInput==NULL)	
	{
		free(NeuronML->L);
		free(NeuronML);
		return NULL;
	}



	NeuronML->Layer=(PdsNVector **)calloc(L->Nel-1,sizeof(PdsNVector *));
	if(NeuronML->Layer==NULL) 
	{
		free(NeuronML->L);
		pds_nivector_free(NeuronML->LayerInput);
		free(NeuronML);
		return NULL;
	}

	for(i=0;i<NeuronML->N;i++)
	{	
		NeuronML->Layer[i]=pds_nvector_new(NeuronML->L[i+1],NeuronML->L[i]);
		if(NeuronML->Layer[i]==NULL)
		{
			for(j=0;j<i;j++)	pds_nvector_free(NeuronML->Layer[i]);
			free(NeuronML->L);
			pds_nivector_free(NeuronML->LayerInput);
			free(NeuronML);
			return NULL;
		}
		
	}

	pds_nvector_connect_input_data(NeuronML->Layer[0],NeuronML->LayerInput);
	for(i=1;i<NeuronML->N;i++)
	{
		pds_nvector_connect_input(NeuronML->Layer[i],NeuronML->Layer[i-1]);
	}

	NeuronML->Error=0.707106781;
	NeuronML->Alpha=0.5;
	NeuronML->MaxIter=0;
	NeuronML->Iter=0;
	NeuronML->ShowPrintf=0;


	return NeuronML;
}


/** \fn PdsNeuronML *pds_neuronml_new_load_data(const char *mlayerfile)
 *  \brief Crea una estructura de tipo PdsNeuronML e inicia todos los elementos.
 *  desde un archivo de texto.
 *  \param[in] mlayerfile Es el archivo de texto de donde se leen los datos.
 *  \return Un puntero al vector de tipo PdsNeuronML.
 *  \ingroup PdsNeuronMLGroup
 */
PdsNeuronML *pds_neuronml_new_load_data(const char *mlayerfile)
{
	PdsNeuronML *NeuronML=NULL;
	FILE *fd=NULL;
	PdsNnNatural i,j;
	PdsNnReal x;
	int tmp;

	if(mlayerfile==NULL)	return NULL;

	fd=fopen(mlayerfile,"r");
	if(fd==NULL)	return NULL;

	// Reservo memoria de la estructura.
	NeuronML=(PdsNeuronML *)calloc(1,sizeof(PdsNeuronML));
	if(NeuronML==NULL)	return NULL;

	if(feof(fd)==0)	
	{
		tmp=fscanf(fd,"%e",&x);
		if(tmp==0)	{free(NeuronML);	return NULL;}
		NeuronML->N=(PdsNnNatural)x;
	}
	else		{free(NeuronML);	return NULL;}

	if(feof(fd)==0)	
	{
		tmp=fscanf(fd,"%e",&(NeuronML->Error));
		if(tmp==0)	{free(NeuronML);	return NULL;}
	}
	else		{free(NeuronML);	return NULL;}

	if(feof(fd)==0)	
	{
		tmp=fscanf(fd,"%e",&(NeuronML->Alpha));
		if(tmp==0)	{free(NeuronML);	return NULL;}
	}
	else		{free(NeuronML);	return NULL;}

	if(feof(fd)==0)	
	{
		tmp=fscanf(fd,"%e",&x);
		if(tmp==0)	{free(NeuronML);	return NULL;}
		NeuronML->MaxIter=(PdsNnNatural)x;
	}
	else		{free(NeuronML);	return NULL;}

	if(feof(fd)==0)	
	{
		tmp=fscanf(fd,"%e",&x);
		if(tmp==0)	{free(NeuronML);	return NULL;}
		NeuronML->Iter=(PdsNnNatural)x;
	}
	else		{free(NeuronML);	return NULL;}

	if(feof(fd)==0)	
	{
		tmp=fscanf(fd,"%e",&x);
		if(tmp==0)	{free(NeuronML);	return NULL;}
		NeuronML->ShowPrintf=(PdsNnBool)x;
	}
	else		{free(NeuronML);	return NULL;}

	// Reservo vector de capas.
	NeuronML->L=(PdsNnNatural*)calloc(NeuronML->N+1,sizeof(PdsNnNatural));

	if(NeuronML->L==NULL)	{free(NeuronML);	return NULL;}


	for(i=0;i<=NeuronML->N;i++)
	{
		x=0;
		if(feof(fd)==0)	
		{
			tmp=fscanf(fd,"%e",&x);
			if(tmp==0)	
			{free(NeuronML->L);	free(NeuronML);	return NULL;}
		}
		else	{free(NeuronML->L);	free(NeuronML);	return NULL;}

		if(x>0)
		{
			if(x>1)		NeuronML->L[i]=(PdsNnNatural)x;
			else		NeuronML->L[i]=1;
		}
		else
		{
			if(x<-1)	NeuronML->L[i]=(PdsNnNatural)(-x);
			else		NeuronML->L[i]=1;
		}
	}

	// Creo vector de neuronas de entrada.
	NeuronML->LayerInput=pds_nivector_new(NeuronML->L[0]);
	if(NeuronML->LayerInput==NULL)	
	{
		free(NeuronML->L);
		free(NeuronML);
		return NULL;
	}


	// Creo vector de neuronas de capas oculta y de salida.
	NeuronML->Layer=(PdsNVector **)calloc(NeuronML->N,sizeof(PdsNVector *));
	if(NeuronML->Layer==NULL) 
	{
		free(NeuronML->L);
		pds_nivector_free(NeuronML->LayerInput);
		free(NeuronML);
		return NULL;
	}

	for(i=0;i<NeuronML->N;i++)
	{	
		NeuronML->Layer[i]=pds_nvector_new(NeuronML->L[i+1],NeuronML->L[i]);
		if(NeuronML->Layer[i]==NULL)
		{
			for(j=0;j<i;j++)	pds_nvector_free(NeuronML->Layer[i]);
			free(NeuronML->L);
			pds_nivector_free(NeuronML->LayerInput);
			free(NeuronML);
			return NULL;
		}
		
	}

	pds_nvector_connect_input_data(NeuronML->Layer[0],NeuronML->LayerInput);
	for(i=1;i<NeuronML->N;i++)
	{
		pds_nvector_connect_input(NeuronML->Layer[i],NeuronML->Layer[i-1]);
	}

	pds_neuronml_fscanf(NeuronML,fd);
	
	fclose(fd);

	return NeuronML;
}


/** \fn int pds_neuronml_save_data(const PdsNeuronML *NeuronML, const char *mlayerfile)
 *  \brief Salva una estructura de tipo PdsNeuronML en un archivo de texto.
 *  \param[in] NeuronML Es la red neuronal multicapa a salvar.
 *  \param[in] mlayerfile Es el archivo de texto donde se guardará la red.
 *  \return TRUE si todo fue bien o FALSE si no. (ejem: NeuronML==NULL o mlayerfile=NULL).
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_save_data(const PdsNeuronML *NeuronML, const char *mlayerfile)
{
	FILE *fd=NULL;
	PdsNnNatural i;

	if(NeuronML==NULL)	return FALSE;
	if(mlayerfile==NULL)	return FALSE;

	fd=fopen(mlayerfile,"w");
	if(fd==NULL)	return FALSE;

	fprintf(fd,"%u\t",NeuronML->N);

	fprintf(fd,"%e\t",NeuronML->Error);

	fprintf(fd,"%e\t",NeuronML->Alpha);

	fprintf(fd,"%u\t",NeuronML->MaxIter);

	fprintf(fd,"%u\t",NeuronML->Iter);

	fprintf(fd,"%u\n",NeuronML->ShowPrintf);

	for(i=0;i<=NeuronML->N;i++)
	{	
		if(i!=NeuronML->N)	fprintf(fd,"%u\t",NeuronML->L[i]);
		else			fprintf(fd,"%u\n",NeuronML->L[i]);
	}

	pds_neuronml_fprintf(NeuronML,fd);

	fclose(fd);

	return TRUE;
}


/** \fn int pds_neuronml_set_max_error(PdsNeuronML *NeuronML,PdsNnReal Error)
 *  \brief Coloca el maximo error aceptado, cualquier valor de error menor
 *  provoca que se detenga el algoritmo de aprendizaje.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Error Máximo error aceptado.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_max_error(PdsNeuronML *NeuronML,PdsNnReal Error)
{
	PdsNnReal e;
	if(NeuronML==NULL)	return FALSE;

	e=fabs(Error);
	if(e!=0.0)	NeuronML->Error=e;
	else		NeuronML->Error=0.05;
	
	return TRUE;
}


/** \fn int pds_neuronml_set_alpha(PdsNeuronML *NeuronML,PdsNnReal Alpha)
 *  \brief Coloca el valor del factor de aprendizaje Alpha.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Alpha Factor de aprendizaje.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_alpha(PdsNeuronML *NeuronML,PdsNnReal Alpha)
{
	if(NeuronML==NULL)	return FALSE;

	NeuronML->Alpha=Alpha;
	
	return TRUE;
}


/** \fn int pds_neuronml_set_max_number_iterations(PdsNeuronML *NeuronML,PdsNnNatural MaxIter)
 *  \brief Coloca el máximo numero de iteraciones aceptado, cualquier valor mayor
 *  provoca que se detenga el algoritmo de aprendizaje. si este número es cero
 *  entonces el algoritmo continua hasta que el error en la salida sea menor que
 *  NeuronML->Error.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] MaxIter Número máximo de iteraciones.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_max_number_iterations(PdsNeuronML *NeuronML,PdsNnNatural MaxIter)
{

	if(NeuronML==NULL)	return FALSE;

	NeuronML->MaxIter=MaxIter;

	return TRUE;
}


/** \fn int pds_neuronml_get_number_iterations(const PdsNeuronML *NeuronML,PdsNnNatural *Iter)
 *  \brief Devuelve el número de iteraciones del ultimo entrenamiento hecho con
 *  la función pds_neuronml_training.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Iter Número de iteraciones del último entrenamiento.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_number_iterations(const PdsNeuronML *NeuronML,PdsNnNatural *Iter)
{
	*Iter=0;

	if(NeuronML==NULL)	return FALSE;

	*Iter=NeuronML->Iter;

	return TRUE;
}


/** \fn int pds_neuronml_enable_printf(PdsNeuronML *NeuronML)
 *  \brief Habilita la muestra en pantalla del avance del aprendizaje.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_enable_printf(PdsNeuronML *NeuronML)
{

	if(NeuronML==NULL)	return FALSE;

	NeuronML->ShowPrintf=1;

	return TRUE;
}


/** \fn int pds_neuronml_init_uniform_synaptic_weight(PdsNeuronML *NeuronML,PdsUniform *RV)
 *  \brief Inicializa todos los pesos usando una variable aleatoria uniforme.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, RV==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_init_uniform_synaptic_weight(PdsNeuronML *NeuronML,PdsUniform *RV)
{
	PdsNnNatural i;
	
	if(NeuronML==NULL)	return FALSE;
	if(RV      ==NULL)	return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		pds_nvector_init_uniform_synaptic_weight(NeuronML->Layer[i],RV);
	}

	return TRUE;
}


/** \fn int pds_neuronml_init_uniform_u(PdsNeuronML *NeuronML,PdsUniform *RV)
 *  \brief Inicializa todos los umbrales U usando una variable aleatoria uniforme.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] RV Variable aleatoria uniforme.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, RV==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_init_uniform_u(PdsNeuronML *NeuronML,PdsUniform *RV)
{
	PdsNnNatural i;
	
	if(NeuronML==NULL)	return FALSE;
	if(RV      ==NULL)	return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		pds_nvector_init_uniform_u(NeuronML->Layer[i],RV);
	}

	return TRUE;
}


/** \fn int pds_neuronml_set_sigma(PdsNeuronML *NeuronML,PdsNnReal Sigma)
 *  \brief Coloca el valor de Sigma en todas las neuronas
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] Sigma Valor de sigma en la función de activación de las neuronas.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, RV==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_set_sigma(PdsNeuronML *NeuronML,PdsNnReal Sigma)
{
	PdsNnNatural i;
	
	if(NeuronML==NULL)	return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		pds_nvector_set_sigma(NeuronML->Layer[i],Sigma);
	}

	return TRUE;
}


/** \fn int pds_neuronml_iterate(PdsNeuronML *NeuronML,const PdsVector *X,PdsVector *Y)
 *  \brief Itera la red neuronal multicapa NeuronML.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] X Datos de entrada de la red neuronal multicapa.
 *  \param[out] Y Datos de salida de la red neuronal multicapa.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, o tamaños de
 *  vectores incompatibles). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_iterate(PdsNeuronML *NeuronML,const PdsVector *X,PdsVector *Y)
{
	PdsNnNatural N,i;

	if(NeuronML==NULL)	return FALSE;

	N=NeuronML->N;
	if(NeuronML->LayerInput->Nel!=X->Nel)	return FALSE;
	if(NeuronML->Layer[N-1]->Nel!=Y->Nel)	return FALSE;

	pds_nivector_set_input(NeuronML->LayerInput,X);

	for(i=0;i<N;i++)	pds_nvector_iterate(NeuronML->Layer[i]);

	pds_nvector_get_output(NeuronML->Layer[N-1],Y);

	return TRUE;
}



/** \fn int pds_neuronml_training(PdsNeuronML *NeuronML,const PdsVector *X,PdsVector *Y)
 *  \brief Entrena la red neuronal multicapa NeuronML. Usa el valor de Y como 
 *  entrenamiento, para finalizar carga el nuevo valor de Y.
 *  \f[ e^{inputs}_i = W[i] e \f].
 *  \f[ W[i] = W[i] + \alpha e f'(\theta) \frac{X[i]}{X^TX} \f].
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[in] X Datos de entrada de la red neuronal multicapa.
 *  \param[in] Y Datos esperados de salida de la red neuronal multicapa.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL, o tamaños de
 *  vectores incompatibles). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_training(PdsNeuronML *NeuronML,const PdsVector *X,PdsVector *Y)
{
	PdsNnNatural N,i,j;
	PdsNnReal e;
	PdsSnReal XtX;

	if(NeuronML==NULL)	return FALSE;

	N=NeuronML->N;
	if(NeuronML->LayerInput->Nel!=X->Nel)	return FALSE;
	if(NeuronML->Layer[N-1]->Nel!=Y->Nel)	return FALSE;

	pds_nivector_set_input(NeuronML->LayerInput,X);

	for(i=0;i<N;i++)	pds_nvector_iterate(NeuronML->Layer[i]);

	pds_nvector_evaluate_e(NeuronML->Layer[N-1],Y);

	pds_nvector_get_rms_error(NeuronML->Layer[N-1],&e);


	j=0;
	while(e>(NeuronML->Error))
	{
		for(i=0;i<N;i++)
		pds_nvector_backpropagate_error(NeuronML->Layer[N-1-i]);

		for(i=0;i<N;i++)
		{
			pds_neuron_get_xtx(NeuronML->Layer[i]->V[0],&XtX);
			pds_nvector_update_weight_norm(NeuronML->Layer[i],NeuronML->Alpha,XtX);
		}

		pds_nivector_iterate(NeuronML->LayerInput);
		for(i=0;i<N;i++)	
		pds_nvector_iterate(NeuronML->Layer[i]);

		pds_nvector_evaluate_e(NeuronML->Layer[N-1],Y);

		pds_nvector_get_rms_error(NeuronML->Layer[N-1],&e);

		j++;
		if((j >= NeuronML->MaxIter)&&(NeuronML->MaxIter!=0))	break;
	}

	pds_nvector_get_output(NeuronML->Layer[N-1],Y);

	NeuronML->Iter=j;

	return TRUE;
}


/** \fn int pds_neuronml_get_hard_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote)
 *  \brief Evalúa una votación de todos los valores de salida Y[0] de las
 *  neuronas de salida de la red neuronal multicapa, se considera como un voto 
 *  positivo "+1" cualquier valor mayor o igual a "0". Se considera como un voto 
 *  negativo "-1" cualquier valor menor a "0". Luego se realiza la suma con signo 
 *  de las votaciones.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[out] Vote Valor de la votación abrupta.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_hard_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote)
{
	*Vote=0;

	if(NeuronML==NULL)		return FALSE;

	pds_nvector_get_hard_vote(NeuronML->Layer[NeuronML->N-1],Vote);

	return TRUE;
}


/** \fn int pds_neuronml_get_soft_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote)
 *  \brief Evalúa una votación de todos los valores de salida Y[0] de las
 *  neuronas del vector de salida de la red neuronal multicapa, se realiza una 
 *  suma simple de todos los valores de salida.
 *  \param[in,out] NeuronML Red neuronal Multi capa a trabajar.
 *  \param[out] Vote Valor de la votación suave.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_get_soft_vote(const PdsNeuronML *NeuronML,PdsNnReal *Vote)
{
	*Vote=0;

	if(NeuronML==NULL)		return FALSE;

	pds_nvector_get_soft_vote(NeuronML->Layer[NeuronML->N-1],Vote);

	return TRUE;
}


/** \fn int pds_neuronml_fprintf(const PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Guarda en un archivo de texto los pesos W[i], el valor de U y Sigma.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[in] NeuronML La estructura multicapa a leer.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_fprintf(const PdsNeuronML *NeuronML, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NeuronML==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		m=pds_nvector_fprintf(NeuronML->Layer[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_neuronml_fscanf(PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Lee de un archivo de texto los pesos W[i], el valor de U  y Sigma.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[out] NeuronML La estructura multicapa a escribir.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_fscanf(PdsNeuronML *NeuronML, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NeuronML==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		if(feof(fd)!=0)	return FALSE;
		m=pds_nvector_fscanf(NeuronML->Layer[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_neuronml_fwrite(const PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Guarda en un archivo binario los pesos W[i], el valor de U y Sigma.
 *  \param[in] NeuronML La estructura multicapa a leer.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_fwrite(const PdsNeuronML *NeuronML, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NeuronML==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		m=pds_nvector_fwrite(NeuronML->Layer[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}


/** \fn int pds_neuronml_fread(PdsNeuronML *NeuronML, FILE *fd)
 *  \brief Lee de un archivo binario los pesos W[i], el valor de U  y Sigma.
 *  \param[out] NeuronML La estructura multicapa a escribir.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NeuronML==NULL o fd==NULL). 
 *  \ingroup PdsNeuronMLGroup
 */
int pds_neuronml_fread(PdsNeuronML *NeuronML, FILE *fd)
{
	PdsNnNatural i;
	int m;

	if(NeuronML==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;i<NeuronML->N;i++)
	{
		if(feof(fd)!=0)	return FALSE;
		m=pds_nvector_fscanf(NeuronML->Layer[i],fd);	
		if(m==FALSE)	return FALSE;
	}

	return TRUE;
}

/** \fn void pds_neuronml_free(PdsNeuronML *Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuronML.
 *  \param[in,out] Neuron La neurona a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronMLGroup
 */
void pds_neuronml_free(PdsNeuronML *NeuronML)
{
	PdsNnNatural i;
	if(NeuronML!=NULL)
	{
		free(NeuronML->L);
		pds_nivector_free(NeuronML->LayerInput);
		for(i=0;i<NeuronML->N;i++)	pds_nvector_free(NeuronML->Layer[i]);
		free(NeuronML);
	}
}


/** \fn void pds_neuronml_destroy(PdsNeuronML **Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuronML, y limpia el puntero con NULL.
 *  \param[in,out] Neuron La neurona a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronMLGroup
 */
void pds_neuronml_destroy(PdsNeuronML **NeuronML)
{
	PdsNnNatural i;
	if((*NeuronML)!=NULL)
	{
		free((*NeuronML)->L);	
		pds_nivector_destroy(&((*NeuronML)->LayerInput));
		for(i=0;i<(*NeuronML)->N;i++)	pds_nvector_destroy(&((*NeuronML)->Layer[i]));
		free((*NeuronML));
		(*NeuronML)=NULL;
	}
}

