/*
 * pdsmathti.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsmath.h>

/** \fn double pds_qfunc(double x)
 *  \brief Evalúa la función Q(x) 
 *  
 *  \f[ y=Q(x)=\frac{1}{\sqrt{2\pi}}\int_{x}^{\infty}exp({-\frac{u^2}{2}})du \f]
 *  \param[in] x Valor de entrada.
 *  \return El valor de Q(x).
 *  \ingroup PdsMathGroup
 */
double pds_qfunc(double x)
{
	double S;

	if(x<0)	return 1.0-pds_qfunc(-x);//problemas de presición aquí.	//

	if(x==0)	return 0.5;
	else
	{
		S=pds_integration_inf(pds_exp22,x,8192);
		//PDS_1_OVER_SQRT_2PI=(1.0/sqrt(2.0*M_PI))
		S=PDS_1_OVER_SQRT_2PI*S;
		if(S>=0.5)	return 0.5;
		else		return S;
	}
}


/** \fn double pds_qfuncinv(double x)
 *  \brief Evalúa la función Q^{-1}(x) , función Q inversa.
 *  
 *  \f[ y=Q^{-1}(x) \f]
 *  \f[ x=Q(y)=\frac{1}{\sqrt{2\pi}}\int_{y}^{\infty}exp({-\frac{u^2}{2}})du \f]
 *  \param[in] x Valor de entrada.
 *  \return El valor de Q^{-1}(x).
 *  \ingroup PdsMathGroup
 */
double pds_qfuncinv(double x)
{
	double a,b,y,r;
	double delta,signo;
	int i,N=1000;

	if(x<=0)	return 1.0/0.0;
	if(x>=1)	return -1.0/0.0;
	if(x==0.5)	return 0.0;

	signo=1.0;
	if(x>0.5)	{x=1.0-x;signo=-1.0;}
	//A partir de aquí todos los valores de x son ahora 0.0<x<0.5 .

	//QLB−KW−1(x) "New Exponential Lower Bounds on the Gaussian Q-Function via Jensen’s Inequality" 2011
	if(x<0.25)	a=sqrt(-(M_PI/2.0)*log(4.0*x));
	else		a=0.0;

	//Chernoff bound "http://en.wikipedia.org/wiki/Q-function"
	b=sqrt(-2.0*log(2.0*x));

	delta=1/10000.0;
	if(delta>(x/1000.0))	delta=(x/1000.0);

	i=0;
	do{
		y=(a+b)*0.5;
		r=pds_qfunc(y)-x;
		if(r>0)
		{
			if((pds_qfunc(a)-x)<0)	b=y;
			else			a=y;
		}
		else
		{
			if((pds_qfunc(a)-x)<0)	a=y;
			else			b=y;
		}
		i++;
	}while(((r>=delta)||(r<=-delta))&&(i<N));

	return y*signo;
}


/** \fn double pds_erf(double x)
 *  \brief Evalúa la función erf(x) 
 *  
 *  \f[ y=erf(x)=\frac{2}{\sqrt{\pi}}\int_{0}^{x}exp({-u^2})du \f]
 *  \param[in] x Valor de entrada.
 *  \return El valor de erf(x).
 *  \ingroup PdsMathGroup
 */
double pds_erf(double x)
{
	double S;

	if(x==0)	return 0.0;
	if(x<0)		return -pds_erf(-x);
	else
	{
		S=pds_integration(pds_exp2,0,x,8192);
		S=PDS_2_OVER_SQRT_PI*S;
		if(S>=1.0)	return 1.0;
		else		return S;
	}
}


/** \fn double pds_erfc(double x)
 *  \brief Evalúa la función erfc(x) 
 *  
 *  \f[ y=erfc(x)=1-\frac{2}{\sqrt{\pi}}\int_{0}^{x}exp({-u^2})du \f]
 *  \param[in] x Valor de entrada.
 *  \return El valor de erfc(x).
 *  \ingroup PdsMathGroup
 */
double pds_erfc(double x)
{
	double S;

	if(x<0)		return 2.0-pds_qfunc(-x);
	if(x==0)	return 1.0;
	else
	{
		S=pds_integration_inf(pds_exp2,x,8192);
		S=PDS_2_OVER_SQRT_PI*S;

		if(S>=1.0)	return 1.0;
		else		return S;
	}
}


/** \fn double pds_sgn(double x)
 *  \brief Evalúa la función signo sgn(x) 
 *  
 *  \f[y=sgn(x)=\f] \f{eqnarray*}{
        1 &si& x>0 \\ 
        0 &si& x=0 \\ 
       -1 &si& x<0
   \f}
 *  \param[in] x Valor de entrada.
 *  \return El valor de sgn(x).
 *  \ingroup PdsMathGroup
 */
double pds_sgn(double x)
{
	if(x<0)		return -1.0;
	if(x>0)		return 1.0;
	if(x==0)	return 0;
}


/** \fn float pds_fsgn(float x)
 *  \brief Evalúa la función signo sgn(x) 
 *  
 *  \f[y=sgn(x)=\f] \f{eqnarray*}{
        1 &si& x>0 \\ 
        0 &si& x=0 \\ 
       -1 &si& x<0
   \f}
 *  \param[in] x Valor de entrada.
 *  \return El valor de sgn(x).
 *  \ingroup PdsMathGroup
 */
float pds_fsgn(float x)
{
	if(x<0)		return -1.0;
	if(x>0)		return 1.0;
	if(x==0)	return 0;
}


/** \fn int pds_isgn(int x)
 *  \brief Evalúa la función signo sgn(x) 
 *  
 *  \f[y=sgn(x)=\f] \f{eqnarray*}{
        1 &si& x>0 \\ 
        0 &si& x=0 \\ 
       -1 &si& x<0
   \f}
 *  \param[in] x Valor de entrada.
 *  \return El valor de sgn(x).
 *  \ingroup PdsMathGroup
 */
int pds_isgn(int x)
{
	if(x<0)		return -1;
	if(x>0)		return 1;
	if(x==0)	return 0;
}

/** \fn double pds_sinc(double x)
 *  \brief Evalúa la función sinc(x)=sin(x)/x 
 *  
 *  \f[y=sinc(x)=\frac{sin(x)}{x}\f] 
 *  \param[in] x Valor de entrada.
 *  \return El valor de sinc(x).
 *  \ingroup PdsMathGroup
 */
double pds_sinc(double x)
{
	if(x==0)	return 1.0;
	else		return sin(x)/x;
}


/** \fn double pds_sigmoid(double x)
 *  \brief Evalúa la función sigmoid(x)=1/(1+e^{-x}) 
 *  
 *  \f[y=sigmoid(x)=\frac{1}{1+e^{-x}}\f] 
 *  \param[in] x Valor de entrada.
 *  \return El valor de sigmoid(x).
 *  \ingroup PdsMathGroup
 */
double pds_sigmoid(double x)
{
	return 1.0/(1+exp(-x));
}


/** \fn double pds_gauss(double x,double U,double Sigma)
 *  \brief Evalúa la función gaussiana, o distribución gaussiana f(x)=N(U,Sigma^2).
 *  
 *  \f[y=\frac{1}{\sqrt{2\pi \sigma^2}}e^{-\frac{(x-U)^2}{2\sigma^2}}\f] 
 *  \param[in] x Valor de entrada.
 *  \param[in] U El valor medio de x.
 *  \param[in] Sigma Es desvío padrón de x.
 *  \return El valor de f(x)=N(U,Sigma^2).
 *  \ingroup PdsMathGroup
 */
double pds_gauss(double x,double U,double Sigma)
{
	return exp(-((x-U)*(x-U))/(2.0*Sigma*Sigma))/(sqrt(2.0*M_PI)*Sigma);
}


/** \fn double pds_gauss2(double x,double U,double Sigma2)
 *  \brief Evalúa la función gaussiana, o distribución gaussiana f(x)=N(U,Sigma2).
 *  
 *  \f[y=\frac{1}{\sqrt{2\pi \sigma^2}}e^{-\frac{(x-U)^2}{2\sigma^2}}\f] 
 *  \param[in] x Valor de entrada.
 *  \param[in] U El valor medio de x.
 *  \param[in] Sigma2 Es la varianza de x, Sigma^2.
 *  \return El valor de f(x)=N(U,Sigma^2).
 *  \ingroup PdsMathGroup
 */
double pds_gauss2(double x,double U,double Sigma2)
{
	return exp(-((x-U)*(x-U))/(2.0*Sigma2))/sqrt(2.0*M_PI*Sigma2);
}


/** \fn double pds_gnorm(double x)
 *  \brief Evalúa la función gaussiana normalizada, o distribución gaussiana f(x)=N(0,1.0).
 *  
 *  \f[y=\frac{1}{\sqrt{2\pi}}e^{-\frac{x^2}{2}}\f] 
 *  \param[in] x Valor de entrada.
 *  \return El valor de f(x)=N(0,1.0).
 *  \ingroup PdsMathGroup
 */
double pds_gnorm(double x)
{
	return exp(-(x*x)/2.0)*PDS_1_OVER_SQRT_2PI;
}


/** \fn double pds_exp22(double x)
 *  \brief Evalúa la función f(x)=exp(-x^2/2).
 *  
 *  \f[y=e^{-\frac{x^2}{2}}\f] 
 *  \param[in] x Valor de entrada.
 *  \return El valor de f(x).
 *  \ingroup PdsMathGroup
 */
double pds_exp22(double x)
{
	return exp(-(x*x)/2.0);
}


/** \fn double pds_exp2(double x)
 *  \brief Evalúa la función f(x)=exp(-x^2).
 *  
 *  \f[y=e^{-x^2}\f] 
 *  \param[in] x Valor de entrada.
 *  \return El valor de f(x).
 *  \ingroup PdsMathGroup
 */
double pds_exp2(double x)
{
	return exp(-(x*x));
}


/** \fn double pds_exp1(double x)
 *  \brief Evalúa la función f(x)=exp(-x).
 *  
 *  \f[y=e^{-x}\f] 
 *  \param[in] x Valor de entrada.
 *  \return El valor de f(x).
 *  \ingroup PdsMathGroup
 */
double pds_exp1(double x)
{
	return exp(-x);
}

/** \fn double pds_hb(double x)
 *  \brief Evalúa la función de entropía binaria Hb(x)=-x*log2(x)-(1-x)*log2(1-x).
 *  
 *  \f[y=H_b(x)=-x log_2(x)-(1-x) log_2(1-x)\f] 
 *  \param[in] x Valor de entrada.
 *  \return El valor de Hb(x).
 *  \ingroup PdsMathGroup
 */
double pds_hb(double x)
{
	double h;
	if(x>=1.0)	return 0.0;
	if(x<=0.0)	return 0.0;

	h=(-x*log(x)-(1.0-x)*log(1.0-x))/PDS_LN2;

	if(h<=0.0)	return 0.0;
	if(h>=1.0)	return 1.0;

	return h;
}

/** \fn double pds_hbinv(double h)
 *  \brief Retorna el valor de x de la función de entropía binaria para un
 *  valor de h dado h=Hb(x)=-x*log2(x)-(1-x)*log2(1-x).
 *  
 *  \f[h=H_b(x)=-x log_2(x)-(1-x) log_2(1-x)\f] 
 *  \param[in] h Valor de entrada.
 *  \return El valor de x en h=Hb(x).
 *  \ingroup PdsMathGroup
 */
double pds_hbinv(double h)
{
	double a,b,p0,h0,dh;
	if(h>=1.0)	return 0.5;
	if(h<=0.0)	return 0.0;

	a=0.0;	b=h/2;	
	p0=0.0;

	//dp=0.001;
	if(h<=0.5)	{dh=h/100000;}
	else		{dh=(1.0-h)/100000;}
	
	do{
		p0=(a+b)/2.0;

		h0=pds_hb(p0);

		if(h>h0)	a=p0;
		else		b=p0;
	}while((fabs(h-h0)>dh));

	return p0;
}

/** \fn double pds_nchoosek(unsigned int n,unsigned int k)
 *  \brief Retorna el combinatorio (n,k)
 *  
 *  \f[ {n \choose k}=\frac{n!}{k!(n-k)!} \f] 
 *  \param[in] n Valor superior del combinatorio.
 *  \param[in] k Valor inferior del combinatorio.
 *  \return El valor del combinatorio (n,k).
 *  \ingroup PdsMathGroup
 */
double pds_nchoosek(unsigned int n,unsigned int k)
{
	double C;
	unsigned int a;
	unsigned int i;

	if((n-k)>k)	a=k;
	else		a=n-k;

	C=1.0;
	for(i=0;i<a;i++)	
	{
		C=C*(n-i)/(a-i);
	}

	return C;
}


/** \fn double pds_binomial(unsigned int n,unsigned int k, double p)
 *  \brief Retorna la distribucion binomial(n,k,p)
 *  
 *  \f[ f(n,k,p)={n \choose k} p^k (1-p)^{n-k}\f] 
 *
 *  \param[in] n Numero total de ensayos
 *  \param[in] k Numero de eventos encontrdos.
 *  \param[in] p probbilidad de contecer un evento en un ensayo.
 *  \return El valor de f(n,k,p).
 *  \ingroup PdsMathGroup
 */
double pds_binomial(unsigned int n,unsigned int k, double p)
{
	double f;

	f=pds_nchoosek(n,k)*pow(p,k)*pow(1-p,n-k);

	return f;
}



/** \fn double pds_integration(double (*f)(double), double a,double b,unsigned int n)
 *  \brief Evalúa la integral de a-->b de la función f(x), aplicando la regla de 
 *  Simpson con n divisiones, si n no es par internamente la función hace n=n+1.
 *  
 *  \f[S_n=\int_{a}^{b}f(x)dx\f] 
 *  \f[h=\frac{b-a}{n}\f] 
 *  \f[x_i=a+h~i\f] 
 *  \f[S_n=\frac{h}{3}(f(x_0)+f(x_n)+4\left [ f(x_1)+f(x_3)+\cdots +f(x_{n-1}) \right ]+2\left [ f(x_2)+f(x_4)+\cdots +f(x_{n-2}) \right ])\f] 
 *  \param[in] f La función a integrar.
 *  \param[in] a Límite inferior de la integral.
 *  \param[in] b Límite superior de la integral.
 *  \param[in] n Es el número de divisiones.
 *  \return El valor de la integral o cero si hubo un error, ejemplo b<a o n<=0.
 *  \ingroup PdsMathGroup
 */
double pds_integration(double (*f)(double), double a,double b,unsigned int n)
{
	double S0,Si,Sp,h;
	int i;

	if(b<a)		return 0.0;
	if(n<=0)	return 0.0;
	if(n%2!=0)	n=n+1;
	h=(b-a)/n;

	S0=(*f)(a)+(*f)(a+h*n);
	
	for(i=1,Si=0;i<n;i=i+2)
	{
		Si=Si+(*f)(a+h*i);
	}
	
	for(i=2,Sp=0;i<n;i=i+2)
	{
		Sp=Sp+(*f)(a+h*i);
	}

	return (S0+4.0*Si+2.0*Sp)*h/3.0;
}


/** \fn double pds_integration2(double (*f)(double), double a,double b, double fa,double fb,unsigned int n)
 *  \brief Evalúa la integral de a-->b de la función f(x), aplicando la regla de 
 *  Simpson con n divisiones, si n no es par internamente la función hace n=n+1.
 *  
 *  \f[S_n=\int_{a}^{b}f(x)dx\f] 
 *  \f[h=\frac{b-a}{n}\f] 
 *  \f[x_i=a+h~i\f] 
 *  \f[S_n=\frac{h}{3}(f(a)+f(b)+4\left [ f(x_1)+f(x_3)+\cdots +f(x_{n-1}) \right ]+2\left [ f(x_2)+f(x_4)+\cdots +f(x_{n-2}) \right ])\f] 
 *  \param[in] f La función a integrar.
 *  \param[in] a Límite inferior de la integral.
 *  \param[in] b Límite superior de la integral.
 *  \param[in] fa El resultado de evaluar f(a).
 *  \param[in] fb El resultado de evaluar f(b).
 *  \param[in] n Es el número de divisiones.
 *  \return El valor de la integral o cero si hubo un error, ejemplo b<a o n<=0.
 *  \ingroup PdsMathGroup
 */
double pds_integration2(double (*f)(double), double a,double b, double fa,double fb,unsigned int n)
{
	double Si,Sp,h;
	int i;

	if(b<a)		return 0.0;
	if(n<=0)	return 0.0;
	if(n%2!=0)	n=n+1;
	h=(b-a)/n;

	
	for(i=1,Si=0;i<n;i=i+2)
	{
		Si=Si+(*f)(a+h*i);
	}
	
	for(i=2,Sp=0;i<n;i=i+2)
	{
		Sp=Sp+(*f)(a+h*i);
	}
	return (fa+fb+4.0*Si+2.0*Sp)*h/3.0;
}



/** \fn double pds_integration_inf(double (*f)(double), double a,unsigned int n)
 *  \brief Evalúa la integral de a-->infinito de la función f(x), aplicando el 
 *  cambio de variable u<--1/(x+1) para integrar de 0-->1/(a+1) y ejecutar luego 
 *  la regla de Simpson con n divisiones, si n no es par internamente la 
 *  función hace n=n+1. Además es necesario que f(infinito)-->0.
 *  
 *  \f[S_n=\int_{a}^{ \infty }f(x)dx\f] 
 *  \f[u=\frac{1}{x+1}\f] 
 *  \f[S_n=\int_{0}^{ \frac{1}{a+1} }\frac{f(\frac{1}{u}-1)}{u^2}du\f] 
 *  \param[in] f La función a integrar.
 *  \param[in] a Límite inferior de la integral. a>=0.
 *  \param[in] n Es el número de divisiones.
 *  \return El valor de la integral o cero si hubo un error, ejemplo b<a o n<=0.
 *  \ingroup PdsMathGroup
 */
double pds_integration_inf(double (*f)(double), double a,unsigned int n)
{
	double S0,Si,Sp,h;
	int i;

	if(a<0.0)	return 0.0;
	if(n<=0)	return 0.0;
	if(n%2!=0)	n=n+1;
	h=1.0/((a+1.0)*n);

	S0=0.0+(*f)(a)*(a+1.0)*(a+1.0);

	for(i=1,Si=0;i<n;i=i+2)
	{
		Si=Si+(*f)((1.0/(h*i))-1.0)/((h*i)*(h*i));
	}
	
	for(i=2,Sp=0;i<n;i=i+2)
	{
		Sp=Sp+(*f)((1.0/(h*i))-1.0)/((h*i)*(h*i));
	}
	return (S0+4.0*Si+2.0*Sp)*h/3.0;
}


