/*
 * extras.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


/** \file extras.h
 *  \author Fernando Pujaico Rivera
 *  \date 26-01-2012
 *  \brief Funciones extras para plotar en octave.
 *  
 */

#ifndef __EXTRAS_H__
#define __EXTRAS_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pds/pdsvector.h>

/** \fn int pds_matrix_copy_vector_col(PdsMatrix *Matrix,const PdsVector *VecSrc,PdsRaNatural col)
 *  \brief Copia vector VecSrc en la columna col de la matriz Matrix.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix(:,col)=VecSrc</b>.
 *  \param[in,out] Matrix La matriz a copiar.
 *  \param[in] VecSrc Vector que se usará como fuente.
 *  \param[in] col Columna que se copiará.
 *  \return TRUE si las matrices existen o FALSE si no.
 */
int pds_matrix_copy_vector_col(PdsMatrix *Matrix,const PdsVector *VecSrc,PdsRaNatural col)
{
	PdsRaNatural i,j;
	PdsRaNatural N;

	if(Matrix==NULL)	return FALSE;
	if(VecSrc==NULL)	return FALSE;
	if(col>=Matrix->Ncol)	return FALSE;

	if( Matrix->Nlin < VecSrc->Nel )	N=Matrix->Nlin;
	else					N=VecSrc->Nel;

	for(i=0;i<N;i++)	Matrix->M[i][col]=VecSrc->V[i];

	return TRUE;
}

/** \fn int pds_octplot_surf_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
 *  \brief Imprime una matriz M en un archivo png usando como intermedio un 
 *  archivo *.m de octave. Usando el comando surf.
 *  \param[in] M Matriz a imprimir.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] labelz Etiqueta del eje Z: Altura.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 */
int pds_octplot_surf_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
{
	int i,j;
	FILE *fd=NULL;
	char *orden=NULL;

	if(M==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(labelz==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"M=[");
	for(i=0 ; i<M->Nlin ; i++)	
	{
		for(j=0 ; j<M->Ncol ; j++)
		{
			if(j!=M->Ncol-1)	fprintf(fd,"%e\t",M->M[i][j]);
			else			fprintf(fd,"%e\n",M->M[i][j]);
		}
	}
	fprintf(fd,"];\n");
	fprintf(fd,"surf(M);\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"zlabel(\'%s\');\n",labelz);
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"tmp=input('<< ..:: Press enter to finish ::.. >>');\n");	// <----OJO----
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}


/** \fn int pds_octplot_pcolor_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
 *  \brief Imprime una matriz M en un archivo png usando como intermedio un 
 *  archivo *.m de octave. Usando el comando pcolor.
 *  \param[in] M Matriz a imprimir.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] labelz Etiqueta del eje Z: Altura.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 */
int pds_octplot_pcolor_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
{
	int i,j;
	FILE *fd=NULL;
	char *orden=NULL;

	if(M==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(labelz==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"M=[");
	for(i=0 ; i<M->Nlin ; i++)	
	{
		for(j=0 ; j<M->Ncol ; j++)
		{
			if(j!=M->Ncol-1)	fprintf(fd,"%e\t",M->M[i][j]);
			else			fprintf(fd,"%e\n",M->M[i][j]);
		}
	}
	fprintf(fd,"];\n");
	fprintf(fd,"pcolor(M);\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"zlabel(\'%s\');\n",labelz);
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}


/** \fn int pds_octplot_image_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
 *  \brief Imprime una matriz M en un archivo png usando como intermedio un 
 *  archivo *.m de octave. Usando el comando image.
 *  \param[in] M Matriz a imprimir.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] labelz Etiqueta del eje Z: Altura.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 */
int pds_octplot_image_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
{
	int i,j;
	FILE *fd=NULL;
	char *orden=NULL;

	if(M==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(labelz==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"M=[");
	for(i=0 ; i<M->Nlin ; i++)	
	{
		for(j=0 ; j<M->Ncol ; j++)
		{
			if(j!=M->Ncol-1)	fprintf(fd,"%e\t",M->M[i][j]);
			else			fprintf(fd,"%e\n",M->M[i][j]);
		}
	}
	fprintf(fd,"];\n");
	fprintf(fd,"image(M);\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"zlabel(\'%s\');\n",labelz);
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}

/** \fn int pds_octplot_vector_in_png(const PdsVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime un vector V en un archivo png usando como intermedio un 
 *  archivo *.m de octave.
 *  \param[in] V Vector a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 */
int pds_octplot_vector_in_png(const PdsVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(V==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"V=[");
	for(i=0 ; i<V->Nel ; i++)	fprintf(fd,"%e\t",V->V[i]);
	fprintf(fd,"];\n");
	fprintf(fd,"plot(V,\'o-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}

/** \fn int pds_octplot_cvector_in_png(const PdsCVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime un vector complejo V en un archivo png usando como intermedio un 
 *  archivo *.m de octave.
 *  \param[in] V Vector complejo a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 */
int pds_octplot_cvector_in_png(const PdsCVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(V==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"Vreal=[");
	for(i=0 ; i<V->Nel ; i++)	fprintf(fd,"%e\t",V->V[i].Real);
	fprintf(fd,"];\n");
	fprintf(fd,"Vimag=[");
	for(i=0 ; i<V->Nel ; i++)	fprintf(fd,"%e\t",V->V[i].Imag);
	fprintf(fd,"];\n");
	fprintf(fd,"plot(Vreal,\'o-\',Vimag,\'o-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"legend('Real','Imag');\n");
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}



#ifdef __cplusplus
}
#endif 


#endif /* __EXTRAS_H__ */

