/*
 * pdsstft.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <pds/pdsfft.h>
#include <pds/pdsstft.h>
#include <pds/pdscvector.h>
#include <pds/pdsvector.h>
#include <pds/pdscn.h>
#include <stdio.h>
#include <math.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsStft                                                ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsStft *pds_stft_new(PdsFtNatural L, PdsFtNatural *N)
 *  \brief Crea una estructura de tipo PdsStft, para generar una STFT de L puntos
 *  de entrada y N puntos de salida, N>=L.
 *  Si N no es potencia de 2, no da error, y se crea una estructura para una STFT
 *  con un  N1, que si es potencia de dos y mayor a N1, (N1>=N). El valor de N mínimo
 *  es N=2. El nuevo valor N1 será cargado a N. N=N1.
 *  El vector W (ventana) se incia con todos los elementos con 1.0 .
 *  \param[in] L Es el número de elementos de entrada de la STFT.
 *  \param[in,out] N Es el número de elementos de salida de la STFT.
 *  \return Un puntero a una estructura de tipo PdsStft. En caso de error devuelve
 *  NULL.
 *  \ingroup PdsStftGroup
 */
PdsStft *pds_stft_new(PdsFtNatural L, PdsFtNatural *N)
{
	PdsStft *STFT=NULL;
	PdsFtNatural i,r,n;

	if(*N<=1)	return NULL;
	if(*N<L)	return NULL;


	STFT=(PdsStft *)calloc(1,sizeof(PdsStft));
	if(STFT==NULL)	return NULL;


	STFT->FFT=pds_fft_new(N);
	if(STFT->FFT==NULL)
	{
		free(STFT);
		return NULL;
	}

	STFT->N=*N;
	STFT->L=L;

	STFT->X=pds_cvector_new(STFT->N);
	if(STFT->X==NULL)
	{
		pds_fft_free(STFT->FFT);
		free(STFT);
		return NULL;
	}

	STFT->W=pds_vector_new(STFT->N);
	if(STFT->W==NULL)
	{
		pds_fft_free(STFT->FFT);
		pds_cvector_free(STFT->X);
		free(STFT);
		return NULL;
	}
	pds_vector_init_value (STFT->W,1.0);
	

	return STFT;
}



/** \fn int pds_stft_set_window_rectangular(PdsStft *STFT)
 *  \brief Copia una ventana rectangular a W de la STFT.
 *  \param[in,out] STFT La estructura a una STFT de N puntos.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem STFT==NULL)
 *  \ingroup PdsStftGroup
 */
int pds_stft_set_window_rectangular(PdsStft *STFT)
{
	PdsFtNatural i;

	if(STFT==NULL)	return FALSE;


	for(i=0;i<STFT->N;i++)
	{
		STFT->W->V[i]=1.0;
	}
	

	return TRUE;
}


/** \fn int pds_stft_set_window_hanning(PdsStft *STFT)
 *  \brief Copia una ventana de Hann a W de la STFT.
 *  \param[in,out] STFT La estructura a una STFT de N puntos.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem STFT==NULL)
 *  \ingroup PdsStftGroup
 */
int pds_stft_set_window_hanning(PdsStft *STFT)
{
	PdsFtNatural i;

	if(STFT==NULL)	return FALSE;


	for(i=0;i<STFT->N;i++)
	{
		STFT->W->V[i]=0.5*( 1.0 - cos((2.0*M_PI*i)/(STFT->N-1.0)) );
	}
	

	return TRUE;
}

/** \fn int pds_stft_set_window_hamming(PdsStft *STFT)
 *  \brief Copia una ventana de Hamming a W de la STFT.
 *  \param[in,out] STFT La estructura a una STFT de N puntos.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem STFT==NULL)
 *  \ingroup PdsStftGroup
 */
int pds_stft_set_window_hamming(PdsStft *STFT)
{
	PdsFtNatural i;

	if(STFT==NULL)	return FALSE;


	for(i=0;i<STFT->N;i++)
	{
		STFT->W->V[i]=0.54-0.46*cos((2.0*M_PI*i)/(STFT->N-1.0));
	}
	

	return TRUE;
}


/** \fn int pds_stft_set_window_gauss(PdsStft *STFT, PdsFtReal Sigma)
 *  \brief Copia una ventana de Gauss a W de la STFT.
 *  \param[in,out] STFT La estructura a una STFT de N puntos.
 *  \param[in] Sigma Valor de sigma debe ser <= 0.5, de lo contrario da error.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem STFT==NULL)
 *  \ingroup PdsStftGroup
 */
int pds_stft_set_window_gauss(PdsStft *STFT, PdsFtReal Sigma)
{
	PdsFtNatural i;

	if(STFT==NULL)	return FALSE;
	if(Sigma>0.5)	return FALSE;

	for(i=0;i<STFT->N;i++)
	{
		STFT->W->V[i]=exp( -0.5 * ((i-(STFT->N-1.0)/2.0)/(Sigma*(STFT->N-1.0)/2.0)) * ((i-(STFT->N-1.0)/2.0)/(Sigma*(STFT->N-1.0)/2.0)) );
	}
	

	return TRUE;
}

/** \fn int pds_stft_evaluate_real(PdsStft *STFT,PdsCVector *Out,const PdsVector *In)
 *  \brief Evalua la STFT de un vetor real. El tamanho del vector Out debe ser 
 *  igual al número de puntos de salida de la STFT, El tamanho del vector In debe ser 
 *  igual al número de puntos de entrada de la STFT.
 *
 *  \f[ \mathbf{STFT_m} \left \{ x[n] \right \} \equiv Out_m[k] = \sum_{n=0}^{N-1} x_m[n]w[n]e^{-j 2 \pi k \frac{n}{N}} \f]
 *  \param[in,out] STFT La estructura a una STFT de N puntos.
 *  \param[out] Out El vector complejo con la STFT de N puntos.
 *  \param[in] In El vector de entrada de tamaño igual a L, al que se le desea 
 *  aplicar la STFT.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem STFT==NULL, In==NULL,
 *  Out==NULL, In->Nel!=L o Out->Nel!=N)
 *  \ingroup PdsStftGroup
 */
int pds_stft_evaluate_real(PdsStft *STFT,PdsCVector *Out,const PdsVector *In)
{
	PdsFtNatural i,M;
	int id;

	if(STFT==NULL)	return FALSE;
	if(Out==NULL)	return FALSE;
	if(In==NULL)	return FALSE;
	if(In->Nel!=STFT->L)	return FALSE;
	if(Out->Nel!=STFT->N)	return FALSE;

	M=STFT->N-STFT->L;

	for(i=M;i<STFT->N;i++)
	{
		STFT->X->V[i].Real=(In->V[i-M]) * (STFT->W->V[i]);
		STFT->X->V[i].Imag=0.0;
	}

	id=pds_fft_evaluate_complex(STFT->FFT,Out,STFT->X);
	if(id==FALSE)	return FALSE;

	for(i=0;i<M;i++)
	{
		STFT->X->V[i].Real=(In->V[STFT->L-M+i]) * (STFT->W->V[i]);
		STFT->X->V[i].Imag=0.0;
	}

	

	return TRUE;
}


/** \fn int pds_stft_evaluate_complex(const PdsStft *STFT,PdsCVector *Out,const PdsCVector *In)
 *  \brief Evalua la STFT de un vetor real. El tamanho del vector Out debe ser 
 *  igual al número de puntos de salida de la STFT, El tamanho del vector In debe ser 
 *  igual al número de puntos de entrada de la STFT.
 *
 *  \f[ \mathbf{STFT_m} \left \{ x[n] \right \} \equiv Out_m[k] = \sum_{n=0}^{N-1} x_m[n]w[n]e^{-j 2 \pi k \frac{n}{N}} \f]
 *  \param[in,out] STFT La estructura a una STFT de N puntos.
 *  \param[out] Out El vector complejo con la STFT de N puntos.
 *  \param[in] In El vector de entrada de tamaño igual a L, al que se le desea 
 *  aplicar la STFT.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem STFT==NULL, In==NULL,
 *  Out==NULL, In->Nel!=L o Out->Nel!=N)
 *  \ingroup PdsStftGroup
 */
int pds_stft_evaluate_complex(const PdsStft *STFT,PdsCVector *Out,const PdsCVector *In)
{
	PdsFtNatural i,M;
	int id;

	if(STFT==NULL)	return FALSE;
	if(Out==NULL)	return FALSE;
	if(In==NULL)	return FALSE;
	if(In->Nel!=STFT->L)	return FALSE;
	if(Out->Nel!=STFT->N)	return FALSE;

	M=STFT->N-STFT->L;

	for(i=M;i<STFT->N;i++)
	{
		STFT->X->V[i].Real=In->V[i-M].Real*STFT->W->V[i];
		STFT->X->V[i].Imag=In->V[i-M].Imag*STFT->W->V[i];
	}

	id=pds_fft_evaluate_complex(STFT->FFT,Out,STFT->X);
	if(id==FALSE)	return FALSE;

	for(i=0;i<M;i++)
	{
		STFT->X->V[i].Real=In->V[STFT->L-M+i].Real*STFT->W->V[i];
		STFT->X->V[i].Imag=In->V[STFT->L-M+i].Imag*STFT->W->V[i];
	}

	return TRUE;
}


/** \fn void pds_stft_free(PdsStft *STFT)
 *  \brief Libera una estructura de tipo puntero PdsStft.
 *  \param[in,out] STFT La STFT a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsStftGroup
 */
void pds_stft_free(PdsStft *STFT)
{
	if(STFT!=NULL)
	{
		pds_vector_free(STFT->W);
		pds_cvector_free(STFT->X);
		pds_fft_free(STFT->FFT);
		free(STFT);
	}
}


/** \fn void pds_stft_destroy(PdsStft **STFT)
 *  \brief Libera una estructura de tipo puntero PdsStft, y carga a la estructura con NULL.
 *  \param[in,out] STFT La STFT a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsStftGroup
 */
void pds_stft_destroy(PdsStft **STFT)
{
	if((*STFT)!=NULL)
	{
		pds_vector_free((*STFT)->W);
		pds_cvector_free((*STFT)->X);
		pds_fft_free((*STFT)->FFT);
		free(*STFT);
		(*STFT)=NULL;
	}
}

