/*
 * pdsgoertzel.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsgoertzel.h
 *  \author Fernando Pujaico Rivera
 *  \date 10-05-2011
 *  \brief Funciones que trabajan con una transformada de Fourier (FT) aplicando el 
 *  algoritmo de Goertzel.

 *  Para que este algoritmo sea ventajoso comparado al
 *  algoritmo FFT, debe cumplirse que la cantidad de puntos np de la FT, que se 
 *  desea averiguar debe cumplir np<3N*log2(N)/(4N+4). Si solo se a creado una 
 *  es estructura PdsGoertzel entonces np=1.  
 *  Estas funciones trabajan con una Fft así.<br>
 *  \image html pdsgoertzel.png "Transformada de Fourier aplicando el algoritmo de Goertzel"
   \f{eqnarray*}
        X[k] &=& \sum_{i=0}^{N-1}x[i]W_N^{ik} \\ 
        W_N^{r} &=& e^{-j  2 \pi r / N } \\
            ~&~&~ \\
            np&<&\frac{3~N~log2(N)}{4(N+1)}
   \f}
 */

#ifndef __PDSGOERTZEL_H__
#define __PDSGOERTZEL_H__

#ifdef __cplusplus
extern "C" {
#endif 


#include <stdio.h>

#include <pds/pdsftglobal.h>
#include <pds/pdscn.h>
#include <pds/pdsvector.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif


/** \defgroup PdsGoertzelGroup Funciones del módulo PdsGoertzel.
 *  \brief Encuentra un elemento de la transformada de Fourier de un vector, 
 *  aplicando el algoritmo de Goertzel.
 *  \image html pdsgoertzel.png "Transformada de Fourier aplicando el algoritmo de Goertzel"
   \f{eqnarray*}
        X[k] &=& \sum_{i=0}^{N-1}x[i]W_N^{ik} \\ 
        W_N^{r} &=& e^{-j  2 \pi r / N }
   \f}
 * @{
 */


/*! \struct PdsGoertzel
 *  \brief La estructura tipo  PdsGoertzel.
 *  Esta estructura genera una Transformada de Fourier de N puntos aplicando el 
 *  algoritmo de Goertzel. Para usar incluir pds/pdsft.h.
 *
 *  PdsGoertzel aplica el algoritmo de Goertzel y evalua solo el elemento 
 *  X[k] de la transformada de Fourier de x[n].
 *  \image html pdsgoertzel.png "Transformada de Fourier aplicando el algoritmo de Goertzel"
   \f{eqnarray*}
        X[k] &=& \sum_{i=0}^{N-1}x[i]W_N^{ik} \\ 
        W_N^{r} &=& e^{-j  2 \pi r / N }
   \f}
 *  \ingroup PdsGoertzelGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! a=2*cos(2*pi*k/N). */
	PdsFtReal a;
	/*! Indice del elemento k-esimo,X[k], de la transformada de Fourier de N puntos,
         *  FT{X}=FourierTransform{X} \n
         *  X[k]=FT{x[n]}_k \n k<N
         */
	PdsFtNatural k;
	/*! Elemento multiplicador final  */
	PdsComplex W;
	/*! Número de elementos, N es potencia de 2, 2^r=N  */
	PdsFtNatural N;
}PdsGoertzel;


/** \fn PdsGoertzel *pds_goertzel_new(PdsFtNatural k,PdsFtNatural N)
 *  \brief Crea una estructura de tipo PdsGoertzel, para generar una FT de N puntos.
 *  Si N no es potencia de 2, no da error, y se crea una estructura para una FT
 *  con un  N1, que si es potencia de dos y mayor a N, (N1>=N). El valor de N mínimo
 *  es N=2 .
 *  \param[in] k Es el elemento de la FT{x[n]}, X[k], que se desea encontrar.
 *  \param[in] N Es el número de elementos de la FT.
 *  \return Un puntero a una estructura de tipo PdsGoertzel. En caso de error devuelve
 *  NULL.
 *  \ingroup PdsGoertzelGroup
 */
PdsGoertzel *pds_goertzel_new(PdsFtNatural k,PdsFtNatural N);


/** \fn int pds_goertzel_real_evaluate(const PdsGoertzel *FT,PdsComplex *X,const PdsVector *In)
 *  \brief Evalúa la transformada de fourier X[k] a un vector real. El tamaño 
 *  del vector In debe ser igual o menor que el número de puntos de la FT, La diferencia 
 *  se llenará con ceros.
 *  \param[in] FT La estructura a una FT de Goertzel de N puntos.
 *  \param[out] X El elemento complejo en la posición k de la FT de N puntos.
 *  \param[in] In El vector de un tamaño menor o igual a N, al que se le desea 
 *  aplicar la FT.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsGoertzelGroup
 */
int pds_goertzel_real_evaluate(const PdsGoertzel *FT,PdsComplex *X,const PdsVector *In);


/** \fn void pds_goertzel_free(PdsGoertzel *FT)
 *  \brief Libera una estructura de tipo puntero PdsGoertzel.
 *  \param[in,out] FT La FT de Goertzel a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsGoertzelGroup
 */
void pds_goertzel_free(PdsGoertzel *FT);


/** \fn void pds_goertzel_destroy(PdsGoertzel **FT)
 *  \brief Libera una estructura de tipo puntero PdsGoertzel, y carga a la estructura con NULL.
 *  \param[in,out] FT La FT de Goertzel a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsGoertzelGroup
 */
void pds_goertzel_destroy(PdsGoertzel **FT);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSGOERTZEL_H__ */ 

