/*
 * pdsspectrograph.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <pds/pdsfft.h>
#include <pds/pdsspectrograph.h>
#include <pds/pdscvector.h>
#include <pds/pdsvector.h>
#include <pds/pdscn.h>
#include <stdio.h>
#include <math.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsSpectrograph                                        ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsSpectrograph *pds_spectrograph_new(PdsFtNatural L, PdsFtNatural *N)
 *  \brief Crea una estructura de tipo PdsSpectrograph, para generar un SPECTROGRAPH de L puntos
 *  de tiempo y M puntos de frecuencia, (N=2M) >=L.
 *  Si N no es potencia de 2, no da error, y se crea una estructura para una SPECTROGRAPH
 *  con un  N1, que si es potencia de dos y mayor a N1, (N1>=N). El valor de M mínimo
 *  es M=2 y N=4. El nuevo valor N1 afectara a M. M=N1/2.
 *  El vector W (ventana) se incia con todos los elementos con 1.0 .
 *  \param[in] L Es el número de muestras de tiempo tomados por el SPECTROGRAPH 
 *  para generar la FFT de N puntos.
 *  \param[in,out] N Es el número de elementos de fft del SPECTROGRAPH. La matriz
 *  de salida del espectrograma debera tener M=N/2 lineas.
 *  \return Un puntero a una estructura de tipo PdsSpectrograph. En caso de error devuelve
 *  NULL.
 *  \ingroup PdsSpectrographGroup
 */
PdsSpectrograph *pds_spectrograph_new(PdsFtNatural L, PdsFtNatural *N)
{
	PdsSpectrograph *SPECTROGRAPH=NULL;
	PdsFtNatural i,r,n;

	if(*N<=3)	return NULL;
	if((*N)<L)	return NULL;
	if(L==0)	return NULL;


	SPECTROGRAPH=(PdsSpectrograph *)calloc(1,sizeof(PdsSpectrograph));
	if(SPECTROGRAPH==NULL)	return NULL;

	SPECTROGRAPH->FFT=pds_fft_new(N);
	if(SPECTROGRAPH->FFT==NULL)
	{
		free(SPECTROGRAPH);
		return NULL;
	}

	SPECTROGRAPH->N=*N;
	SPECTROGRAPH->M=SPECTROGRAPH->N/2;
	SPECTROGRAPH->L=L;

	SPECTROGRAPH->X=pds_cvector_new(SPECTROGRAPH->N);
	if(SPECTROGRAPH->X==NULL)
	{
		pds_fft_free(SPECTROGRAPH->FFT);
		free(SPECTROGRAPH);
		return NULL;
	}

	SPECTROGRAPH->W=pds_vector_new(SPECTROGRAPH->N);
	if(SPECTROGRAPH->W==NULL)
	{
		pds_fft_free(SPECTROGRAPH->FFT);
		pds_cvector_free(SPECTROGRAPH->X);
		free(SPECTROGRAPH);
		return NULL;
	}
	pds_vector_init_value (SPECTROGRAPH->W,1.0);
	
	SPECTROGRAPH->Y=pds_cvector_new(SPECTROGRAPH->N);
	if(SPECTROGRAPH->Y==NULL)
	{
		pds_vector_free(SPECTROGRAPH->W);
		pds_cvector_free(SPECTROGRAPH->X);
		pds_fft_free(SPECTROGRAPH->FFT);
		free(SPECTROGRAPH);
		return NULL;
	}
	
	SPECTROGRAPH->V=pds_vector_new(SPECTROGRAPH->N);
	if(SPECTROGRAPH->V==NULL)
	{
		pds_cvector_free(SPECTROGRAPH->Y);
		pds_vector_free(SPECTROGRAPH->W);
		pds_cvector_free(SPECTROGRAPH->X);
		pds_fft_free(SPECTROGRAPH->FFT);
		free(SPECTROGRAPH);
		return NULL;
	}
	return SPECTROGRAPH;
}



/** \fn int pds_spectrograph_rectangular_window(PdsSpectrograph *SPECTROGRAPH)
 *  \brief Copia una ventana rectangular a W de la SPECTROGRAPH.
 *  \param[in,out] SPECTROGRAPH La estructura a una SPECTROGRAPH de N puntos.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem SPECTROGRAPH==NULL)
 *  \ingroup PdsSpectrographGroup
 */
int pds_spectrograph_rectangular_window(PdsSpectrograph *SPECTROGRAPH)
{
	PdsFtNatural i;

	if(SPECTROGRAPH==NULL)	return FALSE;


	for(i=0;i<SPECTROGRAPH->N;i++)
	{
		SPECTROGRAPH->W->V[i]=1.0;
	}
	

	return TRUE;
}


/** \fn int pds_spectrograph_hann_window(PdsSpectrograph *SPECTROGRAPH)
 *  \brief Copia una ventana de Hann a W de la SPECTROGRAPH.
 *  \param[in,out] SPECTROGRAPH La estructura a una SPECTROGRAPH de N puntos.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem SPECTROGRAPH==NULL)
 *  \ingroup PdsSpectrographGroup
 */
int pds_spectrograph_hann_window(PdsSpectrograph *SPECTROGRAPH)
{
	PdsFtNatural i;

	if(SPECTROGRAPH==NULL)	return FALSE;


	for(i=0;i<SPECTROGRAPH->N;i++)
	{
		SPECTROGRAPH->W->V[i]=0.5*( 1.0 - cos((2.0*M_PI*i)/(SPECTROGRAPH->N-1.0)) );
	}
	

	return TRUE;
}

/** \fn int pds_spectrograph_hamming_window(PdsSpectrograph *SPECTROGRAPH)
 *  \brief Copia una ventana de Hamming a W de la SPECTROGRAPH.
 *  \param[in,out] SPECTROGRAPH La estructura a una SPECTROGRAPH de N puntos.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem SPECTROGRAPH==NULL)
 *  \ingroup PdsSpectrographGroup
 */
int pds_spectrograph_hamming_window(PdsSpectrograph *SPECTROGRAPH)
{
	PdsFtNatural i;

	if(SPECTROGRAPH==NULL)	return FALSE;


	for(i=0;i<SPECTROGRAPH->N;i++)
	{
		SPECTROGRAPH->W->V[i]=0.54-0.46*cos((2.0*M_PI*i)/(SPECTROGRAPH->N-1.0));
	}
	

	return TRUE;
}


/** \fn int pds_spectrograph_gauss_window(PdsSpectrograph *SPECTROGRAPH, PdsFtReal Sigma)
 *  \brief Copia una ventana de Gauss a W de la SPECTROGRAPH.
 *  \param[in,out] SPECTROGRAPH La estructura a una SPECTROGRAPH de N puntos.
 *  \param[in] Sigma Valor de sigma debe ser <= 0.5, de lo contrario da error.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem SPECTROGRAPH==NULL)
 *  \ingroup PdsSpectrographGroup
 */
int pds_spectrograph_gauss_window(PdsSpectrograph *SPECTROGRAPH, PdsFtReal Sigma)
{
	PdsFtNatural i;

	if(SPECTROGRAPH==NULL)	return FALSE;
	if(Sigma>0.5)	return FALSE;

	for(i=0;i<SPECTROGRAPH->N;i++)
	{
		SPECTROGRAPH->W->V[i]=exp( -0.5 * ((i-(SPECTROGRAPH->N-1.0)/2.0)/(Sigma*(SPECTROGRAPH->N-1.0)/2.0)) * ((i-(SPECTROGRAPH->N-1.0)/2.0)/(Sigma*(SPECTROGRAPH->N-1.0)/2.0)) );
	}
	

	return TRUE;
}


/** \fn int pds_spectrograph_real_evaluate(PdsSpectrograph *SPECTROGRAPH,PdsMatrix *Out,const PdsVector *In)
 *  \brief Evalua la SPECTROGRAPH de un vector real. El número de lineas la matriz 
 *  Out debe ser igual N/2=M, Si el tamanho del vector de entrada es menor que
 *  L*Out->Ncol se rrelleno con ceros y se realiza el espectrograma, si es mayor,
 *  solo se usa lo que necesito y el resto se ignora.
 *  \param[in,out] SPECTROGRAPH La estructura a una SPECTROGRAPH de M puntos.
 *  \param[out] Out La matriz real con la SPECTROGRAPH de M puntos.
 *  \param[in] In El vector de entrada al que se le desea aplicar la SPECTROGRAPH.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem SPECTROGRAPH==NULL, In==NULL,
 *  Out==NULL o Out->Nlin!=N/2)
 *  \ingroup PdsSpectrographGroup
 */
int pds_spectrograph_real_evaluate(PdsSpectrograph *SPECTROGRAPH,PdsMatrix *Out,const PdsVector *In)
{
	PdsFtNatural i,j,k,M;
	int id;

	if(SPECTROGRAPH==NULL)	return FALSE;
	if(Out==NULL)		return FALSE;
	if(In==NULL)		return FALSE;
	if(Out->Nlin!=SPECTROGRAPH->N/2)	return FALSE;

	M=SPECTROGRAPH->N-SPECTROGRAPH->L;

	for(j=0;j<Out->Ncol;j++)
	{
		for(i=0;i<SPECTROGRAPH->N;i++)
		{
			k=i+j*SPECTROGRAPH->L;
			if(k >= M)
			{
				if( (k-M) < In->Nel )	
				{
					SPECTROGRAPH->X->V[i].Real=In->V[k-M]*SPECTROGRAPH->W->V[i];
					SPECTROGRAPH->X->V[i].Imag=0.0;
				}
				else
				{
					SPECTROGRAPH->X->V[i].Real=0.0;
					SPECTROGRAPH->X->V[i].Imag=0.0;
				}
			}
		}
		id=pds_fft_complex_evaluate(SPECTROGRAPH->FFT,SPECTROGRAPH->Y,SPECTROGRAPH->X);
		if(id==FALSE)	return FALSE;

		id=pds_cvector_modulus2(SPECTROGRAPH->Y,SPECTROGRAPH->V);
		if(id==FALSE)	return FALSE;

		id=pds_matrix_copy_vector_col(Out,SPECTROGRAPH->V,j);
		if(id==FALSE)	return FALSE;
	}
	
	for(i=0;i<M;i++)
	{
		k=i+j*SPECTROGRAPH->L;
		if(k >= M)
		{
			if( (k-M) < In->Nel )	
			{
				SPECTROGRAPH->X->V[i].Real=In->V[k-M]*SPECTROGRAPH->W->V[i];
				SPECTROGRAPH->X->V[i].Imag=0.0;
			}
			else
			{
				SPECTROGRAPH->X->V[i].Real=0.0;
				SPECTROGRAPH->X->V[i].Imag=0.0;
			}
		}
	}

	return TRUE;
}


/** \fn int pds_spectrograph_complex_evaluate(PdsSpectrograph *SPECTROGRAPH,PdsMatrix *Out,const PdsCVector *In)
 *  \brief Evalua la SPECTROGRAPH de un vector complejo. El número de lineas la matriz 
 *  Out debe ser igual N/2=M, Si el tamanho del vector de entrada es menor que
 *  L*Out->Ncol se rrelleno con ceros y se realiza el espectrograma, si es mayor,
 *  solo se usa lo que necesito y el resto se ignora.
 *  \param[in,out] SPECTROGRAPH La estructura a una SPECTROGRAPH de M puntos.
 *  \param[out] Out La matriz real con la SPECTROGRAPH de M puntos.
 *  \param[in] In El vector de entrada al que se le desea aplicar la SPECTROGRAPH.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem SPECTROGRAPH==NULL, In==NULL,
 *  Out==NULL o Out->Nlin!=N/2)
 *  \ingroup PdsSpectrographGroup
 */
int pds_spectrograph_complex_evaluate(PdsSpectrograph *SPECTROGRAPH,PdsMatrix *Out,const PdsCVector *In)
{
	PdsFtNatural i,j,k,M;
	int id;

	if(SPECTROGRAPH==NULL)	return FALSE;
	if(Out==NULL)		return FALSE;
	if(In==NULL)		return FALSE;
	if(Out->Nlin!=SPECTROGRAPH->N/2)	return FALSE;

	M=SPECTROGRAPH->N-SPECTROGRAPH->L;

	for(j=0;j<Out->Ncol;j++)
	{
		for(i=0;i<SPECTROGRAPH->N;i++)
		{
			k=i+j*SPECTROGRAPH->L;
			if(k >= M)
			{
				if( (k-M) < In->Nel )	
				{
					SPECTROGRAPH->X->V[i].Real=In->V[k-M].Real*SPECTROGRAPH->W->V[i];
					SPECTROGRAPH->X->V[i].Imag=In->V[k-M].Imag*SPECTROGRAPH->W->V[i];
				}
				else
				{
					SPECTROGRAPH->X->V[i].Real=0.0;
					SPECTROGRAPH->X->V[i].Imag=0.0;
				}
			}
		}
		id=pds_fft_complex_evaluate(SPECTROGRAPH->FFT,SPECTROGRAPH->Y,SPECTROGRAPH->X);
		if(id==FALSE)	return FALSE;

		id=pds_cvector_modulus2(SPECTROGRAPH->Y,SPECTROGRAPH->V);
		if(id==FALSE)	return FALSE;

		id=pds_matrix_copy_vector_col(Out,SPECTROGRAPH->V,j);
		if(id==FALSE)	return FALSE;
	}
	
	for(i=0;i<M;i++)
	{
		k=i+j*SPECTROGRAPH->L;
		if(k >= M)
		{
			if( (k-M) < In->Nel )	
			{
				SPECTROGRAPH->X->V[i].Real=In->V[k-M].Real*SPECTROGRAPH->W->V[i];
				SPECTROGRAPH->X->V[i].Imag=In->V[k-M].Imag*SPECTROGRAPH->W->V[i];
			}
			else
			{
				SPECTROGRAPH->X->V[i].Real=0.0;
				SPECTROGRAPH->X->V[i].Imag=0.0;
			}
		}
	}

	return TRUE;
}



/** \fn void pds_spectrograph_free(PdsSpectrograph *SPECTROGRAPH)
 *  \brief Libera una estructura de tipo puntero PdsSpectrograph.
 *  \param[in,out] SPECTROGRAPH La SPECTROGRAPH a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsSpectrographGroup
 */
void pds_spectrograph_free(PdsSpectrograph *SPECTROGRAPH)
{
	if(SPECTROGRAPH!=NULL)
	{
		pds_cvector_free(SPECTROGRAPH->X);
		pds_vector_free(SPECTROGRAPH->W);
		pds_cvector_free(SPECTROGRAPH->Y);
		pds_vector_free(SPECTROGRAPH->V);
		pds_fft_free(SPECTROGRAPH->FFT);
		free(SPECTROGRAPH);
	}
}


/** \fn void pds_spectrograph_destroy(PdsSpectrograph **SPECTROGRAPH)
 *  \brief Libera una estructura de tipo puntero PdsSpectrograph, y carga a la estructura con NULL.
 *  \param[in,out] SPECTROGRAPH La SPECTROGRAPH a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsSpectrographGroup
 */
void pds_spectrograph_destroy(PdsSpectrograph **SPECTROGRAPH)
{
	if((*SPECTROGRAPH)!=NULL)
	{
		pds_cvector_free((*SPECTROGRAPH)->X);
		pds_vector_free((*SPECTROGRAPH)->W);
		pds_cvector_free((*SPECTROGRAPH)->Y);
		pds_vector_free((*SPECTROGRAPH)->V);
		pds_fft_free((*SPECTROGRAPH)->FFT);
		free(*SPECTROGRAPH);
		(*SPECTROGRAPH)=NULL;
	}
}

