/*
 * pdsdfutils.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsdfutils.h
 *  \author Fernando Pujaico Rivera
 *  \date 18-01-2012
 *  \brief Funciones que necesita la biblioteca.
 *  
 */

#ifndef __PDSDFUTILS_H__
#define __PDSDFUTILS_H__

#ifdef __cplusplus
extern "C" {
#endif 


#include <pds/pdsdfglobal.h>
#include <pds/pdsvector.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsDfUtilsGroup Funciones útiles de la biblioteca.
 *  \brief Funciones útiles de la biblioteca.
 *  
 * @{
 */


/** \fn int pds_vector_lowpass_butterworth(PdsVector *hnum,PdsVector *hden,PdsDfReal Wn)
 *  \brief Encuentra el numerador y denominador de un filtro Butterworth pasa bajo. 
 *  Ambos vectores de entrada tienen la misma longitud Nel y Nel-1 es el orden
 *  N del filtro Butterworth
 *
 *  \f[ \left|H(j\Omega )\right|={1 \over {\sqrt {1+(\Omega /\Omega_{c})^{2N}}}}, ~~\forall~ \Omega=2 \pi f, ~ \Omega_{c}=2 \pi f_{c} \f]
 *  \f[ H(j\Omega )H(-j\Omega ) =\left|H(j\Omega )\right|^2\f]
 *
 *  <b>Teoría</b><br>
 *  Usa una transformación bi-linear 
 *  \f[ \Omega \leftarrow -j~ \frac{2}{T_d}~{\frac{z-1}{z+1}} \equiv  \frac{2}{T_d}~tan(\frac{w}{2}) \f]
 *
 *  Con una frecuencia de corte igual a \f$ \Omega_c \leftarrow \frac{2}{T_d}~tan(\frac{W_n \pi}{2}) \f$ 
 *  Por comodidad es escogido \f$  T_d \equiv  2~tan(\frac{W_n \pi}{2})\f$ 
 *
 *  \f[ \left( \frac{\Omega}{\Omega_{c}}\right)^{2n} \leftarrow (-1)^n \left(\frac{2}{T_d}\right)^{2n} \left(\frac{z-1}{z+1}\right)^{2n} \f]
 *
 *  \param[out] hnum Coeficientes del numerador del filtro digital.
 *  \param[out] hden Coeficientes del denominador del filtro digital.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. hnum==NULL, hden==NULL, Wn<0 o Wn>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_vector_lowpass_butterworth(PdsVector *hnum,PdsVector *hden,PdsDfReal Wn);


/** \fn PdsVector *pds_vector_new_butterworth_hnum(PdsDfNatural Order,PdsDfReal Wn)
 *  \brief Encuentra el numerador de un filtro Butterworth pasa bajo de orden Order. 
 *
 *  \f[ \left|H(j\Omega )\right|={1 \over {\sqrt {1+(\Omega /\Omega_{c})^{2N}}}}, ~~\forall~ \Omega=2 \pi f, ~ \Omega_{c}=2 \pi f_{c} \f]
 *  \f[ H(j\Omega )H(-j\Omega ) =\left|H(j\Omega )\right|^2\f]
 *
 *  <b>Teoría</b><br>
 *  Usa una transformación bi-linear 
 *  \f[ \Omega \leftarrow -j~ \frac{2}{T_d}~{\frac{z-1}{z+1}} \equiv  \frac{2}{T_d}~tan(\frac{w}{2}) \f]
 *
 *  Con una frecuencia de corte igual a \f$ \Omega_c \leftarrow \frac{2}{T_d}~tan(\frac{W_n \pi}{2}) \f$ 
 *  Por comodidad es escogido \f$  T_d \equiv  2~tan(\frac{W_n \pi}{2})\f$ 
 *
 *  \f[ \left( \frac{\Omega}{\Omega_{c}}\right)^{2n} \leftarrow (-1)^n \left(\frac{2}{T_d}\right)^{2n} \left(\frac{z-1}{z+1}\right)^{2n} \f]
 *
 *  \param[in] Order Orden del filtro Butterworth.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return Un puntero a un vector numerador o NULL en caso de error. 
 *  (ej. Order==0 Wn>1 o Wn<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_butterworth_hnum(PdsDfNatural Order,PdsDfReal Wn);


/** \fn PdsVector *pds_vector_new_butterworth_hden(PdsDfNatural Order,PdsDfReal Wn)
 *  \brief Encuentra el denominador de un filtro Butterworth pasa bajo de orden Order. 
 *
 *  \f[ \left|H(j\Omega )\right|={1 \over {\sqrt {1+(\Omega /\Omega_{c})^{2N}}}}, ~~\forall~ \Omega=2 \pi f, ~ \Omega_{c}=2 \pi f_{c} \f]
 *  \f[ H(j\Omega )H(-j\Omega ) =\left|H(j\Omega )\right|^2\f]
 *
 *  <b>Teoría</b><br>
 *  Usa una transformación bi-linear 
 *  \f[ \Omega \leftarrow -j~ \frac{2}{T_d}~{\frac{z-1}{z+1}} \equiv  \frac{2}{T_d}~tan(\frac{w}{2}) \f]
 *
 *  Con una frecuencia de corte igual a \f$ \Omega_c \leftarrow \frac{2}{T_d}~tan(\frac{W_n \pi}{2}) \f$ 
 *  Por comodidad es escogido \f$  T_d \equiv  2~tan(\frac{W_n \pi}{2})\f$ 
 *
 *  \f[ \left( \frac{\Omega}{\Omega_{c}}\right)^{2n} \leftarrow (-1)^n \left(\frac{2}{T_d}\right)^{2n} \left(\frac{z-1}{z+1}\right)^{2n} \f]
 *
 *  \param[in] Order  Orden del filtro Butterworth.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return Un puntero a un vector numerador o NULL en caso de error. 
 *  (ej. N==0 Wn>1 o Wn<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_butterworth_hden(PdsDfNatural Order,PdsDfReal Wn);


/** \fn int pds_vector_iir_frequency_response(const PdsVector *hnum,const PdsVector *hden,PdsVector *H)
 *  \brief Encuentra el módulo de la respuesta en frecuencia, normalizada 
 *  de 0 a pi del filtro digital conformado por el numerador hnum y denominador hden.
 *
 *  El numero de puntos analizados en la respuesta en frecuencia, es el número 
 *  de elementos del vector H. Los datos de H serán sobrescritos.
 *
 *  <b>Teoría</b><br>
 *  \f[ H(w)=\left| \frac{hnum(Z=e^{j w})}{hden(Z=e^{j w})}\right| , ~\forall ~ 0 \leq w \leq \pi \f]
 *
 *  \f[ \Omega \leftarrow -j~ \frac{2}{T_d}~{\frac{z-1}{z+1}} \equiv  \frac{2}{T_d}~tan(\frac{w}{2}) \f]
 *  \param[in] hnum Coeficientes del numerador del filtro digital.
 *  \param[in] hden Coeficientes del denominador del filtro digital.
 *  \param[out] H Donde se guardará el módulo cuadrado de la respuesta en 
 *  frecuencia del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. hnum==NULL, hden==NULL o H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_vector_iir_frequency_response(const PdsVector *hnum,const PdsVector *hden,PdsVector *H);


/** \fn int pds_vector_lowpass_rectangular(PdsVector *h,PdsDfReal Wn)
 *  \brief Encuentra los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana rectangular.
 *
 *  <b>Teoría:</b><br>
 *  \f[ h(n)=\frac{sin(W_n \pi (n-\frac{Order}{2}))}{\pi (n-\frac{Order}{2})}, ~\forall~n \in Z^+, 0 \leq n \leq Order \f]
 *
 *  \param[out] h Donde se guardarán los coeficientes del filtro digital.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  Wn indica una frecuencia de corte a Wn*Fs/2, donde Fs es la frecuencia de
 *  muestreo.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL, Wn<0 o Wn>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_vector_lowpass_rectangular(PdsVector *h,PdsDfReal Wn);


/** \fn PdsVector *pds_vector_new_lowpass_rectangular(PdsDfNatural N,PdsDfReal Wn)
 *  \brief Devuelve un vector con los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana rectangular.
 *
 *  <b>Teoría:</b><br>
 *  \f[ h(n)=\frac{sin(W_n \pi (n-\frac{Order}{2}))}{\pi (n-\frac{Order}{2})}, ~\forall~n \in Z^+, 0 \leq n \leq Order \f]
 *
 *  \param[in] N Orden del filtro FIR.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital. 
 *  Wn indica una frecuencia de corte a Wn*Fs/2, donde Fs es la frecuencia de
 *  muestreo.
 *  \return Un puntero a un vector de coeficientes de filtro FIR o NULL en caso de error. 
 *  (ej. N==0 Wn>1 o Wn<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_lowpass_rectangular(PdsDfNatural N,PdsDfReal Wn);


/** \fn int pds_vector_lowpass_hamming(PdsVector *h,PdsDfReal Wn)
 *  \brief Encuentra los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hamming.
 *
 *  <b>Teoría:</b><br>
 *  \f[ h(n)=\frac{sin(W_n \pi (n-\frac{Order}{2}))}{\pi (n-\frac{Order}{2})}hamming(n), ~\forall~n \in Z^+, 0 \leq n \leq Order \f]
 *
 *  \param[out] h Donde se guardarán los coeficientes del filtro digital.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  Wn indica una frecuencia de corte a Wn*Fs/2, donde Fs es la frecuencia de
 *  muestreo.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL, Wn<0 o Wn>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_vector_lowpass_hamming(PdsVector *h,PdsDfReal Wn);


/** \fn PdsVector *pds_vector_new_lowpass_hamming(PdsDfNatural N,PdsDfReal Wn)
 *  \brief Devuelve un vector con los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hamming.
 *
 *  <b>Teoría:</b><br>
 *  \f[ h(n)=\frac{sin(W_n \pi (n-\frac{Order}{2}))}{\pi (n-\frac{Order}{2})}hamming(n), ~\forall~n \in Z^+, 0 \leq n \leq Order \f]
 *
 *  \param[in] N Orden del filtro FIR.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital. 
 *  Wn indica una frecuencia de corte a Wn*Fs/2, donde Fs es la frecuencia de
 *  muestreo.
 *  \return Un puntero a un vector de coeficientes de filtro FIR o NULL en caso de error. 
 *  (ej. N==0 Wn>1 o Wn<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_lowpass_hamming(PdsDfNatural N,PdsDfReal Wn);


/** \fn int pds_vector_lowpass_hanning(PdsVector *h,PdsDfReal Wn)
 *  \brief Encuentra los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hanning.
 *
 *  <b>Teoría:</b><br>
 *  \f[ h(n)=\frac{sin(W_n \pi (n-\frac{Order}{2}))}{\pi (n-\frac{Order}{2})}hanning(n), ~\forall~n \in Z^+, 0 \leq n \leq Order \f]
 *
 *  \param[out] h Donde se guardarán los coeficientes del filtro digital.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  Wn indica una frecuencia de corte a Wn*Fs/2, donde Fs es la frecuencia de
 *  muestreo.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL, Wn<0 o Wn>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_vector_lowpass_hanning(PdsVector *h,PdsDfReal Wn);


/** \fn PdsVector *pds_vector_new_lowpass_hanning(PdsDfNatural N,PdsDfReal Wn)
 *  \brief Devuelve un vector con los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hanning.
 *
 *  <b>Teoría:</b><br>
 *  \f[ h(n)=\frac{sin(W_n \pi (n-\frac{Order}{2}))}{\pi (n-\frac{Order}{2})}hanning(n), ~\forall~n \in Z^+, 0 \leq n \leq Order \f]
 *
 *  \param[in] N Orden del filtro FIR.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital. 
 *  Wn indica una frecuencia de corte a Wn*Fs/2, donde Fs es la frecuencia de
 *  muestreo.
 *  \return Un puntero a un vector de coeficientes de filtro FIR o NULL en caso de error. 
 *  (ej. N==0 Wn>1 o Wn<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_lowpass_hanning(PdsDfNatural N,PdsDfReal Wn);


/** \fn int pds_vector_fir_frequency_response(const PdsVector *h,PdsVector *H)
 *  \brief Encuentra el módulo de la respuesta en frecuencia, normalizada 
 *  de 0 a pi del filtro digital conformado por el numerador h.
 *
 *  <b>Teoría</b><br>
 *  \f[ H(w)=\left|h(Z=e^{j w})\right|, ~\forall ~ 0 \leq w \leq \pi \f]
 *
 *  \param[in] h Coeficientes del filtro digital.
 *  \param[out] H Donde se guardará el módulo cuadrado de la respuesta en 
 *  frecuencia del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL o H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_vector_fir_frequency_response(const PdsVector *h,PdsVector *H);



/** \fn int pds_vector_lowpass_to_highpass(PdsVector *H)
 *  \brief Carga un vector con \f$H(-Z)\f$. Tranforma un filtro pasa bajo con un 
 *  corte en Wc a un filtro pasa alto con un corte en PI-Wc. 
 *  
 *  En realidad lo que hace es un corrimiento de PI de la
 *  respuesta en frecuencia. Que es equivalente a cambiar el valor de Z por -Z.
 *  \param[in,out] H Donde se encuentra el filtro pasa bajo y donde se guardará 
 *  el filtro pasa alto.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_vector_lowpass_to_highpass(PdsVector *H);


/** \fn PdsVector *pds_vector_new_lowpass_to_highpass(const PdsVector *H)
 *  \brief Retorna un vector con \f$H(-Z)\f$. Tranforma un filtro pasa bajo con 
 *  una frecuencia de corte en Wc a un filtro pasa alto con un corte en PI-Wc. 
 *  
 *  En realidad lo que hace es un corrimiento de PI de la
 *  respuesta en frecuencia. Que es equivalente a cambiar el valor de Z por -Z.
 *  \param[in] H Donde se encuentra el filtro pasa bajo y donde se guardará 
 *  el filtro pasa alto.
 *  \return Si todo fue bien retorna un puntero a un vector con el filtro pasa 
 *  alto, o NULL si no. (ej. H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_lowpass_to_highpass(const PdsVector *H);


/** \fn PdsVector *pds_vector_new_lowpass_to_bandpass(const PdsVector *H)
 *  \brief Retorna un vector con \f$H(-Z^2)\f$. Crea a partir de un filtro pasa 
 *  bajoo con un corte en Wc un filtro pasa banda centrado en (PI/2) con ancho de 
 *  banda Wc. 
 *
 *  En realidad lo que hace es comprimir
 *  toda respuesta en frecuencia de [-PI,PI] a [-PI/2,PI/2] y dislocarla a PI/2 y
 *  -PI/2. Que es equivalente a cambiar el valor de Z por -Z^2. Conociendo que Nel
 *  es el numero de elementos de H, entonces el vector entregado tendrá 2N-1 elementos.
 *  \param[in] H Donde se encuentra el filtro pasa bajo.
 *  \return Si todo fue bien retorna un puntero a un vector con el filtro pasa 
 *  banda, o NULL si no. (ej. H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_lowpass_to_bandpass(const PdsVector *H);


/** \fn PdsVector *pds_vector_new_lowpass_to_bandreject(const PdsVector *H)
 *  \brief Retorna un vector con \f$H(Z^2)\f$. Crea a partir de un filtro pasa bajo 
 *  con un corte en Wc un filtro rechaza banda centrado en (PI/2) con ancho de 
 *  rechaza banda PI-Wc. 
 *
 *  En realidad lo que hace es comprimir toda respuesta en frecuencia de [-PI,PI] 
 *  a [-PI/2,PI/2] y dislocarla a 0, PI y -PI. Que es equivalente a cambiar el 
 *  valor de Z por Z^2. Conociendo que Nel es el número de elementos de H, entonces 
 *  el vector entregado tendrá 2N-1 elementos.
 *  \param[in] H Donde se encuentra el filtro pasa bajo.
 *  \return Si todo fue bien retorna un puntero a un vector con el filtro rechaza 
 *  banda, o NULL si no. (ej. H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_lowpass_to_bandreject(const PdsVector *H);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


