/*
 * pdsfirlms.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsfirlms.h
 *  \author Fernando Pujaico Rivera
 *  \date 21-05-2011
 *  \brief Funciones que trabajan con estructuras Filtro Fir lms.
 *  
 *  <br>
 *  \image html filtrobloco.png "Filtro LMS FIR"
 *  \image html lms.png "Filtro LMS FIR"
 *  \image html firlms.png "Actualización de pesos"
 *  \f[ y[n]=\sum_{i=0}^{M}h_ix[n-i] \f].
 *  \f[ e[n]=d[n] - y[n] \f].
 *  \f[ h_i[n+1]=h_i[n] + \mu e[n] x[n-i] \f].
 */

#ifndef __PDSFIRLMS_H__
#define __PDSFIRLMS_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <pds/pdsdfglobal.h>

#include <pds/pdsfir.h>
#include <stdio.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsFirLmsGroup Funciones del módulo PdsFirLms.
 *
 *  <br>
 *  \image html filtrobloco.png "Filtro LMS FIR"
 *  \image html lms.png "Filtro LMS FIR"
 *  \image html firlms.png "Actualización de pesos"
 *  \f[ y[n]=\sum_{i=0}^{M}h_ix[n-i] \f].
 *  \f[ e[n]=d[n] - y[n] \f].
 *  \f[ h_i[n+1]=h_i[n] + \mu e[n] x[n-i] \f].
 * @{
 */


/*! \struct PdsFirLms
 *  \brief Una estructura tipo  PdsFirLms .
 *
 *  Esta estructura genera un filtro FIR.
 *  Para usar incluir pds/pdsdf.h.
 *  <br>
 *  \image html filtrobloco.png "Filtro LMS FIR"
 *  \image html lms.png "Filtro LMS FIR"
 *  \image html firlms.png "Actualización de pesos"
 *  \f[ y[n]=\sum_{i=0}^{M}h_ix[n-i] \f].
 *  \f[ e[n]=d[n] - y[n] \f].
 *  \f[ h_i[n+1]=h_i[n] + \mu e[n] x[n-i] \f].
 *  \ingroup PdsFirLmsGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Es el vector de elementos h_i. */
	PdsDfReal Mhu;
	/*! El número de elementos h_i. */
	PdsDfInteger Work;
	/*! Vector de elemnotos x[n-i]. */
	PdsFir *Fir;
}PdsFirLms;


/** \fn PdsFirLms *pds_fir_lms_new(PdsDfReal Mhu, PdsRaNatural M)
 *  \brief Crea un filtro FIR LMS con parametros h[i] del filtro FIR, con un 
 *  valor inicial de h[i]=1/(1+M). Por defecto el filtro FIR LMS estará 
 *  auto configurandose continuamente, a no ser que se deshabilite con
 *  pds_fir_lms_disable .
 * 
 *  \param[in] Mhu Es el paso de la constante de adaptación.
 *  \param[in] M Es el grado del filtro FIR, h[i], 0<= i <=M.
 *  \return Un puntero a una estructura de tipo PdsFirLms. O NULL en caso de error.
 *  \ingroup PdsFirLmsGroup
 */
PdsFirLms *pds_fir_lms_new(PdsDfReal Mhu, PdsRaNatural M);


/** \fn int pds_fir_lms_disable(PdsFirLms *FirLms)
 *  \brief Deshabilita la reconfiguración de los pesos h[i] del filtro FIR LMS
 *  y se mantienen los ultimos pesos modificados
 * 
 *  \param[in,out] FirLms El filtro FIR LMS a usar.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirLmsGroup
 */
int pds_fir_lms_disable(PdsFirLms *FirLms);


/** \fn int pds_fir_lms_enable(PdsFirLms *FirLms)
 *  \brief Habilita la reconfiguración de los pesos h[i] del filtro FIR LMS.
 * 
 *  \param[in,out] FirLms El filtro FIR LMS a usar.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirLmsGroup
 */
int pds_fir_lms_enable(PdsFirLms *FirLms);


/** \fn int pds_fir_lms_set_mhu(PdsFirLms *FirLms,PdsDfReal Mhu)
 *  \brief Coloca el valor Mhu del filtro FIR LMS.
 * 
 *  \param[in,out] FirLms El filtro FIR LMS a usar.
 *  \param[in] Mhu Factor de aprendizaje Mhu.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirLmsGroup
 */
int pds_fir_lms_set_mhu(PdsFirLms *FirLms,PdsDfReal Mhu);


/** \fn int pds_fir_lms_evaluate_value(PdsFirLms *FirLms,PdsDfReal d,PdsDfReal x,PdsDfReal *e,PdsDfReal *y)
 *  \brief Evalúa el filtro FIR LMS con el valor de entrada "d" e "x", el 
 *  resultado es cargado en "e" e "y".
 * 
 *  \param[in,out] FirLms El filtro FIR LMS a usar.
 *  \param[in] d Es una entrada adicional al filtro FIR LMS.
 *  \param[in] x Es la señal de entrada del filtro FIR.
 *  \param[out] e Es la señal de error, es la diferencia entre "d" e "y".
 *  \param[out] y Es el valor de salida del filtro FIR.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirLmsGroup
 */
int pds_fir_lms_evaluate_value(PdsFirLms *FirLms,PdsDfReal d,PdsDfReal x,PdsDfReal *e,PdsDfReal *y);


/** \fn int pds_fir_lms_evaluate_vector(PdsFirLms *FirLms,const PdsVector *d,const PdsVector *x,PdsVector *e,PdsVector *y)
 *  \brief Evalúa el filtro FIR LMS con el vector de entrada x, el resultado
 *  es cargado en el vector y.
 * 
 *  Se recomienda usar esta función solo cuando x es mucho mayo que FIR->h.
 *  Solo se realizan corrimientos de FIR->X al inicio y al final del vector x
 *  en los casos intermediarios se aprovecha tener el vector y no se efectuan
 *  corrimientos, por lo que es un poco mas rápido que pds_fir_lms_evaluate_value
 *  cuando x es mucho mayo que FIR->h.
 *  \param[in,out] FirLms El filtro FIR a usar.
 *  \param[in] d Es el vector de una entrada adicional al filtro FIR LMS.
 *  \param[in] x Es el vector de la señal de entrada del filtro FIR.
 *  \param[out] e Es el vector de la señal de error, es la diferencia entre "d" e "y".
 *  \param[out] y Es el vector del valor de salida del filtro FIR.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsFirLmsGroup
 */
int pds_fir_lms_evaluate_vector(PdsFirLms *FirLms,const PdsVector *d,const PdsVector *x,PdsVector *e,PdsVector *y);


/** \fn void pds_fir_lms_free(PdsFirLms *FirLms)
 *  \brief Libera el filtro de tipo PdsFirLms.
 *  \param[in] FirLms El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFirLmsGroup
 */
void pds_fir_lms_free(PdsFirLms *FirLms);


/** \fn void pds_fir_lms_destroy(PdsFirLms **FirLms)
 *  \brief Libera el filtro de tipo PdsFirLms. y carga la variable con NULL.
 *  \param[in] FirLms El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFirLmsGroup
 */
void pds_fir_lms_destroy(PdsFirLms **FirLms);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSFIRLMS_H__ */ 

