/*
 * pdsfir.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsfir.h
 *  \author Fernando Pujaico Rivera
 *  \date 21-05-2011
 *  \brief Funciones que trabajan con estructuras Filtro Fir.
 *  
 *  <br>
 *  \image html fir-filter.png "Filtro FIR"
 *  \f[ y[n]=\sum_{i=0}^{N-1}h_ix[n-i] \f].
 *  \f[ \frac{Y[z]}{X[z]} = H[z] = \sum_{i=0}^{N-1}h_iz^{-i} \f].
 */

#ifndef __PDSFIR_H__
#define __PDSFIR_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <pds/pdsdfglobal.h>
#include <pds/pdsvector.h>

#include <stdio.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsFirGroup Funciones del módulo PdsFir.
 *
 *  <br>
 *  \image html fir-filter.png "Filtro FIR"
 *  \f[ y[n]=\sum_{i=0}^{N-1}h_ix[n-i] \f].
 *  \f[ \frac{Y[z]}{X[z]} = H[z] = \sum_{i=0}^{N-1}h_iz^{-i} \f].
 * @{
 */


/*! \struct PdsFir
 *  \brief Una estructura tipo  PdsFir .
 *
 *  Esta estructura genera un filtro FIR.
 *  Para usar incluir pds/pdsdf.h.
 *  <br>
 *  \image html fir-filter.png "Filtro FIR"
 *  \f[ y[n]=\sum_{i=0}^{N-1}h_ix[n-i] \f].
 *  \f[ \frac{Y[z]}{X[z]} = H[z] = \sum_{i=0}^{N-1}h_iz^{-i} \f].
 *  \ingroup PdsFirGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Es el vector de elementos h_i. */
	PdsVector *h;
	/*! El número de elementos h_i. */
	PdsDfNatural N;
	/*! Vector de elemnotos x[n-i]. */
	PdsVector *X;
}PdsFir;


/** \fn PdsFir *pds_fir_new(const PdsVector *h)
 *  \brief Crea un filtro FIR con parámetros h.
 * 
 *  El vector h entregado como parámetro a esta función nunca es modificado.
 *  \param[in] h Vector para los coeficientes de h_i*x[n-i].
 *  \return Un puntero a una estructura de tipo PdsFir. O NULL en caso de error.
 *  \ingroup PdsFirGroup
 */
PdsFir *pds_fir_new(const PdsVector *h);


/** \fn int pds_fir_evaluate_value(PdsFir *FIR,PdsDfReal x,PdsDfReal *y)
 *  \brief Evalúa el filtro FIR con el valor de entrada x, el resultado
 *  es cargado en y.
 * 
 *  En cada iteración se realizan corrimientos para el vector FIR->X.
 *  \param[in,out] FIR El filtro FIR a usar.
 *  \param[in] x El valor de entrada del filtro.
 *  \param[out] y El valor de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirGroup
 */
int pds_fir_evaluate_value(PdsFir *FIR,PdsDfReal x,PdsDfReal *y);


/** \fn int pds_fir_evaluate_vector(PdsFir *FIR,const PdsVector *x,PdsVector *y)
 *  \brief Evalúa el filtro FIR con el vector de entrada x, el resultado
 *  es cargado en el vector y.
 * 
 *  Se recomienda usar esta función solo cuando x es mucho mayo que FIR->h.
 *  Solo se realizan corrimientos de FIR->X al inicio y al final del vector x
 *  en los casos intermediarios se aprovecha tener el vector y no se efectuan
 *  corrimientos, por lo que es un poco mas rápido que pds_fir_evaluate_value
 *  cuando x es mucho mayo que FIR->h.
 *  \param[in,out] FIR El filtro FIR a usar.
 *  \param[in] x El vector de entrada del filtro.
 *  \param[out] y El vector de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsFirGroup
 */
int pds_fir_evaluate_vector(PdsFir *FIR,const PdsVector *x,PdsVector *y);



/** \fn int pds_fir_get_x_value(PdsFir *FIR,PdsDfNatural i,PdsDfReal *x)
 *  \brief Esta funcion retorna el valor de X[n-i].
 * 
 *  Siendo n la muestra actual e i indica la i-esima muestra anterior.
 *
 *  \param[in] FIR El filtro FIR a usar.
 *  \param[in] i El valor i de la posicion de X[n-i] en la entrada del filtro.
 *  \param[out] x El valor de X[n-i].
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirGroup
 */
int pds_fir_get_x_value(PdsFir *FIR,PdsDfNatural i,PdsDfReal *x);

/** \fn int pds_fir_get_h_value(PdsFir *FIR,PdsDfNatural i,PdsDfReal *h)
 *  \brief Esta funcion retorna el valor de h[i].
 * 
 *  Siendo h[i] el factor de ponderacion de X[n-i].
 *
 *  \param[in] FIR El filtro FIR a usar.
 *  \param[in] i El valor i de la posicion de h[i].
 *  \param[out] h El valor de h[i].
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirGroup
 */
int pds_fir_get_h_value(PdsFir *FIR,PdsDfNatural i,PdsDfReal *h);

/** \fn pds_fir_get_dimension(PdsFir *FIR,PdsDfNatural *M)
 *  \brief Esta funcion retorna o grado del filtro FIR.
 *
 *  \param[in] FIR El filtro FIR a usar.
 *  \param[out] M El orden del filtro.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirGroup
 */
int pds_fir_get_dimension(PdsFir *FIR,PdsDfNatural *M);

/** \fn void pds_fir_free(PdsFir *FIR)
 *  \brief Libera el filtro de tipo PdsFir.
 *  \param[in] FIR El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFirGroup
 */
void pds_fir_free(PdsFir *FIR);


/** \fn void pds_fir_destroy(PdsFir **FIR)
 *  \brief Libera el filtro de tipo PdsFir. y carga la variable con NULL.
 *  \param[in] FIR El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFirGroup
 */
void pds_fir_destroy(PdsFir **FIR);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSFIR_H__ */ 

