/*
 * pdslpfir.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <pds/pdslpfir.h>
#include <pds/pdsvector.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsLpFir                                                 ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsLpFir *pds_lpfir_new(const PdsVector *h,PdsDfNatural par)
 *  \brief Crea un filtro Linear-Phase FIR con parametros h.
 * 
 *  El vector h entregado como parámetro a esta función nunca es modificado.
 *  \param[in] h Vector para los coeficientes de h_i*x[n-i].
 *  \param[in] par Indica si el vector H total tiene una cantidad de elementos par.
 *  \return Un puntero a una estructura de tipo PdsLpFir. O NULL en caso de error.
 *  \ingroup PdsLpFirGroup
 */
PdsLpFir *pds_lpfir_new(const PdsVector *h,PdsDfNatural par)
{
	PdsLpFir *LPFIR=NULL;

	if(h==NULL)	return NULL;

	LPFIR=(PdsLpFir *)calloc(1,sizeof(PdsLpFir));
	if(LPFIR==NULL)	return NULL;

	LPFIR->Par=par;

	if(par==TRUE)	LPFIR->N=h->Nel*2;
	else		LPFIR->N=h->Nel*2-1;

	LPFIR->h=pds_vector_new_vector(h);
	if(LPFIR->h==NULL)
	{
		free(LPFIR);
		return NULL;
	}

	LPFIR->X=pds_vector_new(LPFIR->N);
	if(LPFIR->X==NULL)
	{
		pds_vector_free(LPFIR->h);
		free(LPFIR);
		return NULL;
	}

	return LPFIR;
}


/** \fn int pds_lpfir_evaluate_value(PdsLpFir *LPFIR,PdsDfReal x,PdsDfReal *y)
 *  \brief Evalúa el filtro Linear-Phase FIR con el valor de entrada x, el resultado
 *  es cargado en y.
 * 
 *  En cada iteración se realizan corrimientos para el vector LPFIR->X.
 *  \param[in,out] LPFIR El filtro Linear-Phase FIR a usar.
 *  \param[in] x El valor de entrada del filtro.
 *  \param[out] y El valor de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsLpFirGroup
 */
int pds_lpfir_evaluate_value(PdsLpFir *LPFIR,PdsDfReal x,PdsDfReal *y)
{
	PdsRaNatural i;
	PdsDfReal S;

	if(LPFIR==NULL)	return FALSE;

	// Corrimiento de x.
	for(i=(LPFIR->N-1);i>0;i--) 
	{
		LPFIR->X->V[i]=LPFIR->X->V[i-1];
	}
	LPFIR->X->V[0]=x;

	// Evaluo salida y.
	for(i=0,S=0;i<(LPFIR->h->Nel-1);i++)
	{
		S=S+LPFIR->h->V[i]*( LPFIR->X->V[i] + LPFIR->X->V[LPFIR->N-1-i] );
	}

	if(LPFIR->Par==TRUE)
	{
		S=S+LPFIR->h->V[i]*( LPFIR->X->V[i] + LPFIR->X->V[LPFIR->N-1-i] );
	}
	else
	{
		S=S+LPFIR->h->V[i]*( LPFIR->X->V[i] );
	}
	
	*y=S;

	return TRUE;
}


/** \fn int pds_lpfir_evaluate_vector(PdsLpFir *LPFIR,const PdsVector *x,PdsVector *y)
 *  \brief Evalúa el filtro Linear-Phase FIR con el vector de entrada x, el resultado
 *  es cargado en el vector y.
 * 
 *  Se recomienda usar esta función solo cuando x es mucho mayor que LPFIR->h.
 *  Solo se realizan corrimientos de LPFIR->X al inicio y al final del vector x
 *  en los casos intermediarios se aprovecha tener el vector, y no se efectuan
 *  corrimientos, por lo que es un poco mas rápido que pds_lpfir_evaluate_value
 *  cuando x es mucho mayo que LPFIR->h.
 *  \param[in,out] LPFIR El filtro Linear-Phase FIR a usar.
 *  \param[in] x El vector de entrada del filtro.
 *  \param[out] y El vector de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsLpFirGroup
 */
int pds_lpfir_evaluate_vector(PdsLpFir *LPFIR,const PdsVector *x,PdsVector *y)
{
	PdsRaNatural i,j,M,N,Nel,L;
	PdsDfReal S;

	if(LPFIR==NULL)		return FALSE;
	if(x==NULL)		return FALSE;
	if(y==NULL)		return FALSE;
	if(x->Nel!=y->Nel)	return FALSE;
	

	N=LPFIR->N;
	L=LPFIR->h->Nel;
	Nel=x->Nel;

	// Evaluó salida y tramo1.
	for(j=0;(j<N)&&(j<Nel);j++)
	{
		S=0;
		for(i=0;(i<=j)&&(i<L);i++)
		{
			S=S+LPFIR->h->V[i]*x->V[j-i];
		}
		for(   ;(i<=j)       ;i++)
		{
			S=S+LPFIR->h->V[N-i-1]*x->V[j-i];
		}

		for(i=j+1;(i<N)&&(i<L);i++)
		{
			S=S+LPFIR->h->V[i]*LPFIR->X->V[i-(j+1)];
		}	
		for(     ;(i<N)       ;i++)
		{
			S=S+LPFIR->h->V[N-i-1]*LPFIR->X->V[i-(j+1)];
		}	
		y->V[j]=S;
	}
	// Evaluó salida y tramo2.
	for(j=N;j<Nel;j++)
	{
		S=0;
		for(i=0;i<L-1;i++)
		{
			S=S+LPFIR->h->V[i]*( x->V[j-i] + x->V[j-(N-1-i)] );
		}

		//i=L-1
		if(LPFIR->Par==TRUE)
		{
			S=S+LPFIR->h->V[i]*( x->V[j-i] + x->V[j-(N-1-i)] );
		}
		else
		{
			S=S+LPFIR->h->V[i]*( x->V[j-i] );
		}

		y->V[j]=S;
	}
	// Ordenamiento final.
	if(N>Nel)	M=N-Nel;
	else			M=0;
	for(j=0;j<M;j++)
	{
		LPFIR->X->V[N-1-j]=LPFIR->X->V[M-1-j];
	}
	for(j=0;(j<Nel)&&(j<N);j++)
	{
		LPFIR->X->V[j]=x->V[(Nel-1)-j];
	}
	return TRUE;
}


/** \fn void pds_lpfir_free(PdsLpFir *LPFIR)
 *  \brief Libera el filtro de tipo PdsLpFir.
 *  \param[in] LPFIR El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsLpFirGroup
 */
void pds_lpfir_free(PdsLpFir *LPFIR)
{
	if(LPFIR!=NULL)
	{
		pds_vector_free(LPFIR->h);
		pds_vector_free(LPFIR->X);
		free(LPFIR);
	}
}


/** \fn void pds_lpfir_destroy(PdsLpFir **LPFIR)
 *  \brief Libera el filtro de tipo PdsLpFir. y carga la variable con NULL.
 *  \param[in] LPFIR El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsLpFirGroup
 */
void pds_lpfir_destroy(PdsLpFir **LPFIR)
{
	if((*LPFIR)!=NULL)
	{
		pds_vector_free((*LPFIR)->h);
		pds_vector_free((*LPFIR)->X);
		free((*LPFIR));
		(*LPFIR)=NULL;
	}
}


