/*
 * pdsiir.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <pds/pdsiir.h>
#include <pds/pdsvector.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsIir                                                 ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsIir *pds_iir_new(const PdsVector *hnum,const PdsVector *hden)
 *  \brief Crea un filtro IIR con parametros hnum y hden.
 * 
 *  \f[ \frac{Y[z]}{X[z]}=H[z]=\frac{\sum_{i=0}^{P}h_{num}[i]z^{-i}}{\sum_{i=0}^{Q}h_{den}[i]z^{-i}} \f].
 *  Los vectores hnum y hdem entregados como parametros de esta función nunca 
 *  son modificados.
 *  \param[in] hnum Vector para los coeficientes de numerador del filtro.
 *  \param[in] hden Vector para los coeficientes de denominador del filtro.
 *  \return Un puntero a una estructura de tipo PdsIir. O NULL en caso de error.
 *  Si a[0] es igual a cero la funcion da error y retorna NULL.
 *  \ingroup PdsIirGroup
 */
PdsIir *pds_iir_new(const PdsVector *hnum,const PdsVector *hden)
{
	PdsIir *IIR=NULL;
	PdsDfNatural M;

	if(hnum==NULL)		return NULL;
	if(hden==NULL)		return NULL;
	if(hden->V[0]==0)	return NULL;

	IIR=(PdsIir *)calloc(1,sizeof(PdsIir));
	if(IIR==NULL)	return NULL;

	IIR->Q=hden->Nel-1;
	IIR->a=pds_vector_new_vector(hden);
	if(IIR->a==NULL)
	{
		free(IIR);
		return NULL;
	}
	pds_vector_mul_value(IIR->a,-1.0/hden->V[0]);

	IIR->P=hnum->Nel-1;
	IIR->b=pds_vector_new_vector(hnum);
	if(IIR->b==NULL)
	{
		pds_vector_free(IIR->a);
		free(IIR);
		return NULL;
	}
	pds_vector_mul_value(IIR->b,1.0/hden->V[0]);

	if(IIR->P > IIR->Q)	M=IIR->P+1;
	else			M=IIR->Q+1;
	IIR->t=pds_vector_new(M);
	if(IIR->t==NULL)
	{
		pds_vector_free(IIR->b);
		pds_vector_free(IIR->a);
		free(IIR);
		return NULL;
	}

	return IIR;
}


/** \fn int pds_iir_evaluate_value(PdsIir *IIR,PdsDfReal x,PdsDfReal *y)
 *  \brief Evalua el filtro IIR con el valor de entrada x, el resultado
 *  es cargado en y.
 * 
 *  \param[in,out] IIR El filtro IIR a usar.
 *  \param[in] x El valor de entrada del filtro.
 *  \param[out] y El valor de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsIirGroup
 */
int pds_iir_evaluate_value(PdsIir *IIR,PdsDfReal x,PdsDfReal *y)
{
	PdsRaNatural j;
	PdsDfReal S;

	if(IIR==NULL)	return FALSE;

		// Corrimiento de t.
		for(j=(IIR->t->Nel-1);j>0;j--) 
		{
			IIR->t->V[j]=IIR->t->V[j-1];
		}

		S=x;

		// Evaluo vector t.
		for(j=1;j<=IIR->Q;j++)
		{
			S=S+IIR->a->V[j]*IIR->t->V[j];
		}
		IIR->t->V[0]=S;

		// Evaluo salida y.
		for(j=0,S=0;j<=IIR->P;j++)
		{
			S=S+IIR->b->V[j]*IIR->t->V[j];
		}

		*y=S;

	return TRUE;
}


/** \fn int pds_iir_evaluate_vector(PdsIir *IIR,const PdsVector *x,PdsVector *y)
 *  \brief Evalua el filtro IIR con el vector de entrada x, el resultado
 *  es cargado en el vector y.
 * 
 *  \param[in,out] IIR El filtro IIR a usar.
 *  \param[in] x El vector de entrada del filtro.
 *  \param[out] y El vector de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo x o y son NULL ó de 
 *  distinto tamaño.
 *  \ingroup PdsIirGroup
 */
int pds_iir_evaluate_vector(PdsIir *IIR,const PdsVector *x,PdsVector *y)
{
	PdsRaNatural i,j;
	PdsDfReal S;

	if(IIR==NULL)	return FALSE;
	if(x==NULL)		return FALSE;
	if(y==NULL)		return FALSE;
	if(x->Nel!=y->Nel)	return FALSE;

	for(i=0;i<x->Nel;i++)
	{
		// Corrimiento de t.
		for(j=(IIR->t->Nel-1);j>0;j--) 
		{
			IIR->t->V[j]=IIR->t->V[j-1];
		}

		S=x->V[i];

		// Evaluo vector t.
		for(j=1;j<=IIR->Q;j++)
		{
			S=S+IIR->a->V[j]*IIR->t->V[j];
		}
		IIR->t->V[0]=S;

		// Evaluo salida y.
		for(j=0,S=0;j<=IIR->P;j++)
		{
			S=S+IIR->b->V[j]*IIR->t->V[j];
		}

		y->V[i]=S;
	}
	return TRUE;
}


/** \fn void pds_iir_free(PdsIir *IIR)
 *  \brief Libera el filtro de tipo PdsIir.
 *  \param[in] IIR El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsIirGroup
 */
void pds_iir_free(PdsIir *IIR)
{
	if(IIR!=NULL)
	{
		pds_vector_free(IIR->t);
		pds_vector_free(IIR->b);
		pds_vector_free(IIR->a);
		free(IIR);
	}
}


/** \fn void pds_iir_destroy(PdsIir **IIR)
 *  \brief Libera el filtro de tipo PdsIir. y carga la variable con NULL.
 *  \param[in] IIR El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsIirGroup
 */
void pds_iir_destroy(PdsIir **IIR)
{
	if((*IIR)!=NULL)
	{
		pds_vector_free((*IIR)->t);
		pds_vector_free((*IIR)->b);
		pds_vector_free((*IIR)->a);
		free((*IIR));
		(*IIR)=NULL;
	}
}


