/*
 * pdsfir.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <pds/pdsfir.h>
#include <pds/pdsvector.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsFir                                                 ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsFir *pds_fir_new(const PdsVector *h)
 *  \brief Crea un filtro FIR con parametros h.
 * 
 *  El vector h entregado como parámetro a esta función nunca es modificado.
 *  \param[in] h Vector para los coeficientes de h_i*x[n-i].
 *  \return Un puntero a una estructura de tipo PdsFir. O NULL en caso de error.
 *  \ingroup PdsFirGroup
 */
PdsFir *pds_fir_new(const PdsVector *h)
{
	PdsFir *FIR=NULL;

	if(h==NULL)	return NULL;

	FIR=(PdsFir *)calloc(1,sizeof(PdsFir));
	if(FIR==NULL)	return NULL;

	FIR->N=h->Nel;
	FIR->h=pds_vector_new_vector(h);
	if(FIR->h==NULL)
	{
		free(FIR);
		return NULL;
	}

	FIR->X=pds_vector_new(FIR->N);
	if(FIR->X==NULL)
	{
		pds_vector_free(FIR->h);
		free(FIR);
		return NULL;
	}

	return FIR;
}


/** \fn int pds_fir_evaluate_value(PdsFir *FIR,PdsDfReal x,PdsDfReal *y)
 *  \brief Evalúa el filtro FIR con el valor de entrada x, el resultado
 *  es cargado en y.
 * 
 *  En cada iteración se realizan corrimientos para el vector FIR->X.
 *  \param[in,out] FIR El filtro FIR a usar.
 *  \param[in] x El valor de entrada del filtro.
 *  \param[out] y El valor de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirGroup
 */
int pds_fir_evaluate_value(PdsFir *FIR,PdsDfReal x,PdsDfReal *y)
{
	PdsRaNatural i;
	PdsDfReal S;

	if(FIR==NULL)	return FALSE;

	// Corrimiento de x.
	for(i=(FIR->N-1);i>0;i--) 
	{
		FIR->X->V[i]=FIR->X->V[i-1];
	}
	FIR->X->V[0]=x;

	// Evaluo salida y.
	for(i=0,S=0;i<FIR->N;i++)
	{
		S=S+FIR->h->V[i]*FIR->X->V[i];
	}

	*y=S;

	return TRUE;
}


/** \fn int pds_fir_evaluate_vector(PdsFir *FIR,const PdsVector *x,PdsVector *y)
 *  \brief Evalúa el filtro FIR con el vector de entrada x, el resultado
 *  es cargado en el vector y.
 * 
 *  Se recomienda usar esta función solo cuando x es mucho mayo que FIR->h.
 *  Solo se realizan corrimientos de FIR->X al inicio y al final del vector x
 *  en los casos intermediarios se aprovecha tener el vector y no se efectuan
 *  corrimientos, por lo que es un poco mas rápido que pds_fir_evaluate_value
 *  cuando x es mucho mayo que FIR->h.
 *  \param[in,out] FIR El filtro FIR a usar.
 *  \param[in] x El vector de entrada del filtro.
 *  \param[out] y El vector de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsFirGroup
 */
int pds_fir_evaluate_vector(PdsFir *FIR,const PdsVector *x,PdsVector *y)
{
	PdsRaNatural i,j,M,N,Nel;
	PdsDfReal S;

	if(FIR==NULL)		return FALSE;
	if(x==NULL)		return FALSE;
	if(y==NULL)		return FALSE;
	if(x->Nel!=y->Nel)	return FALSE;
	

	N=FIR->N;
	Nel=x->Nel;
	// Evaluó salida y tramo1.
	for(j=0;(j<N)&&(j<Nel);j++)
	{
		S=0;
		for(i=0;i<=j;i++)
		{
			S=S+FIR->h->V[i]*x->V[j-i];
		}
		for(i=j+1;i<N;i++)
		{
			S=S+FIR->h->V[i]*FIR->X->V[i-(j+1)];
		}	
		y->V[j]=S;
	}
	// Evaluó salida y tramo2.
	for(j=N;j<Nel;j++)
	{
		S=0;
		for(i=0;i<=(N-1);i++)
		{
			S=S+FIR->h->V[i]*x->V[j-i];
		}
		y->V[j]=S;
	}
	// Ordenamiento final.
	if(N>Nel)	M=N-Nel;
	else			M=0;
	for(j=0;j<M;j++)
	{
		FIR->X->V[N-1-j]=FIR->X->V[M-1-j];
	}
	for(j=0;(j<Nel)&&(j<N);j++)
	{
		FIR->X->V[j]=x->V[(Nel-1)-j];
	}
	return TRUE;
}


/** \fn int pds_fir_get_x_value(PdsFir *FIR,PdsDfNatural i,PdsDfReal *x)
 *  \brief Esta funcion retorna el valor de X[n-i].
 * 
 *  Siendo n la muestra actual e i indica la i-esima muestra anterior.
 *
 *  \param[in] FIR El filtro FIR a usar.
 *  \param[in] i El valor i de la posicion de X[n-i] en la entrada del filtro.
 *  \param[out] x El valor de X[n-i].
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirGroup
 */
int pds_fir_get_x_value(PdsFir *FIR,PdsDfNatural i,PdsDfReal *x)
{
	PdsDfReal S;

	if(FIR==NULL)	return FALSE;

	if((i>=0)&&(i<FIR->N))	*x=FIR->X->V[i];

	return TRUE;
}

/** \fn int pds_fir_get_h_value(PdsFir *FIR,PdsDfNatural i,PdsDfReal *h)
 *  \brief Esta funcion retorna el valor de h[i].
 * 
 *  Siendo h[i] el factor de ponderacion de X[n-i].
 *
 *  \param[in] FIR El filtro FIR a usar.
 *  \param[in] i El valor i de la posicion de h[i].
 *  \param[out] h El valor de h[i].
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirGroup
 */
int pds_fir_get_h_value(PdsFir *FIR,PdsDfNatural i,PdsDfReal *h)
{
	PdsDfReal S;

	if(FIR==NULL)	return FALSE;

	if((i>=0)&&(i<FIR->N))	*h=FIR->h->V[i];

	return TRUE;
}

/** \fn pds_fir_get_dimension(PdsFir *FIR,PdsDfNatural *M)
 *  \brief Esta funcion retorna o grado del filtro FIR.
 *
 *  \param[in] FIR El filtro FIR a usar.
 *  \param[out] M El orden del filtro.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirGroup
 */
int pds_fir_get_dimension(PdsFir *FIR,PdsDfNatural *M)
{
	PdsDfReal S;

	if(FIR==NULL)	return FALSE;

	*M=FIR->N-1;

	return TRUE;
}


/** \fn void pds_fir_free(PdsFir *FIR)
 *  \brief Libera el filtro de tipo PdsFir.
 *  \param[in] FIR El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFirGroup
 */
void pds_fir_free(PdsFir *FIR)
{
	if(FIR!=NULL)
	{
		pds_vector_free(FIR->h);
		pds_vector_free(FIR->X);
		free(FIR);
	}
}


/** \fn void pds_fir_destroy(PdsFir **FIR)
 *  \brief Libera el filtro de tipo PdsFir. y carga la variable con NULL.
 *  \param[in] FIR El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFirGroup
 */
void pds_fir_destroy(PdsFir **FIR)
{
	if((*FIR)!=NULL)
	{
		pds_vector_free((*FIR)->h);
		pds_vector_free((*FIR)->X);
		free((*FIR));
		(*FIR)=NULL;
	}
}


