/*
 * pdsbvector.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsbvector.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-05-2011
 *  \brief Funciones que trabajan con vectores binarios.
 *  
 *  Estas funciones trabajan con un vector binario de la forma:<br>
 *  \image html pdsbvector.png "Vector de Nel elementos"
 *  \b Nel es el número de elementos.
 */

#ifndef __PDSBVECTOR_H__
#define __PDSBVECTOR_H__

#ifdef __cplusplus
extern "C" {
#endif 


#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

#include <stdio.h>
#include <pds/pdsbaglobal.h>

/** \defgroup PdsBVectorGroup Funciones del módulo PdsBVector.
 * @{
 */


/*! \struct PdsBVector
 *  \brief La estructura tipo  PdsBVector .
 *  Esta estructura genera un vector binario de Nel bits.
 *  Para usar incluir pds/pdsbvector.h.
 *  \image html pdsbvector.png "Vector de Nel elementos"
 *  \b Nel es el número de elementos.
 *  \ingroup PdsBVectorGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! un areglo de [Nel/8] bytes. */
	PdsBaByte *V;
	/*! Número de bits. */
	PdsBaNatural Nel;
	/*! Número de bytes. */
	PdsBaNatural Nbytes;
}PdsBVector;

/** \fn PdsBVector *pds_bvector_new(PdsBaNatural Nel)
 *  \brief Crea una matriz de tipo PdsBVector.
 *  \param[in] Nel Es el número de bits del vector.
 *  \return Un puntero al vector de tipo PdsBVector.
 *  \ingroup PdsBVectorGroup
 */
PdsBVector *pds_bvector_new(PdsBaNatural Nel);


/** \fn int pds_bvector_init(PdsBVector *BVector, PdsBaBit Bit)
 *  \brief Inicia con el valor Bit los Nel bits del vector.
 *  \param[in,out] BVector Es el vector a inicializar.
 *  \param[in] Bit Es el valor de los bits del vector, solo {0,1}.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL o Bit!={0,1}.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_init(PdsBVector *BVector, PdsBaBit Bit);


/** \fn int pds_bvector_init_rand(PdsBVector *BVector, PdsBaReal p)
 *  \brief Inicia el vector con algunos "1"s, cada bit tiene una probabilidad p de ser "1" (P{V_i=1}=p).
 *  \param[in,out] BVector Es el vector a inicializar.
 *  \param[in] p Es la probabilidad de cada bit ser 1.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL, p<0 o p>1.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_init_rand(PdsBVector *BVector, PdsBaReal p);


/** \fn int pds_bvector_cmp(const PdsBVector *BVector1,const PdsBVector *BVector2, PdsBaNatural *n)
 *  \brief Encuentra el número de elementos distintos entre los dos vectores.
 *  \param[in] BVector1 Es uno de los vectores a comparar.
 *  \param[in] BVector2 Es uno de los vectores a comparar.
 *  \param[in] n Es el número de elementos distintos entre los dos vectores.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_cmp(const PdsBVector *BVector1,const PdsBVector *BVector2, PdsBaNatural *n);


/** \fn int pds_bvector_cp(PdsBVector *BVector1,const PdsBVector *BVector2)
 *  \brief Copia el contenido del vector BVector2 a BVector1, ambos vectores 
 *  deben tener el mismo tamaño.
 *  \param[in,out] BVector1 Es el vector en donde se copiará.
 *  \param[in] BVector2 Es el vector a copiar.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_cp(PdsBVector *BVector1,const PdsBVector *BVector2);


/** \fn int pds_bvector_set_natural(PdsBVector *BVector,PdsBaNatural n)
 *  \brief Escribe un número natural codificado en binario, en el vector BVector.
 *  Si el número en binario es mayor que el vector no dará error, simplemente se
 *  copiará hasta donde pueda.
 *  \param[in,out] BVector Es el vector en donde se escribirá.
 *  \param[in] n Es el número natural que se escribirá.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_set_natural(PdsBVector *BVector,PdsBaNatural n);


/** \fn int pds_bvector_get_byte(const PdsBVector *BVector,PdsBaNatural n,PdsBaByte *m)
 *  \brief Obtiene el byte de la posición n del vector BVector y lo carga en m.
 *  \param[in] BVector Es el vector en donde se pedirá el byte.
 *  \param[in] n Es el número de byte a pedir.
 *  \param[out] m Es el byte pedido.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL, o n fuera de rango.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_get_byte(const PdsBVector *BVector,PdsBaNatural n,PdsBaByte *m);


/** \fn int pds_bvector_set_byte(PdsBVector *BVector,PdsBaNatural n,PdsBaByte m)
 *  \brief Escribe en el byte de la posición n del vector BVector, desde m.
 *  \param[in,out] BVector Es el vector en donde se escribirá el byte.
 *  \param[in] n Es el número de byte a escribir.
 *  \param[in] m Es el byte a escribir.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL, o n fuera de rango.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_set_byte(PdsBVector *BVector,PdsBaNatural n,PdsBaByte m);


/** \fn int pds_bvector_get_bit(const PdsBVector *BVector,PdsBaNatural n,PdsBaBit *m)
 *  \brief Obtiene el bit de la posición n del vector BVector y lo carga en m.
 *  \param[in] BVector Es el vector en donde se pedirá el bit.
 *  \param[in] n Es el número de bit a pedir.
 *  \param[out] m Es el bit pedido.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL, o n fuera de rango.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_get_bit(const PdsBVector *BVector,PdsBaNatural n,PdsBaBit *m);


/** \fn int pds_bvector_set_bit(PdsBVector *BVector,PdsBaNatural n,PdsBaBit m)
 *  \brief Escribe el bit de la posición n del vector BVector y lo carga desde m.
 *  \param[in,out] BVector Es el vector en donde se escribirá el bit.
 *  \param[in] n Es el número de bit a escribir.
 *  \param[in] m Es el bit escrito.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL, o n fuera de rango.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_set_bit(PdsBVector *BVector,PdsBaNatural n,PdsBaBit m);



/** \fn int pds_bvector_weight_bvector(const PdsBVector *BVector,PdsBaNatural *n)
 *  \brief Retorna el peso del vector binario BVector.
 *  \param[in] BVector Es un vector binario.
 *  \param[out] n El peso del vector binario BVector.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: El vector es nulo.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_weight_bvector(const PdsBVector *BVector,PdsBaNatural *n);


/** \fn int pds_bvector_mean_bvector(const PdsBVector *BVector,PdsBaReal *m)
 *  \brief Retorna el valor medio del vector binario BVector.
 *  \param[in] BVector Es un vector binario.
 *  \param[out] m El valor medio del vector binario BVector.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: El vector es nulo.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_mean_bvector(const PdsBVector *BVector,PdsBaReal *m);


/** \fn pds_bvector_xor_bvector(PdsBVector *BVector1,const PdsBVector *BVector2)
 *  \brief realiza a operación:  BVector1 = BVector1 XOR BVector2;
 *  \param[out] BVector1 Es un vector binario.
 *  \param[in] BVector2 Es un vector binario.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: la cantidad de
 *  elementos es distinta o algun vector es nulo.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_xor_bvector(PdsBVector *BVector1,const PdsBVector *BVector2);


/** \fn int pds_bvector_xor_bvectors(PdsBVector *BVector1,const PdsBVector *BVector2,const PdsBVector *BVector3)
 *  \brief realiza a operación:  BVector1 = BVector2 XOR BVector3;
 *  \param[out] BVector1 Es un vector binario.
 *  \param[in] BVector2 Es un vector binario.
 *  \param[in] BVector3 Es un vector binario.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: la cantidad de
 *  elementos es distinta o algun vector es nulo.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_xor_bvectors(PdsBVector *BVector1,const PdsBVector *BVector2,const PdsBVector *BVector3);


/** \fn int pds_bvector_cor_bvector(const PdsBVector *VectorX,const PdsBVector *VectorY, PdsBaReal *c)
 *  \brief Devuelve el valor del coeficiente de correlación muestral de los vectores VectorX y VectorY.
 *
 *  \f[ \eta_x=\frac{\sum_{i=0}^{Nel-1} {X_i}}{Nel} \f]
 *  \f[ \eta_y=\frac{\sum_{i=0}^{Nel-1} {Y_i}}{Nel} \f]
 *  \f[ cor(X,Y)=\frac{\sum_{i=0}^{Nel-1} {(X_i -{\eta}_x)(Y_i -{\eta}_y)}}{\sqrt{\sum_{i=0}^{Nel-1} {(X_i -{\eta}_x)^2}}\sqrt{\sum_{i=0}^{Nel-1} {(Y_i -{\eta}_y)^2}}} \f]
 *  \param[in] VectorX El vector en consulta.
 *  \param[in] VectorY El vector en consulta.
 *  \param[out] c El valor de la correlación de los vectores VectorX y VectorY.
 *  \return TRUE si todo fue bien o FALSE si no (ej: VectorX==NULL, VectorY==NULL o
 *  longitudes distintas). 
 *  \ingroup PdsVectorGroup
 */
int pds_bvector_cor_bvector(const PdsBVector *VectorX,const PdsBVector *VectorY, PdsBaReal *c);


/** \fn int pds_bvector_printf(const PdsBVector *BVector)
 *  \brief Imprime en pantalla todos los elementos del vector;
 *  \param[in] BVector Es el vector a mostrar.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BVector==NULL.
 *  \ingroup PdsBVectorGroup
 */
int pds_bvector_printf(const PdsBVector *BVector);


/** \fn void pds_bvector_free(PdsBVector *BVector)
 *  \brief Libera un vector de tipo puntero PdsBVector.
 *  \param[in,out] BVector El vector a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBVectorGroup
 */
void pds_bvector_free(PdsBVector *BVector);


/** \fn void pds_bvector_destroy(PdsBVector **BVector)
 *  \brief Libera un vector de tipo puntero PdsBVector.
 *  \param[in,out] BVector El vector a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBVectorGroup
 */
void pds_bvector_destroy(PdsBVector **BVector);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif
/* FPRNEURON_H */ 

