// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/xception.h"
#include "../libpdrx/conversions.h"
#include "config_impl.h"
#include "diagram_impl.h"

#ifdef USE_BOARD

using namespace LibBoard;

#define	COLOR(rgb)	Color(((rgb >> 16) & 0x0FF), ((rgb >> 8) & 0x0FF), (rgb & 0x0FF))

#define	LINE_THICKNESS	0.25

//=== BoardDiagram =========================================================
BoardDiagram::BoardDiagram ()
	: DiagramImpl()
	, m_canvas()
	, m_shapes()
	, m_background(0x00FFFFFF)
{
	m_diagram.width = m_diagram.height = 0.0;
	m_canvas.setUnit(Board::UPoint);
}

void BoardDiagram::OpenCanvas (size_t width, size_t height, RGB bgColor)
{
	// - set a size,
	// - fill the background

	m_diagram.width = width;
	m_diagram.height = height;
	m_background = bgColor;

	m_canvas.clear(COLOR(m_background));
}

void BoardDiagram::CloseCanvas ()
{
	m_canvas << m_shapes;
}

void BoardDiagram::DrawAxes (const ptime& x_min, const ptime& x_max, XUnit x_unit, double y_min, double y_max, double y_unit, RGB color, const string& text, const ConfigImpl::Periods& periods)
{
	try
	{
		DiagramImpl::DrawAxes(x_min, x_max, x_unit, y_min, y_max, y_unit, color, text, periods);

		// Note: libboard doesn't compute real dimensions for text,
		// if we put text on a shape and ask the shape about the
		// size of the text it returns (0,0), so we must guess the
		// dimensions
		const double char_width = m_diagram.width / 94.0;
		const double char_height = m_diagram.width / 94.0;
		const float font_size = float(m_diagram.width / 60.0);

		// Step 1: iterate over the time periods, make backgrounds
		foreach(const ConfigImpl::Period* pPeriod, periods)
		{
			ptime begin(not_a_date_time), end(not_a_date_time);

			{
				ptime t(pPeriod->m_begin, time_duration(0, 0, 0, 0));
				if (t < x_min)
					begin = x_min;
				else
				{
					if (t > x_max)
						begin = x_max;
					else
						begin = t;
				}
			}

			{
				ptime t(pPeriod->m_end, time_duration(0, 0, 0, 0));
				if (t < x_min)
					end = x_min;
				else
				{
					if (t > x_max)
						end = x_max;
					else
						end = t;
				}
			}

			if (begin != end)
				m_shapes << LibBoard::Rectangle(CalcX(begin), CalcY(y_max), CalcX(end) - CalcX(begin), CalcY(y_max) - CalcY(y_min), COLOR(pPeriod->m_color), COLOR(pPeriod->m_color), 0.0);
		}

		// Step 2: x-axis
		const double x_0 = CalcX(m_xmin);
		m_shapes << Line(x_0, 0.0, m_diagram.width, 0.0, COLOR(color), LINE_THICKNESS);

		const double dash_height = m_diagram.height / 100.0; // one percent of m_diagram.height
		if (m_xunit == months)
		{
			// the problem with months is that they differ in
			// their lengths, so we must rather increment the
			// month in the date of the first day than adding
			// just a fixed length as in all the other cases
			//  below
			for (ptime t(m_xmin); t <= m_xmax; )
			{
				string s1(lexical_cast<string>(t)), s2;
				s1.erase(7);
				s2 = s1;
				s1 += "-01 00:00:00.000";
				t = time_from_string(s1);
				if (t >= m_xmin)
				{
					double x = CalcX(t);
					m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
					Text text(x + (char_height / 2.0), - dash_height - (s2.length() * char_width), s2, Fonts::Helvetica, font_size, COLOR(color));
					text.rotate(1.5707963); // 90°
					m_shapes << text;
				}
				const date& d = t.date();
				greg_month m = d.month();
				t = ptime((m == 12) ? date(d.year() + 1, 1, 1) : date(d.year(), m + 1, 1), time_duration(0, 0, 0, 0));
			}
		}
		else
		{
			for (ptime t(m_xmin); t <= m_xmax; t += m_increment)
			{
				string s1(lexical_cast<string>(t)), s2;
				switch (m_xunit)
				{
					case years:	s1.erase(4);  s2 = s1; s1 += "-01-01 00:00:00.000"; break;
					case days:	s1.erase(10); s2 = s1; s1 += " 00:00:00.000"; break;
					case hours:	s1.erase(13); s2 = s1; s1 += ":00:00.000"; break;
					case minutes:	s1.erase(16); s2 = s1; s1 += ":00.000"; break;
					default:	break;
				}
				t = time_from_string(s1);
				if (t >= m_xmin)
				{
					double x = CalcX(t);
					m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
					RGB c = (m_sundays != (RGB)-1 && m_xunit == days && t.date().day_of_week() == 0) ? m_sundays : color;
					Text text(x + (char_height / 2.0), - dash_height - (s2.length() * char_width), s2, Fonts::Helvetica, font_size, COLOR(c));
					text.rotate(1.5707963); // 90°
					m_shapes << text;
				}
			}
		}

		// Step 3: y-axis, much easier because of simple values
		//
		// we must draw the y-axis some pixels longer on the upper
		// end to avoid the top most number to be clipped, the shape
		// will not know about the fact that the top most number has
		// a text height that extends the axis
		//
		m_shapes << Line(x_0, CalcY(m_ymin), x_0, CalcY(m_ymax) + 5.0, COLOR(color), LINE_THICKNESS);

		const double dash_width = m_diagram.width / 100.0; // one percent of m_diagram.width

		double left = x_0;
		double top = CalcY(y_max);
		const string& yf = FormatY();
		for (double t = GetFirstYAxisItem(); t <= m_ymax; t += m_yunit)
		{
			double x = x_0 - dash_width;
			double y = CalcY(t);
			m_shapes << Line(x, y, x_0, y, COLOR(color), LINE_THICKNESS);

			string s((format(yf) % t).str());
			x -= (s.length() * char_width);
			y -= (char_height / 2.0);
			m_shapes << Text(x, y, s, Fonts::Helvetica, font_size, COLOR(color));

			if (left > x)
				left = x;
			if (top < y + char_height)
				top = y + char_height;
		}

		// text above the y-axis
		if (!text.empty())
		{
			m_shapes << LibBoard::Dot(left, top + char_height + 6.0, COLOR(m_background), (float)1.0);
			m_shapes << Text(left, top + 5.0, text, Fonts::Helvetica, font_size, COLOR(color));
		}
	}
	CATCH_RETHROW("")
}

void BoardDiagram::DrawAxes (Diagram::FoldInterval fi, double y_min, double y_max, double y_unit, RGB color, const string& text)
{
	try
	{
		DiagramImpl::DrawAxes(fi, y_min, y_max, y_unit, color, text);

		// Note: libboard doesn't compute real dimensions for text,
		// if we put text on a shape and ask the shape about the
		// size of the text it returns (0,0), so we must guess the
		// dimensions
		const double char_width = m_diagram.width / 94.0;
		const double char_height = m_diagram.width / 94.0;
		const float font_size = float(m_diagram.width / 60.0);

		// Step 1: x-axis
		const double x_0 = CalcX(m_xmin);
		m_shapes << Line(x_0, 0.0, m_diagram.width, 0.0, COLOR(color), LINE_THICKNESS);

		const double dash_height = m_diagram.height / 100.0; // one percent of m_diagram.height
		if (m_xunit == months)
		{
			// the problem with months is that they differ in
			// their lengths, so we must rather increment the
			// month in the date of the first day than adding
			// just a fixed length as in all the other cases
			// below
			for (ptime t(m_xmin); t <= m_xmax; )
			{
				double x = CalcX(t);
				m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
				const string& s = Fold(lexical_cast<string>(t), fi);
				Text text(x + (char_height / 2.0), - dash_height - (s.length() * char_width), s, Fonts::Helvetica, font_size, COLOR(color));
				text.rotate(1.5707963); // 90°
				m_shapes << text;
				const date& d = t.date();
				greg_month m = d.month();
				t = ptime((m == 12) ? date(d.year() + 1, 1, 1) : date(d.year(), m + 1, 1), time_duration(0, 0, 0, 0));
			}
		}
		else
		{
			for (ptime t(m_xmin); t <= m_xmax; t += m_increment)
			{
				double x = CalcX(t);
				m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
				const string& s = Fold(lexical_cast<string>(t), fi);
				Text text(x + (char_height / 2.0), - dash_height - (s.length() * char_width), s, Fonts::Helvetica, font_size, COLOR(color));
				text.rotate(1.5707963); // 90°
				m_shapes << text;
			}
		}

		// Step 2: y-axis, much easier because of simple values
		//
		// we must draw the y-axis some pixels longer on the upper
		// end to avoid the top most number to be clipped, the shape
		// will not know about the fact that the top most number has
		// a text height that extends the axis
		//
		m_shapes << Line(x_0, CalcY(m_ymin), x_0, CalcY(m_ymax) + 5.0, COLOR(color), LINE_THICKNESS);

		const double dash_width = m_diagram.width / 100.0; // one percent of m_diagram.width

		double left = x_0;
		double top = CalcY(y_max);
		const string& yf = FormatY();
		for (double t = GetFirstYAxisItem(); t <= m_ymax; t += m_yunit)
		{
			double x = x_0 - dash_width;
			double y = CalcY(t);
			m_shapes << Line(x, y, x_0, y, COLOR(color), LINE_THICKNESS);

			string s((format(yf) % t).str());
			x -= (s.length() * char_width);
			y -= (char_height / 2.0);
			m_shapes << Text(x, y, s, Fonts::Helvetica, font_size, COLOR(color));

			if (left > x)
				left = x;
			if (top < y + char_height)
				top = y + char_height;
		}

		// text above the y-axis
		if (!text.empty())
		{
			m_shapes << LibBoard::Dot(left, top + char_height + 6.0, COLOR(m_background), (float)1.0);
			m_shapes << Text(left, top + 5.0, text, Fonts::Helvetica, font_size, COLOR(color));
		}
	}
	CATCH_RETHROW("")
}

void BoardDiagram::DrawCurve (const Selection& s, RGB color, LineStyle style, double thickness, size_t splineValues)
{
	try
	{
		const size_t* pDimensions = s.shape(); // [0] rows, [1] columns

		if (pDimensions[0] == 1 && any_cast<string>(s[0][0]).empty())
			THROW("selection contains just one single value which is the result of an aggregation, you must ensure the selection to have multiple values to draw a curve");

		if (thickness == 0.0)
			thickness = LINE_THICKNESS;
		else
		{
			// we want a thickness of 1.0 to draw a line with
			// about 1 pixel width, so we must adjust this a bit
			// because libboard draws already a really thick
			// line using a thickness of 1.0
			thickness /= 2.0;
		}

		if (style == zigzag)
		{
			// in this special case (which is by far the most
			// wanted) we use a Polyline construct instead of
			// many, many Line constructs, we also avoid to have
			// three following equal dots in a line by erasing
			// the middle one, this results in much smaller
			// output files
			size_t row = 0;
			while (row < pDimensions[0])
			{
				XYs xys;
				for ( ; row < pDimensions[0]; row++)
				{
					Dot current(any_cast<string>(s[row][0]), s[row][1]);
					if (current.m_timestamp < m_xmin)
						continue;
					if (!current)
						break;
					XY xy(CalcX(current.m_timestamp), CalcY(any_cast<double>(current.m_value)));
					size_t n = xys.size();
					if (n > 1)
					{
						double y_2 = xys[n - 2].second;
						double y_1 = xys[n - 1].second;
						double y = xy.second;
						if (y_2 == y_1 && y_1 == y)
							xys.erase(--xys.end());
					}
					xys.push_back(xy);
				}

				if (!xys.empty())
				{
					if (splineValues > 1)
						Spline(xys, splineValues);

					LibBoard::Polyline polyline(false, COLOR(color), Color(false), (float)thickness);
					foreach(const XY& xy, xys)
					{
						polyline << Point(xy.first, xy.second);
					}

					m_shapes << polyline;
				}
			}
		}
		else
		{
			Dot last;
			for (size_t row = 0; row < pDimensions[0]; row++)
			{
				Dot current(any_cast<string>(s[row][0]), s[row][1]);

				switch (style)
				{
					case symbol_dot:
					{
						if (current)
						{
							if (current.m_timestamp < m_xmin)
								continue;
							double x = CalcX(current.m_timestamp);
							double y = CalcY(any_cast<double>(current.m_value));
							m_shapes << LibBoard::Dot(x, y, COLOR(color), (float)thickness);
						}
						break;
					}
					case symbol_plus:
					{
						if (current)
						{
							if (current.m_timestamp < m_xmin)
								continue;
							double x = CalcX(current.m_timestamp);
							double y = CalcY(any_cast<double>(current.m_value));
							const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
							m_shapes << Line(x - extent, y, x + extent, y, COLOR(color), (float)thickness);
							m_shapes << Line(x, y - extent, x, y + extent, COLOR(color), (float)thickness);
						}
						break;
					}
					case symbol_vbar:
					{
						if (current)
						{
							if (current.m_timestamp < m_xmin)
								continue;
							double x = CalcX(current.m_timestamp);
							double y = CalcY(any_cast<double>(current.m_value));
							const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
							m_shapes << Line(x, y - extent, x, y + extent, COLOR(color), (float)thickness);
						}
						break;
					}
					case symbol_hbar:
					{
						if (current)
						{
							if (current.m_timestamp < m_xmin)
								continue;
							double x = CalcX(current.m_timestamp);
							double y = CalcY(any_cast<double>(current.m_value));
							const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
							m_shapes << Line(x - extent, y, x + extent, y, COLOR(color), (float)thickness);
						}
						break;
					}
					case symbol_x:
					{
						if (current)
						{
							if (current.m_timestamp < m_xmin)
								continue;
							double x = CalcX(current.m_timestamp);
							double y = CalcY(any_cast<double>(current.m_value));
							const double extent = m_diagram.width / 200.0; // half a percent of m_diagram.width
							m_shapes << Line(x - extent, y - extent, x + extent, y + extent, COLOR(color), (float)thickness);
							m_shapes << Line(x - extent, y + extent, x + extent, y - extent, COLOR(color), (float)thickness);
						}
						break;
					}
					case symbol_X:
					{
						if (current)
						{
							if (current.m_timestamp < m_xmin)
								continue;
							double x = CalcX(current.m_timestamp);
							double y = CalcY(any_cast<double>(current.m_value));
							const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
							m_shapes << Line(x - extent, y - extent, x + extent, y + extent, COLOR(color), (float)thickness);
							m_shapes << Line(x - extent, y + extent, x + extent, y - extent, COLOR(color), (float)thickness);
						}
						break;
					}
					case symbol_circle:
					{
						if (current)
						{
							if (current.m_timestamp < m_xmin)
								continue;
							double x = CalcX(current.m_timestamp);
							double y = CalcY(any_cast<double>(current.m_value));
							const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
							m_shapes << Circle(x, y, extent, COLOR(color), Color(false), (float)thickness);
						}
						break;
					}
					case symbol_square:
					{
						if (current)
						{
							if (current.m_timestamp < m_xmin)
								continue;
							double x = CalcX(current.m_timestamp);
							double y = CalcY(any_cast<double>(current.m_value));
							const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
							m_shapes << LibBoard::Rectangle(x - extent / 2.0, y - extent / 2.0, extent, extent, COLOR(color), Color(false), (float)thickness);
						}
						break;
					}
					default: // zick-zack
					{
						if (current.m_timestamp < m_xmin)
							continue;
						if (last && current)
						{
							m_shapes << Line(
								CalcX(last.m_timestamp), CalcY(any_cast<double>(last.m_value)),
								CalcX(current.m_timestamp), CalcY(any_cast<double>(current.m_value)),
								COLOR(color), (float)thickness
							);
						}
						break;
					}
				}

				if (current)
					last = current;
			}
		}
	}
	CATCH_RETHROW("")
}

void BoardDiagram::DrawStairs (const Selection& s, RGB color, double thickness)
{
	try
	{
		const size_t* pDimensions = s.shape(); // [0] rows, [1] columns

		if (pDimensions[0] == 1 && any_cast<string>(s[0][0]).empty())
			THROW("selection contains just one single value which is the result of an aggregation, you must ensure the selection to have multiple values to draw a curve");

		if (thickness == 0.0)
			thickness = LINE_THICKNESS;
		else
		{
			// we want a thickness of 1.0 to draw a line with
			// about 1 pixel width, so we must adjust this a bit
			// because libboard draws already a really thick
			// line using a thickness of 1.0
			thickness /= 2.0;
		}

		size_t row = 0;
		while (row < pDimensions[0])
		{
			vector<Point> points;
			for ( ; row < pDimensions[0]; row++)
			{
				Dot current(any_cast<string>(s[row][0]), s[row][1]);

				if (current.m_timestamp < m_xmin)
					continue;

				if (!current)
					break;

				Point point(CalcX(current.m_timestamp), CalcY(any_cast<double>(current.m_value)));
				size_t n = points.size();
				if (n > 1)
				{
					double y_2, y_1, y, x_dummy;
					points[n - 2].get(x_dummy, y_2);
					points[n - 1].get(x_dummy, y_1);
					point.get(x_dummy, y);
					if (y_2 == y_1 && y_1 == y)
						points.erase(--points.end());
				}
				points.push_back(point);
			}

			if (!points.empty())
			{
				LibBoard::Polyline polyline(false, COLOR(color), Color(false), (float)thickness);
				vector<Point>::const_iterator I, J;
				J = I = points.begin();
				J++;
				polyline << *I;
				while (J != points.end())
				{
					const Point& p1 = *I++;
					const Point& p2 = *J++;
					double x1, y1, x2, y2;
					p1.get(x1, y1);
					p2.get(x2, y2);
					polyline << Point(x2, y1);
					polyline << p2;
				}
				m_shapes << polyline;
			}
		}
	}
	CATCH_RETHROW("")
}

void BoardDiagram::DrawVLine (const ptime& timestamp, RGB color, double thickness)
{
	try
	{
		double x = CalcX(timestamp);
		m_shapes << Line(x, CalcY(m_ymin), x, CalcY(m_ymax), COLOR(color), (float)((thickness == 0.0) ? LINE_THICKNESS : (thickness / 2.0)));
	}
	CATCH_RETHROW("")
}

void BoardDiagram::DrawVLine (const time_duration& time, RGB color, double thickness)
{
	try
	{
		DrawVLine(ptime(date(9999, 1, 1), time), color, thickness);
	}
	CATCH_RETHROW("")
}

void BoardDiagram::DrawHLine (double value, RGB color, double thickness)
{
	try
	{
		if (value < m_ymin || value > m_ymax)
			return;

		double y = CalcY(value);
		m_shapes << Line(CalcX(m_xmin), y, m_diagram.width, y, COLOR(color), (float)((thickness == 0.0) ? LINE_THICKNESS : (thickness / 2.0)));
	}
	CATCH_RETHROW("")
}

void BoardDiagram::DrawBar (const ptime& t, double value, int bar1, int bar2, RGB color, bool fix)
{
	try
	{
		double x = CalcX(t);
		double y = CalcY(value);
		double x_width = (fix) ? (CalcX(t + m_increment) - x) / double(bar2) : (CalcX(min(t + m_increment, m_xmax)) - x) / double(bar2);
		x += x_width * double(bar1 - 1);

		if (x >= 0)
			m_shapes << LibBoard::Rectangle(x, y, x_width, y - CalcY(m_ymin), COLOR(color), COLOR(color), 0);
	}
	CATCH_RETHROW("")
}

double BoardDiagram::CalcX (const ptime& x) const
{
	try
	{
		double a = (x - m_xmin).total_seconds();
		double b = (m_xmax - m_xmin).total_seconds();
		return (a / b) * m_diagram.width;
	}
	CATCH_RETHROW("")
}

double BoardDiagram::CalcY (double y) const
{
	try
	{
		double a = y - m_ymin;
		double b = m_ymax - m_ymin;
		return (a / b) * m_diagram.height;
	}
	CATCH_RETHROW("")
}

//=== DiagramSVG ===========================================================
DiagramSVG::DiagramSVG ()
	: BoardDiagram()
{
}

void DiagramSVG::SaveAs (const string& filename) const
{
	m_canvas.saveSVG(filename.c_str(), Board::BoundingBox);
}

//=== DiagramEPS ===========================================================
DiagramEPS::DiagramEPS ()
	: BoardDiagram()
{
}

void DiagramEPS::SaveAs (const string& filename) const
{
	m_canvas.saveEPS(filename.c_str(), Board::BoundingBox);
}

//=== DiagramXFig ==========================================================
DiagramXFig::DiagramXFig ()
	: BoardDiagram()
{
}

void DiagramXFig::SaveAs (const string& filename) const
{
	m_canvas.saveFIG(filename.c_str(), Board::BoundingBox);
}

#endif // USE_BOARD
