// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/xception.h"
#include "../libpdrx/conversions.h"
#include "../libpdrx/config.h"
#include "db.h"
#include "out_impl.h"

//=== OutputXML ============================================================
OutputXML::OutputXML (const string& option_key, const string& filename)
	: OutputImpl(option_key)
	, m_filename(filename)
{
}

#define	INDENT	string(indent * 2, ' ')

	static string transform_into_xml (const string& s)
	{
		string result;
		for (size_t i = 0; i < s.length(); i++)
		{
			switch (s[i])
			{
				case '<':	result += "&lt;"; break;
				case '>':	result += "&gt;"; break;
				case '&':	result += "&amp;"; break;
				case '\'':	result += "&apos;"; break;
				case '"':	result += "&quot;"; break;
				default:	result += s[i]; break;
			}
		}
		return result;
	}

void OutputXML::Do (const Config& config, const Database& database) const throw (Xception)
{
	bool verbose = config.GetBoolOption("verbose");
	if (verbose)
		encoded::cout << "exporting database as XML" << endl;

	ofstream ofs(m_filename.c_str(), ios::out);
	if (!ofs.good())
		THROW(format("error creating output file %s") % m_filename);

	ofs << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << endl;
	ofs << "<pdr>" << endl;

	// get the collection names
	Database::Collections collections;
	database.GetCollections(collections);

	size_t indent = 0;

	// dump data
	indent++;
	foreach (const string& collection, collections)
	{
		char type = database.GetCollectionType(collection);

		ofs << INDENT << "<collection name=\"" << collection << "\" type=\"";
		switch (type)
		{
			case 'n':	ofs << "numeric"; break;
			case 'r':	ofs << "ratio"; break;
			case 't':	ofs << "text"; break;
			default:	break;
		}
		ofs << "\" purpose=\"" << database.GetCollectionPurpose(collection) << "\" unit=\"" << database.GetCollectionUnit(collection) << "\">" << endl;

		indent++;
		Database::CollectionItems items;
		database.GetCollectionItems(collection, items);
		if (!items.empty())
		{
			foreach (const Database::CollectionItems::value_type& vt, items)
			{
				ofs << INDENT << "<item datetime=\"" << lexical_cast<string>(vt.first) << "\" value=\"";
				switch (type)
				{
					case 'n':	ofs << any_cast<double>(vt.second); break;
					case 'r':	ofs << lexical_cast<string>(any_cast<Ratio>(vt.second)); break;
					case 't':	ofs << transform_into_xml(any_cast<string>(vt.second)); break;
					default:	break;
				}
				ofs << "\" />" << endl;
			}
		}
		indent--;

		ofs << INDENT << "</collection>" << endl;
	}
	indent--;

	ofs << "</pdr>" << endl;
}
