// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/xception.h"
#include "../libpdrx/conversions.h"
#include "../libpdrx/config.h"
#include "db.h"
#include "in_impl.h"

#ifdef USE_ETPAN

// on WIN32 libetpan conflicts with boost, so we must define some
// identifiers here before libetpan makes conflicting typedefs
#ifdef WIN32
#define uint16_t unsigned __int16
#define uint32_t unsigned __int32
#define int32_t __int32
#endif

#include <libetpan/libetpan.h>

//=== ImapMailClient =======================================================
ImapMailClient::ImapMailClient (const string& option_key)
	: InputImpl(option_key)
{
}

	#define	NO_ETPAN_ERROR(r)	(r == MAILIMAP_NO_ERROR || r == MAILIMAP_NO_ERROR_AUTHENTICATED || r == MAILIMAP_NO_ERROR_NON_AUTHENTICATED)

	static bool has_attr (struct mailimap_msg_att* msg_att, int attr)
	{
		for (clistiter* c1 = clist_begin(msg_att->att_list) ; c1 != NULL ; c1 = clist_next(c1))
		{
			struct mailimap_msg_att_item* item = (struct mailimap_msg_att_item*)clist_content(c1);
			if (item->att_type == MAILIMAP_MSG_ATT_ITEM_DYNAMIC)
			{
				struct mailimap_msg_att_dynamic* p = item->att_data.att_dyn;
				for (clistiter* c2 = clist_begin(p->att_list) ; c2 != NULL ; c2 = clist_next(c2))
				{
					struct mailimap_flag_fetch* ff = (struct mailimap_flag_fetch*)clist_content(c2);
					switch (ff->fl_type)
					{
						case MAILIMAP_FLAG_FETCH_OTHER:
						{
							if (ff->fl_flag->fl_type == attr)
								return true;
							break;
						}
						default:
							break;
					}
				}
			}
		}

		return false;
	}

	static uint32_t get_uid (struct mailimap_msg_att* msg_att)
	{
		for (clistiter* cur = clist_begin(msg_att->att_list) ; cur != NULL ; cur = clist_next(cur))
		{
			struct mailimap_msg_att_item* item = (struct mailimap_msg_att_item*)clist_content(cur);
			if (item->att_type == MAILIMAP_MSG_ATT_ITEM_STATIC && item->att_data.att_static->att_type == MAILIMAP_MSG_ATT_UID)
				return item->att_data.att_static->att_data.att_uid;
		}

		return 0;
	}

	static string get_msg_content (clist* fetch_result)
	{
		for (clistiter* c1 = clist_begin(fetch_result); c1 != NULL; c1 = clist_next(c1))
		{
			struct mailimap_msg_att* msg_att = (struct mailimap_msg_att*)clist_content(c1);
			for (clistiter* c2 = clist_begin(msg_att->att_list) ; c2 != NULL ; c2 = clist_next(c2))
			{
				struct mailimap_msg_att_item* item = (struct mailimap_msg_att_item*)clist_content(c2);
				if (item->att_type == MAILIMAP_MSG_ATT_ITEM_STATIC && item->att_data.att_static->att_type == MAILIMAP_MSG_ATT_BODY_SECTION)
					return string(item->att_data.att_static->att_data.att_body_section->sec_body_part, 0, item->att_data.att_static->att_data.att_body_section->sec_length);
			}
		}

		return string();
	}

	static string get_header (const string& msg_content, const string& header_name)
	{
		stringstream ss(msg_content);

		regex rx(header_name + ": (.*)");
		string line;
		while (getline(ss, line))
		{
			trim(line);
			if (line.empty())
				break;
			smatch mr;
			if (regex_match(line, mr, rx, boost::match_not_dot_newline))
				return string(mr[1]);
		}

		return string();
	}

	static string get_body (const string& msg_content)
	{
		// the body is all text behind the list of headers which
		// is terminated by an empty line

		stringstream ss(msg_content);

		string line;
		while (getline(ss, line))
		{
			trim(line);
			if (line.empty())
				break;
		}

		string content;
		while (getline(ss, line))
		{
			trim(line);
			content += line + '\n';
		}
		return content;
	}

	static void handle_message (struct mailimap* imap, uint32_t uid, const string& subject, Database& database, bool keep, bool verbose, const Database::Collections& collections, bool& data, bool& rejects)
	{
		struct mailimap_set* set = mailimap_set_new_single(uid);
		struct mailimap_fetch_type* fetch_type = mailimap_fetch_type_new_fetch_att_list_empty();
		struct mailimap_section* section = mailimap_section_new(NULL);
		struct mailimap_fetch_att* fetch_att = mailimap_fetch_att_new_body_peek_section(section);
		mailimap_fetch_type_new_fetch_att_list_add(fetch_type, fetch_att);

		clist* fetch_result; // mailimap_msg_att
		int result = mailimap_uid_fetch(imap, set, fetch_type, &fetch_result);
		if (NO_ETPAN_ERROR(result))
		{
			string msg_content = get_msg_content(fetch_result);
			if (!msg_content.empty())
			{
				// check subject
				const string& s = InputImpl::ConvertMailSubject(get_header(msg_content, "Subject"));
				if (s != subject)
					return;

				data = true;

				// extract the timestamp
				const ptime& timestamp = lexical_cast_mime_date(get_header(msg_content, "Date"));

				// extract the Content-Transfer-Encoding
				string content_transfer_encoding(get_header(msg_content, "Content-Transfer-Encoding"));
				to_lower(content_transfer_encoding);

				// get and decode the body
				stringstream ss;
				if (content_transfer_encoding == "base64")
				{
					const string& s = get_body(msg_content);
					size_t index = 0;
					char* buffer;
					size_t len;
					if (mailmime_base64_body_parse(s.c_str(), s.length(), &index, &buffer, &len) == 0)
						ss << buffer;
					mmap_string_unref(buffer);
				}
				else
				{
					if (content_transfer_encoding == "quoted-printable")
					{
						const string& s = get_body(msg_content);
						size_t index = 0;
						char* buffer;
						size_t len;
						if (mailmime_quoted_printable_body_parse(s.c_str(), s.length(), &index, &buffer, &len, 0) == 0)
							ss << buffer;
						mmap_string_unref(buffer);
					}
					else
					{
						// 7bit, 8bit, binary
						const string& s = get_body(msg_content);
						size_t index = 0;
						char* buffer;
						size_t len;
						if (mailmime_binary_body_parse(s.c_str(), s.length(), &index, &buffer, &len) == 0)
							ss << buffer;
						mmap_string_unref(buffer);
					}
				}

				// parse line by line, every line an expression
				string line;
				while (getline(ss, line))
				{
					trim(line);
					if (line.empty())
						continue;
					try
					{
						Database::CollectionsItems items;
						InputImpl::Parse(line, timestamp, verbose, collections, items);
						database.AddCollectionsItems(items);
					}
					catch (const Xception& x)
					{
						encoded::cerr << x.Message(Xception::brief) << endl;
						database.AddRejected(timestamp, line);
						rejects = true;
					}
				}
			}

			// finally apply flags to the message on the server
			struct mailimap_flag_list* flag_list = mailimap_flag_list_new_empty();
			mailimap_flag_list_add(flag_list, mailimap_flag_new_seen());
			if (!keep)
				mailimap_flag_list_add(flag_list, mailimap_flag_new_deleted());
			struct mailimap_store_att_flags* store_att_flags = mailimap_store_att_flags_new(1, 1, flag_list);
			mailimap_uid_store(imap, set, store_att_flags);
			mailimap_store_att_flags_free(store_att_flags);
		}

		mailimap_fetch_list_free(fetch_result);
	}

void ImapMailClient::Do (const Config& config, Database& database) const throw (Xception)
{
	bool verbose = config.GetBoolOption("verbose");
	if (verbose)
		encoded::cout << "looking for mail (IMAP)" << endl;

	// get configuration data
	const string& server = config.GetStringOption(m_option_key + ".server");
	if (server.empty())
		THROW(format("missing specification in configuration file: %s.server)") % m_option_key);

	int port = (config.HasOption(m_option_key + ".port"))
		? (int)config.GetDoubleOption(m_option_key + ".port")
		: 143;

	const string& account = config.GetStringOption(m_option_key + ".account");
	if (account.empty())
		THROW(format("missing specification in configuration file: %s.account)") % m_option_key);

	const string& password = config.GetStringOption(m_option_key + ".password");

	const string& folder = (config.HasOption(m_option_key + ".folder"))
		? config.GetStringOption(m_option_key + ".folder")
		: "INBOX";

	const string& subject = config.GetStringOption(m_option_key + ".subject");

	bool keep = config.GetBoolOption(m_option_key + ".keep");
	bool expunge = config.GetBoolOption(m_option_key + ".expunge");

	// request the IMAP server
	string msg;
	bool data = false;
	bool rejects = false;
	{
		Database::Collections collections;
		database.GetCollections(collections);

		struct mailimap* imap = mailimap_new(0, NULL);

		int result = (port == 993)
			? mailimap_ssl_connect(imap, server.c_str(), port)
			: mailimap_socket_connect(imap, server.c_str(), port);

		if (NO_ETPAN_ERROR(result))
		{
			result = mailimap_login(imap, account.c_str(), password.c_str());
			if (NO_ETPAN_ERROR(result))
			{
				result = mailimap_select(imap, folder.c_str());
				if (NO_ETPAN_ERROR(result))
				{
					struct mailimap_set* set = mailimap_set_new_interval(1, 0);

					struct mailimap_fetch_type* fetch_type = mailimap_fetch_type_new_fetch_att_list_empty();
					mailimap_fetch_type_new_fetch_att_list_add(fetch_type, mailimap_fetch_att_new_uid());
					mailimap_fetch_type_new_fetch_att_list_add(fetch_type, mailimap_fetch_att_new_flags());

					clist* fetch_result;
					result = mailimap_fetch(imap, set, fetch_type, &fetch_result);
					if (NO_ETPAN_ERROR(result))
					{
						for (clistiter* cur = clist_begin(fetch_result); cur != NULL; cur = clist_next(cur))
						{
							struct mailimap_msg_att* msg_att = (struct mailimap_msg_att*)clist_content(cur);
							if (!has_attr(msg_att, MAILIMAP_FLAG_DELETED) && !has_attr(msg_att, MAILIMAP_FLAG_SEEN))
							{
								uint32_t uid = get_uid(msg_att);
								if (uid != 0)
									handle_message(imap, uid, subject, database, keep, verbose, collections, data, rejects);
							}
						}
					}
					else
						; // do nothing, no data

					mailimap_fetch_list_free(fetch_result);

					if (expunge)
						mailimap_expunge(imap);
				}
				else
					msg = (format("could not open mail folder %s on IMAP server") % folder).str();
				mailimap_logout(imap);
			}
			else
				msg = (format("could not authenticate to IMAP server %s") % server).str();
		}
		else
			msg = (format("could not connect to IMAP server %s on port %d") % server % port).str();

		mailimap_free(imap);
	}

	if (verbose && !data)
		encoded::cout << "    no data on server" << endl;

	if (rejects)
		encoded::cerr << "!!! at least one expression has been rejected, try -r to list rejections !!!" << endl;

	if (!msg.empty())
		THROW(msg);
}

#endif
