// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace boost::filesystem;

#include "../libpdrx/xception.h"
#include "../libpdrx/config.h"

#include "http.h"

#include <Poco/HMACEngine.h>
#include <Poco/SHA1Engine.h>
#include <Poco/Base64Encoder.h>

#ifdef _WIN32
#include <shellapi.h>
#endif

#ifdef DEBUG
#define DEBUG_REQUEST
#define DEBUG_HEADERS
#define DEBUG_RESPONSE
#define DEBUG_IGNORE_HTTP_STATUS_CODE
#endif

// note: we don't use the encoded streams here for debug output, we want to
// see everything exactly "as is"

//=== HttpClient ===========================================================
HttpClient::HttpClient (const string& host, const string& proxy)
	: m_host(host)
	, m_io_service()
	, m_endpoint()
	, m_socket(m_io_service)
	, m_ctx(m_io_service, asio::ssl::context::sslv23)
	, m_pSslStream(NULL)
{
	m_ctx.set_options(asio::ssl::context::default_workarounds | asio::ssl::context::no_sslv2);
	m_ctx.set_verify_mode(asio::ssl::context::verify_none);

	// let a resolver build an endpoint_iterator
	system::error_code error;
	asio::ip::tcp::resolver::iterator endpoint_iterator;
	if (!proxy.empty())
	{
		// split the proxy, the proxy has normally a port
		static const regex rx("([^:]+):(.+)?");
		smatch mr;
		if (regex_match(proxy, mr, rx))
		{
			const string proxy_server(mr[1]);
			const string proxy_port(mr[2]);

			asio::ip::tcp::resolver resolver(m_io_service);
			asio::ip::tcp::resolver::query query(
				proxy_server,
				(proxy_port.empty())
					? string("https")
					: proxy_port,
				(proxy_port.empty())
					? (asio::ip::tcp::resolver::query::passive | asio::ip::tcp::resolver::query::address_configured)
					: (asio::ip::tcp::resolver::query::passive | asio::ip::tcp::resolver::query::address_configured | asio::ip::tcp::resolver::query::numeric_service)
			);
			endpoint_iterator = resolver.resolve(query, error);
		}
	}
	else
	{
		asio::ip::tcp::resolver resolver(m_io_service);
		asio::ip::tcp::resolver::query query(host, "https");
		endpoint_iterator = resolver.resolve(query, error);
	}

	// get the real endpoint we use from now on
	if (error == 0)
	{
		// we try to open a temporary socket for this, if this is
		// successful we take this endpoint
		asio::ip::tcp::socket socket(m_io_service);
		asio::ip::tcp::resolver::iterator end;
		while (endpoint_iterator != end)
		{
			socket.connect(*endpoint_iterator, error);
			if (socket.is_open())
			{
				m_endpoint = *endpoint_iterator;
				socket.close();
				break;
			}
			else
				endpoint_iterator++;
		}
	}

	// if we have an endpoint open m_socket and do all the SSL stuff
	if (!m_endpoint.address().to_string().empty())
		Start();
}

HttpClient::~HttpClient ()
{
	if (Connected())
		Stop();
}

void HttpClient::Start ()
{
	// don't throw anything here, constructor!

	system::error_code error;
	m_socket.connect(m_endpoint, error);
	if (m_socket.is_open())
	{
		asio::socket_base::keep_alive option(true);
		m_socket.set_option(option);

		m_pSslStream = new asio::ssl::stream<asio::ip::tcp::socket&>(m_socket, m_ctx);
		m_pSslStream->handshake(asio::ssl::stream_base::client, error);
		if (error != 0)
		{
			delete m_pSslStream;
			m_pSslStream = NULL;
			m_socket.close();
		}
	}
}

void HttpClient::Stop ()
{
	delete m_pSslStream;
	m_pSslStream = NULL;
	m_socket.close();
}

bool HttpClient::Connected () const
{
	return m_socket.is_open() && m_pSslStream;
}

string HttpClient::Request (const string& method, const string& filename, const string& additionalHeaders /*= ""*/, const string& body /*= ""*/)
{
	// step 1: send the request
	{
		string r;

		// note: the filename here does neither include "https://"
		// nor the server name, we use a plain filename here with
		// absolute path and parameters
		r += method + " " + filename + " HTTP/1.1\r\n";

		r += "Host: " + m_host + "\r\n";
		r += "Accept: */*\r\n";
		r += "Connection: keep-alive\r\n";
		r += "Content-Type: application/x-www-form-urlencoded\r\n";
		r += "Content-Length: " + lexical_cast<string>(body.size()) + "\r\n";
		if (!additionalHeaders.empty())
			r += additionalHeaders;
		trim(r);
		r += "\r\n\r\n"; // yes, twice
		r += body;

#ifdef DEBUG_REQUEST
		cout << "----------" << endl;
		cout << r;
		cout << "----------" << endl;
#endif

		asio::write(*m_pSslStream, asio::buffer(r));
	}

	// step2: receive the response from server
	string output;
	{
		string line;
		system::error_code error;

		asio::streambuf r_streambuf;
		std::istream r_istream(&r_streambuf);

		asio::read_until(*m_pSslStream, r_streambuf, "\r\n", error);
		if (error !=0)
			THROW("error reading HTTP response");

		// check the first line
		string http_version;
		r_istream >> http_version;
		unsigned int status_code;
		r_istream >> status_code;
		string status_message;
		getline(r_istream, status_message);
#ifdef DEBUG_HEADERS
		cout << http_version << ' ' << status_code << status_message << endl;
#endif
		if (!r_istream || http_version.substr(0, 5) != "HTTP/")
			THROW("invalid response from HTTP server (no HTTP!)");
#ifndef DEBUG_IGNORE_HTTP_STATUS_CODE
		if (status_code != 200)
			THROW(format("invalid status in HTTP response: %d") % status_code);
#endif

		// examine the header lines
		asio::read_until(*m_pSslStream, r_streambuf, "\r\n\r\n", error);
		if (error !=0)
			THROW("error reading HTTP response headers");

		size_t content_length = 0;
		while (getline(r_istream, line) && line != "\r")
		{
#ifdef DEBUG_HEADERS
			cout << line << endl;
#endif
			if (line.find("Content-Length: ") == 0)
			{
				line.erase(0, 16);
				line.erase(line.size() - 1, 1); // \r
				content_length = lexical_cast<size_t>(line);
			}
		}
#ifdef DEBUG_HEADERS
		cout << endl;
#endif

		// read the body

		// note: if we got a content length we read exactly a block
		// of text with that length, this is probably much faster
		// than reading until the end of the stream as seen in the
		// else clause

		ostringstream oss;
		if (content_length > 0)
		{
			if (r_streambuf.size() > 0)
			{
				content_length -= r_streambuf.size();
				oss << &r_streambuf;
			}
			if (content_length > 0 && asio::read(*m_pSslStream, r_streambuf, asio::transfer_exactly(content_length), error))
				oss << &r_streambuf;
		}
		else
		{
			if (r_streambuf.size() > 0)
				oss << &r_streambuf;
			while (asio::read(*m_pSslStream, r_streambuf, asio::transfer_at_least(1), error))
				oss << &r_streambuf;
		}
		if (error != 0 && error != asio::error::eof && error != asio::error::shut_down)
			THROW("error reading HTTP response body");
		output = oss.str();
	}
#ifdef DEBUG_RESPONSE
	cout << output << endl;
	cout << "----------" << endl;
#endif

	return output;
}

//=== OAuthHttpClient ======================================================
namespace OAuth
{
	string Base64Encode (const Poco::DigestEngine::Digest& digest)
	{
		ostringstream oss;

		Poco::Base64Encoder enc(oss);
		foreach(unsigned char c, digest)
		{
			enc << c;
		}
		enc.close();

		return oss.str();
	}

	string UrlEncode (const string& s)
	{
		string result;
		foreach(char c, s)
		{
			if (('0' <= c && c <= '9') || ('A' <= c && c <= 'Z') || ('a' <= c && c <= 'z') || c == '~' || c == '-' || c == '_' || c == '.')
				result += c;
			else
				result += (format("%%%02X") % (int)c).str();
		}
		return result;
	}

	class Params
	{
		typedef map<string, string> Map;

		Map m_map;

		public:

		Params ();

		Params& operator () (const string& key, const string& value);

		string AsList () const;
		string AsBaseString (const string& method, const string& endpoint) const;
		string AsAuthorizationHeader () const;
	};

	Params::Params ()
		: m_map()
	{
	}

	Params& Params::operator () (const string& key, const string& value)
	{
		m_map.insert(Map::value_type(key, value));
		return *this;
	}

	string Params::AsList () const
	{
		string t;
		foreach(const Map::value_type& vt, m_map)
		{
			if (!t.empty())
				t += '&';
			t += vt.first + '=' + vt.second;
		}
		return t;
	}

	string Params::AsBaseString (const string& method, const string& endpoint) const
	{
		return method + '&' + UrlEncode(endpoint) + '&' + UrlEncode(AsList());
	}

	string Params::AsAuthorizationHeader () const
	{
		string result("OAuth ");
		string t;
		foreach(const Map::value_type& vt, m_map)
		{
			if (!t.empty())
				t += ", ";
			t += vt.first + "=\"" + vt.second + "\"";
		}
		return result + t;
	}

	string CreateNonce () // a nonce (just a random string)
	{
		string nonce;
		const char* t = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
		for (int i = 0; i < 16; i++)
			nonce += t[rand() % (sizeof(t) - 1)];
		return nonce;
	}

	string CreateTimestamp ()
	{
		time_t t;
		time(&t);
		return (format("%d") % t).str();
	}
} // namespace OAuth

OAuthHttpClient::OAuthHttpClient (const string& host, const string& proxy, const string& consumer_key, const string& consumer_secret, const Config& config, const string& option_key)
	: HttpClient(host, proxy)
	, m_consumer_key(consumer_key)
	, m_consumer_secret(consumer_secret)
	, m_private_keys_filename()
#ifndef _WIN32
	, m_browser_executable(config.GetStringOption("browser"))
#endif
	, m_access_token()
	, m_access_token_secret()
	, m_create_private_keys_file()
{
	path cfg(config.GetConfigFile());
	string fn;
	if (cfg.filename().string().find(".") == 0)
		fn = ".pdr_";
	fn += option_key + ".keys";
	m_private_keys_filename = (cfg.parent_path() / fn).string();

	// we try to open the private keys file here, if we don't have such
	// a file at the moment the keys remain empty, this causes an OAuth
	// authentication later to get these keys
	ifstream ifs(m_private_keys_filename.c_str(), ios::in);
	if (ifs.good())
	{
		string line;
		getline(ifs, line);
		trim(line);
		static const regex rx("([^,]+),(.+)");
		smatch mr;
		if (regex_match(line, mr, rx))
		{
			m_access_token = mr[1];
			m_access_token_secret = mr[2];
		}
	}
}

OAuthHttpClient::~OAuthHttpClient ()
{
	// if we created the private keys during this session save them here
	// for later use
	if (m_create_private_keys_file)
	{
		ofstream ofs(m_private_keys_filename.c_str(), ios::out);
		if (ofs.good())
			ofs << m_access_token << "," << m_access_token_secret << endl;
	}
}

	static void split (const string& s, vector<string>& values)
	{
		char_separator<char> sep("&");
		tokenizer<char_separator<char> > tok(s, sep);
		foreach(string token, tok)
		{
			trim(token);
			if (!token.empty())
				values.push_back(token);
		}
	}

string OAuthHttpClient::Request (const string& method, const string& filename, const string& additionalHeaders /*= ""*/, const string& body /*= ""*/)
{
	// note: the sense of this overwrite is to generate SIGNED requests
	// instead of normal, none signed requests

	// first we need to separate the query parameters behind a '?' from
	// the filename for signing, so we get a short_filename without any
	// parameters and query_params
	string short_filename;
	vector<string> query_params;
	{
		string::size_type pos = filename.find("?");
		if (pos != string::npos)
		{
			short_filename = string(filename, 0, pos);
			split(string(filename, pos + 1), query_params);
		}
		else
			short_filename = filename;
	}

	// now we generate a signed Authorization header, the signature is
	// computed over the HTTP-method, the short (!) filename and the
	// OAuth parameters combined with the separated query parameters
	string authorization_header;
	{
		OAuth::Params params;
		params
			("oauth_consumer_key",		m_consumer_key)
			("oauth_nonce",			OAuth::CreateNonce())
			("oauth_signature_method",	"HMAC-SHA1")
			("oauth_timestamp",		OAuth::CreateTimestamp())
			("oauth_token",			m_access_token)
			("oauth_version",		"1.0")
		;
		foreach(const string& query_param, query_params)
		{
			static const regex rx("([^=]+)=(.+)");
			smatch mr;
			if (regex_match(query_param, mr, rx))
			{
				params
					(mr[1], mr[2]);
			}
		}

		string oauth_signature;
		{
			const string& text = params.AsBaseString(method, string("https://") + m_host + short_filename);
			const string& key = m_consumer_secret + '&' + m_access_token_secret;

			Poco::HMACEngine<Poco::SHA1Engine> hmac(key);
			hmac.update(text);
			oauth_signature = OAuth::UrlEncode(OAuth::Base64Encode(hmac.digest()));
		}

		params
			("oauth_signature",		oauth_signature);

		authorization_header = "Authorization: " + params.AsAuthorizationHeader() + "\r\n";
	}

	// combine the Authorization header with user defined headers
	string headers(additionalHeaders);
	trim(headers);
	if (!headers.empty())
		headers += "\r\n";
	headers += authorization_header;

	// now run the normal request mechanism
	return HttpClient::Request(method, filename, headers, body);
}

void OAuthHttpClient::Authenticate (const string& basepath)
{
	// this method gets the request token and request token secret and
	// transforms them into an access token and access token secret
	// which will later be stored permanently in the private keys file

	if (m_access_token.empty() || m_access_token_secret.empty())
	{
		cout << "    no private keys found, interactive authentication needed" << endl;

		// step 1: get a request token
		string request_token, request_token_secret;

		{
			const string endpoint(basepath + "/oauth/request_token");

			OAuth::Params params;
			params
				("oauth_callback",		"oob")
				("oauth_consumer_key",		m_consumer_key)
				("oauth_nonce",			OAuth::CreateNonce())
				("oauth_signature_method",	"HMAC-SHA1")
				("oauth_timestamp",		OAuth::CreateTimestamp())
				("oauth_version",		"1.0")
			;

			string oauth_signature;
			{
				const string& text = params.AsBaseString("POST", string("https://") + m_host + endpoint);
				const string& key = m_consumer_secret + '&';

				Poco::HMACEngine<Poco::SHA1Engine> hmac(key);
				hmac.update(text);
				oauth_signature = OAuth::UrlEncode(OAuth::Base64Encode(hmac.digest()));
			}

			params
				("oauth_signature",		oauth_signature);

			string header("Authorization: ");
			header += params.AsAuthorizationHeader();

			const string& response = HttpClient::Request("POST", endpoint, header);

			const regex rx("oauth_token=([^&]+)&oauth_token_secret=([^&]+)&oauth_callback_confirmed=true");
			smatch mr;
			if (!regex_match(response, mr, rx))
				THROW("could not get request_token");
			request_token = mr[1];
			request_token_secret = mr[2];
		}

		// step 2: authorize the user (the interactive thing)

		// (because we don't know what the user does and how long it
		// takes we must close the socket, otherwise it will surely
		// timeout after some seconds)
		Stop();

		string oauth_verifier;
		{
			const string endpoint(string("https://") + m_host + basepath + "/oauth/authorize?oauth_token=" + request_token);
#ifdef _WIN32
			::ShellExecute(0, "open", endpoint.c_str(), "", "", SW_SHOW);
#else
			if (m_browser_executable.empty())
			{
				THROW(
					"You didn't specify a browser executable. You need a browser to do an "
					"interactive authentication on a web page. Use the pdr command line parameter "
					"--browser (or -? for general information)"
				);
			}

			std::system((m_browser_executable + " " + endpoint).c_str());
#endif
			cout << "        PIN (displayed in browser): ";
			getline(cin, oauth_verifier);
			trim(oauth_verifier);
		}

		// (now open the socket and do the SSL stuff again)
		Start();

		// step 3: change the request_token against an access token
		{
			const string endpoint(basepath + "/oauth/access_token");

			OAuth::Params params;
			params
				("oauth_consumer_key",		m_consumer_key)
				("oauth_nonce",			OAuth::CreateNonce())
				("oauth_signature_method",	"HMAC-SHA1")
				("oauth_timestamp",		OAuth::CreateTimestamp())
				("oauth_token",			request_token)
				("oauth_verifier",		oauth_verifier)
				("oauth_version",		"1.0")
			;

			string oauth_signature;
			{
				const string& text = params.AsBaseString("POST", string("https://") + m_host + endpoint);
				const string& key = m_consumer_secret + '&' + request_token_secret;

				Poco::HMACEngine<Poco::SHA1Engine> hmac(key);
				hmac.update(text);
				oauth_signature = OAuth::UrlEncode(OAuth::Base64Encode(hmac.digest()));
			}

			params
				("oauth_signature",		oauth_signature);

			string header("Authorization: ");
			header += params.AsAuthorizationHeader();

			const string& response = HttpClient::Request("POST", endpoint, header);

			const regex rx("oauth_token=([^&]+)&oauth_token_secret=([^&]+).*");
			smatch mr;
			if (!regex_match(response, mr, rx))
				THROW("could not get access_token");
			m_access_token = mr[1];
			m_access_token_secret = mr[2];

			cout	<< "        access_token: " << m_access_token << endl
				<< "        access_token_secret: " << m_access_token_secret << endl
				<< "    success" << endl;

			m_create_private_keys_file = true;
		}
	}
}

string OAuthHttpClient::GetUniqueFeedIdentifier () const
{
	return m_access_token + ',' + m_access_token_secret;
}
