// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

#include <Poco/Data/Common.h>
#include <Poco/Data/RecordSet.h>

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace Poco::Data;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/xception.h"
#include "../libpdrx/conversions.h"
#include "../libpdrx/config.h"
#include "db_impl.h"

//=== PocoDatabaseImpl (abstract base class) ===============================
PocoDatabaseImpl::PocoDatabaseImpl (const string& connect, bool verbose, const string& KEY, Connector* pConnector)
	: PocoDatabase(connect, verbose, KEY, pConnector)
{
}

PocoDatabaseImpl::~PocoDatabaseImpl ()
{
}

void PocoDatabaseImpl::Connect () throw (Xception)
{
	return PocoDatabase::Connect();
}

string PocoDatabaseImpl::GetNewTblname () const
{
	try
	{
		set<string> tblnames;
		foreach (const CollectionMetaInfo& cmi, m_collectionMetaInfos)
		{
			tblnames.insert(cmi.m_tblname);
		}
		set<string>::const_iterator I = tblnames.end();
		string s(*(--I));
		s.erase(0, 1); // C
		int i = lexical_cast<int>(s);
		return (format("C%d") % ++i).str();
	}
	CATCH_RETHROW("could not get new table name")
}

char PocoDatabaseImpl::GetCollectionTypeFromValue (const any& a) const
{
	if (a.type() == typeid(double))
		return 'n';

	if (a.type() == typeid(Ratio))
		return 'r';

	if (a.type() == typeid(string))
		return 't';

	THROW("invalid collection type");
}

void PocoDatabaseImpl::ListCollections () throw (Xception)
{
	if (m_verbose)
		encoded::cout << "listing collections" << endl;

	struct fill {
		const string& m_s;
		fill (const string& s)
			: m_s(s)
		{
		}
		string operator () (int idx)
		{
			static const size_t widths[] = {6, 9, 7, 7, 21, 21, 18}; // real column widths
			size_t width = widths[idx];
			if (m_s[0] == '-')
				return string(width - 2, '-') + "  ";
			else
			{
				wstring result(lexical_cast<wstring>(m_s));
				while (result.length() <= width)
					result += L' ';
				if (result.length() > width)
					result.erase(width - 1);
				result += L' '; // + 1 space as column delimiter
				return lexical_cast<string>(result);
			}
		}
	};

	try
	{
		encoded::cout << "  " << fill("name")(0) << fill("type")(1) << fill("table")(2) << fill("recs")(3) << fill("first")(4) << fill("last")(5) << fill("unit")(6) << "purpose" << endl;
		encoded::cout << "  " << fill("-")(0) << fill("-")(1) << fill("-")(2) << fill("-")(3) << fill("-")(4) << fill("-")(5) << fill("-")(6) << "-------" << endl;

		foreach (const CollectionMetaInfo& cmi, m_collectionMetaInfos)
		{
			encoded::cout << "  " << fill(cmi.m_collection)(0);

			switch (cmi.m_type)
			{
				case 'n':	encoded::cout << fill("numeric")(1); break;
				case 'r':	encoded::cout << fill("ratio")(1); break;
				case 't':	encoded::cout << fill("text")(1); break;
				default:	break;
			}

			encoded::cout << fill(cmi.m_tblname)(2);

			int count;
			string min, max;
			(*m_pSession) << "select count(*),min(t),max(t) from " << cmi.m_tblname << ";", into(count), into(min), into(max), now;
			encoded::cout << fill(lexical_cast<string>(count))(3) << fill(min)(4) << fill(max)(5);

			encoded::cout << fill(cmi.m_unit)(6);
			encoded::cout << cmi.m_purpose << endl;
		}
	}
	CATCH_RETHROW("could not list collections")
}

void PocoDatabaseImpl::AddCollection (const string& name) throw (Xception) // "name[, n|r|t[, unit[, purpose]]]"
{
	try
	{
		// split collection name, type, unit and purpose
		string n, t, p, u;
		{
			regex rx("\\s*([^,\\s]+)(?:\\s*,\\s*([nrt])\\s*(?:,([^,]+)(?:,(.+))?)?)?");
			smatch mr;
			if (!regex_match(name, mr, rx))
				THROW(format("illegal collection specification: %s") % name);
			n = mr[1];
			t = mr[2];
			if (t.empty())
				t = "n"; // numeric
			u = mr[3];
			trim(u);
			p = mr[4];
			trim(p);
		}

		// check for existing collection
		try
		{
			const CollectionMetaInfo& cmi = GetCollectionMetaInfo(n);
			if (cmi.m_type == t[0])
			{
				if (cmi.m_purpose != p)
				{
					UpdatePurpose(n, p);
					const_cast<CollectionMetaInfo&>(cmi).m_purpose = p;
				}
				return;
			}
			else
				THROW(format("collection %s already exists but differs in type: %c <-> %s") % n % cmi.m_type % t);
		}
		catch (...)
		{
			// ok, collection doesn't exist
		}

		if (m_verbose)
		{
			encoded::cout << "adding collection " << n;
			if (!p.empty())
				encoded::cout << " (" << p << ')';
			encoded::cout << endl;
		}

		// build new table name
		const string& tblname = GetNewTblname();

		// do the database specific SQL job
		CreateCollectionInSchema(n, t[0], tblname, p, u);

		// create a CollectionMetaInfo
		CollectionMetaInfo cmi = {n, t[0], tblname, p, u};
		m_collectionMetaInfos.insert(cmi);
	}
	CATCH_RETHROW(format("could not create collection: %s") % name)
}

void PocoDatabaseImpl::DeleteCollection (const string& name) throw (Xception)
{
	if (m_verbose)
		encoded::cout << "deleting collection" << endl;

	try
	{
		// check for built-in collection
		if (name == "*" || name == "#")
			THROW(format("cannot delete built-in collection: %s") % name);

		// check for unknown collection
		const CollectionMetaInfo& cmi = GetCollectionMetaInfo(name);

		// do the database specific SQL job
		DropCollectionFromSchema(cmi.m_tblname);

		// forget the CollectionMetaInfo
		m_collectionMetaInfos.erase(cmi);
	}
	CATCH_RETHROW(format("could not delete collection: %s") % name)
}

void PocoDatabaseImpl::DeleteAllCollections () throw (Xception)
{
	if (m_verbose)
		encoded::cout << "deleting all collections" << endl;

	try
	{
		DBTransactor transactor(m_pSession, m_transactionCounter);

		CollectionMetaInfos to_delete;
		foreach (const CollectionMetaInfo& cmi, m_collectionMetaInfos)
		{
			if (cmi.m_collection != "*" && cmi.m_collection != "#")
				to_delete.insert(cmi);
		}
		foreach (const CollectionMetaInfo& cmi, to_delete)
		{
			(*m_pSession) << "drop table " << cmi.m_tblname << ";", now;
			m_collectionMetaInfos.erase(cmi);
		}

		(*m_pSession) << "delete from TCollections where name not in ('*','#');", now;
		(*m_pSession) << "update TCollections set purpose = NULL;", now;
		(*m_pSession) << "delete from C0;", now;
		(*m_pSession) << "delete from C1;", now;

		transactor.Commit();
	}
	CATCH_RETHROW("could not delete collections")
}

void PocoDatabaseImpl::ListRejections () throw (Xception)
{
	if (m_verbose)
		encoded::cout << "listing rejections" << endl;

	struct fill {
		const string& m_s;
		fill (const string& s)
			: m_s(s)
		{
		}
		string operator () (int idx)
		{
			static const size_t widths[] = {20}; // real column widths
			size_t width = widths[idx];
			wstring result(lexical_cast<wstring>(m_s));
			while (result.length() < width)
				result += L' ';
			if (result.length() > width)
				result.erase(width - 1);
			result += L' '; // + 1 space as column delimiter
			return lexical_cast<string>(result);
		}
	};

	try
	{
		Statement select(*m_pSession);
		select << "select t,expr from TRejected order by t;";
		select.execute();

		RecordSet rs(select);
		if (rs.moveFirst())
		{
			encoded::cout << "  " << fill("timestamp")(0) << "expression" << endl;
			do {
				encoded::cout << "  " << fill(rs.value(0).convert<string>())(0) << rs.value(1).convert<string>() << endl;
			} while (rs.moveNext());
		}
	}
	CATCH_RETHROW("could not list rejections")
}

void PocoDatabaseImpl::AddRejected (const ptime& timestamp, const string& expr) throw (Xception)
{
	try
	{
		const string& t = lexical_cast<string>(timestamp);
		(*m_pSession) << "insert into TRejected values (null,'" << t << "','" << expr << "');", now;
	}
	CATCH_RETHROW("could not insert rejection")
}

void PocoDatabaseImpl::DeleteRejected (const ptime& timestamp) throw (Xception)
{
	try
	{
		const string& t = lexical_cast<string>(timestamp);
		(*m_pSession) << "delete from TRejected where t='" << t << "';", now;
	}
	CATCH_RETHROW("could not delete rejection")
}

void PocoDatabaseImpl::DeleteAllRejections () throw (Xception)
{
	if (m_verbose)
		encoded::cout << "deleting all rejections" << endl;

	try
	{
		(*m_pSession) << "delete from TRejected;", now;
	}
	CATCH_RETHROW("could not delete rejections")
}

char PocoDatabaseImpl::GetCollectionType (const string& name) const throw (Xception)
{
	return GetCollectionMetaInfo(name).m_type;
}

string PocoDatabaseImpl::GetCollectionPurpose (const string& name) const throw (Xception)
{
	return GetCollectionMetaInfo(name).m_purpose;
}

string PocoDatabaseImpl::GetCollectionUnit (const string& name) const throw (Xception)
{
	return GetCollectionMetaInfo(name).m_unit;
}

void PocoDatabaseImpl::GetCollections (Collections& collections) const throw (Xception)
{
	foreach (const CollectionMetaInfo& cmi, m_collectionMetaInfos)
	{
		collections.insert(cmi.m_collection);
	}
}

void PocoDatabaseImpl::GetCollectionItems (const string& name, CollectionItems& items) const throw (Xception)
{
	try
	{
		const CollectionMetaInfo& cmi = GetCollectionMetaInfo(name);

		Statement select(*m_pSession);
		select << "select ";
		switch (cmi.m_type)
		{
			case 'r':	select << "t,n,d"; break;
			default:	select << "t,v"; break;
		}
		select << " from " << cmi.m_tblname << ";";
		select.execute();

		RecordSet rs(select);
		if (rs.moveFirst())
		{
			do {
				ptime t = lexical_cast<ptime>(rs.value(0).convert<string>());
				any a;
				switch (cmi.m_type)
				{
					case 'n':	a = rs.value(1).convert<double>(); break;
					case 'r':	a = Ratio(rs.value(1).convert<double>(), rs.value(2).convert<double>()); break;
					case 't':	a = rs.value(1).convert<string>(); break;
				}
				items.insert(Database::CollectionItems::value_type(t, a));
			} while (rs.moveNext());
		}
	}
	CATCH_RETHROW("could not select collection data")
}

void PocoDatabaseImpl::AddCollectionsItems (const CollectionsItems& items) throw (Xception)
{
	try
	{
		DBTransactor transactor(m_pSession, m_transactionCounter);

		foreach (const CollectionsItems::value_type& vt, items)
		{
			InsertOrUpdateCollectionItem(GetCollectionMetaInfo(vt.first), vt.second);
		}

		transactor.Commit();
	}
	CATCH_RETHROW("could not add collection data")
}

void PocoDatabaseImpl::DeleteCollectionsItems (const ptime& timestamp, const Collections& collections)
{
	try
	{
		DBTransactor transactor(m_pSession, m_transactionCounter);

		const string& t = lexical_cast<string>(timestamp);
		foreach (const CollectionMetaInfo& cmi, m_collectionMetaInfos)
		{
			if (collections.empty() || collections.find(cmi.m_collection) != collections.end())
				(*m_pSession) << "delete from " << cmi.m_tblname << " where t='" << t << "';", now;
		}

		transactor.Commit();
	}
	CATCH_RETHROW("could not delete collection data")
}

ptime PocoDatabaseImpl::GetLastRssUpdate (const string& feed) const throw (Xception)
{
	try
	{
		string t;
		(*m_pSession) << "select t from TLastRssUpdates where f='" << feed << "';", into(t), now;

		return (t.empty()) ? ptime() : lexical_cast<ptime>(t);
	}
	CATCH_RETHROW("could not get last RSS update")
}

void PocoDatabaseImpl::SetLastRssUpdate (const string& feed, const ptime& timestamp) const throw (Xception)
{
	try
	{
		string t;
		(*m_pSession) << "select t from TLastRssUpdates where f='" << feed << "';", into(t), now;

		if (t.empty())
			(*m_pSession) << "insert into TLastRssUpdates (f,t) values ('" << feed << "','" << lexical_cast<string>(timestamp) << "');", now;
		else
			(*m_pSession) << "update TLastRssUpdates set t='" << lexical_cast<string>(timestamp) << "' where f='" << feed << "';", now;
	}
	CATCH_RETHROW("could not set last RSS update")
}

ptime PocoDatabaseImpl::GetYoungestCollectionItemAtAll () const throw (Xception)
{
	try
	{
		ptime timestamp;
		foreach (const CollectionMetaInfo& cmi, m_collectionMetaInfos)
		{
			string max;
			(*m_pSession) << "select max(t) from " << cmi.m_tblname << ";", into(max), now;

			if (!max.empty())
			{
				ptime t = lexical_cast<ptime>(max);
				if (timestamp == not_a_date_time || timestamp < t)
					timestamp = t;
			}
		}
		return timestamp;
	}
	CATCH_RETHROW("could not select collection data")
}

void PocoDatabaseImpl::GenerateExpressions (Expressions& expressions, const ptime& timestamp, const string& rejected) const throw (Xception)
{
	try
	{
		typedef multimap<ptime, pair<any, string> > Data;

		// collect data from the collections
		Data data;
		{
			foreach (const CollectionMetaInfo& cmi, m_collectionMetaInfos)
			{
				// build the SQL string
				string sql("select t,");
				switch (cmi.m_type)
				{
					case 'r':	sql += "n,d"; break;
					default:	sql += "v"; break;
				}
				sql += " from " + cmi.m_tblname;
				if (timestamp != not_a_date_time)
					sql += " where t='" + lexical_cast<string>(timestamp) + "'";
				sql += " order by 1;";

				// execute
				Statement select(*m_pSession);
				select << sql;
				select.execute();

				// fetch data
				RecordSet rs(select);
				if (rs.moveFirst())
				{
					do {
						pair<any, string> value;
						switch (cmi.m_type)
						{
							case 'n':	value.first = rs.value(1).convert<double>(); break;
							case 'r':	value.first = Ratio(rs.value(1).convert<double>(), rs.value(2).convert<double>()); break;
							default:	value.first = rs.value(1).convert<string>(); break;
						}
						value.second = cmi.m_collection;
						data.insert(Data::value_type(lexical_cast<ptime>(rs.value(0).convert<string>()), value));
					} while (rs.moveNext());
				}
			}
		}

		// build expressions from collection data
		Data::const_iterator I = data.begin();
		while (I != data.end())
		{
			const ptime timestamp = (*I).first;

			string expr;
			string comment;
			Data::const_iterator J = data.upper_bound(timestamp);
			while (I != J)
			{
				const Data::value_type& vt = *I++;
				const any& a = vt.second.first;
				switch (GetCollectionTypeFromValue(a))
				{
					case 'n':
					{
						expr += (format(" %g%s") % any_cast<double>(a) % vt.second.second).str();
						break;
					}
					case 'r':
					{
						expr += (format(" %g%s%g") % any_cast<Ratio>(a).m_numerator % vt.second.second % any_cast<Ratio>(a).m_denominator).str();
						break;
					}
					case 't':
					{
						if (vt.second.second != "#")
							expr += (format(" \"%s\"%s") % any_cast<string>(a) % vt.second.second).str();
						else
							comment = any_cast<string>(a);
						break;
					}
					default:
						break;
				}
			}
			if (expr[0] == ' ')
				expr.erase(0, 1);
			if (!comment.empty())
				expr += " ; " + comment;

			expressions.insert(Database::Expressions::value_type(timestamp, expr));
		}

		// collect data also from rejections
		{
			// build the SQL string
			string sql("select t,expr from TRejected");
			if (timestamp != not_a_date_time)
				sql += " where t='" + lexical_cast<string>(timestamp) + "'";
			sql += " order by 1;";

			// execute
			Statement select(*m_pSession);
			select << sql;
			select.execute();

			// fetch data
			RecordSet rs(select);
			if (rs.moveFirst())
			{
				do {
					const ptime& timestamp = lexical_cast<ptime>(rs.value(0).convert<string>());
					const string& expr = rejected + rs.value(1).convert<string>();
					expressions.insert(Database::Expressions::value_type(timestamp, expr));
				} while (rs.moveNext());
			}
		}
	}
	CATCH_RETHROW("could not generate expressions from collection data")
}
