// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "datatypes.h"
#include "xception.h"
#include "conversions.h"
#include "config.h"

#include <Poco/TextEncoding.h>
#include <Poco/UnicodeConverter.h>
#include <Poco/StreamConverter.h>

namespace boost {

	// ptime, date, time_duration <-> string
	template<> boost::posix_time::ptime lexical_cast (const std::string& arg)
	{
		std::string s(arg);
		s[10] = 'T';
		s.erase(4, 1); // '-'
		s.erase(6, 1); // '-'
		s.erase(11, 1); // ':'
		s.erase(13, 1); // ':'
		return boost::posix_time::from_iso_string(s);
	}

	boost::posix_time::ptime lexical_cast_literal_month (const std::string& arg)
	{
		typedef map<string, string> Months;
		static Months months;
		if (months.empty())
		{
			months.insert(Months::value_type("Jan", "01"));
			months.insert(Months::value_type("Feb", "02"));
			months.insert(Months::value_type("Mar", "03"));
			months.insert(Months::value_type("Apr", "04"));
			months.insert(Months::value_type("May", "05"));
			months.insert(Months::value_type("Jun", "06"));
			months.insert(Months::value_type("Jul", "07"));
			months.insert(Months::value_type("Aug", "08"));
			months.insert(Months::value_type("Sep", "09"));
			months.insert(Months::value_type("Oct", "10"));
			months.insert(Months::value_type("Nov", "11"));
			months.insert(Months::value_type("Dec", "12"));
		}

		string t(arg);
		t.replace(5, 3, months[string(t, 5, 3)]);
		return lexical_cast<boost::posix_time::ptime>(t);
	}

	boost::posix_time::ptime lexical_cast_mime_date (const std::string& arg) throw (Xception)
	{
		// Wed, 16 Nov 2011 12:57:23 +0000
		static const regex rx("^(?:.+, )?([0-9]+) ([A-Za-z]+) ([0-9]+) ([0-9]+:[0-9]+:[0-9]+) ([+-][0-9]{4})$");
		//                                  1          2         3                4                  5

		smatch mr;
		if (regex_match(arg, mr, rx))
		{
			string s(mr[1]);
			if (s.length() == 1)
				s.insert(0, "0");

			ptime timestamp = lexical_cast_literal_month((format("%s-%s-%s %s") % mr[3] % mr[2] % s % mr[4]).str());

			// if needed, convert to UTC
			int dt_zone = lexical_cast<int>(mr[5]);
			if (dt_zone != 0)
			{
				int diff_hours = dt_zone / 100;
				int diff_min = dt_zone - diff_hours * 100;
				timestamp -= time_duration(diff_hours, diff_min, 0, 0);
			}

			// now localize
			typedef boost::date_time::c_local_adjustor<ptime> local_adj;
			return local_adj::utc_to_local(timestamp);
		}
		else
			THROW(format("invalid date format: %s") % arg);
	}

	template<> boost::gregorian::date lexical_cast (const std::string& arg)
	{
		boost::gregorian::date d = boost::gregorian::from_string(arg);
		return d;
	}

	template<> std::string lexical_cast (const boost::posix_time::ptime& arg)
	{
		std::string s(boost::posix_time::to_iso_extended_string(arg));
		if (s[0] != 'n') // "not_a_date_time"
		{
			s[10] = ' '; // replace 'T'
			s.erase(19); // cut milliseconds
		}
		return s;
	}

	template<> std::string lexical_cast (const boost::gregorian::date& arg)
	{
		std::string s(boost::gregorian::to_iso_extended_string(arg));
		return s;
	}

	template<> std::string lexical_cast (const boost::posix_time::time_duration& arg)
	{
		std::string s(boost::posix_time::to_simple_string(arg));
		std::string::size_type pos = s.find('.');
		if (pos != std::string::npos)
			s.erase(pos); // cut milliseconds
		return s;
	}

	// Ratio <-> string
	template<> Ratio lexical_cast (const std::string& arg) throw (Xception)
	{
		string::size_type pos = arg.find('/');
		if (pos != string::npos)
			return Ratio(lexical_cast<double>(string(arg, 0, pos)), lexical_cast<double>(string(arg, pos + 1)));
		else
			THROW(format("not a Ratio value: %s") % arg);
	}

	template<> string lexical_cast (const Ratio& arg) throw (Xception)
	{
		return (format("%g/%g") % arg.m_numerator % arg.m_denominator).str();
	}

	// wstring <-> string
	template<> std::wstring lexical_cast (const std::string& input)
	{
		std::wstring output;
		Poco::UnicodeConverter::toUTF16(input, output);
		return output;
	}

	template<> std::string lexical_cast (const std::wstring& input)
	{
		std::string output;
		Poco::UnicodeConverter::toUTF8(input, output);
		return output;
	}
}

namespace encoded {

	// this code is inspired from the Poco::StreamBufConverter class,
	// the reason why we implement own classes here is that we don't
	// know the native encoding to be used in the moment when the
	// cout and cin objects are created, we get the native encoding
	// later by reading the config file, but Poco::StreamBufConverter
	// needs hard const references not changable later

	class ostreambuf: public Poco::UnbufferedStreamBuf
	{
		std::ostream*	m_pOstr;
		int		m_defaultChar;
		unsigned char	m_buffer[Poco::TextEncoding::MAX_SEQUENCE_LENGTH];
		int		m_sequenceLength;
		int		m_pos;

		public:

		ostreambuf (std::ostream& ostr, int defaultChar = '?');

		protected:

		virtual int writeToDevice (char c);
	};

	ostreambuf::ostreambuf (std::ostream& ostr, int defaultChar)
		: Poco::UnbufferedStreamBuf()
		, m_pOstr(&ostr)
		, m_defaultChar(defaultChar)
		, m_sequenceLength(0)
		, m_pos(0)
	{
	}

	int ostreambuf::writeToDevice (char c)
	{
		const Poco::TextEncoding& inEncoding = InternalEncoding();
		const Poco::TextEncoding& outEncoding = NativeEncoding();

		m_buffer[m_pos++] = (unsigned char)c;
		if (m_sequenceLength == 0 || m_sequenceLength == m_pos)
		{
			int n = inEncoding.queryConvert(m_buffer, m_pos);
			if (n >= -1)
			{
				int uc = n;
				if (n == -1)
					return -1;

				int n = outEncoding.convert(uc, m_buffer, sizeof(m_buffer));
				if (n == 0)
					n = outEncoding.convert(m_defaultChar, m_buffer, sizeof(m_buffer));

				m_pOstr->write((char*)m_buffer, n);
				m_pOstr->flush();
				m_sequenceLength = 0;
				m_pos = 0;
			}
			else
				m_sequenceLength = -n;
		}

		return charToInt(c);
	}

	class istreambuf: public Poco::UnbufferedStreamBuf
	{
		std::istream*	m_pIstr;
		int		m_defaultChar;
		unsigned char	m_buffer[Poco::TextEncoding::MAX_SEQUENCE_LENGTH];
		int		m_sequenceLength;
		int		m_pos;

		public:

		istreambuf (std::istream& istr, int defaultChar = '?');

		protected:

		virtual int readFromDevice ();
	};

	istreambuf::istreambuf (std::istream& istr, int defaultChar)
		: Poco::UnbufferedStreamBuf()
		, m_pIstr(&istr)
		, m_defaultChar(defaultChar)
		, m_sequenceLength(0)
		, m_pos(0)
	{
	}

	int istreambuf::readFromDevice ()
	{
		if (m_pos < m_sequenceLength)
			return m_buffer[m_pos++];

		m_pos = 0;
		m_sequenceLength = 0;
		int c = m_pIstr->get();
		if (c == -1)
			return -1;

		const Poco::TextEncoding& inEncoding = NativeEncoding();
		const Poco::TextEncoding& outEncoding = InternalEncoding();

		int uc;
		m_buffer [0] = (unsigned char)c;
		int n = inEncoding.queryConvert(m_buffer, 1);
		int read = 1;

		while (n < -1)
		{
			m_pIstr->read((char*)m_buffer + read, -n - read);
			read = -n;
			n = inEncoding.queryConvert(m_buffer, -n);
		}

		uc = (n <= -1) ? m_defaultChar : n;

		m_sequenceLength = outEncoding.convert(uc, m_buffer, sizeof(m_buffer));
		if (m_sequenceLength == 0)
			m_sequenceLength = outEncoding.convert(m_defaultChar, m_buffer, sizeof(m_buffer));
		if (m_sequenceLength == 0)
			return -1;
		else
			return m_buffer[m_pos++];
	}

	ostreambuf oscb(std::cout);
	istreambuf iscb(std::cin);

	std::ostream cout(&oscb);
	std::ostream cerr(&oscb);
	std::istream cin(&iscb);
}
