// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "common.h"

using namespace std;

#include "ascii_with_codepage.h"
#include "Poco/String.h"

//=== ASCIICodepage437Encoding =============================================
const char* ASCIICodepage437Encoding::_names[] =
{
	"ASCII/437",
	"ASCII/CP437",
	"ASCII/Codepage 437",
	NULL
};

const Poco::TextEncoding::CharacterMap ASCIICodepage437Encoding::_charMap =
{
			// this is nothing but 7bit ASCII
	/* 00 */	0x0000, 0x0001, 0x0002, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007, 0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
	/* 10 */	0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017, 0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
	/* 20 */	0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027, 0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
	/* 30 */	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
	/* 40 */	0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
	/* 50 */	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
	/* 60 */	0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
	/* 70 */	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x007f,

			// these are the characters from codepage 437 that
			// look like a letter, we do not encode all the
			// pseudo graphic characters here
	/* 80 */	0x00C7,	0x00FC,	0x00E9,	0x00E2,	0x00E4,	0x00E0,	0x00E5,	0x00E7,	0x00EA,	0x00EB,	0x00E8,	0x00EF,	0x00EE,	0x00EC,	0x00C4,	0x00C5,
 	/* 90 */	0x00C9,	0x00E6,	0x00C6,	0x00F4,	0x00F6,	0x00F2,	0x00FB,	0x00F9,	0x00FF,	0x00D6,	0x00DC,	0x00A2,	0x00A3,	0x00A5,	0x20A7,	0x0192,
 	/* A0 */	0x00E1,	0x00ED,	0x00F3,	0x00FA,	0x00F1,	0x00D1,	0x00AA,	0x00BA,	-1,	-1,	-1,	-1,	-1,	0x00A1,	-1,	-1,
 	/* B0 */	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
 	/* C0 */	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
 	/* D0 */	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
 	/* E0 */	0x03B1,	0x00DF,	0x0393,	0x03C0,	0x03A3,	0x03C3,	0x00B5,	0x03C4,	0x03A6,	0x0398,	0x03A9,	0x03B4,	0x221E,	0x03C6,	0x03B5,	0x2229,
 	/* F0 */	0x2261,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
};

ASCIICodepage437Encoding::ASCIICodepage437Encoding ()
	: Poco::TextEncoding()
{
}

const char* ASCIICodepage437Encoding::canonicalName () const
{
	return _names[0];
}

bool ASCIICodepage437Encoding::isA (const std::string& encodingName) const
{
	for (const char** name = _names; *name; name++)
	{
		if (Poco::icompare(encodingName, *name) == 0)
			return true;
	}
	return false;
}

const Poco::TextEncoding::CharacterMap& ASCIICodepage437Encoding::characterMap () const
{
	return _charMap;
}

int ASCIICodepage437Encoding::convert (const unsigned char* bytes) const
{
	return _charMap[*bytes];
}

int ASCIICodepage437Encoding::convert (int ch, unsigned char* bytes, int length) const
{
	if (ch >= 0 && ch <= 127)
	{
		if (bytes && length >= 1)
			*bytes = (unsigned char) ch;
		return 1;
	}
	else switch (ch)
	{
		case 0x00C7:	if (bytes && length >= 1) *bytes = 0x80; return 1;
		case 0x00FC:	if (bytes && length >= 1) *bytes = 0x81; return 1;
		case 0x00E9:	if (bytes && length >= 1) *bytes = 0x82; return 1;
		case 0x00E2:	if (bytes && length >= 1) *bytes = 0x83; return 1;
		case 0x00E4:	if (bytes && length >= 1) *bytes = 0x84; return 1;
		case 0x00E0:	if (bytes && length >= 1) *bytes = 0x85; return 1;
		case 0x00E5:	if (bytes && length >= 1) *bytes = 0x86; return 1;
		case 0x00E7:	if (bytes && length >= 1) *bytes = 0x87; return 1;
		case 0x00EA:	if (bytes && length >= 1) *bytes = 0x88; return 1;
		case 0x00EB:	if (bytes && length >= 1) *bytes = 0x89; return 1;
		case 0x00E8:	if (bytes && length >= 1) *bytes = 0x8A; return 1;
		case 0x00EF:	if (bytes && length >= 1) *bytes = 0x8B; return 1;
		case 0x00EE:	if (bytes && length >= 1) *bytes = 0x8C; return 1;
		case 0x00EC:	if (bytes && length >= 1) *bytes = 0x8D; return 1;
		case 0x00C4:	if (bytes && length >= 1) *bytes = 0x8E; return 1;
		case 0x00C5:	if (bytes && length >= 1) *bytes = 0x8F; return 1;
		case 0x00C9:	if (bytes && length >= 1) *bytes = 0x90; return 1;
		case 0x00E6:	if (bytes && length >= 1) *bytes = 0x91; return 1;
		case 0x00C6:	if (bytes && length >= 1) *bytes = 0x92; return 1;
		case 0x00F4:	if (bytes && length >= 1) *bytes = 0x93; return 1;
		case 0x00F6:	if (bytes && length >= 1) *bytes = 0x94; return 1;
		case 0x00F2:	if (bytes && length >= 1) *bytes = 0x95; return 1;
		case 0x00FB:	if (bytes && length >= 1) *bytes = 0x96; return 1;
		case 0x00F9:	if (bytes && length >= 1) *bytes = 0x97; return 1;
		case 0x00FF:	if (bytes && length >= 1) *bytes = 0x98; return 1;
		case 0x00D6:	if (bytes && length >= 1) *bytes = 0x99; return 1;
		case 0x00DC:	if (bytes && length >= 1) *bytes = 0x9A; return 1;
		case 0x00A2:	if (bytes && length >= 1) *bytes = 0x9B; return 1;
		case 0x00A3:	if (bytes && length >= 1) *bytes = 0x9C; return 1;
		case 0x00A5:	if (bytes && length >= 1) *bytes = 0x9D; return 1;
		case 0x20A7:	if (bytes && length >= 1) *bytes = 0x9E; return 1;
		case 0x0192:	if (bytes && length >= 1) *bytes = 0x9F; return 1;
		case 0x00E1:	if (bytes && length >= 1) *bytes = 0xA0; return 1;
		case 0x00ED:	if (bytes && length >= 1) *bytes = 0xA1; return 1;
		case 0x00F3:	if (bytes && length >= 1) *bytes = 0xA2; return 1;
		case 0x00FA:	if (bytes && length >= 1) *bytes = 0xA3; return 1;
		case 0x00F1:	if (bytes && length >= 1) *bytes = 0xA4; return 1;
		case 0x00D1:	if (bytes && length >= 1) *bytes = 0xA5; return 1;
		case 0x00AA:	if (bytes && length >= 1) *bytes = 0xA6; return 1;
		case 0x00BA:	if (bytes && length >= 1) *bytes = 0xA7; return 1;

		case 0x00A1:	if (bytes && length >= 1) *bytes = 0xAD; return 1;

		case 0x03B1:	if (bytes && length >= 1) *bytes = 0xE0; return 1;
		case 0x00DF:	if (bytes && length >= 1) *bytes = 0xE1; return 1;
		case 0x0393:	if (bytes && length >= 1) *bytes = 0xE2; return 1;
		case 0x03C0:	if (bytes && length >= 1) *bytes = 0xE3; return 1;
		case 0x03A3:	if (bytes && length >= 1) *bytes = 0xE4; return 1;
		case 0x03C3:	if (bytes && length >= 1) *bytes = 0xE5; return 1;
		case 0x00B5:	if (bytes && length >= 1) *bytes = 0xE6; return 1;
		case 0x03C4:	if (bytes && length >= 1) *bytes = 0xE7; return 1;
		case 0x03A6:	if (bytes && length >= 1) *bytes = 0xE8; return 1;
		case 0x0398:	if (bytes && length >= 1) *bytes = 0xE9; return 1;
		case 0x03A9:	if (bytes && length >= 1) *bytes = 0xEA; return 1;
		case 0x03B4:	if (bytes && length >= 1) *bytes = 0xEB; return 1;
		case 0x221E:	if (bytes && length >= 1) *bytes = 0xEC; return 1;
		case 0x03C6:	if (bytes && length >= 1) *bytes = 0xED; return 1;
		case 0x03B5:	if (bytes && length >= 1) *bytes = 0xEE; return 1;
		case 0x2229:	if (bytes && length >= 1) *bytes = 0xEF; return 1;
		case 0x2261:	if (bytes && length >= 1) *bytes = 0xF0; return 1;

		default:	return 0;
	}
}

int ASCIICodepage437Encoding::queryConvert (const unsigned char* bytes, int length) const
{
	if (1 <= length)
		return _charMap [*bytes];
	else
		return -1;
}

int ASCIICodepage437Encoding::sequenceLength (const unsigned char* bytes, int length) const
{
	return 1;
}

//=== ASCIICodepage850Encoding =============================================
const char* ASCIICodepage850Encoding::_names[] =
{
	"ASCII/850",
	"ASCII/CP850",
	"ASCII/Codepage 850",
	NULL
};

const Poco::TextEncoding::CharacterMap ASCIICodepage850Encoding::_charMap =
{
			// this is nothing but 7bit ASCII
	/* 00 */	0x0000, 0x0001, 0x0002, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007, 0x0008, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
	/* 10 */	0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017, 0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f,
	/* 20 */	0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027, 0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
	/* 30 */	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
	/* 40 */	0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
	/* 50 */	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
	/* 60 */	0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
	/* 70 */	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x007f,

			// these are the characters from codepage 850 that
			// look like a letter, we do not encode all the
			// pseudo graphic characters here
	/* 80 */	0x00C7,	0x00FC,	0x00E9,	0x00E2,	0x00E4,	0x00E0,	0x00E5,	0x00E7,	0x00EA,	0x00EB,	0x00E8,	0x00EF,	0x00EE,	0x00EC,	0x00C4,	0x00C5,
 	/* 90 */	0x00C9,	0x00E6,	0x00C6,	0x00F4,	0x00F6,	0x00F2,	0x00FB,	0x00F9,	0x00FF,	0x00D6,	0x00DC,	0x00F8,	0x00A3,	0x00D8,	-1,	0x0192,
 	/* A0 */	0x00E1,	0x00ED,	0x00F3,	0x00FA,	0x00F1,	0x00D1,	0x00AA,	0x00BA,	-1,	-1,	-1,	-1,	-1,	0x00A1,	-1,	-1,
 	/* B0 */	-1,	-1,	-1,	-1,	-1,	0x00C1,	0x00C2,	0x00C0,	-1,	-1,	-1,	-1,	-1,	0x00A2,	0x00A5,	-1,
 	/* C0 */	-1,	-1,	-1,	-1,	-1,	-1,	0x00E3,	0x00C3,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
 	/* D0 */	0x00F0,	0x00D0,	0x00CA,	0x00CB,	0x00C8,	0x0131,	0x00CD,	0x00CE,	0x0407,	-1,	-1,	-1,	-1,	-1,	0x00CC,	-1,
 	/* E0 */	0x00D3,	0x00DF,	0x00D4,	0x00D2,	0x00F5,	0x00D5,	0x00B5,	0x00DE,	0x00FE,	0x00DA,	0x00DB,	0x00D9,	0x00FD,	0x00DD,	-1,	-1,
 	/* F0 */	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,	-1,
};

ASCIICodepage850Encoding::ASCIICodepage850Encoding ()
	: Poco::TextEncoding()
{
}

const char* ASCIICodepage850Encoding::canonicalName () const
{
	return _names[0];
}

bool ASCIICodepage850Encoding::isA (const std::string& encodingName) const
{
	for (const char** name = _names; *name; name++)
	{
		if (Poco::icompare(encodingName, *name) == 0)
			return true;
	}
	return false;
}

const Poco::TextEncoding::CharacterMap& ASCIICodepage850Encoding::characterMap () const
{
	return _charMap;
}

int ASCIICodepage850Encoding::convert (const unsigned char* bytes) const
{
	return _charMap[*bytes];
}

int ASCIICodepage850Encoding::convert (int ch, unsigned char* bytes, int length) const
{
	if (ch >= 0 && ch <= 127)
	{
		if (bytes && length >= 1)
			*bytes = (unsigned char) ch;
		return 1;
	}
	else switch (ch)
	{
		case 0x00C7:	if (bytes && length >= 1) *bytes = 0x80; return 1;
		case 0x00FC:	if (bytes && length >= 1) *bytes = 0x81; return 1;
		case 0x00E9:	if (bytes && length >= 1) *bytes = 0x82; return 1;
		case 0x00E2:	if (bytes && length >= 1) *bytes = 0x83; return 1;
		case 0x00E4:	if (bytes && length >= 1) *bytes = 0x84; return 1;
		case 0x00E0:	if (bytes && length >= 1) *bytes = 0x85; return 1;
		case 0x00E5:	if (bytes && length >= 1) *bytes = 0x86; return 1;
		case 0x00E7:	if (bytes && length >= 1) *bytes = 0x87; return 1;
		case 0x00EA:	if (bytes && length >= 1) *bytes = 0x88; return 1;
		case 0x00EB:	if (bytes && length >= 1) *bytes = 0x89; return 1;
		case 0x00E8:	if (bytes && length >= 1) *bytes = 0x8A; return 1;
		case 0x00EF:	if (bytes && length >= 1) *bytes = 0x8B; return 1;
		case 0x00EE:	if (bytes && length >= 1) *bytes = 0x8C; return 1;
		case 0x00EC:	if (bytes && length >= 1) *bytes = 0x8D; return 1;
		case 0x00C4:	if (bytes && length >= 1) *bytes = 0x8E; return 1;
		case 0x00C5:	if (bytes && length >= 1) *bytes = 0x8F; return 1;
		case 0x00C9:	if (bytes && length >= 1) *bytes = 0x90; return 1;
		case 0x00E6:	if (bytes && length >= 1) *bytes = 0x91; return 1;
		case 0x00C6:	if (bytes && length >= 1) *bytes = 0x92; return 1;
		case 0x00F4:	if (bytes && length >= 1) *bytes = 0x93; return 1;
		case 0x00F6:	if (bytes && length >= 1) *bytes = 0x94; return 1;
		case 0x00F2:	if (bytes && length >= 1) *bytes = 0x95; return 1;
		case 0x00FB:	if (bytes && length >= 1) *bytes = 0x96; return 1;
		case 0x00F9:	if (bytes && length >= 1) *bytes = 0x97; return 1;
		case 0x00FF:	if (bytes && length >= 1) *bytes = 0x98; return 1;
		case 0x00D6:	if (bytes && length >= 1) *bytes = 0x99; return 1;
		case 0x00DC:	if (bytes && length >= 1) *bytes = 0x9A; return 1;
		case 0x00F8:	if (bytes && length >= 1) *bytes = 0x9B; return 1;
		case 0x00A3:	if (bytes && length >= 1) *bytes = 0x9C; return 1;
		case 0x00D8:	if (bytes && length >= 1) *bytes = 0x9D; return 1;

		case 0x0192:	if (bytes && length >= 1) *bytes = 0xA0; return 1;
		case 0x00E1:	if (bytes && length >= 1) *bytes = 0xA1; return 1;
		case 0x00ED:	if (bytes && length >= 1) *bytes = 0xA2; return 1;
		case 0x00F3:	if (bytes && length >= 1) *bytes = 0xA3; return 1;
		case 0x00FA:	if (bytes && length >= 1) *bytes = 0xA4; return 1;
		case 0x00F1:	if (bytes && length >= 1) *bytes = 0xA5; return 1;
		case 0x00D1:	if (bytes && length >= 1) *bytes = 0xA6; return 1;
		case 0x00AA:	if (bytes && length >= 1) *bytes = 0xA7; return 1;
		case 0x00BA:	if (bytes && length >= 1) *bytes = 0xA8; return 1;

		case 0x00A1:	if (bytes && length >= 1) *bytes = 0xAD; return 1;

		case 0x00C1:	if (bytes && length >= 1) *bytes = 0xB5; return 1;
		case 0x00C2:	if (bytes && length >= 1) *bytes = 0xB6; return 1;
		case 0x00C0:	if (bytes && length >= 1) *bytes = 0xB7; return 1;

		case 0x00A2:	if (bytes && length >= 1) *bytes = 0xBD; return 1;
		case 0x00A5:	if (bytes && length >= 1) *bytes = 0xBE; return 1;

		case 0x00E3:	if (bytes && length >= 1) *bytes = 0xC6; return 1;
		case 0x00C3:	if (bytes && length >= 1) *bytes = 0xC7; return 1;

		case 0x00F0:	if (bytes && length >= 1) *bytes = 0xD0; return 1;
		case 0x00D0:	if (bytes && length >= 1) *bytes = 0xD1; return 1;
		case 0x00CA:	if (bytes && length >= 1) *bytes = 0xD2; return 1;
		case 0x00CB:	if (bytes && length >= 1) *bytes = 0xD3; return 1;
		case 0x00C8:	if (bytes && length >= 1) *bytes = 0xD4; return 1;
		case 0x0131:	if (bytes && length >= 1) *bytes = 0xD5; return 1;
		case 0x00CD:	if (bytes && length >= 1) *bytes = 0xD6; return 1;
		case 0x00CE:	if (bytes && length >= 1) *bytes = 0xD7; return 1;
		case 0x0407:	if (bytes && length >= 1) *bytes = 0xD8; return 1;

		case 0x00CC:	if (bytes && length >= 1) *bytes = 0xDE; return 1;

		case 0x00D3:	if (bytes && length >= 1) *bytes = 0xE0; return 1;
		case 0x00DF:	if (bytes && length >= 1) *bytes = 0xE1; return 1;
		case 0x00D4:	if (bytes && length >= 1) *bytes = 0xE2; return 1;
		case 0x00D2:	if (bytes && length >= 1) *bytes = 0xE3; return 1;
		case 0x00F5:	if (bytes && length >= 1) *bytes = 0xE4; return 1;
		case 0x00D5:	if (bytes && length >= 1) *bytes = 0xE5; return 1;
		case 0x00B5:	if (bytes && length >= 1) *bytes = 0xE6; return 1;
		case 0x00DE:	if (bytes && length >= 1) *bytes = 0xE7; return 1;
		case 0x00FE:	if (bytes && length >= 1) *bytes = 0xE8; return 1;
		case 0x00DA:	if (bytes && length >= 1) *bytes = 0xE9; return 1;
		case 0x00DB:	if (bytes && length >= 1) *bytes = 0xEA; return 1;
		case 0x00D9:	if (bytes && length >= 1) *bytes = 0xEB; return 1;
		case 0x00FD:	if (bytes && length >= 1) *bytes = 0xEC; return 1;
		case 0x00DD:	if (bytes && length >= 1) *bytes = 0xED; return 1;

		default:	return 0;
	}
}

int ASCIICodepage850Encoding::queryConvert (const unsigned char* bytes, int length) const
{
	if (1 <= length)
		return _charMap [*bytes];
	else
		return -1;
}

int ASCIICodepage850Encoding::sequenceLength (const unsigned char* bytes, int length) const
{
	return 1;
}
