// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/encoding.h"
#include "../libpdrx/build.h"
#include "config_impl.h"
#include "db.h"
#include "out.h"

//=== ConfigImpl ===========================================================
ConfigImpl::ConfigImpl ()
	: Config()
	, m_periods()
{
	m_od_visible.add_options()

		// common options
		("help,?",			"show this help screen and quit")
		("version,V",			"display version number and quit")
		("verbose,v",			"produce verbose output on stdout")
		("config,c", value<string>(),	"use specific configuration file")
		("midnight,m", value<string>(),	"define what is midnight, arg is hh:mm[:ss]")
		("now,n", value<string>(),	"define what is \"now\", arg is CCYY-MM-DD[-hh:mm[:ss]]")
		("interactive,i",		"don't generate any output, start interactive")
		("fast,f",			"cache internal informations in memory, needs much more RAM")
	;

	// further we need named options for the list of the configured
	// outputs and for the sum of all the given command line outputs,
	// but we don't want to have this on the help screen
	m_od_cmdline.add_options() ("outputs", value<string>(), "");
	m_od_cmdline.add_options() ("cmd_line_outputs", value<string>(), "");

	m_od_cmdline.add_options() ("cmd_line_args", value<vector<string> >(), "");
	m_pod.add("cmd_line_args", -1); // unlimited
}

ConfigImpl::~ConfigImpl ()
{
	foreach(Period* pPeriod, m_periods)
	{
		delete pPeriod;
	}
}

	struct fo_cmd_line_arg: public unary_function<option&, bool> {
		string& m_cmd_line_outputs;
		fo_cmd_line_arg (string& cmd_line_outputs)
			: m_cmd_line_outputs(cmd_line_outputs)
		{
		}
		result_type operator () (argument_type a)
		{
			if (a.string_key == "cmd_line_args")
			{
				if (!m_cmd_line_outputs.empty())
					m_cmd_line_outputs += ',';
				m_cmd_line_outputs += a.value[0];
				return true;
			}
			else
				return false;
		}
	};

void ConfigImpl::ModifyCmdLineOptions (parsed_options& po)
{
	// in pdx we transform all the given command line arguments (the
	// arguments, not the options) into a list of output names

	string cmd_line_outputs;
	{
		vector<option>::iterator I = remove_if(po.options.begin(), po.options.end(), fo_cmd_line_arg(cmd_line_outputs));
		if (I != po.options.end())
			po.options.erase(I, po.options.end());
	}

	if (!cmd_line_outputs.empty())
	{
		vector<string> values;
		values.push_back(cmd_line_outputs);
		po.options.push_back(option("cmd_line_outputs", values));
	}
}

	struct fo_period: public unary_function<option&, bool> {
		ConfigImpl::Periods& m_periods;
		fo_period (ConfigImpl::Periods& periods)
			: m_periods(periods)
		{
		}
		result_type operator () (argument_type a)
		{
			if (a.string_key == "period")
			{
				foreach(const string& s, a.value)
				{
					try
					{
						//                                       1            2            3                    4            5            6                    7
						static const regex rx_period("[ \t]*([0-9]{1,4})-([0-9]{1,2})-([0-9]{1,2})[ \t.,;]+([0-9]{1,4})-([0-9]{1,2})-([0-9]{1,2})[ \t,;]+#([0-9A-Fa-f]+).*");
						smatch mr;
						if (regex_match(s, mr, rx_period, boost::match_not_dot_newline))
						{
							auto_ptr<ConfigImpl::Period> period(new ConfigImpl::Period);
							int year, month, day;
							year = lexical_cast<int>(mr[1]);
							month = lexical_cast<int>(mr[2]);
							day = lexical_cast<int>(mr[3]);
							period->m_begin = date(year, month, day);
							year = lexical_cast<int>(mr[4]);
							month = lexical_cast<int>(mr[5]);
							day = lexical_cast<int>(mr[6]);
							period->m_end = date(year, month, day);
							istringstream(mr[7]) >> hex >> period->m_color;
							m_periods.push_back(period.release());
						}
						else
							throw 1;
					}
					catch (...)
					{
						cerr << "*** warning: syntax error in time period: " << s << ", ignoring this" << endl;
					}
				}
				return true;
			}
			else
				return false;
		}
	};

void ConfigImpl::ModifyCfgFileOptions (parsed_options& po)
{
	// transform the periods into Periods, too
	vector<option>::iterator I = remove_if(po.options.begin(), po.options.end(), fo_period(m_periods));
	if (I != po.options.end())
		po.options.erase(I, po.options.end());
}

const ConfigImpl::Periods& ConfigImpl::GetPeriods () const
{
	return m_periods;
}

//=== main =================================================================
int main (int argc, char* argv[])
{
	try
	{
		InitEncodings();

		// parse the command line and the config file
		auto_ptr<Config> config(ConfigFactory<ConfigImpl>::Create(argc, argv));

		// handle the very simple cases
		if (config->GetBoolOption("version"))
		{
			cout	<< "pdx - personal data expert, " << VERSION << endl
				<< "(C) 2010-2011 Torsten Mueller, Bern, Switzerland" << endl
				<< '(' << BUILD << ')' << endl
				<< "This program is free software under the terms of the GNU General Public License v2 or later." << endl;
			return 0;
		}

		if (config->GetBoolOption("help"))
		{
			cout << "pdx - personal data expert" << endl;
			cout << "usage: pdx [options] [outputs]" << endl;
			config->Help(cout);
			return 0;
		}

		// not very simple, work with the database
		{
			auto_ptr<Database> database(DBFactory::Create(*config));
			OutputFactory factory;

			if (config->GetBoolOption("interactive"))
				auto_ptr<Output>(factory.GetOutputInteractive())->Do(*config, *database);
			else
			{
				// create output
				bool verbose = config->GetBoolOption("verbose");
				if (verbose)
					cout << "generating output ";

				const OutputFactory::Outputs& outputs = factory.GetOutputs(*config);
				foreach (const Output* pOutput, outputs)
				{
					if (verbose)
						cout << ':' << flush;
					try
					{
						pOutput->Do(*config, *database);
					}
					catch (...)
					{
						if (verbose)
							cout << endl;
						throw;
					}
				}
				if (verbose)
					cout << endl;
			}
		}

		return 0;
	}
	catch (const Xception& e)
	{
		cerr << "*** fatal: " << e.Message() << endl;
		return 1;
	}
	catch (...)
	{
		cerr << "*** fatal: internal error" << endl;
		return 1;
	}
}
