// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "config_impl.h"
#include "out_ftree.h"
#include "diagram.h"

//=== FContext classes =====================================================
FContext::FContext (const Config& config, const Database& database)
	: m_config(config)
	, m_database(database)
{
}

FContext::~FContext ()
{
}

FDiagramContext::FDiagramContext (const Config& config, const Database& database, Diagram& diagram)
	: FContext(config, database)
	, m_diagram(diagram)
{
}

//=== FTree classes ========================================================

//--- FTree (abstract base class) ------------------------------------------
FTree::FTree ()
{
}

FTree::~FTree ()
{
}

//--- FOneLiner (abstract base class) --------------------------------------
FOneLiner::FOneLiner ()
	: FTree()
{
}

//--- the classes for constants --------------------------------------------
FDouble::FDouble (double value)
	: FConst<double>(value)
{
}

void FDouble::Dump (ostream& os, int indent) const throw (Xception)
{
	if (indent != -1)
		os << string(indent * 2, ' ');
	os << format("%g") % m_value;
}

FDateTime::FDateTime (const ptime& value)
	: FConst<ptime>(value)
{
}

void FDateTime::Dump (ostream& os, int indent) const throw (Xception)
{
	if (indent != -1)
		os << string(indent * 2, ' ');
	os << lexical_cast<string>(m_value);
}

FDate::FDate (const date& value)
	: FConst<date>(value)
{
}

void FDate::Dump (ostream& os, int indent) const throw (Xception)
{
	if (indent != -1)
		os << string(indent * 2, ' ');
	os << lexical_cast<string>(m_value);
}

FTime::FTime (const time_duration& value)
	: FConst<time_duration>(value)
{
}

void FTime::Dump (ostream& os, int indent) const throw (Xception)
{
	if (indent != -1)
		os << string(indent * 2, ' ');
	os << lexical_cast<string>(m_value);
}

FString::FString (const string& value)
	: FConst<string>(value)
{
}

void FString::Dump (ostream& os, int indent) const throw (Xception)
{
	if (indent != -1)
		os << string(indent * 2, ' ');
	os << '"' << m_value << '"';
}

FFormatSpec::FFormatSpec (const fstring& value)
	: FConst<fstring>(value)
{
}

void FFormatSpec::Dump (ostream& os, int indent) const throw (Xception)
{
	if (indent != -1)
		os << string(indent * 2, ' ');
	os << '<' << m_value << '>';
}

FInt::FInt (int value)
	: FConst<int>(value)
{
}

void FInt::Dump (ostream& os, int indent) const throw (Xception)
{
	if (indent != -1)
		os << string(indent * 2, ' ');
	os << m_value;
}

FColor::FColor (size_t value)
	: FConst<size_t>(value)
{
}

void FColor::Dump (ostream& os, int indent) const throw (Xception)
{
	if (indent != -1)
		os << string(indent * 2, ' ');
	os << '#' << format("%X") % m_value;
}

//--- FKeyword -------------------------------------------------------------
FKeyword::FKeyword (const string& name)
	: FOneLiner()
	, m_name(name)
{
}

const type_info* FKeyword::Type () const throw (Xception)
{
	return &typeid(kstring);
}

void FKeyword::Dump (ostream& os, int indent) const throw (Xception)
{
	if (indent != -1)
		os << string(indent * 2, ' ');
	os << m_name;
}

any FKeyword::Evaluate (FContext& ) const throw (Xception)
{
	return any(m_name);
}

//--- FExpr ----------------------------------------------------------------
FImplFactory FExpr::fImplFactory;

FExpr::FExpr ()
	: FTree()
	, m_func()
	, m_params()
	, m_pFImpl(NULL)
{
}

FExpr::~FExpr ()
{
	foreach (FTree* pParam, m_params)
	{
		delete pParam;
	}
}

const type_info* FExpr::Type () const throw (Xception)
{
	if (!m_pFImpl)
	{
		FImpl::ParamTypes paramtypes;
		foreach (FTree* pParam, m_params)
		{
			paramtypes.push_back(pParam->Type());
		}
		m_pFImpl = fImplFactory.Match(m_func, paramtypes);
	}

	return m_pFImpl->m_returntype;
}

void FExpr::Dump (ostream& os, int indent) const throw (Xception)
{
	if (indent != -1)
		os << string(indent * 2, ' ');
	os << '(' << m_func;
	if (m_params.empty())
		os << ')' << endl;
	else
	{
		bool on_same_line = true;
		bool all_on_same_line = true;
		foreach (FTree* pParam, m_params)
		{
			if (dynamic_cast<FOneLiner*>(pParam))
			{
				if (on_same_line)
				{
					os << ' ';
					pParam->Dump(os, -1);
				}
				else
				{
					pParam->Dump(os, indent + 1);
					on_same_line = true;
				}
			}
			else
			{
				os << endl;
				pParam->Dump(os, indent + 1);
				all_on_same_line = on_same_line = false;
			}
		}
		if (all_on_same_line)
			os << ')' << endl;
		else
		{
			if (on_same_line)
				os << endl;
			if (indent != -1)
				os << string(indent * 2, ' ');
			os << ')' << endl;
		}
	}
}

any FExpr::Evaluate (FContext& context) const throw (Xception)
{
	if (!m_pFImpl)
	{
		FImpl::ParamTypes paramtypes;
		foreach (FTree* pParam, m_params)
		{
			paramtypes.push_back(pParam->Type());
		}
		m_pFImpl = fImplFactory.Match(m_func, paramtypes);
	}

	return m_pFImpl->Evaluate(context, m_params);
}

void FExpr::SetFunction (const string& func)
{
	m_func = func;
}

void FExpr::AddParam (FTree* pFTree)
{
	m_params.push_back(pFTree);
}

//=== helper functions =====================================================
ostream& operator << (ostream& os, const FTree& fTree) throw (Xception)
{
	fTree.Dump(os, 0);
	return os;
}

ostream& operator << (ostream& os, const any& a) throw (Xception)
{
	if (a.empty())
		return os << "nil";

	const type_info& ti = a.type();

	if (ti == typeid(double))
		return os << format("%g") % any_cast<double>(a);

	if (ti == typeid(Ratio))
		return os << (format("%g") % any_cast<Ratio>(a).m_numerator) << '/' << (format("%g") % any_cast<Ratio>(a).m_denominator);

	if (ti == typeid(string))
		return os << any_cast<string>(a);

	if (ti == typeid(ptime))
		return os << lexical_cast<string>(any_cast<ptime>(a));

	if (ti == typeid(date))
		return os << lexical_cast<string>(any_cast<date>(a));

	if (ti == typeid(time_duration))
		return os << lexical_cast<string>(any_cast<time_duration>(a));

	if (ti == typeid(int))
		return os << any_cast<int>(a);

	if (ti == typeid(shared_ptr<Selection>))
	{
		shared_ptr<Selection> selection = any_cast<shared_ptr<Selection> >(a);
		const size_t* pDimensions = selection->shape();
		for (size_t row = 0; row < pDimensions[0]; row++)
		{
			if (row > 0)
				os << endl;
			for (size_t column = 0; column < pDimensions[1]; column++)
			{
				if (column > 0)
					os << '\t';
				os << (*selection)[row][column]; // any!
			}
		}
		return os;
	}

	throw Xception("expression returns unknown type");
}
