// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace boost::filesystem;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/encoding.h"
#include "config_impl.h"
#include "diagram_impl.h"

#ifdef USE_CAIRO

// We don't know how big the image becomes in the end. So we draw onto
// a surface with a size bigger than wanted called the "canvas".
// The diagram will always stay in the upper right corner of the
// canvas while it's left and bottom sides have large initial margins
// for drawing the axes and the numbers on them. Later we will cut off
// the usused space. For this we need to know the left bottom corner
// of our drawing, called the "origin". The origin will be updated
// after every drawing operation and moves towards the left bottom
// corner of the canvas.

// (we need single color values between 0.0 and 1.0)
#define	RED(c)		((double)(unsigned char)(c >> 16)) / 255.0
#define	GREEN(c)	((double)(unsigned char)(c >>  8)) / 255.0
#define	BLUE(c)		((double)(unsigned char)(c      )) / 255.0

//=== CairoDiagram =========================================================
CairoDiagram::CairoDiagram (bool antialias)
	: DiagramImpl()
	, m_pSurface(NULL)
	, m_pContext(NULL)
	, m_bgcolor(0x00FFFFFF)
	, m_antialias(antialias)
	, m_thickness((antialias) ? 0.5 : 1.0)
{
	m_canvas.width = m_canvas.height = m_diagram.width = m_diagram.height = 0;
	m_origin.x = m_origin.y = 0.0;
	m_margin.left = m_margin.top = m_margin.right = m_margin.bottom = 0.0;
}

CairoDiagram::~CairoDiagram ()
{
	if (m_pContext)
		cairo_destroy(m_pContext);
	if (m_pSurface)
		cairo_surface_destroy(m_pSurface);
}

void CairoDiagram::OpenCanvas (size_t width, size_t height, RGB bgColor)
{
	m_diagram.width = (int)width;
	m_diagram.height = (int)height;

	m_bgcolor = bgColor;

	// compute the margins
	{
		cairo_surface_t* pSurface = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, width, height);
		cairo_t* pContext = cairo_create(pSurface);

		cairo_set_font_size(pContext, width / 40);
		cairo_font_options_t* pOptions = cairo_font_options_create();
		cairo_get_font_options(pContext, pOptions);
		cairo_font_options_set_antialias(pOptions, CAIRO_ANTIALIAS_NONE);
		cairo_set_font_options(pContext, pOptions);
		cairo_font_options_destroy(pOptions);

		cairo_text_extents_t extents;
		cairo_text_extents(pContext, " CCCC-MM-DD hh:mm:ss ", &extents);
		m_margin.left = m_margin.bottom = extents.width;
		m_margin.right = m_margin.top = extents.height / 2.0 + 1.0;
		m_margin.top += extents.height + 5; // for the text above the y-axis

		cairo_destroy(pContext);
		cairo_surface_destroy(pSurface);
	}

	// set the canvas size
	m_canvas.width = int(width) + int(m_margin.left + m_margin.right);
	m_canvas.height = int(height) + int(m_margin.top + m_margin.bottom);

	// initially we put the origin into the upper right corner of
	// the canvas
	m_origin.x = (double)m_canvas.width;
	m_origin.y = 0.0;

	// create the surface for the canvas
	m_pSurface = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, m_canvas.width, m_canvas.height);
	m_pContext = cairo_create(m_pSurface);

	// set some over all properties
	cairo_antialias_t aa = (m_antialias) ? CAIRO_ANTIALIAS_DEFAULT : CAIRO_ANTIALIAS_NONE;
	cairo_set_antialias(m_pContext, aa);

	cairo_set_font_size(m_pContext, width / 40);
	cairo_font_options_t* pOptions = cairo_font_options_create();
	cairo_get_font_options(m_pContext, pOptions);
	cairo_font_options_set_antialias(pOptions, aa);
	cairo_set_font_options(m_pContext, pOptions);
	cairo_font_options_destroy(pOptions);

	// from now on we draw the contents into a group
	cairo_push_group(m_pContext);

	// fill the entire image with a solid background
	cairo_set_source_rgb(m_pContext, 1.0, 1.0, 1.0);
	cairo_paint(m_pContext);

	cairo_new_path(m_pContext);
}

void CairoDiagram::CloseCanvas ()
{
	// now we create a resulting surface with the real size
	cairo_surface_t* pSurface = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, m_canvas.width - (int)m_origin.x + 1, (int)m_origin.y + 1);
	cairo_t* pContext = cairo_create(pSurface);

	// get all the drawn stuff from the canvas as a pattern
	cairo_pattern_t* pPattern = cairo_pop_group(m_pContext);

	// paint the pattern again onto the resulting surface
	cairo_matrix_t m;
	cairo_matrix_init_translate(&m, m_origin.x, 0.0);
	cairo_pattern_set_matrix(pPattern, &m);
	cairo_set_source(pContext, pPattern);
	cairo_pattern_destroy(pPattern);
	cairo_paint(pContext);
	cairo_destroy(pContext);

	// now cleanup the context and the canvas, we don't draw
	// anymore, hold only the resulting surface
	cairo_destroy(m_pContext);
	m_pContext = NULL;
	cairo_surface_destroy(m_pSurface);
	m_pSurface = pSurface;
}

void CairoDiagram::DrawAxes (const ptime& x_min, const ptime& x_max, XUnit x_unit, double y_min, double y_max, double y_unit, RGB color, const string& text, const ConfigImpl::Periods& periods)
{
	DiagramImpl::DrawAxes(x_min, x_max, x_unit, y_min, y_max, y_unit, color, text, periods);

	const double x_0 = CalcX(m_xmin);
	const double y_0 = CalcY(m_ymin);

	// Step 0: fill the background of the diagram
	cairo_set_source_rgb(m_pContext, RED(m_bgcolor), GREEN(m_bgcolor), BLUE(m_bgcolor));
	cairo_rectangle(m_pContext, x_0, m_margin.top, (double)m_diagram.width, (double)m_diagram.height);
	cairo_fill(m_pContext);
	UpdateOrigin(x_0, y_0);

	// Step 1: iterate over the time periods, make backgrounds
	foreach(const ConfigImpl::Period* pPeriod, periods)
	{
		ptime begin(not_a_date_time), end(not_a_date_time);

		{
			ptime t(pPeriod->m_begin, time_duration(0, 0, 0, 0));
			if (t < x_min)
				begin = x_min;
			else
			{
				if (t > x_max)
					begin = x_max;
				else
					begin = t;
			}
		}

		{
			ptime t(pPeriod->m_end, time_duration(0, 0, 0, 0));
			if (t < x_min)
				end = x_min;
			else
			{
				if (t > x_max)
					end = x_max;
				else
					end = t;
			}
		}

		if (begin != end)
		{
			cairo_set_source_rgb(m_pContext, RED(pPeriod->m_color), GREEN(pPeriod->m_color), BLUE(pPeriod->m_color));
			cairo_rectangle(m_pContext, CalcX(begin), CalcY(y_min), CalcX(end) - CalcX(begin), CalcY(y_max) - CalcY(y_min));
			cairo_fill(m_pContext);
		}
	}

	// Step 2: x-axis
	cairo_set_source_rgb(m_pContext, RED(color), GREEN(color), BLUE(color));
	cairo_set_line_width(m_pContext, m_thickness);
	move_to(x_0, y_0);
	line_to(x_0 + (double)m_diagram.width, y_0);

	const double dash_height = m_diagram.height / 100.0; // one percent of m_height
	if (m_xunit == months)
	{
		// the problem with months is that they differ in their
		// lengths, so we must rather increment the month in the
		// date of the first day than adding just a fixed length as
		// in all the other cases below
		for (ptime t(m_xmin); t <= m_xmax; )
		{
			string s1(lexical_cast<string, ptime>(t)), s2;
			s1.erase(7);
			s2 = s1;
			s1 += "-01 00:00:00.000";
			t = time_from_string(s1);
			if (t >= m_xmin)
			{
				double x = CalcX(t);
				move_to(x, y_0);
				line_to(x, y_0 + dash_height);
				text_x(x, y_0, dash_height, s2);
			}
			const date& d = t.date();
			greg_month m = d.month();
			t = ptime((m == 12) ? date(d.year() + 1, 1, 1) : date(d.year(), m + 1, 1), time_duration(0, 0, 0, 0));
		}
	}
	else
	{
		for (ptime t(m_xmin); t <= m_xmax; t += m_increment)
		{
			string s1(lexical_cast<string, ptime>(t)), s2;
			switch (m_xunit)
			{
				case years:	s1.erase(4);  s2 = s1; s1 += "-01-01 00:00:00.000"; break;
				case days:	s1.erase(10); s2 = s1; s1 += " 00:00:00.000"; break;
				case hours:	s1.erase(13); s2 = s1; s1 += ":00:00.000"; break;
				case minutes:	s1.erase(16); s2 = s1; s1 += ":00.000"; break;
				default:	break;
			}
			t = time_from_string(s1);
			if (t >= m_xmin)
			{
				double x = CalcX(t);
				move_to(x, y_0);
				line_to(x, y_0 + dash_height);
				if (m_sundays != (RGB)-1 && m_xunit == days && t.date().day_of_week() == 0)
				{
					cairo_set_source_rgb(m_pContext, RED(m_sundays), GREEN(m_sundays), BLUE(m_sundays));
					text_x(x, y_0, dash_height, s2);
					cairo_set_source_rgb(m_pContext, RED(color), GREEN(color), BLUE(color));
				}
				else
					text_x(x, y_0, dash_height, s2);
			}
		}
	}

	// Step 3: y-axis, much easier because of simple values
	move_to(x_0, y_0);
	line_to(x_0, y_0 - m_diagram.height);

	const double dash_width = m_diagram.width / 100.0; // one percent of m_width

	const string& yf = FormatY();
	for (double t = m_ymin; t <= m_ymax; t += m_yunit)
	{
		double y = CalcY(t);
		move_to(x_0 - dash_width, y);
		line_to(x_0, y);

		string s((format(yf) % t).str());
		text_y(x_0, y, dash_width, s);
	}

	// text above the y-axis
	if (!text.empty())
	{
		cairo_text_extents_t extents;
		cairo_text_extents(m_pContext, text.c_str(), &extents);
		move_to(m_origin.x, CalcY(y_max) - (extents.height / 2) - 5);
		UpdateOrigin(m_origin.x, CalcY(y_max) + (extents.height / 2) + 5);
		cairo_show_text(m_pContext, text.c_str());
	}

	cairo_stroke(m_pContext);
}

void CairoDiagram::DrawAxes (Diagram::FoldInterval fi, double y_min, double y_max, double y_unit, RGB color, const string& text)
{
	DiagramImpl::DrawAxes(fi, y_min, y_max, y_unit, color, text);

	const double x_0 = CalcX(m_xmin);
	const double y_0 = CalcY(m_ymin);

	// Step 0: fill the background of the diagram
	cairo_set_source_rgb(m_pContext, RED(m_bgcolor), GREEN(m_bgcolor), BLUE(m_bgcolor));
	cairo_rectangle(m_pContext, x_0, m_margin.top, (double)m_diagram.width, (double)m_diagram.height);
	cairo_fill(m_pContext);
	UpdateOrigin(x_0, y_0);

	// Step 1: x-axis
	cairo_set_source_rgb(m_pContext, RED(color), GREEN(color), BLUE(color));
	cairo_set_line_width(m_pContext, m_thickness);
	move_to(x_0, y_0);
	line_to(x_0 + (double)m_diagram.width, y_0);

	const double dash_height = m_diagram.height / 100.0; // one percent of m_diagram.height
	if (m_xunit == months)
	{
		// the problem with months is that they differ in their
		// lengths, so we must rather increment the month in the
		// date of the first day than adding just a fixed length as
		// in all the other cases below
		for (ptime t(m_xmin); t <= m_xmax; )
		{
			double x = CalcX(t);
			move_to(x, y_0);
			line_to(x, y_0 + dash_height);

			const string& s = Fold(lexical_cast<string, ptime>(t), fi);
			text_x(x, y_0, dash_height, s);

			const date& d = t.date();
			greg_month m = d.month();
			t = ptime((m == 12) ? date(d.year() + 1, 1, 1) : date(d.year(), m + 1, 1), time_duration(0, 0, 0, 0));
		}
	}
	else
	{
		for (ptime t(m_xmin); t <= m_xmax; t += m_increment)
		{
			double x = CalcX(t);
			move_to(x, y_0);
			line_to(x, y_0 + dash_height);

			const string& s = Fold(lexical_cast<string, ptime>(t), fi);
			text_x(x, y_0, dash_height, s);
		}
	}

	// Step 2: y-axis, much easier because of simple values
	move_to(x_0, y_0);
	line_to(x_0, y_0 - m_diagram.height);

	const double dash_width = m_diagram.width / 100.0; // one percent of m_width

	const string& yf = FormatY();
	for (double t = m_ymin; t <= m_ymax; t += m_yunit)
	{
		double y = CalcY(t);
		move_to(x_0 - dash_width, y);
		line_to(x_0, y);

		string s((format(yf) % t).str());
		text_y(x_0, y, dash_width, s);
	}

	// text above the y-axis
	if (!text.empty())
	{
		cairo_text_extents_t extents;
		cairo_text_extents(m_pContext, text.c_str(), &extents);
		move_to(m_origin.x, CalcY(y_max) - (extents.height / 2) - 5);
		UpdateOrigin(m_origin.x, CalcY(y_max) + (extents.height / 2) + 5);
		cairo_show_text(m_pContext, text.c_str());
	}

	cairo_stroke(m_pContext);
}

void CairoDiagram::DrawCurve (const Selection& s, RGB color, LineStyle style, double thickness, bool continuous)
{
	const size_t* pDimensions = s.shape(); // [0] rows, [1] columns

	if (pDimensions[0] == 1 && any_cast<string>(s[0][0]).empty())
		throw Xception("selection contains just one single value which is the result of an aggregation, you must ensure the selection to have multiple values to draw a curve");

	if (thickness == 0.0)
		thickness = m_thickness;

	cairo_set_source_rgb(m_pContext, RED(color), GREEN(color), BLUE(color));

	Dot last;
	for (size_t row = 0; row < pDimensions[0]; row++)
	{
		Dot current(any_cast<string>(s[row][0]), s[row][1]);

		switch (style)
		{
			case symbol_dot:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					cairo_set_line_width(m_pContext, thickness);
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					move_to(x - thickness / 2.0, y);
					line_to(x + thickness / 2.0, y);
					cairo_stroke(m_pContext);
				}
				break;
			}
			case symbol_plus:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					cairo_set_line_width(m_pContext, thickness);
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					move_to(x - extent, y);
					line_to(x + extent, y);
					move_to(x, y - extent);
					line_to(x, y + extent);
					cairo_stroke(m_pContext);
				}
				break;
			}
			case symbol_vbar:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					cairo_set_line_width(m_pContext, thickness);
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					move_to(x, y - extent);
					line_to(x, y + extent);
					cairo_stroke(m_pContext);
				}
				break;
			}
			case symbol_hbar:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					cairo_set_line_width(m_pContext, thickness);
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					move_to(x - extent, y);
					line_to(x + extent, y);
					cairo_stroke(m_pContext);
				}
				break;
			}
			case symbol_x:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					cairo_set_line_width(m_pContext, thickness);
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 200.0; // half a percent of m_diagram.width
					move_to(x - extent, y - extent);
					line_to(x + extent, y + extent);
					move_to(x - extent, y + extent);
					line_to(x + extent, y - extent);
					cairo_stroke(m_pContext);
				}
				break;
			}
			case symbol_X:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					cairo_set_line_width(m_pContext, thickness);
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					move_to(x - extent, y - extent);
					line_to(x + extent, y + extent);
					move_to(x - extent, y + extent);
					line_to(x + extent, y - extent);
					cairo_stroke(m_pContext);
				}
				break;
			}
			case symbol_circle:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					cairo_set_line_width(m_pContext, thickness);
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					cairo_arc(m_pContext, x, y, extent / 2.0, 0.0, 2.0 * M_PI);
					UpdateOrigin(x - extent / 2.0, y + extent / 2.0);
					cairo_stroke(m_pContext);
				}
				break;
			}
			case symbol_square:
			{
				if (current)
				{
					if (current.m_timestamp < m_xmin)
						continue;
					cairo_set_line_width(m_pContext, thickness);
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_diagram.width / 100.0; // one percent of m_diagram.width
					cairo_rectangle(m_pContext, x - extent / 2.0, y - extent / 2.0, extent, extent);
					UpdateOrigin(x - extent / 2.0, y + extent / 2.0);
					cairo_stroke(m_pContext);
				}
				break;
			}
			default: // zick-zack
			{
				if (current.m_timestamp < m_xmin)
					continue;
				if (last && current)
				{
					cairo_set_line_width(m_pContext, thickness);
					move_to(CalcX(last.m_timestamp), CalcY(any_cast<double>(last.m_value)));
					line_to(CalcX(current.m_timestamp), CalcY(any_cast<double>(current.m_value)));
					cairo_stroke(m_pContext);
				}
				break;
			}
		}

		if (current || !continuous)
			last = current;
	}
}

void CairoDiagram::DrawStairs (const Selection& s, RGB color, double thickness)
{
	const size_t* pDimensions = s.shape(); // [0] rows, [1] columns

	if (pDimensions[0] == 1 && any_cast<string>(s[0][0]).empty())
		throw Xception("selection contains just one single value which is the result of an aggregation, you must ensure the selection to have multiple values to draw a curve");

	if (thickness == 0.0)
		thickness = m_thickness;

	cairo_set_source_rgb(m_pContext, RED(color), GREEN(color), BLUE(color));

	Dot last;
	for (size_t row = 0; row < pDimensions[0]; row++)
	{
		Dot current(any_cast<string>(s[row][0]), s[row][1]);

		if (current.m_timestamp < m_xmin)
			continue;
		if (last && current)
		{
			cairo_set_line_width(m_pContext, thickness);
			move_to(CalcX(last.m_timestamp), CalcY(any_cast<double>(last.m_value)));
			line_to(CalcX(current.m_timestamp), CalcY(any_cast<double>(last.m_value)));
			line_to(CalcX(current.m_timestamp), CalcY(any_cast<double>(current.m_value)));
			cairo_stroke(m_pContext);
		}

		if (current)
			last = current;
	}
}

void CairoDiagram::DrawVLine (const ptime& timestamp, RGB color, double thickness)
{
	cairo_set_source_rgb(m_pContext, RED(color), GREEN(color), BLUE(color));
	cairo_set_line_width(m_pContext, (float)((thickness == 0.0) ? m_thickness : thickness));
	double x = CalcX(timestamp);
	double y_0 = CalcY(m_ymin);
	move_to(x, y_0);
	line_to(x, y_0 - m_diagram.height);
	cairo_stroke(m_pContext);
}

void CairoDiagram::DrawVLine (const time_duration& time, RGB color, double thickness)
{
	DrawVLine(ptime(date(9999, 1, 1), time), color, thickness);
}

void CairoDiagram::DrawHLine (double value, RGB color, double thickness)
{
	cairo_set_source_rgb(m_pContext, RED(color), GREEN(color), BLUE(color));
	cairo_set_line_width(m_pContext, (float)((thickness == 0.0) ? m_thickness : thickness));
	double x_0 = CalcX(m_xmin);
	double y = CalcY(value);
	move_to(x_0, y);
	line_to(x_0 + m_diagram.width, y);
	cairo_stroke(m_pContext);
}

void CairoDiagram::DrawBar (const ptime& t, double value, int bar1, int bar2, RGB color, bool fix)
{
	double x = CalcX(t);
	double y = CalcY(value);
	double x_width = (fix) ? (CalcX(t + m_increment) - x) / double(bar2) : (CalcX(min(t + m_increment, m_xmax)) - x) / double(bar2);
	x += x_width * double(bar1 - 1);

	if (x >= m_margin.left)
	{
		cairo_set_source_rgb(m_pContext, RED(color), GREEN(color), BLUE(color));
		cairo_rectangle(m_pContext, x, y, x_width, CalcY(m_ymin) - y);
		cairo_fill(m_pContext);
		UpdateOrigin(x, CalcY(m_ymin));
		cairo_stroke(m_pContext);
	}
}

void CairoDiagram::UpdateOrigin (double x, double y)
{
	if (x < m_origin.x)
		m_origin.x = x;
	if (y > m_origin.y)
		m_origin.y = y;
}

double CairoDiagram::CalcX (const ptime& x) const
{
	double a = (x - m_xmin).total_seconds();
	double b = (m_xmax - m_xmin).total_seconds();
	return (a / b) * m_diagram.width + m_margin.left;
}

double CairoDiagram::CalcY (double y) const
{
	double a = y - m_ymin;
	double b = m_ymax - m_ymin;
	return m_diagram.height - (a / b) * m_diagram.height + m_margin.top;
}

void CairoDiagram::move_to (double x, double y)
{
	cairo_move_to(m_pContext, x, y);
	UpdateOrigin(x, y);
}

void CairoDiagram::line_to (double x, double y)
{
	cairo_line_to(m_pContext, x, y);
	UpdateOrigin(x, y);
}

void CairoDiagram::text_x (double x, double y, double dash_height, const string& s)
{
	cairo_save(m_pContext);

	cairo_text_extents_t extents;
	cairo_text_extents(m_pContext, s.c_str(), &extents);
	move_to(x + (extents.height / 2.0), y + dash_height + 2 + extents.width);
	cairo_rotate(m_pContext, -1.5707963); // -90°
	cairo_show_text(m_pContext, s.c_str());

	cairo_restore(m_pContext);
}

void CairoDiagram::text_y (double x, double y, double dash_width, const string& s)
{
	cairo_text_extents_t extents;
	cairo_text_extents(m_pContext, s.c_str(), &extents);
	move_to(x - dash_width - 2 - extents.width, y + (extents.height / 2.0));
	cairo_show_text(m_pContext, s.c_str());
}

//=== DiagramPNG ===========================================================
DiagramPNG::DiagramPNG (bool antialias)
	: CairoDiagram(antialias)
{
}

void DiagramPNG::SaveAs (const string& filename) const
{
	cairo_status_t st = cairo_surface_write_to_png(m_pSurface, filename.c_str());
	if (st != CAIRO_STATUS_SUCCESS)
		throw Xception(cairo_status_to_string(st));
}

#endif // USE_CAIRO
