// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/encoding.h"
#include "config_impl.h"
#include "diagram_impl.h"

//=== DiagramImpl ==========================================================
DiagramImpl::DiagramImpl ()
	: m_sundays((RGB)-1)
	, m_xmin(not_a_date_time)
	, m_xmax(not_a_date_time)
	, m_xunit((XUnit)-1)
	, m_ymin(0.0)
	, m_ymax(0.0)
	, m_yunit(0.0)
	, m_increment(not_a_date_time)
{
}

void DiagramImpl::SetSundaysColor (RGB color)
{
	m_sundays = color;
}

void DiagramImpl::DrawAxes (const ptime& x_min, const ptime& x_max, XUnit x_unit, double y_min, double y_max, double y_unit, RGB color, const string& text, const ConfigImpl::Periods& /*periods*/)
{
	m_xmin = x_min;
	m_xmax = x_max;
	m_xunit = x_unit;
	m_ymin = y_min;
	m_ymax = y_max;
	m_yunit = y_unit;

	switch (m_xunit)
	{
		case years:	m_increment = posix_time::hours(365 * 24); break;
		case months:	m_increment = posix_time::hours(30 * 24); break;
		case days:	m_increment = posix_time::hours(24); break;
		case hours:	m_increment = posix_time::hours(1); break;
		case minutes:	m_increment = posix_time::minutes(1); break;
		default:	m_increment = posix_time::seconds(1); break;
	}
}

void DiagramImpl::DrawAxes (Diagram::FoldInterval fi, double y_min, double y_max, double y_unit, RGB color, const string& text)
{
	// for folding we must handle dates in the year 9999, this is a bit
	// special but indeed no problem, we just need an according axis
	switch (fi)
	{
		case year:
		{
			m_xmin = ptime(date(9999,  Jan, 1), time_duration(0, 0, 0));
			m_xmax = ptime(date(10000, Jan, 1), time_duration(0, 0, 0));
			m_xunit = months;
			m_increment = posix_time::hours(30 * 24);
			break;
		}
		case month:
		{
			m_xmin = ptime(date(9999, Jan,  1), time_duration(0, 0, 0));
			m_xmax = ptime(date(9999, Feb,  1), time_duration(0, 0, 0));
			m_xunit = days;
			m_increment = posix_time::hours(24);
			break;
		}
		case day:
		{
			m_xmin = ptime(date(9999, Jan,  1), time_duration(0, 0, 0));
			m_xmax = ptime(date(9999, Jan,  2), time_duration(0, 0, 0));
			m_xunit = hours;
			m_increment = posix_time::hours(1);
			break;
		}
		case hour:
		{
			m_xmin = ptime(date(9999, Jan,  1), time_duration(0, 0, 0));
			m_xmax = ptime(date(9999, Jan,  1), time_duration(1, 0, 0));
			m_xunit = minutes;
			m_increment = posix_time::minutes(1);
			break;
		}
		case minute:
		{
			m_xmin = ptime(date(9999, Jan,  1), time_duration(0, 0, 0));
			m_xmax = ptime(date(9999, Jan,  1), time_duration(0, 1, 0));
			m_xunit = seconds;
			m_increment = posix_time::seconds(1);
			break;
		}
	}

	m_ymin = y_min;
	m_ymax = y_max;
	m_yunit = y_unit;
}

void DiagramImpl::DrawBars (const Selection& s, RGB color, int bar1, int bar2, const time_duration& midnight)
{
	const size_t* pDimensions = s.shape(); // [0] rows, [1] columns

	if (pDimensions[0] == 1 && any_cast<string>(s[0][0]).empty())
		throw Xception("selection contains just one single value which is the result of an aggregation, you must ensure the selection to have multiple values to draw bars");

	for (size_t row = 0; row < pDimensions[0]; row++)
	{
		Dot current(any_cast<string>(s[row][0]), s[row][1]);

		if (current)
		{
			ptime t(current.m_timestamp - midnight);
			{
				string s(lexical_cast<string>(t));
				switch (m_xunit)
				{
					case years:	s.erase(4);  s += "-01-01 00:00:00.000"; break;
					case days:	s.erase(10); s += " 00:00:00.000"; break;
					case hours:	s.erase(13); s += ":00:00.000"; break;
					case minutes:	s.erase(16); s += ":00.000"; break;
					default:	break;
				}
				t = time_from_string(s);
			}

			if (bar1 != 0 && bar2 != 0 && bar1 <= bar2)
				DrawBar(t, any_cast<double>(current.m_value), bar1, bar2, color, true);
			else
			{
				static bool darken = false;
				RGB c = (darken)
					? (size_t(double((color >> 16) & 0xFF) * 0.9) << 16) + (size_t(double((color >> 8) & 0xFF) * 0.9) << 8) + size_t(double(color & 0xFF) * 0.9)
					: color;
				darken = !darken;

				ptime tend = t + m_increment;
				vector<double> values;
				while (row < pDimensions[0] && (lexical_cast<ptime>(any_cast<string>(s[row][0])) - midnight) < tend)
					values.push_back(any_cast<double>(s[row++][1]));
				row--;

				int b2 = values.size();
				for (int b1 = 1; b1 <= b2; b1++)
					DrawBar(t, values[b1 - 1], b1, b2, c, false);
			}
		}
	}
}

string DiagramImpl::Fold (const string& timestamp, Diagram::FoldInterval fi)
{
	string result(timestamp);
	switch (fi)
	{
		case Diagram::year:	result.erase(0,  5); result.erase(2); break;
		case Diagram::month:	result.erase(0,  8); result.erase(2); break;
		case Diagram::day:	result.erase(0, 11); result.erase(2); break;
		case Diagram::hour:	result.erase(0, 14); result.erase(2); break;
		case Diagram::minute:	result.erase(0, 17); break;
	}
	return result;
}

string DiagramImpl::FormatY () const
{
	stringstream ss;
	ss << m_yunit;
	string s(ss.str());
	string::size_type pos = s.find('.');
	if (pos != string::npos)
	{
		s.erase(0, pos + 1);
		return (format("%%1.%df") % s.length()).str();
	}
	else
		return "%1.0f";
}
