/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2006, 2007, 2008, 2009, 2010  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-09-12 08:17:51 -0600 (Sun, 12 Sep 2010) $
--   $Revision: 4722 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  Also this file was created in the
--   alpha_05_branch so check the logs for this file in it too.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <assert.h>
#include <dirent.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "md5.h"
#include "sha1.h"
#include "sha256.h"
#include "sha512.h"

#include "crc32.h"
#include "disk_io.h"
#include "objectify.h"
#include "time_stamp.h"
#include "security.h"          /* define functions to read and write encrypted blocks */
#include "strlcxx.h"           /* in case strlcpy and strlcat are not provided by the system */


/* Map linux file modification time name in stat struct */
#ifndef HAVE_STRUCT_STAT_ST_MTIMESPEC
#define st_mtimespec st_mtim
#endif

#define DATA_STORAGE_SIZE (FILE_BLOCK_SIZE - 12)   /* 4 bytes each for flags, id, checksum */

typedef struct file_block {
    uint8 flags[4];
    ObjRef id;
    uint8 checksum[4];
    uint8 storage[DATA_STORAGE_SIZE];
} DataObject;



static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


static inline uint64 file_size_to_uint64(uint8 size[5])
{
    return ((uint64)size[0] << 32) |
	   ((uint64)size[1] << 24) |
	   ((uint64)size[2] << 16) |
	   ((uint64)size[3] << 8)  |
	    (uint64)size[4];
}


uint32 nwos_estimate_blocks_for_file(const char* path)
{
    struct stat stat_struct;
    uint32 result = 0;
    off_t num_blocks;

    /* if file fails to open just return zero */

    if (stat(path, &stat_struct) == 0)
    {
	num_blocks = stat_struct.st_size / DATA_STORAGE_SIZE + 14;

	num_blocks += num_blocks / 60;

	if (num_blocks < 0xffffffffLL)
	{
	    result = (uint32) num_blocks;
	}
	else
	{
	    result = 0xffffffff;
	}
    }

    return result;
}


/* This function copies a path string and removes any extra slashes, so there are no "//" or "." in the path.         */
/* It will not return with a trailing slash.  If there are more ".."'s in the path than subdirectories it will return */
/* the result.  Examples:                                                                                             */
/*                           "subdir1/./subdir2/"          -> "subdir1/subdir2"                                       */
/*                           "subdir1/subdir2/.."          -> "subdir1"                                               */
/*                           "subdir1/subdir2/../../"      -> "."                                                     */
/*                           "subdir1/subdir2/../../.."    -> ".."                                                    */
/*                           "subdir1/subdir2/../../../.." -> "../.."                                                 */

void nwos_normalize_path(char* dst, char* src, size_t size)
{
    int i = 0;

    assert(src[0] != '/');                                      /* cannot be an absolute path */
    assert(strlen(src) + 1 < size);

    while (src[0] == '.' && src[1] == '/') src += 2;

    while (*src != '\0')
    {
	assert(i < size);

	dst[i++] = *src++;

	while (dst[i-1] == '/' && *src == '/') src++;

	if (i == 2 && dst[0] == '.' && dst[1] == '/') i = 0;  /* eliminate any "./" in the path */ 

	if (i > 2 && dst[i-3] == '/' && dst[i-2] == '.' && dst[i-1] == '/') i = i - 2;  /* eliminate any "/./" in the path */ 

	if (i > 3 && dst[i-4] == '/' && dst[i-3] == '.' && dst[i-2] == '.' && dst[i-1] == '/') /* eliminate "/../" in the path */ 
	{
	    if (i < 6 || (i == 6 && memcmp(dst, "..", 2) != 0) || (i > 6 && memcmp(&dst[i-7], "/..", 3) != 0))
	    {
		i = i - 4;
		while (i > 0 && dst[i-1] != '/') i--;
	    }
	}

	assert(i < 2 || dst[0] != '.' || dst[1] != '/');
    }

    if (i > 0 && dst[i-2] == '/' && dst[i-1] == '.') i = i - 2;   /* eliminate "/." ending the path */ 

    if (i > 2 && dst[i-3] == '/' && dst[i-2] == '.' && dst[i-1] == '.')  /* eliminate "/.." ending the path */ 
    {
	if (i < 5 || (i == 5 && memcmp(dst, "..", 2) != 0) || (i > 5 && memcmp(&dst[i-6], "/..", 3) != 0))
	{
	    i = i - 3;
	    while (i > 0 && dst[i-1] != '/') i--;
	}
    }

    if (dst[i-1] == '/')
    {
	i--;
    }

    if (i == 0)
    {
	dst[i++] = '.';
    }

    assert(i < size);

    dst[i] = '\0';

    assert(dst[0] != '\0');      /* should not be an empty string */

    assert(dst[i-1] != '/');     /* should not end with a '/' */

    assert(dst[0] != '.' || dst[1] != '/');     /* if the first char is a dot it should not be ./ */

    char *p = dst;

    while (strlen(p) > 3 && memcmp(p, "../", 3) == 0) p += 3;

    if (strlen(p) > 3)
    {
	assert(strstr(p, "/../") == NULL);
    }

    assert(i < 2 || dst[i-1] != '.' || dst[i-2] != '/');   /* should not end with a "/." */
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* File path (name) object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static bool find_file_path(bool public, const char* path, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    ObjRef path_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    size_t length;
    int i;

    length = strlen(path);

    assert(length <= 255);

    void_reference(ref);   /* in case we don't find it */


    /* for now return false if file path private class doesn't exist yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("FILE PATH", &path_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("FILE PATH", &path_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("FILE PATH", &path_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&path_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &path_class_ref))
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_path_object_size));

	    /* remember ptr_path_obj points to the kludge buffer */

	    if (ptr_path_obj->count == length && strncmp((char*)ptr_path_obj->storage, path, length) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


bool nwos_find_public_file_path(const char* path, ObjRef* ref)
{
    return find_file_path(true, path, ref);
}


bool nwos_find_file_path(const char* path, ObjRef* ref)
{
    return find_file_path(nwos_in_public_mode(), path, ref);
}


/* returns the number of path and file association objects that refer to this path */
int nwos_number_of_files_for_path(ObjRef* path_ref)
{
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    C_struct_Path_And_File_Association assoc_obj;
    ObjRef assoc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    int result = 0;


    if (nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref))
    {
	assert(nwos_read_variable_sized_object_from_disk(path_ref, kludge, sizeof(kludge), &get_path_object_size));

	/* remember ptr_path_obj points to the kludge buffer */

	ref_list = nwos_malloc_reference_list(&ptr_path_obj->header.object.references);

	num_refs = ref_list->common_header.num_refs;

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&object_class, &assoc_class_ref))
	    {
		assert(nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj)));

		assert(is_same_object(&assoc_obj.path, path_ref));

		if (is_void_reference(&assoc_obj.header.object.next_version))
		{
		    result++;
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;   /* return number of path_and_file_associations we found that refer to this path */
}



/* Find existing path or create new */

ObjCreateResult nwos_create_file_path(const char* path, ObjRef* ref)
{
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    size_t length;
    ObjRef path_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    int i;
#ifndef PUBLIC_MODE
    ObjRef public_ref;
    char msg[128];
#endif

    length = strlen(path);

    assert(length <= 255);
    
    /* first find out if we already have this path */

    if (!nwos_find_file_path(path, ref))   /* didn't find it */
    {
	memset(kludge, 0, sizeof(kludge));  /* zero it out */

	/* remember ptr_path_obj points to the kludge buffer */

#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("FILE PATH", &path_class_ref));
#else
	nwos_find_or_create_private_class_definition("FILE PATH", &path_class_ref);
#endif

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_path_obj->header.common, ref, &path_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_file_path(path, &public_ref))
	{
	    copy_reference(&ptr_path_obj->header.object.clone_of, &public_ref);
	    snprintf(msg, sizeof(msg), "file path: %s  ref: %08x  clone_of: %08x",
		     path, nwos_ref_to_word(ref), nwos_ref_to_word(&public_ref));
	    nwos_log(msg);
	}
#endif
	ptr_path_obj->count = length;
	for (i = 0; i < length; i++) ptr_path_obj->storage[i] = path[i];

	nwos_create_reference_list(ref, &ptr_path_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_path_obj->header.object, sizeof(ObjectHeader), ptr_path_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_path_obj->count, sizeof(C_struct_File_Path) + length - sizeof(EveryObject), ptr_path_obj->header.common.data_chksum);

	nwos_write_object_to_disk(ref, kludge, sizeof(C_struct_File_Path) + length);

	nwos_add_to_references(ref, &path_class_ref);

#ifdef VERIFY_WRITE
	/* NOTE: these don't work in the mixed 32/64 bit reference environment, because nwos_write_block changes the upper 4 bytes of the reference */
	ptr_path_obj = malloc(sizeof(C_struct_File_Path) + length);
	assert(nwos_read_object_from_disk(ref, ptr_path_obj, sizeof(C_struct_File_Path) + length));
	assert(memcmp(kludge, ptr_path_obj, sizeof(C_struct_File_Path) + length) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_path_object_size));  /* read the other way */
	assert(memcmp(ptr_path_obj, kludge, sizeof(C_struct_File_Path) + length) == 0);

	free(ptr_path_obj);
	ptr_path_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}



bool nwos_file_path_to_string(ObjRef* ref, char* string, size_t size)
{
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* ptr_path_obj;
    int i;
    size_t path_obj_size;

    /* read the path object into the kludge buffer and then after we know what size it is malloc space and copy it there */
    assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_path_object_size));

    /* point to the kludge buffer temporarily to get the size of the object */
    ptr_path_obj = (C_struct_File_Path*)kludge;

    path_obj_size = sizeof(C_struct_File_Path) + ptr_path_obj->count;
    /* printf("path_obj_size: %d\n", path_obj_size); */
    assert(path_obj_size > sizeof(C_struct_File_Path));

    /* now we have the size allocate memory for it and copy it there */
    ptr_path_obj = malloc(path_obj_size);
    memcpy(ptr_path_obj, kludge, path_obj_size);

    for (i = 0; i < ptr_path_obj->count; i++) 
    {
	assert(i < size);
	string[i] = ptr_path_obj->storage[i];
    }
    string[i] = '\0';

    free(ptr_path_obj);

    return true;
}


bool nwos_file_path_to_path_and_file_association(ObjRef* path_ref, int index, ObjRef* assoc_ref)
{
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    ReferenceList* ref_list;
    int num_refs;
    ObjRef object_class;
    ObjRef assoc_class_ref;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_File file_obj;
    int i;
    int count;

    if (!nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref)) return false;


    /* read the path object into the kludge buffer and then after we know what size it is malloc space and copy it there */
    assert(nwos_read_variable_sized_object_from_disk(path_ref, kludge, sizeof(kludge), &get_path_object_size));

    ref_list = nwos_malloc_reference_list(&ptr_path_obj->header.object.references);

    num_refs = ref_list->common_header.num_refs;

    count = 0;

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj)));

	    assert(is_same_object(&assoc_obj.path, path_ref));

	    assert(nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj)));

	    /* don't count any that have newer versions */

	    if (is_void_reference(&file_obj.header.object.next_version))
	    {
		if (count == index)    /* this is the one we want */
		{
		    copy_reference(assoc_ref, &ref_list->references[i]);
		    break;
		}
		else
		{
		    count++;
		}
	    }
	}
    }

    //    memcpy(file_ref, &ptr_path_obj->file, sizeof(*file_ref));

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i < num_refs);
}


/*-------------------------------------------------------------------------------------------------------------------*/
/* MD5 object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static bool find_md5(bool public, uint8 md5sum[MD5_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    ObjRef md5_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    C_struct_MD5sum md5_object;
    int i;

    void_reference(ref);  /* in case we don't find it */


    /* for now return false if the md5sum private class doesn't exist yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("MD5SUM", &md5_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("MD5SUM", &md5_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("MD5SUM", &md5_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&md5_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &md5_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &md5_object, sizeof(md5_object)));

	    if (memcmp(md5_object.md5sum, md5sum, MD5_DIGEST_SIZE) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


bool nwos_find_public_md5(uint8 md5sum[MD5_DIGEST_SIZE], ObjRef* ref)
{
    return find_md5(true, md5sum, ref);
}


/* find an md5 based upon what mode we are in */

bool nwos_find_md5(uint8 md5sum[MD5_DIGEST_SIZE], ObjRef* ref)
{
    return find_md5(nwos_in_public_mode(), md5sum, ref);
}


/* Find existing MD5sum or create new */

ObjCreateResult nwos_create_md5(uint8 md5sum[MD5_DIGEST_SIZE], ObjRef* ref)
{
    ObjRef md5_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    C_struct_MD5sum md5_object;
#ifdef VERIFY_WRITE
    C_struct_MD5sum* ptr_md5_obj;
#endif
#ifndef PUBLIC_MODE
    ObjRef public_ref;
    char msg[128];

    assert(!nwos_in_public_mode());
#endif

    /* first find out if we already have this md5 */

    if (!nwos_find_md5(md5sum, ref))   /* didn't find it */
    {
#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("MD5SUM", &md5_class_ref));
#else
	nwos_find_or_create_private_class_definition("MD5SUM", &md5_class_ref);
#endif

	memset(&md5_object, 0, sizeof(md5_object));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&md5_object.header.common, ref, &md5_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_md5(md5sum, &public_ref))
	{
	    copy_reference(&md5_object.header.object.clone_of, &public_ref);
	    snprintf(msg, sizeof(msg), "md5sum - ref: %08x  clone_of: %08x",
		     nwos_ref_to_word(ref), nwos_ref_to_word(&public_ref));
	    nwos_log(msg);
	}
#endif
	memcpy(md5_object.md5sum, md5sum, sizeof(md5_object.md5sum));

	nwos_create_reference_list(ref, &md5_object.header.object.references);

	nwos_crc32_calculate((uint8*) &md5_object.header.object, sizeof(ObjectHeader), md5_object.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &md5_object.md5sum, sizeof(C_struct_MD5sum) - sizeof(EveryObject), md5_object.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &md5_object, sizeof(md5_object));

	nwos_add_to_references(ref, &md5_class_ref);

#ifdef VERIFY_WRITE
	/* NOTE: these don't work in the mixed 32/64 bit reference environment, because nwos_write_block changes the upper 4 bytes of the reference */
	ptr_md5_obj = malloc(sizeof(C_struct_MD5sum));
	assert(nwos_read_object_from_disk(ref, ptr_md5_obj, sizeof(C_struct_MD5sum)));
	assert(memcmp(&md5_object, ptr_md5_obj, sizeof(C_struct_MD5sum)) == 0);
	free(ptr_md5_obj);
	ptr_md5_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* SHA1 object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static bool find_sha1(bool public, uint8 sha1sum[SHA1_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    ObjRef sha1_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    C_struct_SHA1sum sha1_object;
    int i;

    void_reference(ref);  /* in case we don't find it */

    /* for now return false if the sha1sum private class doesn't exist yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("SHA1SUM", &sha1_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("SHA1SUM", &sha1_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("SHA1SUM", &sha1_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&sha1_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &sha1_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &sha1_object, sizeof(sha1_object)));

	    if (memcmp(sha1_object.sha1sum, sha1sum, SHA1_DIGEST_SIZE) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


bool nwos_find_public_sha1(uint8 sha1sum[SHA1_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha1(true, sha1sum, ref);
}


/* find an sha1 based upon what mode we are in */

bool nwos_find_sha1(uint8 sha1sum[SHA1_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha1(nwos_in_public_mode(), sha1sum, ref);
}



/* Find existing SHA1sum or create new */

ObjCreateResult nwos_create_sha1(uint8 sha1sum[SHA1_DIGEST_SIZE], ObjRef* ref)
{
    ObjRef sha1_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    C_struct_SHA1sum sha1_object;
#ifdef VERIFY_WRITE
    C_struct_SHA1sum* ptr_sha1_obj;
#endif
#ifndef PUBLIC_MODE
    ObjRef public_ref;
    char msg[128];

    assert(!nwos_in_public_mode());
#endif

    /* first find out if we already have this sha1 */

    if (!nwos_find_sha1(sha1sum, ref))   /* didn't find it */
    {
#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("SHA1SUM", &sha1_class_ref));
#else
	nwos_find_or_create_private_class_definition("SHA1SUM", &sha1_class_ref);
#endif

	memset(&sha1_object, 0, sizeof(sha1_object));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&sha1_object.header.common, ref, &sha1_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_sha1(sha1sum, &public_ref))
	{
	    copy_reference(&sha1_object.header.object.clone_of, &public_ref);
	    snprintf(msg, sizeof(msg), "sha1sum - ref: %08x  clone_of: %08x",
		     nwos_ref_to_word(ref), nwos_ref_to_word(&public_ref));
	    nwos_log(msg);
	}
#endif
	memcpy(sha1_object.sha1sum, sha1sum, sizeof(sha1_object.sha1sum));

	nwos_create_reference_list(ref, &sha1_object.header.object.references);

	nwos_crc32_calculate((uint8*) &sha1_object.header.object, sizeof(ObjectHeader), sha1_object.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &sha1_object.sha1sum, sizeof(C_struct_SHA1sum) - sizeof(EveryObject), sha1_object.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &sha1_object, sizeof(sha1_object));

	nwos_add_to_references(ref, &sha1_class_ref);

#ifdef VERIFY_WRITE
	/* NOTE: these don't work in the mixed 32/64 bit reference environment, because nwos_write_block changes the upper 4 bytes of the reference */
	ptr_sha1_obj = malloc(sizeof(C_struct_SHA1sum));
	assert(nwos_read_object_from_disk(ref, ptr_sha1_obj, sizeof(C_struct_SHA1sum)));
	assert(memcmp(&sha1_object, ptr_sha1_obj, sizeof(C_struct_SHA1sum)) == 0);
	free(ptr_sha1_obj);
	ptr_sha1_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* SHA256 object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static bool find_sha256(bool public, uint8 sha256sum[SHA256_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    ObjRef sha256_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    C_struct_SHA256sum sha256_object;
    int i;

    void_reference(ref);  /* in case we don't find it */

    /* for now return false if the sha256sum private class doesn't exist yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("SHA256SUM", &sha256_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("SHA256SUM", &sha256_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("SHA256SUM", &sha256_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&sha256_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &sha256_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &sha256_object, sizeof(sha256_object)));

	    if (memcmp(sha256_object.sha256sum, sha256sum, SHA256_DIGEST_SIZE) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


bool nwos_find_public_sha256(uint8 sha256sum[SHA256_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha256(true, sha256sum, ref);
}


/* find an sha256 based upon what mode we are in */

bool nwos_find_sha256(uint8 sha256sum[SHA256_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha256(nwos_in_public_mode(), sha256sum, ref);
}




/* Find existing SHA256sum or create new */

ObjCreateResult nwos_create_sha256(uint8 sha256sum[SHA256_DIGEST_SIZE], ObjRef* ref)
{
    ObjRef sha256_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    C_struct_SHA256sum sha256_object;
#ifdef VERIFY_WRITE
    C_struct_SHA256sum* ptr_sha256_obj;
#endif
#ifndef PUBLIC_MODE
    ObjRef public_ref;
    char msg[128];

    assert(!nwos_in_public_mode());
#endif

    /* first find out if we already have this sha256 */

    if (!nwos_find_sha256(sha256sum, ref))   /* didn't find it */
    {
#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("SHA256SUM", &sha256_class_ref));
#else
	nwos_find_or_create_private_class_definition("SHA256SUM", &sha256_class_ref);
#endif

	memset(&sha256_object, 0, sizeof(sha256_object));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&sha256_object.header.common, ref, &sha256_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_sha256(sha256sum, &public_ref))
	{
	    copy_reference(&sha256_object.header.object.clone_of, &public_ref);
	    snprintf(msg, sizeof(msg), "sha256sum - ref: %08x  clone_of: %08x",
		     nwos_ref_to_word(ref), nwos_ref_to_word(&public_ref));
	    nwos_log(msg);
	}
#endif
	memcpy(sha256_object.sha256sum, sha256sum, sizeof(sha256_object.sha256sum));

	nwos_create_reference_list(ref, &sha256_object.header.object.references);

	nwos_crc32_calculate((uint8*) &sha256_object.header.object, sizeof(ObjectHeader), sha256_object.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &sha256_object.sha256sum, sizeof(C_struct_SHA256sum) - sizeof(EveryObject), sha256_object.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &sha256_object, sizeof(sha256_object));

	nwos_add_to_references(ref, &sha256_class_ref);

#ifdef VERIFY_WRITE
	/* NOTE: these don't work in the mixed 32/64 bit reference environment, because nwos_write_block changes the upper 4 bytes of the reference */
	ptr_sha256_obj = malloc(sizeof(C_struct_SHA256sum));
	assert(nwos_read_object_from_disk(ref, ptr_sha256_obj, sizeof(C_struct_SHA256sum)));
	assert(memcmp(&sha256_object, ptr_sha256_obj, sizeof(C_struct_SHA256sum)) == 0);
	free(ptr_sha256_obj);
	ptr_sha256_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* SHA512 object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static bool find_sha512(bool public, uint8 sha512sum[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    ObjRef sha512_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    C_struct_SHA512sum sha512_object;
    int i;

    void_reference(ref);  /* in case we don't find it */

    /* for now return false if the sha512sum private class doesn't exist yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("SHA512SUM", &sha512_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("SHA512SUM", &sha512_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("SHA512SUM", &sha512_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&sha512_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &sha512_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &sha512_object, sizeof(sha512_object)));

	    if (memcmp(sha512_object.sha512sum, sha512sum, SHA512_DIGEST_SIZE) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


bool nwos_find_public_sha512(uint8 sha512sum[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha512(true, sha512sum, ref);
}


/* find an sha512 based upon what mode we are in */

bool nwos_find_sha512(uint8 sha512sum[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha512(nwos_in_public_mode(), sha512sum, ref);
}




/* Find existing SHA512sum or create new */

ObjCreateResult nwos_create_sha512(uint8 sha512sum[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    ObjRef sha512_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    C_struct_SHA512sum sha512_object;
#ifdef VERIFY_WRITE
    C_struct_SHA512sum* ptr_sha512_obj;
#endif
#ifndef PUBLIC_MODE
    ObjRef public_ref;
    char msg[128];

    assert(!nwos_in_public_mode());
#endif

    /* first find out if we already have this sha512 */

    if (!nwos_find_sha512(sha512sum, ref))   /* didn't find it */
    {
#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("SHA512SUM", &sha512_class_ref));
#else
	nwos_find_or_create_private_class_definition("SHA512SUM", &sha512_class_ref);
#endif

	memset(&sha512_object, 0, sizeof(sha512_object));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&sha512_object.header.common, ref, &sha512_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_sha512(sha512sum, &public_ref))
	{
	    copy_reference(&sha512_object.header.object.clone_of, &public_ref);
	    snprintf(msg, sizeof(msg), "sha512sum - ref: %08x  clone_of: %08x",
		     nwos_ref_to_word(ref), nwos_ref_to_word(&public_ref));
	    nwos_log(msg);
	}
#endif
	memcpy(sha512_object.sha512sum, sha512sum, sizeof(sha512_object.sha512sum));

	nwos_create_reference_list(ref, &sha512_object.header.object.references);

	nwos_crc32_calculate((uint8*) &sha512_object.header.object, sizeof(ObjectHeader), sha512_object.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &sha512_object.sha512sum, sizeof(C_struct_SHA512sum) - sizeof(EveryObject), sha512_object.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &sha512_object, sizeof(sha512_object));

	nwos_add_to_references(ref, &sha512_class_ref);

#ifdef VERIFY_WRITE
	/* NOTE: these don't work in the mixed 32/64 bit reference environment, because nwos_write_block changes the upper 4 bytes of the reference */
	ptr_sha512_obj = malloc(sizeof(C_struct_SHA512sum));
	assert(nwos_read_object_from_disk(ref, ptr_sha512_obj, sizeof(C_struct_SHA512sum)));
	assert(memcmp(&sha512_object, ptr_sha512_obj, sizeof(C_struct_SHA512sum)) == 0);
	free(ptr_sha512_obj);
	ptr_sha512_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* File object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static uint64 checksum_file(const char* path, uint8 md5_digest[MD5_DIGEST_SIZE],
			    uint8 sha1_digest[SHA1_DIGEST_SIZE],
			    uint8 sha256_digest[SHA256_DIGEST_SIZE],
			    uint8 sha512_digest[SHA512_DIGEST_SIZE])
{
    FILE* fp;
    struct md5_ctx md5_context;    /* MD5 checksum context */
    struct sha1_ctx sha1_context;
    struct sha256_ctx sha256_context;
    struct sha512_ctx sha512_context;
    size_t bytes_read;
    uint8 buffer[4096];
    uint64 file_length;

    fp = fopen(path, "rb");

    file_length = 0;

    md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */
    sha1_init_ctx(&sha1_context);
    sha256_init_ctx(&sha256_context);
    sha512_init_ctx(&sha512_context);

    bytes_read = sizeof(buffer);   /* go through loop at least once */

    while (bytes_read == sizeof(buffer))
    {
	bytes_read = fread(buffer, sizeof(uint8), sizeof(buffer), fp);

	if (bytes_read < sizeof(buffer))    /* eof or error */
	{
	    if (ferror(fp))
	    {
		perror(path);
		exit(1);            /* FIX THIS - needs to clean up and exit gracefully!! */
	    }
	}

	file_length = file_length + bytes_read;

	if (bytes_read > 0)   /* we have data to store */
	{
	    md5_process_bytes(buffer, bytes_read, &md5_context);    /* include this data in the md5 checksum */
	    sha1_process_bytes(buffer, bytes_read, &sha1_context);    /* include this data in the sha1 checksum */
	    sha256_process_bytes(buffer, bytes_read, &sha256_context); /* include this data in the sha256 checksum */
	    sha512_process_bytes(buffer, bytes_read, &sha512_context); /* include this data in the sha512 checksum */
	}
    }

    fclose(fp);

    md5_finish_ctx(&md5_context, md5_digest);   /* finish computing the md5 sum */
    sha1_finish_ctx(&sha1_context, sha1_digest);
    sha256_finish_ctx(&sha256_context, sha256_digest);
    sha512_finish_ctx(&sha512_context, sha512_digest);

    return file_length;
}



/* find a matching file from the md5sum */

static bool find_matching_file_from_size_and_checksums(bool public, uint64 file_size, uint8 md5_digest[MD5_DIGEST_SIZE], uint8 sha1_digest[SHA1_DIGEST_SIZE], uint8 sha256_digest[SHA256_DIGEST_SIZE], uint8 sha512_digest[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    ObjRef file_class_ref;
    ObjRef object_class;
    ObjRef md5sum_ref;
    ObjRef sha1sum_ref;
    ObjRef sha256sum_ref;
    ObjRef sha512sum_ref;
    uint64 size;
    EveryObject checksum_header;
    C_struct_File file_obj;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    int missing_checksums;   /* bit mask of any invalid checksums */
    bool result = false;
    static char* valid_strings[16] = { "MD5 SHA1 SHA256 SHA512", "SHA1 SHA256 SHA512",   "MD5 SHA256 SHA512",   "SHA256 and SHA512 only", 
				       "MD5 SHA1 SHA512",        "SHA1 and SHA512 only", "MD5 and SHA512 only", "SHA512 only", 
				       "MD5 SHA1 SHA256",        "SHA1 and SHA256 only", "MD5 and SHA256 only", "SHA256 only", 
				       "MD5 and SHA1 only",      "SHA1 only",            "MD5 only",            "BUG in system", };

    void_reference(ref);  /* in case we don't find it */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("FILE", &file_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("FILE", &file_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("FILE", &file_class_ref)) return false;
    }
#endif

    // NOTE: must run all of these (I.E. don't use || )
    if (find_md5(public, md5_digest, &md5sum_ref) | find_sha1(public, sha1_digest, &sha1sum_ref) | find_sha256(public, sha256_digest, &sha256sum_ref) | find_sha512(public, sha512_digest, &sha512sum_ref))
    {
	if (nwos_verbosity_level != 0)
	{
	    printf("Found existing checksums:\n");

	    printf("    MD5: %02x%02x%02x%02x\n", md5sum_ref.id[0],    md5sum_ref.id[1],    md5sum_ref.id[2],    md5sum_ref.id[3]);
	    printf("   SHA1: %02x%02x%02x%02x\n", sha1sum_ref.id[0],   sha1sum_ref.id[1],   sha1sum_ref.id[2],   sha1sum_ref.id[3]);
	    printf(" SHA256: %02x%02x%02x%02x\n", sha256sum_ref.id[0], sha256sum_ref.id[1], sha256sum_ref.id[2], sha256sum_ref.id[3]);
	    printf(" SHA512: %02x%02x%02x%02x\n", sha512sum_ref.id[0], sha512sum_ref.id[1], sha512sum_ref.id[2], sha512sum_ref.id[3]);
	}

	if (!is_void_reference(&sha512sum_ref))
	{
	    nwos_read_object_headers_from_disk(&sha512sum_ref, &checksum_header);
	}
	else if (!is_void_reference(&sha256sum_ref))
	{
	    nwos_read_object_headers_from_disk(&sha256sum_ref, &checksum_header);
	}
	else if (!is_void_reference(&sha1sum_ref))
	{
	    nwos_read_object_headers_from_disk(&sha1sum_ref, &checksum_header);
	}
	else if (!is_void_reference(&md5sum_ref))
	{
	    nwos_read_object_headers_from_disk(&md5sum_ref, &checksum_header);
	}

	ref_list = nwos_malloc_reference_list(&checksum_header.object.references);
	num_refs = ref_list->common_header.num_refs;

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&object_class, &file_class_ref))
	    {
		assert(nwos_read_object_from_disk(&ref_list->references[i], &file_obj, sizeof(file_obj)));

		size = file_size_to_uint64(file_obj.size);

		if (nwos_verbosity_level != 0) printf("size: %llu\n", size);

		if (size == file_size || (public & (size == 0)))
		{
		    if (nwos_verbosity_level != 0)
		    {
			printf("   file: %08x\n", nwos_ref_to_word(&ref_list->references[i]));
			printf("    MD5: %08x\n", nwos_ref_to_word(&file_obj.md5sum));
			printf("   SHA1: %08x\n", nwos_ref_to_word(&file_obj.sha1sum));
			printf(" SHA256: %08x\n", nwos_ref_to_word(&file_obj.sha256sum));
			printf(" SHA512: %08x\n", nwos_ref_to_word(&file_obj.sha512sum));
		    }

		    if ((is_void_reference(&file_obj.md5sum) != is_void_reference(&md5sum_ref)) |
			(is_void_reference(&file_obj.sha1sum) != is_void_reference(&sha1sum_ref)) |
			(is_void_reference(&file_obj.sha256sum) != is_void_reference(&sha256sum_ref)) |
			(is_void_reference(&file_obj.sha512sum) != is_void_reference(&sha512sum_ref)))
		    {
			fprintf(stderr, "\n");
			fprintf(stderr, "BUG: void references in File object don't match checksums found:");
			fprintf(stderr, "    MD5: %08x-%08x\n", nwos_ref_to_word(&file_obj.md5sum), nwos_ref_to_word(&md5sum_ref));
			fprintf(stderr, "   SHA1: %08x-%08x\n", nwos_ref_to_word(&file_obj.sha1sum), nwos_ref_to_word(&sha1sum_ref));
			fprintf(stderr, " SHA256: %08x-%08x\n", nwos_ref_to_word(&file_obj.sha256sum), nwos_ref_to_word(&sha256sum_ref));
			fprintf(stderr, " SHA512: %08x-%08x\n", nwos_ref_to_word(&file_obj.sha512sum), nwos_ref_to_word(&sha512sum_ref));
		    }

		    if (is_same_object(&file_obj.md5sum, &md5sum_ref) &&
			is_same_object(&file_obj.sha1sum, &sha1sum_ref) &&
			is_same_object(&file_obj.sha256sum, &sha256sum_ref) &&
			is_same_object(&file_obj.sha512sum, &sha512sum_ref))
		    {
			missing_checksums = is_void_reference(&file_obj.md5sum);
			missing_checksums |= is_void_reference(&file_obj.sha1sum) << 1;
			missing_checksums |= is_void_reference(&file_obj.sha256sum) << 2;
			missing_checksums |= is_void_reference(&file_obj.sha512sum) << 3;

			if (missing_checksums != 0)
			{
			    printf("[%s] ", valid_strings[missing_checksums]);
			}

			copy_reference(ref, &ref_list->references[i]);   /* run with it */
			break;
		    }
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;

	result = i < num_refs;
    }

    return result;
}


bool nwos_find_public_matching_file_from_size_and_checksums(uint64 file_size, uint8 md5_digest[MD5_DIGEST_SIZE], uint8 sha1_digest[SHA1_DIGEST_SIZE], uint8 sha256_digest[SHA256_DIGEST_SIZE], uint8 sha512_digest[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    return find_matching_file_from_size_and_checksums(true, file_size, md5_digest, sha1_digest, sha256_digest, sha512_digest, ref);
}


/* find an sha512 based upon what mode we are in */

bool nwos_find_matching_file_from_size_and_checksums(uint64 file_size, uint8 md5_digest[MD5_DIGEST_SIZE], uint8 sha1_digest[SHA1_DIGEST_SIZE], uint8 sha256_digest[SHA256_DIGEST_SIZE], uint8 sha512_digest[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    return find_matching_file_from_size_and_checksums(nwos_in_public_mode(), file_size, md5_digest, sha1_digest, sha256_digest, sha512_digest, ref);
}



/* check to see if the file given has the data stored in the system */

bool nwos_file_is_stored(ObjRef* assoc_ref)
{
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_File file_obj;

    assert(nwos_read_object_from_disk(assoc_ref, &assoc_obj, sizeof(assoc_obj)));

    assert(nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj)));

    /* if the block_list is not empty then the data is stored */

    return !is_void_reference(&file_obj.block_list);
}


/* Create a file record with just an MD5 checksum, for Public Objects */

#ifdef PUBLIC_MODE
ObjCreateResult nwos_create_file_with_only_md5(const char* file_name, uint8 md5_digest[MD5_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_File file_obj;
    ObjRef file_class_ref;
    ObjRef path_ref;
    ObjRef file_ref;
    TimeStamp mod_time;

    assert(nwos_find_public_class_definition("FILE", &file_class_ref));

    assert(nwos_create_file_path(file_name, &path_ref) == CREATED_NEW);

    nwos_generate_new_id(&file_ref);

    memset(&file_obj, 0, sizeof(file_obj));  /* zero it out */

    nwos_fill_in_common_header(&file_obj.header.common, &file_ref, &file_class_ref);

    nwos_create_reference_list(&file_ref, &file_obj.header.object.references);

    assert(nwos_create_md5(md5_digest, &file_obj.md5sum) == CREATED_NEW);

    nwos_crc32_calculate((uint8*) &file_obj.header.object, sizeof(ObjectHeader), file_obj.header.common.header_chksum);
    nwos_crc32_calculate((uint8*) &file_obj.size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj.header.common.data_chksum);

    nwos_write_object_to_disk(&file_ref, &file_obj, sizeof(file_obj));

    nwos_add_to_references(&file_ref, &file_obj.md5sum);   /* add to this md5sum object's reference list */

    nwos_add_to_references(&file_ref, &file_class_ref);

    memset(mod_time, 0, sizeof(mod_time));

    return nwos_find_or_create_path_and_file_association(&path_ref, &file_ref, mod_time, ref);
}


ObjCreateResult nwos_create_file_with_duplicate_md5(const char* file_name, uint8 md5_digest[MD5_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_MD5sum md5_obj;
    C_struct_File file_obj;
    ObjRef file_class_ref;
    ObjRef path_ref;
    ObjRef md5_ref;
    ObjRef file_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    TimeStamp mod_time;

    assert(nwos_find_public_class_definition("FILE", &file_class_ref));

    assert(nwos_create_file_path(file_name, &path_ref) == CREATED_NEW);

    assert(nwos_find_public_md5(md5_digest, &md5_ref));

    /* find the file object for this MD5 sum */

    assert(nwos_read_object_from_disk(&md5_ref, &md5_obj, sizeof(md5_obj)));

    ref_list = nwos_malloc_reference_list(&md5_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    void_reference(&file_ref);

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &file_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &file_obj, sizeof(file_obj)));

	    assert(is_same_object(&file_obj.md5sum, &md5_ref));

	    copy_reference(&file_ref, &ref_list->references[i]);

	    break;
	}
    }

    assert(i < num_refs);
    assert(!is_void_reference(&file_ref));

    /* for now, verify that there was only one file object for this md5sum */

    for (i++ ; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	assert(!is_same_object(&object_class, &file_class_ref));
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    memset(mod_time, 0, sizeof(mod_time));

    return nwos_find_or_create_path_and_file_association(&path_ref, &file_ref, mod_time, ref);
}


ObjCreateResult nwos_create_file_with_only_sha1(const char* file_name, uint8 sha1_digest[SHA1_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_File file_obj;
    ObjRef file_class_ref;
    ObjRef path_ref;
    ObjRef file_ref;
    TimeStamp mod_time;

    assert(nwos_find_public_class_definition("FILE", &file_class_ref));

    assert(nwos_create_file_path(file_name, &path_ref) == CREATED_NEW);

    nwos_generate_new_id(&file_ref);

    memset(&file_obj, 0, sizeof(file_obj));  /* zero it out */

    nwos_fill_in_common_header(&file_obj.header.common, &file_ref, &file_class_ref);

    nwos_create_reference_list(&file_ref, &file_obj.header.object.references);

    assert(nwos_create_sha1(sha1_digest, &file_obj.sha1sum) == CREATED_NEW);

    nwos_crc32_calculate((uint8*) &file_obj.header.object, sizeof(ObjectHeader), file_obj.header.common.header_chksum);
    nwos_crc32_calculate((uint8*) &file_obj.size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj.header.common.data_chksum);

    nwos_write_object_to_disk(&file_ref, &file_obj, sizeof(file_obj));

    nwos_add_to_references(&file_ref, &file_obj.sha1sum);   /* add to this sha1sum object's reference list */

    nwos_add_to_references(&file_ref, &file_class_ref);

    memset(mod_time, 0, sizeof(mod_time));

    return nwos_find_or_create_path_and_file_association(&path_ref, &file_ref, mod_time, ref);
}


ObjCreateResult nwos_create_file_with_duplicate_sha1(const char* file_name, uint8 sha1_digest[SHA1_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_SHA1sum sha1_obj;
    C_struct_File file_obj;
    ObjRef file_class_ref;
    ObjRef path_ref;
    ObjRef sha1_ref;
    ObjRef file_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    TimeStamp mod_time;

    assert(nwos_find_public_class_definition("FILE", &file_class_ref));

    assert(nwos_create_file_path(file_name, &path_ref) == CREATED_NEW);

    assert(nwos_find_public_sha1(sha1_digest, &sha1_ref));

    /* find the file object for this SHA1 sum */

    assert(nwos_read_object_from_disk(&sha1_ref, &sha1_obj, sizeof(sha1_obj)));

    ref_list = nwos_malloc_reference_list(&sha1_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    void_reference(&file_ref);

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &file_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &file_obj, sizeof(file_obj)));

	    assert(is_same_object(&file_obj.sha1sum, &sha1_ref));

	    copy_reference(&file_ref, &ref_list->references[i]);

	    break;
	}
    }

    assert(i < num_refs);
    assert(!is_void_reference(&file_ref));

    /* for now, verify that there was only one file object for this sha1sum */

    for (i++ ; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	assert(!is_same_object(&object_class, &file_class_ref));
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    memset(mod_time, 0, sizeof(mod_time));

    return nwos_find_or_create_path_and_file_association(&path_ref, &file_ref, mod_time, ref);
}


ObjCreateResult nwos_add_info_to_existing_file(const char* file_name, ObjRef* assoc_ref)
{
    struct stat stat_struct;
    C_struct_Class_Definition class_def_obj;
    C_struct_File file_obj;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_MD5sum md5_obj;
    ObjRef assoc_class_ref;
    ObjRef path_ref;
    ObjRef object_class;
    uint64 file_length;
    TimeStamp mod_time;
    ReferenceList* ref_list;
    int num_refs;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    uint8 sha1_digest[SHA1_DIGEST_SIZE];
    uint8 sha256_digest[SHA256_DIGEST_SIZE];
    uint8 sha512_digest[SHA512_DIGEST_SIZE];
    int i;

    if (stat(file_name, &stat_struct) != 0)
    {
	perror(file_name);
	return ERROR_OCCURRED;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return ERROR_OCCURRED;
    }

    if (stat_struct.st_size > 1099511627775LL)
    {
	fprintf(stderr, "Cannot handle files larger than 1 terabyte at this time\n");
	return ERROR_OCCURRED;
    }

    if (!nwos_find_file_path(file_name, &path_ref))
    {
	fprintf(stderr, "%s is not in the public objects", file_name);
	return ERROR_OCCURRED;
    }

    assert(nwos_find_public_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));

    nwos_read_class_definition(&assoc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("association num_refs: %d\n", num_refs);

    void_reference(assoc_ref);

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj)));

	    if (is_same_object(&assoc_obj.path, &path_ref))
	    {
		assert(is_void_reference(assoc_ref));   /* error if there is more than one for now */

		copy_reference(assoc_ref, &ref_list->references[i]);
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    printf("Association: %08x\n", nwos_ref_to_word(assoc_ref));

    assert(!is_void_reference(assoc_ref));   /* path found but no association?? */

    assert(nwos_read_object_from_disk(assoc_ref, &assoc_obj, sizeof(assoc_obj)));

    assert(is_same_object(&assoc_obj.path, &path_ref));

    memset(mod_time, 0, sizeof(mod_time));
    assert(memcmp(assoc_obj.modification_time, mod_time, sizeof(mod_time)) == 0);

    /* first put the modification time into the association object */

    nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, assoc_obj.modification_time);

    nwos_crc32_calculate((uint8*) &assoc_obj.path, sizeof(C_struct_Path_And_File_Association) - sizeof(EveryObject), assoc_obj.header.common.data_chksum);

    nwos_overwrite_object_to_disk(assoc_ref, &assoc_obj, sizeof(assoc_obj));

    /* now do the file stuff */
    
    assert(nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj)));

    assert(!is_void_reference(&file_obj.md5sum));
    assert(is_void_reference(&file_obj.sha1sum));
    assert(is_void_reference(&file_obj.sha256sum));
    assert(is_void_reference(&file_obj.sha512sum));
    assert((file_obj.size[0] | file_obj.size[1] | file_obj.size[2] | file_obj.size[3] | file_obj.size[4]) == 0);


    printf("reading file: %s ", file_name);
    fflush(stdout);

    file_length = checksum_file(file_name, md5_digest, sha1_digest, sha256_digest, sha512_digest);

    printf("length: %llu\n", file_length);

    nwos_log_md5sum(file_name, md5_digest);
    nwos_log_sha1sum(file_name, sha1_digest);
    nwos_log_sha256sum(file_name, sha256_digest);
    nwos_log_sha512sum(file_name, sha512_digest);

    assert(nwos_read_object_from_disk(&file_obj.md5sum, &md5_obj, sizeof(md5_obj)));

    if (memcmp(md5_obj.md5sum, md5_digest, sizeof(md5_digest)) != 0)
    {
	fprintf(stderr, "Error: md5sum of file doesn't match MD5 in file:\n");

	fprintf(stderr, "   ");
	for (i = 0; i < MD5_DIGEST_SIZE; i++) fprintf(stderr, "%02x", md5_obj.md5sum[i]);
	fprintf(stderr, " from MD5SUMS file\n");

	fprintf(stderr, "   ");
	for (i = 0; i < MD5_DIGEST_SIZE; i++) fprintf(stderr, "%02x", md5_digest[i]);
	fprintf(stderr, " %s\n", file_name);

	return ERROR_OCCURRED;
    }

    /* store file size as 4 bytes in big endian order */
    file_obj.size[0] = file_length >> 32;
    file_obj.size[1] = file_length >> 24;
    file_obj.size[2] = file_length >> 16;
    file_obj.size[3] = file_length >> 8;
    file_obj.size[4] = file_length;

    if (nwos_create_sha1(sha1_digest, &file_obj.sha1sum) != CREATED_NEW)
    {
	fprintf(stderr, "WARNING: SHA1 already existed: %08x\n", nwos_ref_to_word(&file_obj.sha1sum));
    }

    if (nwos_create_sha256(sha256_digest, &file_obj.sha256sum) != CREATED_NEW)
    {
	fprintf(stderr, "WARNING: SHA256 already existed: %08x\n", nwos_ref_to_word(&file_obj.md5sum));
    }

    if (nwos_create_sha512(sha512_digest, &file_obj.sha512sum) != CREATED_NEW)
    {
	fprintf(stderr, "WARNING: SHA512 already existed: %08x\n", nwos_ref_to_word(&file_obj.md5sum));
    }

    nwos_crc32_calculate((uint8*) &file_obj.size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj.header.common.data_chksum);

    nwos_overwrite_object_to_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));

    nwos_add_to_references(&assoc_obj.file, &file_obj.sha1sum);   /* add to this sha1sum object's reference list */
    nwos_add_to_references(&assoc_obj.file, &file_obj.sha256sum);  /* add to this sha256sum object's reference list */
    nwos_add_to_references(&assoc_obj.file, &file_obj.sha512sum);  /* add to this sha512sum object's reference list */

    return FOUND_EXISTING;
}
#endif


/* Create a file record (doesn't store data of file, just time, md5, etc.) */

ObjCreateResult nwos_create_file_without_storing_data(const char* directory, const char* file_name, ObjRef* ref)
{
    size_t length;
    struct stat stat_struct;
    ObjRef file_class_ref;
    ObjRef path_ref;
    ObjRef file_ref;
    uint64 file_length;
    C_struct_File file_obj;
    TimeStamp mod_time;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    uint8 sha1_digest[SHA1_DIGEST_SIZE];
    uint8 sha256_digest[SHA256_DIGEST_SIZE];
    uint8 sha512_digest[SHA512_DIGEST_SIZE];
    char path[PATH_MAX];
#ifndef PUBLIC_MODE
    ObjRef public_file_ref;
    char msg[128];
#endif

    length = strlcpy(path, directory, PATH_MAX);
    if (length + 1 >= PATH_MAX)
    {
	fprintf(stderr, "path too long (> %u bytes): %s\n", PATH_MAX, directory);
	return ERROR_OCCURRED;
    }

    if (path[length - 1] != '/')   /* the directory didn't end with a '/' so add one */
    {
	path[length] = '/';
	path[length + 1] = '\0';
    }

    if (strlcat(path, file_name, PATH_MAX) >= PATH_MAX)
    {
	fprintf(stderr, "path too long (> %u bytes): %s/%s\n", PATH_MAX, directory, file_name);
	return ERROR_OCCURRED;
    }

    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return ERROR_OCCURRED;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return ERROR_OCCURRED;
    }

    if (stat_struct.st_size > 1099511627775LL)
    {
	fprintf(stderr, "Cannot handle files larger than 1 terabyte at this time\n");
	return ERROR_OCCURRED;
    }


    length = strlen(file_name);

    assert(length <= 255);
    
    printf("creating file record: %s ", file_name);
    fflush(stdout);

    file_length = checksum_file(path, md5_digest, sha1_digest, sha256_digest, sha512_digest);

    printf("length: %llu\n", file_length);

    nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

    nwos_log_md5sum(file_name, md5_digest);
    nwos_log_sha1sum(file_name, sha1_digest);
    nwos_log_sha256sum(file_name, sha256_digest);
    nwos_log_sha512sum(file_name, sha512_digest);


    /* see if there is a file that matches completely */

    if (!nwos_find_matching_file_from_size_and_checksums(file_length, md5_digest, sha1_digest, sha256_digest, sha512_digest, &file_ref))
    {
	nwos_generate_new_id(&file_ref);

	memset(&file_obj, 0, sizeof(file_obj));  /* zero it out */

#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("FILE", &file_class_ref));
#else
	nwos_find_or_create_private_class_definition("FILE", &file_class_ref);
#endif

	nwos_fill_in_common_header(&file_obj.header.common, &file_ref, &file_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_matching_file_from_size_and_checksums(file_length, md5_digest, sha1_digest, sha256_digest, sha512_digest, &public_file_ref))
	{
	    copy_reference(&file_obj.header.object.clone_of, &public_file_ref);
	    snprintf(msg, sizeof(msg), "file - ref: %08x  clone_of: %08x",
		     nwos_ref_to_word(ref), nwos_ref_to_word(&public_file_ref));
	    nwos_log(msg);
	}
#endif
	/* store file size as 4 bytes in big endian order */
	file_obj.size[0] = file_length >> 32;
	file_obj.size[1] = file_length >> 24;
	file_obj.size[2] = file_length >> 16;
	file_obj.size[3] = file_length >> 8;
	file_obj.size[4] = file_length;

	nwos_create_md5(md5_digest, &file_obj.md5sum);

	nwos_create_sha1(sha1_digest, &file_obj.sha1sum);

	nwos_create_sha256(sha256_digest, &file_obj.sha256sum);
	nwos_create_sha512(sha512_digest, &file_obj.sha512sum);

	nwos_create_reference_list(&file_ref, &file_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &file_obj.header.object, sizeof(ObjectHeader), file_obj.header.common.header_chksum);
	nwos_crc32_calculate((uint8*) &file_obj.size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj.header.common.data_chksum);

	nwos_write_object_to_disk(&file_ref, &file_obj, sizeof(file_obj));

	nwos_add_to_references(&file_ref, &file_obj.md5sum);   /* add to this md5sum object's reference list */
	nwos_add_to_references(&file_ref, &file_obj.sha1sum);   /* add to this sha1sum object's reference list */
	nwos_add_to_references(&file_ref, &file_obj.sha256sum);  /* add to this sha256sum object's reference list */
	nwos_add_to_references(&file_ref, &file_obj.sha512sum);  /* add to this sha512sum object's reference list */

	nwos_add_to_references(&file_ref, &file_class_ref);
    }

    /* at this point we have a file object, and we need to either find or create a path object */

    assert(nwos_create_file_path(file_name, &path_ref) != ERROR_OCCURRED);

    return nwos_find_or_create_path_and_file_association(&path_ref, &file_ref, mod_time, ref);
}


static uint64 read_file(const char* path,
			uint8 md5_digest[MD5_DIGEST_SIZE],
			uint8 sha1_digest[SHA1_DIGEST_SIZE],
			uint8 sha256_digest[SHA256_DIGEST_SIZE],
			uint8 sha512_digest[SHA512_DIGEST_SIZE],
			ObjRef* block_list_ref)
{
    FILE* fp;
    size_t bytes_read;
    uint64 file_length;
    ObjRef data_block_ref;
    DataObject data_obj;
    struct md5_ctx md5_context;    /* MD5 checksum context */
    struct sha1_ctx sha1_context;
    struct sha256_ctx sha256_context;
    struct sha512_ctx sha512_context;
    uint8 ivec[IVEC_SIZE];
    int seq = 0;
    char msg[128];


    memset(ivec, 0, sizeof(ivec));

    /* create a reference list that is a list of the data blocks for the file */
    nwos_create_reference_list(NULL, block_list_ref);

    snprintf(msg, sizeof(msg), "read_file %s  block_list: %02x%02x%02x%02x", path,
	   block_list_ref->id[0], block_list_ref->id[1], block_list_ref->id[2], block_list_ref->id[3]);
    nwos_log(msg);


    /* now open the file and start creating the data block objects */

    fp = fopen(path, "rb");

    file_length = 0;

    md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */
    sha1_init_ctx(&sha1_context);
    sha256_init_ctx(&sha256_context);
    sha512_init_ctx(&sha512_context);

    memset(data_obj.flags, 0, sizeof(data_obj.flags));

    bytes_read = DATA_STORAGE_SIZE;   /* go through loop at least once */

    while (bytes_read == DATA_STORAGE_SIZE)
    {
	bytes_read = fread(data_obj.storage, sizeof(uint8), DATA_STORAGE_SIZE, fp);

	if (bytes_read < DATA_STORAGE_SIZE)    /* eof or error */
	{
	    if (ferror(fp))
	    {
		perror(path);
		exit(1);            /* FIX THIS - needs to clean up and exit gracefully!! */
	    }
	}

	file_length = file_length + bytes_read;

	if (bytes_read > 0)   /* we have data to store */
	{
	    md5_process_bytes(data_obj.storage, bytes_read, &md5_context);    /* include this data in the md5 checksum */
	    sha1_process_bytes(data_obj.storage, bytes_read, &sha1_context);    /* include this data in the sha1 checksum */
	    sha256_process_bytes(data_obj.storage, bytes_read, &sha256_context);    /* include this data in the sha256 checksum */
	    sha512_process_bytes(data_obj.storage, bytes_read, &sha512_context);    /* include this data in the sha512 checksum */

	    nwos_generate_new_id(&data_block_ref);

	    memcpy(&data_obj.id, &data_block_ref, sizeof(ObjRef));

#if 0
	    printf("data block seq: %d ref: %02x%02x%02x%02x\n", seq, data_block_ref.id[0], data_block_ref.id[1], data_block_ref.id[2], data_block_ref.id[3]);
#endif

	    nwos_crc32_calculate((uint8*) &data_obj.storage, bytes_read, data_obj.checksum);

	    /* fill in the remainder with random data */
	    while (bytes_read < DATA_STORAGE_SIZE) data_obj.storage[bytes_read++] = random();

	    nwos_write_block_to_disk_and_encrypt(&data_block_ref, &data_obj, sizeof(data_obj), ivec, seq);

	    nwos_add_to_reference_list(&data_block_ref, block_list_ref);

	    seq++;
	}
    }

    fclose(fp);

    snprintf(msg, sizeof(msg), "length: %llu", file_length);
    nwos_log(msg);
    printf("%s\n", msg);

    md5_finish_ctx(&md5_context, md5_digest);   /* finish computing the md5 sum */
    sha1_finish_ctx(&sha1_context, sha1_digest);
    sha256_finish_ctx(&sha256_context, sha256_digest);
    sha512_finish_ctx(&sha512_context, sha512_digest);

    return file_length;
}



/* Create new file - returns FOUND_EXISTING if an exact file match already existed      */
/*          (exact match meaning size, md5, sha1, date, and path are all identical).    */
/*                                                                                      */
/* There are several possible cases of duplication that have to be dealt with:          */
/*   Path is the same but the file is different (file with same name as another file).  */
/*   File is the same but the name is different (copy of a file with a different name). */
/*   Path and File are the same but the date is different.                              */
/*   Everything is identical, but the file isn't stored in the system yet.              */
/*   Everything is identical.                                                           */
/*                                                                                      */
/*  Matching  Matching  Matching  Matching   Data    Allow      Action      Return                            */
/*    Path     Assoc      Time      File    Stored  Multiple                                                  */
/*     T         T         X         T        T        X      Do Nothing  FOUND_EXISTING                      */
/*     T         T         X         T        F        X      Import Data IMPORTED_DATA                       */

/*     T         F         X         T        X        X      Do Nothing  DUPLICATE_FILE                      */

/*     F         X         X         T        X        T      Create Path CREATED_NEW_PATH_FOR_EXISTING_FILE  */
/*     F         X         X         T        X        F      Do Nothing  DUPLICATE_FILE                      */

/*     T         X         X         F        X        X      Do Nothing  DUPLICATE_PATH                      */
/*     F         X         X         F        X        X      Create all  CREATED_NEW                         */
/*                                                                                                            */

ObjCreateResult nwos_create_file(const char* path, ObjRef* ref, bool allow_multiple_paths)
{
    size_t length;
    struct stat stat_struct;
    bool file_found;
    bool path_found;
    ObjRef file_class_ref;
    ObjRef path_ref;
    ObjRef file_ref;
    uint64 file_length;
    C_struct_File file_obj;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    uint8 sha1_digest[SHA1_DIGEST_SIZE];
    uint8 sha256_digest[SHA256_DIGEST_SIZE];
    uint8 sha512_digest[SHA512_DIGEST_SIZE];
    uint8 md5_digest2[MD5_DIGEST_SIZE];
    uint8 sha1_digest2[SHA1_DIGEST_SIZE];
    uint8 sha256_digest2[SHA256_DIGEST_SIZE];
    uint8 sha512_digest2[SHA512_DIGEST_SIZE];
    TimeStamp mod_time;

    void_reference(ref);   /* in case it fails */


    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return ERROR_OCCURRED;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return ERROR_OCCURRED;
    }

    if (stat_struct.st_size > 8 * 1073741824LL)
    {
	fprintf(stderr, "Cannot store files larger than 8 gigabytes at this time\n");
	return ERROR_OCCURRED;
    }

    length = strlen(path);

    assert(length <= 255);


    /* first find out if we already have this file stored */

    file_length = checksum_file(path, md5_digest, sha1_digest, sha256_digest, sha512_digest);

    file_found = nwos_find_matching_file_from_size_and_checksums(file_length, md5_digest, sha1_digest, sha256_digest, sha512_digest, &file_ref);


    /* second find out if we already have this path */

    path_found = nwos_find_file_path(path, &path_ref);


    if (file_found && path_found)
    {
	if (nwos_find_path_and_file_association(&path_ref, &file_ref, NULL, ref))
	{
	    assert(nwos_read_object_from_disk(&file_ref, &file_obj, sizeof(file_obj))); /* make sure we already have the data */

	    if (is_void_reference(&file_obj.block_list))   /* file data is not stored in the system */
	    {
		file_length = read_file(path, md5_digest2, sha1_digest2, sha256_digest2, sha512_digest2, &file_obj.block_list);

		/* verify that the checksums match what they were before */
		assert(memcmp(md5_digest, md5_digest2, sizeof(md5_digest)) == 0);
		assert(memcmp(sha1_digest, sha1_digest2, sizeof(sha1_digest)) == 0);
		assert(memcmp(sha256_digest, sha256_digest2, sizeof(sha256_digest)) == 0);
		assert(memcmp(sha512_digest, sha512_digest2, sizeof(sha512_digest)) == 0);

		if (is_void_reference(&file_obj.md5sum))
		{
		    nwos_create_md5(md5_digest, &file_obj.md5sum);
		    nwos_add_to_references(&file_ref, &file_obj.md5sum);   /* add to the md5sum object's reference list */
		    nwos_log_md5sum(path, md5_digest);
		}

		if (is_void_reference(&file_obj.sha1sum))
		{
		    nwos_create_sha1(sha1_digest, &file_obj.sha1sum);
		    nwos_add_to_references(&file_ref, &file_obj.sha1sum);   /* add to the sha1sum object's reference list */
		    nwos_log_sha1sum(path, sha1_digest);
		}

		if (is_void_reference(&file_obj.sha256sum))
		{
		    nwos_create_sha256(sha256_digest, &file_obj.sha256sum);
		    nwos_add_to_references(&file_ref, &file_obj.sha256sum); /* add to the sha256sum object's reference list */
		    nwos_log_sha256sum(path, sha256_digest);
		}

		if (is_void_reference(&file_obj.sha512sum))
		{
		    nwos_create_sha512(sha512_digest, &file_obj.sha512sum);
		    nwos_add_to_references(&file_ref, &file_obj.sha512sum); /* add to the sha512sum object's reference list */
		    nwos_log_sha512sum(path, sha512_digest);
		}

		nwos_crc32_calculate((uint8*) &file_obj.size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj.header.common.data_chksum);

		nwos_overwrite_object_to_disk(&file_ref, &file_obj, sizeof(file_obj));

		return IMPORTED_DATA;
	    }

	    return FOUND_EXISTING;
	}

	return DUPLICATE_FILE;
    }
    else if (file_found)   /* matching file was found, but path was not */
    {
	if (allow_multiple_paths)
	{
	    assert(nwos_create_file_path(path, &path_ref) == CREATED_NEW);

	    nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

	    assert(nwos_find_or_create_path_and_file_association(&path_ref, &file_ref, mod_time, ref) == CREATED_NEW);

	    return CREATED_NEW_PATH_FOR_EXISTING_FILE;
	}

	return DUPLICATE_FILE;
    }
    else if (path_found)   /* matching path was found, but file was not */
    {
	return DUPLICATE_PATH;
    }
    else
    {
	printf("creating file: %s ", path);
	fflush(stdout);

	assert(nwos_create_file_path(path, &path_ref) == CREATED_NEW);

	memset(&file_obj, 0, sizeof(file_obj));  /* zero it out */

	nwos_find_or_create_private_class_definition("FILE", &file_class_ref);

	nwos_generate_new_id(&file_ref);

	nwos_fill_in_common_header(&file_obj.header.common, &file_ref, &file_class_ref);

	file_length = read_file(path, md5_digest, sha1_digest, sha256_digest, sha512_digest, &file_obj.block_list);

	nwos_create_md5(md5_digest, &file_obj.md5sum);
	nwos_create_sha1(sha1_digest, &file_obj.sha1sum);
	nwos_create_sha256(sha256_digest, &file_obj.sha256sum);
	nwos_create_sha512(sha512_digest, &file_obj.sha512sum);

	nwos_add_to_references(&file_ref, &file_obj.md5sum);   /* add to this md5sum object's reference list */
	nwos_add_to_references(&file_ref, &file_obj.sha1sum);   /* add to this md5sum object's reference list */
	nwos_add_to_references(&file_ref, &file_obj.sha256sum);   /* add to this md5sum object's reference list */
	nwos_add_to_references(&file_ref, &file_obj.sha512sum);   /* add to this md5sum object's reference list */

	nwos_log_md5sum(path, md5_digest);
	nwos_log_sha1sum(path, sha1_digest);
	nwos_log_sha256sum(path, sha256_digest);
	nwos_log_sha512sum(path, sha512_digest);


	/* store file size as 4 bytes in big endian order */
	file_obj.size[0] = file_length >> 32;
	file_obj.size[1] = file_length >> 24;
	file_obj.size[2] = file_length >> 16;
	file_obj.size[3] = file_length >> 8;
	file_obj.size[4] = file_length;

	nwos_create_reference_list(&file_ref, &file_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &file_obj.header.object, sizeof(ObjectHeader), file_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &file_obj.size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj.header.common.data_chksum);

	nwos_write_object_to_disk(&file_ref, &file_obj, sizeof(file_obj));

	nwos_add_to_references(&file_ref, &file_class_ref);

#if 0
	ptr_file_obj = malloc(sizeof(C_struct_File) + num_blocks * sizeof(ObjRef));
	assert(nwos_read_object_from_disk(ref, ptr_file_obj, sizeof(C_struct_File) + num_blocks * sizeof(ObjRef)));
	assert(memcmp(kludge, ptr_file_obj, sizeof(C_struct_File) + num_blocks * sizeof(ObjRef)) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_file_object_size));  /* read the other way */
	assert(memcmp(ptr_file_obj, kludge, sizeof(C_struct_File_Path) + length) == 0);
#endif

	nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

	assert(nwos_find_or_create_path_and_file_association(&path_ref, &file_ref, mod_time, ref) == CREATED_NEW);
    }

    return CREATED_NEW;
}


/* Add new revision of existing file - returns DUPLICATE_FILE if an exact file match already existed */
/*          (exact match meaning size, md5, sha1, date, and path are all identical).    */

ObjCreateResult nwos_add_new_revision_of_file(const char* path, ObjRef* ref)
{
    size_t length;
    struct stat stat_struct;
    ObjRef assoc_class_ref;
    ObjRef file_class_ref;
    ObjRef path_ref;
    ObjRef file_ref;
    EveryObject header;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    ObjRef object_class;
    uint64 file_length;
    C_struct_File new_file_obj;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_Path_And_File_Association old_assoc_obj;
    C_struct_Path_And_File_Association new_assoc_obj;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    uint8 sha1_digest[SHA1_DIGEST_SIZE];
    uint8 sha256_digest[SHA256_DIGEST_SIZE];
    uint8 sha512_digest[SHA512_DIGEST_SIZE];
    TimeStamp mod_time;

    void_reference(ref);   /* in case it fails */


    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return ERROR_OCCURRED;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return ERROR_OCCURRED;
    }

    if (stat_struct.st_size > 8 * 1073741824LL)
    {
	fprintf(stderr, "Cannot handle files larger than 8 gigabytes at this time\n");
	return ERROR_OCCURRED;
    }

    length = strlen(path);

    assert(length <= 255);


    /* first find out if we already have this path */

    if (!nwos_find_file_path(path, &path_ref))   /* this file must already exist */
    {
	return PATH_NOT_FOUND;
    }


    /* second find out if we already have this file stored */

    file_length = checksum_file(path, md5_digest, sha1_digest, sha256_digest, sha512_digest);

    if (nwos_find_matching_file_from_size_and_checksums(file_length, md5_digest, sha1_digest, sha256_digest, sha512_digest, &file_ref))
    {
	if (nwos_find_path_and_file_association(&path_ref, &file_ref, NULL, ref))
	{
	    return DUPLICATE_FILE;
	}
    }

    nwos_read_object_headers_from_disk(&path_ref, &header);

    ref_list = nwos_malloc_reference_list(&header.object.references);
    num_refs = ref_list->common_header.num_refs;

    /* Must make sure the File class is up to date before searching for the old file, */
    /* in case it is of the old flavor (it has to be updated before rewritting it.    */

    nwos_find_or_create_private_class_definition("FILE", &file_class_ref);

    assert(nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));

    memset(&old_assoc_obj, 0, sizeof(old_assoc_obj));  /* zero it out */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj)));

	    assert(is_same_object(&assoc_obj.path, &path_ref));

	    if (is_void_reference(&assoc_obj.header.object.next_version))
	    {
		if (!is_void_reference(&old_assoc_obj.header.common.id))  /* already found a file */
		{
		    fprintf(stderr, "Cannot handle multiple files with the same name at this time\n");
		    return MULTIPLE_FILES;
		}

		memcpy(&old_assoc_obj, &assoc_obj, sizeof(old_assoc_obj));
	    }
	}
    }

    printf("adding file revision: %s\n", path);

    printf("previous assoc: %08x  time: %s\n", nwos_ref_to_word(&old_assoc_obj.header.common.id),
	   nwos_time_stamp_to_string(old_assoc_obj.modification_time));

    nwos_generate_new_id(&file_ref);

    printf("new file: %08x  ", nwos_ref_to_word(&file_ref));
    fflush(stdout);

    memset(&new_file_obj, 0, sizeof(new_file_obj));  /* zero it out */

    nwos_fill_in_common_header(&new_file_obj.header.common, &file_ref, &file_class_ref);

    file_length = read_file(path, md5_digest, sha1_digest, sha256_digest, sha512_digest, &new_file_obj.block_list);

    nwos_create_md5(md5_digest, &new_file_obj.md5sum);
    nwos_create_sha1(sha1_digest, &new_file_obj.sha1sum);
    nwos_create_sha256(sha256_digest, &new_file_obj.sha256sum);
    nwos_create_sha512(sha512_digest, &new_file_obj.sha512sum);

    nwos_add_to_references(&file_ref, &new_file_obj.md5sum);   /* add to this md5sum object's reference list */
    nwos_add_to_references(&file_ref, &new_file_obj.sha1sum);   /* add to this md5sum object's reference list */
    nwos_add_to_references(&file_ref, &new_file_obj.sha256sum);   /* add to this sha256 object's reference list */
    nwos_add_to_references(&file_ref, &new_file_obj.sha512sum);   /* add to this sha512 object's reference list */

    nwos_log_md5sum(path, md5_digest);
    nwos_log_sha1sum(path, sha1_digest);
    nwos_log_sha256sum(path, sha256_digest);
    nwos_log_sha512sum(path, sha512_digest);


    /* store file size as 4 bytes in big endian order */
    new_file_obj.size[0] = file_length >> 32;
    new_file_obj.size[1] = file_length >> 24;
    new_file_obj.size[2] = file_length >> 16;
    new_file_obj.size[3] = file_length >> 8;
    new_file_obj.size[4] = file_length;

    nwos_create_reference_list(&file_ref, &new_file_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &new_file_obj.header.object, sizeof(ObjectHeader), new_file_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &new_file_obj.size[0], sizeof(C_struct_File) - sizeof(EveryObject), new_file_obj.header.common.data_chksum);

    nwos_write_object_to_disk(&file_ref, &new_file_obj, sizeof(new_file_obj));

    nwos_add_to_references(&file_ref, &new_file_obj.header.common.class_definition);

    nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

    assert(nwos_find_or_create_path_and_file_association(&path_ref, &file_ref, mod_time, ref) == CREATED_NEW);

    /* update the old association object's next_version */

    copy_reference(&old_assoc_obj.header.object.next_version, ref);

    nwos_crc32_calculate((uint8*) &old_assoc_obj.header.object, sizeof(ObjectHeader), old_assoc_obj.header.common.header_chksum);

    nwos_overwrite_object_to_disk(&old_assoc_obj.header.common.id, &old_assoc_obj, sizeof(old_assoc_obj));

    /* update the new association object's prev_version */

    assert(nwos_read_object_from_disk(ref, &new_assoc_obj, sizeof(new_assoc_obj)));

    copy_reference(&new_assoc_obj.header.object.prev_version, &old_assoc_obj.header.common.id);

    nwos_crc32_calculate((uint8*) &new_assoc_obj.header.object, sizeof(ObjectHeader), new_assoc_obj.header.common.header_chksum);

    nwos_overwrite_object_to_disk(ref, &new_assoc_obj, sizeof(new_assoc_obj));

    return CREATED_NEW_REVISION;
}


/* Check a file's md5sum */

CheckFileResult nwos_check_file_checksums(const char* path)
{
    size_t length;
    struct stat stat_struct;
    FILE* fp;
    size_t bytes_read;
    ObjRef path_ref;
    ObjRef file_ref;
    uint8 buffer[4096];
    uint64 file_length;
    int num_blocks;
    C_struct_File file_obj;
    struct md5_ctx md5_context;    /* MD5 checksum context */
    struct sha1_ctx sha1_context;
    struct sha256_ctx sha256_context;
    struct sha512_ctx sha512_context;
    C_struct_MD5sum md5_object;
    C_struct_SHA1sum sha1_object;
    C_struct_SHA256sum sha256_object;
    C_struct_SHA512sum sha512_object;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    uint8 sha1_digest[SHA1_DIGEST_SIZE];
    uint8 sha256_digest[SHA256_DIGEST_SIZE];
    uint8 sha512_digest[SHA512_DIGEST_SIZE];
    TimeStamp mod_time;
    CheckFileResult result = File_Not_Found;

    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return ERROR_OCCURRED;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return ERROR_OCCURRED;
    }

    if (stat_struct.st_size > 1099511627775LL)
    {
	fprintf(stderr, "Cannot handle files larger than 1 terabyte at this time\n");
	return ERROR_OCCURRED;
    }

    length = strlen(path);

    assert(length <= 255);
    
    /* first find out if we already have this path */

    assert(false);   /* need to fix file_path to assoc below */

    if (nwos_find_file_path(path, &path_ref))   /* it's ok, found the file */
    {
#if 0
	nwos_file_path_to_file(&path_ref, &file_ref);
#endif
	assert(nwos_read_object_from_disk(&file_ref, &file_obj, sizeof(file_obj)));  /* read the file object */

	file_length = file_size_to_uint64(file_obj.size);

	printf("file size: %llu\n", file_length);
	/* now open the file and start creating the data block objects */

	fp = fopen(path, "rb");

	file_length = 0;
	num_blocks = 0;

	md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */
	sha1_init_ctx(&sha1_context);   /* and the SHA1 context */
	sha256_init_ctx(&sha256_context);   /* and the SHA256 context */
	sha512_init_ctx(&sha512_context);   /* and the SHA512 context */

	bytes_read = sizeof(buffer);   /* go through loop at least once */

	while (bytes_read == sizeof(buffer))
	{
	    bytes_read = fread(buffer, sizeof(uint8), sizeof(buffer), fp);

	    if (bytes_read < sizeof(buffer))    /* eof or error */
	    {
		if (ferror(fp))
		{
		    perror(path);
		    exit(1);            /* FIX THIS - needs to clean up and exit gracefully!! */
		}
	    }

	    file_length = file_length + bytes_read;

	    if (bytes_read > 0)   /* we have data to store */
	    {
		md5_process_bytes(buffer, bytes_read, &md5_context);    /* include this data in the md5 checksum */
		sha1_process_bytes(buffer, bytes_read, &sha1_context);    /* and in the sha1 checksum */
		sha256_process_bytes(buffer, bytes_read, &sha256_context);    /* and in the sha256 checksum */
		sha512_process_bytes(buffer, bytes_read, &sha512_context);    /* and in the sha512 checksum */
	    }
	}

	fclose(fp);

	printf("length: %llu\n", file_length);

	md5_finish_ctx(&md5_context, md5_digest);   /* finish computing the md5 sum */
	sha1_finish_ctx(&sha1_context, sha1_digest);  /* and the sha1 sum */
	sha256_finish_ctx(&sha256_context, sha256_digest);  /* and the sha256 sum */
	sha512_finish_ctx(&sha512_context, sha512_digest);  /* and the sha512 sum */


	fflush(stdout);

	if (file_length != file_size_to_uint64(file_obj.size))
	{
	    printf("file size mismatch\n");
	}

	nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

	assert(false);
#if 0
	if (memcmp(file_obj.modification_time, mod_time, sizeof(TimeStamp)) != 0)
	{
	    printf("time mismatch\n");
	}
#endif
	assert(nwos_read_object_from_disk(&file_obj.md5sum, &md5_object, sizeof(md5_object)));

	if (memcmp(md5_digest, md5_object.md5sum, sizeof(md5_digest)) == 0)
	{
	    int j;
	    printf("MD5 sum OK: ");
	    for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_digest[j]);
	    printf("\n");

	    result = Checksums_Match;    /* found file and MD5 was ok */
	}
	else
	{
	    int j;
	    printf("MD5 checksum error, expected: ");
	    for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_object.md5sum[j]);
	    printf("\n                    received: ");
	    for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_digest[j]);
	    printf("\n");

	    result = MD5_Sum_Mismatch;    /* found file but MD5 was bad */
	}


	assert(nwos_read_object_from_disk(&file_obj.sha1sum, &sha1_object, sizeof(sha1_object)));

	if (memcmp(sha1_digest, sha1_object.sha1sum, sizeof(sha1_digest)) == 0)
	{
	    int j;
	    printf("SHA1 sum OK: ");
	    for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_digest[j]);
	    printf("\n");

	    /* MD5 compare set result, since SHA1 was ok leave it */
	}
	else
	{
	    int j;
	    printf("SHA1 checksum error, expected: ");
	    for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_object.sha1sum[j]);
	    printf("\n                    received: ");
	    for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_digest[j]);
	    printf("\n");

	    if (result == Checksums_Match)
	    {
		result = SHA1_Sum_Mismatch;    /* found file but SHA1 was bad */
	    }
	    else
	    {
		result = Checksums_Mismatch;
	    }
	}


	assert(nwos_read_object_from_disk(&file_obj.sha256sum, &sha256_object, sizeof(sha256_object)));

	if (memcmp(sha256_digest, sha256_object.sha256sum, sizeof(sha256_digest)) == 0)
	{
	    int j;
	    printf("SHA256 sum OK: ");
	    for (j = 0; j < SHA256_DIGEST_SIZE; j++) printf("%02x", sha256_digest[j]);
	    printf("\n");

	    /* MD5 compare set result, since SHA256 was ok leave it */
	}
	else
	{
	    int j;
	    printf("SHA256 checksum error, expected: ");
	    for (j = 0; j < SHA256_DIGEST_SIZE; j++) printf("%02x", sha256_object.sha256sum[j]);
	    printf("\n                    received: ");
	    for (j = 0; j < SHA256_DIGEST_SIZE; j++) printf("%02x", sha256_digest[j]);
	    printf("\n");

	    if (result == Checksums_Match)
	    {
		result = SHA256_Sum_Mismatch;    /* found file but SHA256 was bad */
	    }
	    else
	    {
		result = Checksums_Mismatch;
	    }
	}


	assert(nwos_read_object_from_disk(&file_obj.sha512sum, &sha512_object, sizeof(sha512_object)));

	if (memcmp(sha512_digest, sha512_object.sha512sum, sizeof(sha512_digest)) == 0)
	{
	    int j;
	    printf("SHA512 sum OK: ");
	    for (j = 0; j < SHA512_DIGEST_SIZE; j++) printf("%02x", sha512_digest[j]);
	    printf("\n");

	    /* MD5 compare set result, since SHA512 was ok leave it */
	}
	else
	{
	    int j;
	    printf("SHA512 checksum error, expected: ");
	    for (j = 0; j < SHA512_DIGEST_SIZE; j++) printf("%02x", sha512_object.sha512sum[j]);
	    printf("\n                    received: ");
	    for (j = 0; j < SHA512_DIGEST_SIZE; j++) printf("%02x", sha512_digest[j]);
	    printf("\n");

	    if (result == Checksums_Match)
	    {
		result = SHA512_Sum_Mismatch;    /* found file but SHA512 was bad */
	    }
	    else
	    {
		result = Checksums_Mismatch;
	    }
	}
    }

    return result;   /* didn't find file */
}



/* Read file, if path is null it just checks the file without writing it to disk */

static void dump_file_block_error(int seq, uint8 chksum[4], DataObject *data_obj, const char* which_encrypt)
{
    int j;

    if (which_encrypt == NULL)
    {
	printf("checksum error in file block - sequence: %d calculated: %02x%02x%02x%02x\n\n",
	       seq, chksum[0], chksum[1], chksum[2], chksum[3]);
    }
    else
    {
	printf("checksum error in file block - sequence: %d (%s encryption) calculated: %02x%02x%02x%02x\n\n",
	       seq, which_encrypt, chksum[0], chksum[1], chksum[2], chksum[3]);
    }

    for (j = 0; j < sizeof(DataObject); j++)
    {
	printf("%02x%c", ((uint8*)data_obj)[j], (j % 16 == 15) ? '\n' : ' ');
    }

    printf("\n");
}


bool nwos_restore_file(ObjRef* assoc_ref, const char* path)
{
    bool result = true;
    FILE* fp = NULL;
    int i;
    uint64 file_length;
    uint32 num_blocks;
    C_struct_File file_obj;
    C_struct_Path_And_File_Association assoc_obj;
    DataObject data_obj;
    DataObject data_obj_new_encrypt;
    struct timeval tv[2];      /* [0] is access time and [1] is modification time, see man utimes */
    struct md5_ctx md5_context;           /* MD5 checksum context */
    struct sha1_ctx sha1_context;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    uint8 sha1_digest[SHA1_DIGEST_SIZE];
    C_struct_MD5sum md5_object;
    C_struct_SHA1sum sha1_object;
    ReferenceList* ref_list_ptr;
    uint8 ivec[IVEC_SIZE];
    uint8 chksum[4];
    uint8 chksum_new_encrypt[4];
    bool use_old_decryption = false;


    memset(ivec, 0, sizeof(ivec));

    assert(nwos_read_object_from_disk(assoc_ref, &assoc_obj, sizeof(assoc_obj)));  /* read the path and file assoc. object */

    assert(nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj)));  /* read the file object */

    /* if the path is non-null open the file and start writing the data block objects */

    if (path != NULL)
    {
	fp = fopen(path, "wb");

	if (fp == NULL)
	{
	    perror(path);
	    result = false;     /* could not open the file, don't continue */
	}
    }

    if (result)
    {
	file_length = file_size_to_uint64(file_obj.size);

	printf("file size: %llu\n", file_length);

	ref_list_ptr = nwos_malloc_reference_list(&file_obj.block_list);

	num_blocks = (file_length + DATA_STORAGE_SIZE - 1) / DATA_STORAGE_SIZE;

	assert(num_blocks == ref_list_ptr->common_header.num_refs);

//    printf("num_blocks: %u  block_list: %02x%02x%02x%02x\n", num_blocks,
//	   file_obj.block_list.id[0],
//	   file_obj.block_list.id[1],
//	   file_obj.block_list.id[2],
//	   file_obj.block_list.id[3]);


	i = 0;

	md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */
	sha1_init_ctx(&sha1_context);   /* and the SHA1 context */

	while (file_length > DATA_STORAGE_SIZE)   /* do all except the last block */
	{
	    if (use_old_decryption)
	    {
		assert(nwos_read_block_from_disk_and_old_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, i));
	    }
	    else
	    {
		assert(nwos_read_block_from_disk_and_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, i));
	    }

	    nwos_crc32_calculate((uint8*) &data_obj.storage, sizeof(data_obj.storage), chksum);

	    if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
	    {
		if (i > 0 || use_old_decryption)
		{
		    dump_file_block_error(i, chksum, &data_obj, NULL);
		    exit(1);
		}

		/* if we get to here, it is the first block so it may have used the old encryption */
		/* save the result of the new encryption in case we have to dump */
		memcpy(&data_obj_new_encrypt, &data_obj, sizeof(data_obj_new_encrypt));
		memcpy(chksum_new_encrypt, chksum, sizeof(chksum_new_encrypt));

		memset(ivec, 0, sizeof(ivec));
		assert(nwos_read_block_from_disk_and_old_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, i));
		    
		nwos_crc32_calculate((uint8*) &data_obj.storage, sizeof(data_obj.storage), chksum);

		if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
		{
		    dump_file_block_error(i, chksum, &data_obj, "old");
		    dump_file_block_error(i, chksum_new_encrypt, &data_obj_new_encrypt, "new");
		    exit(1);
		}

		use_old_decryption = true;
	    }

	    md5_process_bytes(data_obj.storage, DATA_STORAGE_SIZE, &md5_context);    /* include this data in the md5 checksum */
	    sha1_process_bytes(data_obj.storage, DATA_STORAGE_SIZE, &sha1_context);    /* and in the sha1 checksum */

	    if (fp != NULL && fwrite(data_obj.storage, sizeof(uint8), DATA_STORAGE_SIZE, fp) != DATA_STORAGE_SIZE)
	    {
		perror(path);
		exit(1);
	    }

	    file_length -= DATA_STORAGE_SIZE;

	    i++;
	}

	if (use_old_decryption)
	{
	    assert(nwos_read_block_from_disk_and_old_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, i));
	}
	else
	{
	    assert(nwos_read_block_from_disk_and_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, i));
	}

	nwos_crc32_calculate((uint8*) &data_obj.storage, file_length, chksum);

	if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
	{
	    if (i > 0)   /* this is not the first block, so we knew which encryption to use, so this is a real error */
	    {
		dump_file_block_error(i, chksum, &data_obj, NULL);
		exit(1);
	    }

	    /* if we get to here it is the first block so we need to try the old decryption */
	    /* save the result of the new encryption in case we have to dump */
	    memcpy(&data_obj_new_encrypt, &data_obj, sizeof(data_obj_new_encrypt));
	    memcpy(chksum_new_encrypt, chksum, sizeof(chksum_new_encrypt));

	    memset(ivec, 0, sizeof(ivec));
	    assert(nwos_read_block_from_disk_and_old_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, i));
		    
	    nwos_crc32_calculate((uint8*) &data_obj.storage, file_length, chksum);

	    if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
	    {
		dump_file_block_error(i, chksum, &data_obj, "old");
		dump_file_block_error(i, chksum_new_encrypt, &data_obj_new_encrypt, "new");
		exit(1);
	    }
	}

	md5_process_bytes(data_obj.storage, file_length, &md5_context);    /* include this data in the md5 checksum */
	sha1_process_bytes(data_obj.storage, file_length, &sha1_context);    /* and in the sha1 checksum */

	if (fp != NULL)
	{
	    if (fwrite(data_obj.storage, sizeof(uint8), file_length, fp) != file_length)
	    {
		perror(path);
		exit(1);
	    }

	    if (fclose(fp) != 0)
	    {
		perror(path);
		exit(1);
	    }

	    if (result)   /* the file was ok, restore it's modification time */
	    {
		nwos_convert_time_stamp_to_timeval(assoc_obj.modification_time, &tv[1]);
		tv[0] = tv[1];  /* copy modification time into access time since we don't save the access time */
		utimes(path, tv);
	    }
	}

	md5_finish_ctx(&md5_context, md5_digest);   /* finish computing the md5 sum */
	sha1_finish_ctx(&sha1_context, sha1_digest);  /* and the sha1 sum */

	if (result)
	{
	    assert(nwos_read_object_from_disk(&file_obj.md5sum, &md5_object, sizeof(md5_object)));

	    if (memcmp(md5_digest, md5_object.md5sum, sizeof(md5_digest)) == 0)
	    {
		int j;
		printf("MD5 sum OK: ");
		for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_digest[j]);
		printf("\n");
	    }
	    else
	    {
		int j;
		printf("MD5 checksum error, expected: ");
		for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_object.md5sum[j]);
		printf("\n                    received: ");
		for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_digest[j]);
		printf("\n");

		result = false;
	    }

	    assert(nwos_read_object_from_disk(&file_obj.sha1sum, &sha1_object, sizeof(sha1_object)));

	    if (memcmp(sha1_digest, sha1_object.sha1sum, sizeof(sha1_digest)) == 0)
	    {
		int j;
		printf("SHA1 sum OK: ");
		for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_digest[j]);
		printf("\n");
	    }
	    else
	    {
		int j;
		printf("SHA1 checksum error, expected: ");
		for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_object.sha1sum[j]);
		printf("\n                    received: ");
		for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_digest[j]);
		printf("\n");

		result = false;
	    }
	}

	nwos_free_reference_list(ref_list_ptr);
	ref_list_ptr = NULL;
    }

    return result;
}


/* Compare file - does not verify checksums (just compares the data) */

bool nwos_file_is_identical(ObjRef* assoc_ref, const char* path, MatchCode match)
{
    bool result = true;
    FILE* fp;
    int i;
    struct stat stat_struct;
    uint64 file_length;
    uint32 num_blocks;
    C_struct_File file_obj;
    C_struct_Path_And_File_Association assoc_obj;
    DataObject data_obj;
    struct timeval tv;
    ReferenceList* ref_list_ptr;
    uint8 ivec[IVEC_SIZE];
    uint8 chksum[4];
    uint8 buffer[DATA_STORAGE_SIZE];
    bool use_old_decryption = false;

    memset(ivec, 0, sizeof(ivec));

    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return false;
    }

    assert(nwos_read_object_from_disk(assoc_ref, &assoc_obj, sizeof(assoc_obj)));  /* read the path and file assoc. object */

    assert(nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj)));  /* read the file object */

    file_length = file_size_to_uint64(file_obj.size);

    printf("file size: %llu\n", file_length);

    if (stat_struct.st_size != file_length)
    {
	fprintf(stderr, "File size mismatch\n");
	return false;
    }

    ref_list_ptr = nwos_malloc_reference_list(&file_obj.block_list);

    num_blocks = (file_length + DATA_STORAGE_SIZE - 1) / DATA_STORAGE_SIZE;

    assert(num_blocks == ref_list_ptr->common_header.num_refs);

//    printf("num_blocks: %u  block_list: %02x%02x%02x%02x\n", num_blocks,
//	   file_obj.block_list.id[0],
//	   file_obj.block_list.id[1],
//	   file_obj.block_list.id[2],
//	   file_obj.block_list.id[3]);


    /* now open the file and start writing the data block objects */

    fp = fopen(path, "rb");

    if (fp == NULL)
    {
	perror(path);
	exit(1);
    }
    else  /* file opened ok */
    {
	i = 0;

	while (file_length > DATA_STORAGE_SIZE)   /* do all except the last block */
	{
	    if (use_old_decryption)
	    {
		assert(nwos_read_block_from_disk_and_old_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, i));
	    }
	    else
	    {
		assert(nwos_read_block_from_disk_and_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, i));
	    }

	    nwos_crc32_calculate((uint8*) &data_obj.storage, sizeof(data_obj.storage), chksum);

	    if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
	    {
		if (i == 0)   /* if this is the first time, check for old decryption */
		{
		    memset(ivec, 0, sizeof(ivec));
		    assert(nwos_read_block_from_disk_and_old_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, i));

		    nwos_crc32_calculate((uint8*) &data_obj.storage, sizeof(data_obj.storage), chksum);
		}

		if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
		{
		    printf("checksum error in file block - sequence: %d\n", i);
		    exit(1);
		}

		use_old_decryption = true;
	    }

	    if (fread(buffer, sizeof(uint8), sizeof(buffer), fp) != DATA_STORAGE_SIZE)
	    {
		perror(path);
		exit(1);
	    }

	    if (memcmp(&data_obj.storage, buffer, DATA_STORAGE_SIZE) != 0)
	    {
		result = false;
		break;
	    }

	    file_length -= DATA_STORAGE_SIZE;

	    i++;
	}


	if (result)   /* file compared ok thus far */
	{
	    if (use_old_decryption)
	    {
		assert(nwos_read_block_from_disk_and_old_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, i));
	    }
	    else
	    {
		assert(nwos_read_block_from_disk_and_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, i));
	    }

	    nwos_crc32_calculate((uint8*) &data_obj.storage, file_length, chksum);

	    if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
	    {
		if (i == 0)   /* if this is the first time, check for old decryption */
		{
		    memset(ivec, 0, sizeof(ivec));
		    assert(nwos_read_block_from_disk_and_old_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, i));

		    nwos_crc32_calculate((uint8*) &data_obj.storage, sizeof(data_obj.storage), chksum);
		}

		if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
		{
		    printf("checksum error in file block - sequence: %d\n", i);
		    exit(1);
		}
	    }

	    if (fread(buffer, sizeof(uint8), file_length, fp) != file_length)
	    {
		perror(path);
		exit(1);
	    }

	    if (memcmp(&data_obj.storage, buffer, file_length) != 0)
	    {
		result = false;
	    }
	}

	if (fclose(fp) != 0)
	{
	    perror(path);
	    exit(1);
	}

	if (result && (match & IgnoreTime) == 0)   /* the file was ok, compare it's modification time */
	{
	    nwos_convert_time_stamp_to_timeval(assoc_obj.modification_time, &tv);

	    if ((match & IgnoreVfat) == 0)
	    {
		if (stat_struct.st_mtimespec.tv_sec != tv.tv_sec || stat_struct.st_mtimespec.tv_nsec / 1000 != tv.tv_usec)
		{
		    printf("Time values mismatch - file %u.%06u  stored: %u.%06u\n",
			   (uint32)stat_struct.st_mtimespec.tv_sec, (uint32)stat_struct.st_mtimespec.tv_nsec / 1000, (uint32)tv.tv_sec, (uint32)tv.tv_usec);
		    result = false;
		}
	    }
	    else    /* only check time accurate to VFAT (2 seconds) */ 
	    {
		if (stat_struct.st_mtimespec.tv_sec != ((uint32)tv.tv_sec & 0xfffffffe))
		{
		    printf("Time values mismatch - file %u  stored: %u\n", (uint32)stat_struct.st_mtimespec.tv_sec, (uint32)tv.tv_sec);
		    result = false;
		}

	    }
	}
    }

    nwos_free_reference_list(ref_list_ptr);
    ref_list_ptr = NULL;

    return result;
}


/* Returns true if the file is stored on some backup media */

bool nwos_file_has_backup(ObjRef* assoc_ref)
{
    ObjRef disc_list_class_ref;
    ObjRef disc_copy_class_ref;
    ObjRef class_ref;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_Disc_Copy copy_obj;
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_disc_obj = (C_struct_Disc_List*)kludge;
    ReferenceList* assoc_ref_list_ptr;
    ReferenceList* disc_ref_list_ptr;
    int num_assoc_refs;
    int num_disc_refs;
    int i;
    int j;
    int count;

    if (!nwos_find_private_class_definition("DISC LIST", &disc_list_class_ref)) return false;
    if (!nwos_find_private_class_definition("DISC COPY", &disc_copy_class_ref)) return false;

    assert(nwos_read_object_from_disk(assoc_ref, &assoc_obj, sizeof(assoc_obj)));  /* read the path and file assoc. object */

    assoc_ref_list_ptr = nwos_malloc_reference_list(&assoc_obj.header.object.references);

    num_assoc_refs = assoc_ref_list_ptr->common_header.num_refs;

    for (i = 0; i < num_assoc_refs; i++)
    {
	nwos_get_object_class(&assoc_ref_list_ptr->references[i], &class_ref);

	if (is_same_object(&class_ref, &disc_list_class_ref))
	{
	    assert(nwos_read_variable_sized_object_from_disk(&assoc_ref_list_ptr->references[i], kludge, sizeof(kludge), &nwos_get_disc_list_object_size));

	    disc_ref_list_ptr = nwos_malloc_reference_list(&ptr_disc_obj->header.object.references);

	    /* as long as we had to read the object, verify that this file is in the list */

	    count = nwos_decode_variable_sized_count(ptr_disc_obj->count);
	    for (j = 0; j < count; j++)
	    {
		if (is_same_object(assoc_ref, &ptr_disc_obj->files[j])) break;
	    }
	    assert(j < count);   /* something very wrong if it wasn't in the list */

	    /* now find any disc copies that exist */

	    num_disc_refs = disc_ref_list_ptr->common_header.num_refs;

	    for (j = 0; j < num_disc_refs; j++)
	    {
		nwos_get_object_class(&disc_ref_list_ptr->references[j], &disc_copy_class_ref);

		if (is_same_object(&class_ref, &disc_list_class_ref))
		{
		    assert(nwos_read_object_from_disk(&disc_ref_list_ptr->references[j], &copy_obj, sizeof(copy_obj)));
		    assert(is_same_object(&assoc_ref_list_ptr->references[i], &copy_obj.disc_list));
		    break;
		}
	    }
	    
	    nwos_free_reference_list(disc_ref_list_ptr);
	    disc_ref_list_ptr = NULL;

	    if (j < num_disc_refs) break;  /* found a disc_copy in references */
	}
    }

    nwos_free_reference_list(assoc_ref_list_ptr);
    assoc_ref_list_ptr = NULL;

    return i < num_assoc_refs;;
}


/*-------------------------------------------------------------------------------------------------------------------*/
/* File association (Path <-> File) object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

bool nwos_find_path_and_file_association_by_index(ObjRef* path_ref, ObjRef* file_ref, int index, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Path_And_File_Association assoc_obj;
    ObjRef assoc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    int count;

    void_reference(ref);  /* in case we don't find it */


    /* for now just return false if there is no private class definition yet */

    if (!nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref)) return false;


    nwos_read_class_definition(&assoc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    count = 0;

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj)));

	    if (is_same_object(&assoc_obj.file, file_ref) && is_same_object(&assoc_obj.path, path_ref))   /* found a match */
	    {
		if (count == index)
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    break;
		}
		else
		{
		    count++;
		}
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


static bool find_path_and_file_association(bool public, ObjRef* path_ref, ObjRef* file_ref, TimeStamp mod_time, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Path_And_File_Association assoc_obj;
    ObjRef assoc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;

    void_reference(ref);  /* in case we don't find it */


    /* for now just return false if there is no private class definition yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));
#else
    if (public)
    {
	assert(nwos_reference_type(path_ref) == Public_Reference);
	assert(nwos_reference_type(file_ref) == Public_Reference);
	if (!nwos_find_public_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref)) return false;
    }
    else
    {
	assert(nwos_reference_type(path_ref) == Private_Reference);
	assert(nwos_reference_type(file_ref) == Private_Reference);
	if (!nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&assoc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj)));

	    if (is_same_object(&assoc_obj.file, file_ref) && is_same_object(&assoc_obj.path, path_ref))   /* found a match */
	    {
		if (mod_time == NULL || memcmp(assoc_obj.modification_time, mod_time, sizeof(TimeStamp)) == 0)
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    break;
		}
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


bool nwos_find_public_path_and_file_association(ObjRef* path_ref, ObjRef* file_ref, TimeStamp mod_time, ObjRef* ref)
{
    assert(nwos_reference_type(path_ref) == Public_Reference);
    assert(nwos_reference_type(file_ref) == Public_Reference);

    return find_path_and_file_association(true, path_ref, file_ref, mod_time, ref);
}


bool nwos_find_path_and_file_association(ObjRef* path_ref, ObjRef* file_ref, TimeStamp mod_time, ObjRef* ref)
{
    return find_path_and_file_association(nwos_in_public_mode(), path_ref, file_ref, mod_time, ref);
}



bool nwos_find_matching_path_and_file_association(const char* dir, const char* file_name, ObjRef* ref, MatchCode match)
{
    size_t length;
    struct stat stat_struct;
    uint64 file_length;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    uint8 sha1_digest[SHA1_DIGEST_SIZE];
    uint8 sha256_digest[SHA256_DIGEST_SIZE];
    uint8 sha512_digest[SHA512_DIGEST_SIZE];
    ObjRef path_ref;
    ObjRef file_ref;
    TimeStamp mod_time;
    uint odd_second;
    bool path_found;
    bool file_found;
    bool assoc_found;
    char path[PATH_MAX];

    void_reference(ref);  /* in case we don't find it */

    if (dir == NULL)      /* no directory specified */
    {
	if (strlcpy(path, file_name, PATH_MAX) >= PATH_MAX)
	{
	    fprintf(stderr, "path too long (> %u bytes): %s\n", PATH_MAX, file_name);
	    return false;
	}
    }
    else
    {
	length = strlcpy(path, dir, PATH_MAX);

	if (length + 1 >= PATH_MAX)
	{
	    fprintf(stderr, "path too long (> %u bytes): %s\n", PATH_MAX, dir);
	    return false;
	}

	if (path[length - 1] != '/')   /* the directory didn't end with a '/' so add one */
	{
	    path[length] = '/';
	    path[length + 1] = '\0';
	}

	if (strlcat(path, file_name, PATH_MAX) >= PATH_MAX)
	{
	    fprintf(stderr, "path too long (> %u bytes): %s/%s\n", PATH_MAX, dir, file_name);
	    return false;
	}
    }

    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return false;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return false;
    }

    if (stat_struct.st_size > 1099511627775LL)
    {
	fprintf(stderr, "Cannot handle files larger than 1 terabyte at this time\n");
	return false;
    }


    length = strlen(file_name);

    assert(length <= 255);
    
    file_length = checksum_file(path, md5_digest, sha1_digest, sha256_digest, sha512_digest);

    path_found = nwos_find_file_path(file_name, &path_ref);

    file_found = nwos_find_matching_file_from_size_and_checksums(file_length, md5_digest, sha1_digest, sha256_digest, sha512_digest, &file_ref);

    if (path_found && file_found)
    {
	assoc_found = nwos_find_path_and_file_association(&path_ref, &file_ref, NULL, ref);

	if ((match & IgnoreTime) != 0 || !assoc_found)
	{
	    printf("(found path: yes  file: yes  association: %s) ", assoc_found ? "yes" : "no");
	}
	else
	{
	    nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

	    assoc_found = nwos_find_path_and_file_association(&path_ref, &file_ref, mod_time, ref);

	    if (!assoc_found & ((match & IgnoreVfat) != 0))    /* if didn't find it and ignore VFAT inaccuracy is set */
	    {
		odd_second = nwos_extract_second_from_time_stamp(mod_time) + 1;      /* try again with odd seconds */
		nwos_insert_second_into_time_stamp(odd_second, mod_time);

		assoc_found = nwos_find_path_and_file_association(&path_ref, &file_ref, mod_time, ref);
	    }

	    printf("(found path: yes  file: yes  association: yes  time: %s) ", assoc_found ? "yes" : "no");
	}
    }
    else
    {
	printf("(found path: %s  file: %s) ", path_found ? "yes" : "no", file_found ? "yes" : "no");
	assoc_found = false;
    }

    return assoc_found;
}



/* Find existing file association or create new */

ObjCreateResult nwos_find_or_create_path_and_file_association(ObjRef* path_ref, ObjRef* file_ref, TimeStamp mod_time, ObjRef* ref)
{
    C_struct_Path_And_File_Association assoc_obj;
#ifdef VERIFY_WRITE
    C_struct_Path_And_File_Association temp;
#endif
    ObjRef assoc_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
#ifndef PUBLIC_MODE
    EveryObject path_header;
    EveryObject file_header;
    ObjRef public_assoc_ref;
    char msg[128];
#endif


    /* first find out if we already have this association */

    if (!nwos_find_path_and_file_association(path_ref, file_ref, mod_time, ref))   /* didn't find existing association */
    {
	memset(&assoc_obj, 0, sizeof(assoc_obj));  /* zero it out */

#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));
#else
	nwos_find_or_create_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref);
#endif

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&assoc_obj.header.common, ref, &assoc_class_ref);

#ifndef PUBLIC_MODE
	nwos_read_object_headers_from_disk(path_ref, &path_header);
	nwos_read_object_headers_from_disk(file_ref, &file_header);
	if (!is_void_reference(&path_header.object.clone_of) && !is_void_reference(&file_header.object.clone_of) &&
	    nwos_find_public_path_and_file_association(&path_header.object.clone_of, &file_header.object.clone_of, NULL, &public_assoc_ref))
	{
	    copy_reference(&assoc_obj.header.object.clone_of, &public_assoc_ref);
	    snprintf(msg, sizeof(msg), "Create path and file association - path: %08x  file: %08x  assoc: %08x",
		     nwos_ref_to_word(path_ref), nwos_ref_to_word(file_ref), nwos_ref_to_word(ref));
	    nwos_log(msg);
	    snprintf(msg, sizeof(msg), "                          public - path: %08x  file: %08x  assoc: %08x",
		     nwos_ref_to_word(&path_header.object.clone_of), nwos_ref_to_word(&file_header.object.clone_of),
		     nwos_ref_to_word(&public_assoc_ref));
	    nwos_log(msg);
	}
#endif
	copy_reference(&assoc_obj.path, path_ref);
	copy_reference(&assoc_obj.file, file_ref);
	memcpy(assoc_obj.modification_time, mod_time, sizeof(assoc_obj.modification_time));

	nwos_create_reference_list(ref, &assoc_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &assoc_obj.header.object, sizeof(ObjectHeader), assoc_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &assoc_obj.path, sizeof(C_struct_Path_And_File_Association) - sizeof(EveryObject), assoc_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &assoc_obj, sizeof(assoc_obj));

	nwos_add_to_references(ref, &assoc_class_ref);
	nwos_add_to_references(ref, file_ref);
	nwos_add_to_references(ref, path_ref);

	/* read back to verify */
#ifdef VERIFY_WRITE
	/* NOTE: these don't work in the mixed 32/64 bit reference environment, because nwos_write_block changes the upper 4 bytes of the reference */
	assert(nwos_read_object_from_disk(ref, &temp, sizeof(temp)));
	assert(memcmp(&temp, &assoc_obj, sizeof(C_struct_Path_And_File_Association)) == 0);
#endif

	result = CREATED_NEW;
    }

    return result;
}


/*-------------------------------------------------------------------------------------------------------------------*/
/* Directory Processing */
/*-------------------------------------------------------------------------------------------------------------------*/

const char* nwos_check_invalid_type(mode_t mode)
{
    if (S_ISCHR(mode))
    {
	return "character device";
    }

    if (S_ISBLK(mode))
    {
	return "block device";
    }

    if (S_ISFIFO(mode))
    {
	return "FIFO (named pipe)";
    }

    if (S_ISLNK(mode))
    {
	return "symbolic link";
    }

    if (S_ISSOCK(mode))
    {
	return "socket";
    }

    return NULL;
}


void nwos_check_directory_is_all_ok(const char* path)
{
    struct stat stat_struct;
    DIR* dp;
    struct dirent *dir_entry;
    const char* invalid_type;
    const char* p;
    char full_path[PATH_MAX];
    int i;

    dp = opendir(path);
    if (dp == NULL)
    {
	perror(path);
	exit(1);
    }

    dir_entry = readdir(dp);
    while (dir_entry != NULL)
    {
	if (strcmp(dir_entry->d_name, ".") != 0 && strcmp(dir_entry->d_name, "..") != 0)
	{
	    // Construct the full path name

	    i = 0;

	    for (p = path; *p != '\0' && i < PATH_MAX; p++) full_path[i++] = *p;

	    if (i < PATH_MAX) full_path[i++] = '/';

	    for (p = dir_entry->d_name; *p != '\0' && i < PATH_MAX; p++) full_path[i++] = *p;

	    if (i < PATH_MAX)
	    {
		full_path[i] = '\0';
	    }
	    else
	    {
		fprintf(stderr, "ERROR: path %s/%s exceeds maximum size of a path (PATH_MAX = %d)\n",
			path, dir_entry->d_name, PATH_MAX);
		exit(1);
	    }

	    if (lstat(full_path, &stat_struct) != 0)
	    {
		perror(full_path);
		exit(1);
	    }

	    invalid_type = nwos_check_invalid_type(stat_struct.st_mode);

	    if (invalid_type != NULL)
	    {
		fprintf(stderr, "ERROR: %s is a %s.\n", full_path, invalid_type);
		exit(1);
	    }

	    if (S_ISDIR(stat_struct.st_mode))    // it's not a normal file, investigate further
	    {
		nwos_check_directory_is_all_ok(full_path);
	    }    
	    else if (!S_ISREG(stat_struct.st_mode))
	    {
		fprintf(stderr, "ERROR: %s is unrecognized type of file.\n", full_path);
		exit(1);
	    }
	}

	dir_entry = readdir(dp);
    }

    if (closedir(dp) != 0)
    {
	perror(path);
	exit(1);
    }
}


/*-------------------------------------------------------------------------------------------------------------------*/
/* File Object Updates */
/*-------------------------------------------------------------------------------------------------------------------*/

void nwos_update_file_001_object_to_current(void* object, size_t size)
{
    C_struct_File_001 file_001_obj;
    C_struct_File* file_obj_ptr = (C_struct_File*) object;
    ObjRef file_class_ref;

    memcpy(&file_001_obj, object, sizeof(file_001_obj));  /* save the 001 data */

    /* change the class definition to the latest */
#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("FILE", &file_class_ref));
#else
    assert(nwos_find_private_class_definition("FILE", &file_class_ref));

    /* if a the newer class doesn't exist yet, change the class definition to the latest public */
    if (is_same_object(&file_class_ref, &file_obj_ptr->header.common.class_definition))
    {
	assert(nwos_find_public_class_definition("FILE", &file_class_ref));
    }
#endif
    copy_reference(&file_obj_ptr->header.common.class_definition, &file_class_ref);

    file_obj_ptr->size[0] = 0;  /* most significant byte didn't exist before */
    file_obj_ptr->size[1] = file_001_obj.size[0];
    file_obj_ptr->size[2] = file_001_obj.size[1];
    file_obj_ptr->size[3] = file_001_obj.size[2];
    file_obj_ptr->size[4] = file_001_obj.size[3];

    copy_reference(&file_obj_ptr->md5sum,     &file_001_obj.md5sum);
    copy_reference(&file_obj_ptr->sha1sum,    &file_001_obj.sha1sum);
    void_reference(&file_obj_ptr->sha256sum);
    void_reference(&file_obj_ptr->sha512sum);
    copy_reference(&file_obj_ptr->media,      &file_001_obj.media);
    copy_reference(&file_obj_ptr->block_list, &file_001_obj.block_list);

    nwos_crc32_calculate((uint8*) &file_obj_ptr->size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj_ptr->header.common.data_chksum);
}


void nwos_update_file_002_object_to_current(void* object, size_t size)
{
    C_struct_File_002 file_002_obj;
    C_struct_File* file_obj_ptr = (C_struct_File*) object;
    ObjRef file_class_ref;

    memcpy(&file_002_obj, object, sizeof(file_002_obj));  /* save the 002 data */

    /* change the class definition to the latest */
#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("FILE", &file_class_ref));
#else
    assert(nwos_find_private_class_definition("FILE", &file_class_ref));

    /* if a the newer class doesn't exist yet, change the class definition to the latest public */
    if (is_same_object(&file_class_ref, &file_obj_ptr->header.common.class_definition))
    {
	assert(nwos_find_public_class_definition("FILE", &file_class_ref));
    }
#endif
    copy_reference(&file_obj_ptr->header.common.class_definition, &file_class_ref);

    file_obj_ptr->size[0] = file_002_obj.size[0];
    file_obj_ptr->size[1] = file_002_obj.size[1];
    file_obj_ptr->size[2] = file_002_obj.size[2];
    file_obj_ptr->size[3] = file_002_obj.size[3];
    file_obj_ptr->size[4] = file_002_obj.size[4];

    copy_reference(&file_obj_ptr->md5sum,     &file_002_obj.md5sum);
    copy_reference(&file_obj_ptr->sha1sum,    &file_002_obj.sha1sum);
    copy_reference(&file_obj_ptr->sha256sum,  &file_002_obj.sha256sum);
    void_reference(&file_obj_ptr->sha512sum);
    copy_reference(&file_obj_ptr->media,      &file_002_obj.media);
    copy_reference(&file_obj_ptr->block_list, &file_002_obj.block_list);

    nwos_crc32_calculate((uint8*) &file_obj_ptr->size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj_ptr->header.common.data_chksum);
}


