/*
--          This file is part of the New World OS and Objectify projects
--         Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-03-16 07:53:57 -0600 (Tue, 16 Mar 2010) $
--   $Revision: 4630 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file and the objectify.c file which
--   this file was derived from.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>

#include "log.h"
#include "mem_alloc.h"
#include "strlcxx.h"           /* in case strlcpy and strlcat are not provided by the system */
#include "user_config.h"


static bool log_disabled = false;


/******************************************/
/* Log file - THIS NEEDS TO GO AWAY SOON. */
/******************************************/

void append_date_from_time_string(char* dst, char* time_str, size_t length)
{
    int i;
    int month;
    char* months[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

    for (i = 0; i < length; i++) if (dst[i] == '\0') break;

    assert((i + 9) < length);

    dst[i++] = time_str[20];
    dst[i++] = time_str[21];
    dst[i++] = time_str[22];
    dst[i++] = time_str[23];

    for (month = 0; month < 12; month++) if (memcmp(months[month], &time_str[4], 3) == 0) break;

    assert(month < 12);

    month++;

    if (month < 10)
    {
	dst[i++] = '0';
	dst[i++] = '0' + month;
    }
    else
    {
	dst[i++] = '1';
	dst[i++] = '0' + month - 10;
    }

    if (time_str[8] == ' ')
    {
	dst[i++] = '0';
    }
    else
    {
	dst[i++] = time_str[8];
    }
    dst[i++] = time_str[9];

    dst[i++] = '\0';
}


void nwos_log(char* str)
{
    struct timeval tv;
    FILE* fp;
    char time_str[32];
    static char* log_file_path;
    size_t path_len;
    const char* path;
    char* p;

    if (log_disabled) return;    /* if logging has been disabled, just return */

    gettimeofday(&tv, NULL);
    assert(strlcpy(time_str, ctime((time_t*)&tv.tv_sec), sizeof(time_str)) < sizeof(time_str));
    *strchr(time_str, '\n') = '\0';

    if (log_file_path == NULL)
    {
	path = nwos_get_log_file_path();

	if (path == NULL || *path == '\0')       /* no log file specified */
	{
	    log_disabled = true;
	    return;
	}
	
	/* NOTE: this is a bad thing to do, it temporarily modifies */
	/* the string returned by nwos_get_log_file_path.           */
        /* We cheat the system by calling strchr.                   */

	p = strchr(path, '/');
	assert(p != NULL);
	p = strchr(p + 1, '/');
	while (p != NULL)
	{
	    *p = '\0';
	    if (mkdir(path, 0755) != 0 && errno != EEXIST)
	    {
		perror(path);
		exit(1);
	    }
	    *p = '/';

	    p = strchr(p + 1, '/');
	}

	path_len = strlen(path) + 1 + 8 + 4 + 1;

	log_file_path = malloc(path_len);
	assert(log_file_path != NULL);

	strlcpy(log_file_path, path, path_len);
	strlcat(log_file_path, "-", path_len);
	append_date_from_time_string(log_file_path, time_str, path_len);
	strlcat(log_file_path, ".txt", path_len);

	/* printf("LOG FILE: %s\n", log_file_path); */
    }

    fp = fopen(log_file_path, "a");
    if (fp == NULL)
    {
	perror(log_file_path);
	exit(1);
    }
    fprintf(fp, "%s: %s\n", time_str, str);
    fclose(fp);
}


void nwos_log_arguments(int argc, char* argv[])
{
    int i;
    int arg;
    char* src;
    char* msg;
    char* prog_str = "Program:";
    size_t msg_size;

    if (argc == 0 || argv == NULL)
    {
	assert(argc == 0 && argv == NULL);    /* if one is zero, they should both be */
	log_disabled = true;
	return;
    }

    assert(argc > 0);
    assert(argv != NULL);

    /* calculate how large the message needs to be */

    msg_size = strlen(prog_str) + 1;

    for (arg = 0; arg < argc; arg++)
    {
	msg_size = msg_size + strlen(argv[arg]) + 1;
    }

    msg = nwos_malloc(msg_size);

    strlcpy(msg, prog_str, msg_size);

    i = strlen(msg);

    assert(msg[i] == '\0');

    for (arg = 0; arg < argc; arg++)
    {
	assert(i < msg_size);

	msg[i] = ' ';
	i++;

	src = argv[arg];

	while (*src != '\0')
	{
	    assert(i < msg_size);

	    msg[i] = *src;
	    src++;
	    i++;
	}
    }

    assert(i < msg_size);

    msg[i] = '\0';

    nwos_log(msg);

    nwos_free(msg);
}


static char hexdigit[16] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };

void nwos_log_md5sum(const char* file_name, uint8 md5_digest[MD5_DIGEST_SIZE])
{
    char msg[12 + FILENAME_MAX + MD5_DIGEST_SIZE * 2] = "MD5 (";
    int prefix_len;
    int i;

    strlcat(msg, file_name, sizeof(msg));
    strlcat(msg, ") = ", sizeof(msg));

    prefix_len = strlen(msg);

    assert(prefix_len + MD5_DIGEST_SIZE < sizeof(msg));

    for (i = 0; i < MD5_DIGEST_SIZE; i++)
    {
	msg[prefix_len + i * 2] = hexdigit[md5_digest[i] >> 4];
	msg[prefix_len + i * 2 + 1] = hexdigit[md5_digest[i] & 0xf];
    }

    msg[prefix_len + i * 2] = '\0';

    nwos_log(msg);
}


void nwos_log_sha1sum(const char* file_name, uint8 sha1_digest[SHA1_DIGEST_SIZE])
{
    char msg[12 + FILENAME_MAX + SHA1_DIGEST_SIZE * 2] = "SHA1 (";
    int prefix_len;
    int i;

    strlcat(msg, file_name, sizeof(msg));
    strlcat(msg, ") = ", sizeof(msg));

    prefix_len = strlen(msg);

    assert(prefix_len + SHA1_DIGEST_SIZE < sizeof(msg));

    for (i = 0; i < SHA1_DIGEST_SIZE; i++)
    {
	msg[prefix_len + i * 2] = hexdigit[sha1_digest[i] >> 4];
	msg[prefix_len + i * 2 + 1] = hexdigit[sha1_digest[i] & 0xf];
    }

    msg[prefix_len + i * 2] = '\0';

    nwos_log(msg);
}


void nwos_log_sha256sum(const char* file_name, uint8 sha256_digest[SHA256_DIGEST_SIZE])
{
    char msg[16 + FILENAME_MAX + SHA256_DIGEST_SIZE * 2] = "SHA256 (";
    int prefix_len;
    int i;

    strlcat(msg, file_name, sizeof(msg));
    strlcat(msg, ") = ", sizeof(msg));

    prefix_len = strlen(msg);

    assert(prefix_len + SHA256_DIGEST_SIZE < sizeof(msg));

    for (i = 0; i < SHA256_DIGEST_SIZE; i++)
    {
	msg[prefix_len + i * 2] = hexdigit[sha256_digest[i] >> 4];
	msg[prefix_len + i * 2 + 1] = hexdigit[sha256_digest[i] & 0xf];
    }

    msg[prefix_len + i * 2] = '\0';

    nwos_log(msg);
}


void nwos_log_sha512sum(const char* file_name, uint8 sha512_digest[SHA512_DIGEST_SIZE])
{
    char msg[16 + FILENAME_MAX + SHA512_DIGEST_SIZE * 2] = "SHA512 (";
    int prefix_len;
    int i;

    strlcat(msg, file_name, sizeof(msg));
    strlcat(msg, ") = ", sizeof(msg));

    prefix_len = strlen(msg);

    assert(prefix_len + SHA512_DIGEST_SIZE < sizeof(msg));

    for (i = 0; i < SHA512_DIGEST_SIZE; i++)
    {
	msg[prefix_len + i * 2] = hexdigit[sha512_digest[i] >> 4];
	msg[prefix_len + i * 2 + 1] = hexdigit[sha512_digest[i] & 0xf];
    }

    msg[prefix_len + i * 2] = '\0';

    nwos_log(msg);
}


