/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2008, 2009, 2010  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-09-11 12:32:09 -0600 (Sat, 11 Sep 2010) $
--   $Revision: 4713 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"
#include "strlcxx.h"


#define MAX_ENTRIES 1024

typedef uint8 SHA1[SHA1_DIGEST_SIZE];

typedef struct {
    char* file_name;
    int line_number;
    char* name;
    SHA1 sha1;
    bool duplicate_sha1;
} File_and_SHA1;


static File_and_SHA1 entries[MAX_ENTRIES];
static int entry = 0;


static uint8 convert_char_to_hex_value(char c)
{
    uint8 result = 0xff;

    if (isxdigit(c))
    {
	if (isdigit(c))
	{
	    result = c - '0';
	}
	else if (isupper(c))
	{
	    result = c - 'A' + 10;
	}
	else
	{
	    result = c - 'a' + 10;
	}
    }

    return result;
}


int process_file(char* file_name)
{
    int line_number = 0;
    int i;
    size_t j;
    FILE* fp;
    char line[128];
    char *p;
    uint8 upper;
    uint8 lower;
    size_t len;


    fp = fopen(file_name, "r");

    if (fp == NULL)
    {
	perror(file_name);
	exit(1);
    }

    while (fgets(line, sizeof(line), fp) != NULL)
    {
	assert(entry < MAX_ENTRIES);

	line_number++;

	entries[entry].file_name = file_name;
	entries[entry].line_number = line_number;
	entries[entry].duplicate_sha1 = false;

	p = line;

	for (i = 0; i < sizeof(SHA1); i++)
	{
	    upper = convert_char_to_hex_value(*p++);

	    if (upper < 16)
	    {
		lower = convert_char_to_hex_value(*p++);
	    }

	    if (upper > 15 || lower > 15)
	    {
		break;
	    }

	    entries[entry].sha1[i] = (upper << 4) | lower;
	}

	len = 0;

	if (i == sizeof(SHA1) && *p++ == ' ')
	{
	    if (*p == ' ' || *p == '*')
	    {
		p++;                      /* point to the first character of the file name */
		*strchr(p, '\n') = '\0';  /* replace the newline with a null */
		len = strlen(p);
	    }
	}

	if (len == 0)
	{
	    fprintf(stderr, "invalid line: %s\n", line);
	    fclose(fp);
	    exit(1);
	}

	/* Check for duplicates in the list */

	for (i = 0; i < entry; i++)
	{
	    if (strcmp(entries[i].name, p) == 0)   /* duplicate name */
	    {
		if (memcmp(entries[i].sha1, entries[entry].sha1, sizeof(SHA1)) == 0)
		{
		    fprintf(stderr, "Warning: duplicate entry: %s %4d: %s\n", file_name, line_number, line);
		    break;
		}
		else
		{
		    fprintf(stderr, "Error: duplicate file name with different checksum:\n");

		    fprintf(stderr, "%s %4d: ", entries[i].file_name, entries[i].line_number);
		    for (j = 0; j < sizeof(SHA1); j++) fprintf(stderr, "%02x", entries[i].sha1[j]);
		    fprintf(stderr, "  %s\n", entries[i].name);

		    fprintf(stderr, "%s %4d: %s\n", file_name, line_number, line);

		    fclose(fp);
		    exit(1);
		}
	    }
	}

	if (i == entry)  /* no duplicates were found */
	{
	    entries[entry].name = malloc(len + 1);

	    if (entries[entry].name == NULL)
	    {
		perror("allocating memory for name");
		fclose(fp);
		exit(1);
	    }

	    strlcpy(entries[entry].name, p, len + 1);

	    entry++;
	}
    }

    if (ferror(fp))
    {
	perror(file_name);
	fclose(fp);
	exit(1);
    }

    fclose(fp);

    printf("%s: %d\n", file_name, line_number);
    
    return line_number;
}


void remove_entry(int entry_num)
{
    int i;
    size_t j;

    for (i = entry_num; i < entry - 1; i++)
    {
	entries[i].file_name = entries[i+1].file_name;
	entries[i].line_number = entries[i+1].line_number;
	entries[i].name = entries[i+1].name;
	for (j = 0; j < sizeof(SHA1); j++) entries[i].sha1[j] = entries[i+1].sha1[j];
    }

    entry--;
}


int main(int argc, char* argv[])
{
    ObjRef path_ref;
    ObjRef assoc_class_ref;
    ObjRef file_class_ref;
    ObjRef assoc_ref;
    ObjRef file_ref;
    ObjRef sha1_ref;
    ObjRef object_class;
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_File file_obj;
    C_struct_SHA1sum sha1_obj;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    int j;
    size_t k;
    ObjCreateResult result;
    int duplicate_sha1s = 0;
    bool same_sha1_means_same_file = false;


    /***********************************************/
    /* First, parse all the command line arguments */
    /***********************************************/

    if (argc < 2)
    {
	fprintf(stderr, "usage: %s [--same-sha1-means-same-file] sha1_file(s)\n", argv[0]);
	exit(1);
    }

    for (i = 1; i < argc; i++)
    {
	if (*argv[i] == '-')
	{
	    if (strcmp(argv[i], "--same-sha1-means-same-file") == 0)
	    {
		same_sha1_means_same_file = true;
	    }
	    else
	    {
		fprintf(stderr, "unknown option: %s\n", argv[i]);
		fprintf(stderr, "usage: %s [--same-sha1-means-same-file] sha1_file(s)\n", argv[0]);
	    }
	}
	else if (process_file(argv[i]) == 0)
	{
	    fprintf(stderr, "Empty file: %s\n", argv[1]);
	    exit(1);
	}
    }


    /***************************************************************/
    /* Check for duplicate SHA1 checksums with different file names */
    /***************************************************************/

    for (i = 0; i < entry - 1; i++)
    {
	for (j = i + 1; j < entry; j++)
	{
	    if (memcmp(entries[i].sha1, entries[j].sha1, sizeof(SHA1)) == 0)     /* duplicate SHA1 with different file name */
	    {
		fprintf(stderr, "%s: duplicate SHA1 with different file name:\n", same_sha1_means_same_file ? "Warning" : "Error");

		fprintf(stderr, "%s %4d: ", entries[i].file_name, entries[i].line_number);
		for (k = 0; k < sizeof(SHA1); k++) fprintf(stderr, "%02x", entries[i].sha1[k]);
		fprintf(stderr, "  %s\n", entries[i].name);

		fprintf(stderr, "%s %4d: ", entries[j].file_name, entries[j].line_number);
		for (k = 0; k < sizeof(SHA1); k++) fprintf(stderr, "%02x", entries[j].sha1[k]);
		fprintf(stderr, "  %s\n", entries[j].name);

		if (same_sha1_means_same_file)
		{
		    entries[j].duplicate_sha1 = true;
		}
		else
		{
		    duplicate_sha1s++;
		}
	    }
	}
    }

    if (duplicate_sha1s > 0)
    {
	exit(1);
    }


    /**********************/
    /* Start up Objectify */
    /**********************/

    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(PUBLIC, NULL);


    /*********************************************************************************/
    // Now check for duplicate files and checksums that are already in the database. */
    /*********************************************************************************/

    assert(nwos_find_public_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));
    assert(nwos_find_public_class_definition("FILE", &file_class_ref));

    for (i = 0; i < entry; i++)
    {
	if (nwos_find_file_path(entries[i].name, &path_ref))
	{
	    assert(nwos_read_variable_sized_object_from_disk(&path_ref, kludge, sizeof(kludge), &nwos_get_object_size));

	    /* remember ptr_path_obj points to the kludge buffer */

	    ref_list = nwos_malloc_reference_list(&ptr_path_obj->header.object.references);

	    num_refs = ref_list->common_header.num_refs;

	    void_reference(&assoc_ref);

	    for (j = 0; j < num_refs; j++)
	    {
		nwos_get_object_class(&ref_list->references[j], &object_class);

		if (is_same_object(&object_class, &assoc_class_ref))
		{
		    assert(nwos_read_object_from_disk(&ref_list->references[j], &assoc_obj, sizeof(assoc_obj)));

		    assert(is_same_object(&assoc_obj.path, &path_ref));

		    copy_reference(&assoc_ref, &ref_list->references[j]);

		    break;
		}
	    }

	    assert(j < num_refs);
	    assert(!is_void_reference(&assoc_ref));

	    /* for now, verify that there was only one association object for this name */

	    for (j++ ; j < num_refs; j++)
	    {
		nwos_get_object_class(&ref_list->references[j], &object_class);

		assert(!is_same_object(&object_class, &assoc_class_ref));
	    }

	    nwos_free_reference_list(ref_list);
	    ref_list = NULL;

	    assert(nwos_read_object_from_disk(&assoc_ref, &assoc_obj, sizeof(assoc_obj)));
	    assert(nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj)));
	    assert(nwos_read_object_from_disk(&file_obj.sha1sum, &sha1_obj, sizeof(sha1_obj)));

	    if (memcmp(entries[i].sha1, sha1_obj.sha1sum, sizeof(SHA1)) == 0)
	    {
		fprintf(stderr, "Warning: duplicate entry already in database:\n");

		fprintf(stderr, "%s %4d: ", entries[i].file_name, entries[i].line_number);
		for (j = 0; j < sizeof(SHA1); j++) fprintf(stderr, "%02x", entries[i].sha1[j]);
		fprintf(stderr, "  %s\n", entries[i].name);

		remove_entry(i);
		i--;
	    }
	    else
	    {
		fprintf(stderr, "Error: duplicate entry already in database with different checksum:\n");

		fprintf(stderr, "%s %4d: ", entries[i].file_name, entries[i].line_number);
		for (j = 0; j < sizeof(SHA1); j++) fprintf(stderr, "%02x", entries[i].sha1[j]);
		fprintf(stderr, "  %s\n", entries[i].name);

		for (j = 0; j < sizeof(SHA1); j++) fprintf(stderr, "%02x", sha1_obj.sha1sum[j]);
		fprintf(stderr, "  %s\n", entries[i].name);

		nwos_terminate_objectify();
		exit(1);
	    }
	}
	else if (nwos_find_sha1(entries[i].sha1, &sha1_ref))
	{
	    /* first, find the file object for this SHA1 sum */

	    assert(nwos_read_object_from_disk(&sha1_ref, &sha1_obj, sizeof(sha1_obj)));

	    ref_list = nwos_malloc_reference_list(&sha1_obj.header.object.references);

	    num_refs = ref_list->common_header.num_refs;

	    void_reference(&file_ref);

	    for (j = 0; j < num_refs; j++)
	    {
		nwos_get_object_class(&ref_list->references[j], &object_class);

		if (is_same_object(&object_class, &file_class_ref))
		{
		    assert(nwos_read_object_from_disk(&ref_list->references[j], &file_obj, sizeof(file_obj)));

		    assert(is_same_object(&file_obj.sha1sum, &sha1_ref));

		    copy_reference(&file_ref, &ref_list->references[j]);

		    break;
		}
	    }

	    assert(j < num_refs);
	    assert(!is_void_reference(&file_ref));

	    /* for now, verify that there was only one file object for this sha1sum */

	    for (j++ ; j < num_refs; j++)
	    {
		nwos_get_object_class(&ref_list->references[j], &object_class);

		assert(!is_same_object(&object_class, &file_class_ref));
	    }

	    nwos_free_reference_list(ref_list);
	    ref_list = NULL;

	    /* second, find the association object for this file */

	    ref_list = nwos_malloc_reference_list(&file_obj.header.object.references);

	    num_refs = ref_list->common_header.num_refs;

	    void_reference(&assoc_ref);

	    for (j = 0; j < num_refs; j++)
	    {
		nwos_get_object_class(&ref_list->references[j], &object_class);

		if (is_same_object(&object_class, &assoc_class_ref))
		{
		    assert(nwos_read_object_from_disk(&ref_list->references[j], &assoc_obj, sizeof(assoc_obj)));

		    assert(is_same_object(&assoc_obj.file, &file_ref));

		    copy_reference(&assoc_ref, &ref_list->references[j]);

		    break;
		}
	    }

	    assert(j < num_refs);
	    assert(!is_void_reference(&assoc_ref));

	    /* for now, verify that there was only one association object for this sha1sum */

	    for (j++ ; j < num_refs; j++)
	    {
		nwos_get_object_class(&ref_list->references[j], &object_class);

		assert(!is_same_object(&object_class, &assoc_class_ref));
	    }

	    nwos_free_reference_list(ref_list);
	    ref_list = NULL;

	    assert(nwos_read_object_from_disk(&assoc_ref, &assoc_obj, sizeof(assoc_obj)));
	    assert(nwos_read_variable_sized_object_from_disk(&assoc_obj.path, kludge, sizeof(kludge), &nwos_get_object_size));

	    fprintf(stderr, "%s: file with duplicate SHA1 checksum already in Public Objects:\n", same_sha1_means_same_file ? "Warning" : "Error");

	    fprintf(stderr, "%s %4d: ", entries[i].file_name, entries[i].line_number);
	    for (j = 0; j < sizeof(SHA1); j++) fprintf(stderr, "%02x", entries[i].sha1[j]);
	    fprintf(stderr, "  %s\n", entries[i].name);

	    for (j = 0; j < sizeof(SHA1); j++) fprintf(stderr, "%02x", sha1_obj.sha1sum[j]);
	    fprintf(stderr, "  %s\n", ptr_path_obj->storage);

	    if (same_sha1_means_same_file)
	    {
		entries[i].duplicate_sha1 = true;
	    }
	    else
	    {
		duplicate_sha1s++;
	    }
	}
    }
	    
    if (duplicate_sha1s > 0)
    {
	nwos_terminate_objectify();
	exit(1);
    }


    /******************************************/
    /* Finally, add entries to Public Objects */
    /******************************************/

    for (i = 0; i < entry; i++)
    {
	int j;

	for (j = 0; j < sizeof(SHA1); j++)
	{
	    printf("%02x", entries[i].sha1[j]);
	}
	printf("  %s\n", entries[i].name);

	if (entries[i].duplicate_sha1)
	{
	    result = nwos_create_file_with_duplicate_sha1(entries[i].name, entries[i].sha1, &assoc_ref);

	    if (result == CREATED_NEW)
	    {
		printf("created duplicate file and path association: %02x%02x%02x%02x\n", 
		       assoc_ref.id[0], assoc_ref.id[1], assoc_ref.id[2], assoc_ref.id[3]);
	    }
	    else
	    {
		printf("result: %d\n", result);
	    }
	}
	else
	{
	    result = nwos_create_file_with_only_sha1(entries[i].name, entries[i].sha1, &assoc_ref);

	    if (result == CREATED_NEW)
	    {
		printf("created new file and path association: %02x%02x%02x%02x\n", 
		       assoc_ref.id[0], assoc_ref.id[1], assoc_ref.id[2], assoc_ref.id[3]);
	    }
	    else
	    {
		printf("result: %d\n", result);
	    }
	}

	assert(!is_void_reference(&assoc_ref));
    }


    /***********************/
    /* Finish up Objectify */
    /***********************/

    nwos_terminate_objectify();

    return 0;
}


