/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2008, 2009, 2010  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-03-31 07:22:42 -0600 (Wed, 31 Mar 2010) $
--   $Revision: 4651 $
--
*/

#include <assert.h>
#include <limits.h>    /* define HOST_NAME_MAX */
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/resource.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>

#include "progress_bar.h"
#include "types.h"


#define DEFAULT_PRIORITY 1

#define SECONDS_IN_A_WEEK (60 * 60 * 24 * 7)


struct
{
  uint8 block[256];
} sequence[4096];


static bool control_c_received = false;


static void control_c_handler(int signal_num)
{
    signal(SIGINT, SIG_DFL);    /* restore to default control-C handler */

    fputs("^C", stdout);     /* indicate that control-C was received */
    fflush(stdout);

    control_c_received = true;
}

static void print_usage(char *prog_name)
{
    fprintf(stderr, "usage: %s [--use-dev-random |--use-dev-srandom |--use-dev-urandom] [--seconds min:max]\n", prog_name);
    fprintf(stderr, "\n");
    fprintf(stderr, "The minimum seconds must be at least 60 and the maximum seconds must be at least 30 more.\n");
}



int main(int argc, char* argv[])
{
    struct timespec ts;
    int i;
    int j;
    int k;
    int argi;
    int num_spins;
    int start_time;
    int seconds_to_shuffle;
    int start_shuffle;
    uint64 num_shuffles;
    int elapsed_time;
    char hostname[HOST_NAME_MAX];
    char filename[HOST_NAME_MAX+16];
    FILE* fp;
    uint8 swap;
    char* ptr;
    long int long_int;
    char* random_device = NULL;
    uint random_word = 0;
    int min_seconds = 0;
    int max_seconds = 0;


    /* REMEMBER: in the real program to generate a key, using the random device should be the default! */

    if (argc > 4)
    {
	print_usage(argv[0]);
	exit(1);
    }

    for (argi = 1; argi < argc; argi++)
    {
	if (strcmp(argv[argi], "--seconds") == 0)
	{
	    if (min_seconds != 0 || max_seconds != 0)
	    {
		fprintf(stderr, "Seconds cannot be specified twice\n");
		print_usage(argv[0]);
		exit(1);
	    }
	    argi++;

	    long_int = strtol(argv[argi], &ptr, 10);

	    if (*ptr != ':')
	    {
		fprintf(stderr, "Invalid minimum seconds specified\n");
		print_usage(argv[0]);
		exit(1);
	    }

	    if (long_int < 60 || long_int > SECONDS_IN_A_WEEK - 60)
	    {
		fprintf(stderr, "Minimum seconds must be between 60 and %d.\n", SECONDS_IN_A_WEEK - 60);
		print_usage(argv[0]);
		exit(1);
	    }

	    min_seconds = (int) long_int;

	    long_int = strtol(ptr + 1, &ptr, 10);

	    if (*ptr != '\0')
	    {
		fprintf(stderr, "Invalid maximum seconds specified\n");
		print_usage(argv[0]);
		exit(1);
	    }

	    if (long_int < min_seconds + 30 || long_int > SECONDS_IN_A_WEEK)
	    {
		fprintf(stderr, "Maximum seconds must be between %d and %d.\n", min_seconds + 30, SECONDS_IN_A_WEEK);
		print_usage(argv[0]);
		exit(1);
	    }

	    max_seconds = (int) long_int;
	}
	else if (strcmp(argv[argi], "--use-dev-random") == 0)
	{
	    if (random_device != NULL)
	    {
		fprintf(stderr, "Only one random device can be specified\n");
		print_usage(argv[0]);
		exit(1);
	    }
	    random_device = "/dev/random";
	}
	else if (strcmp(argv[argi], "--use-dev-srandom") == 0)   /* for OpenBSD */
	{
	    if (random_device != NULL)
	    {
		fprintf(stderr, "Only one random device can be specified\n");
		print_usage(argv[0]);
		exit(1);
	    }
	    random_device = "/dev/srandom";
	}
	else if (strcmp(argv[argi], "--use-dev-urandom") == 0)
	{
	    if (random_device != NULL)
	    {
		fprintf(stderr, "Only one random device can be specified\n");
		print_usage(argv[0]);
		exit(1);
	    }
	    random_device = "/dev/urandom";
	}
	else
	{
	    fprintf(stderr, "Unknown option: %s\n", argv[argi]);
	    print_usage(argv[0]);
	    exit(1);
	}
    }

    if (setpriority(PRIO_PROCESS, 0, DEFAULT_PRIORITY) != 0)
    {
	fprintf(stderr, "WARNING: setting process priority to %d failed!\n", DEFAULT_PRIORITY);
    }

    printf("Process priority: %d\n", getpriority(PRIO_PROCESS, 0));

    clock_gettime(CLOCK_REALTIME, &ts);
    start_time = (int)ts.tv_sec;

    if (random_device != NULL)
    {
	fp = fopen(random_device, "r");

	if (fp == NULL)
	{
	    perror(random_device);
	    exit(1);
	}

	printf("reading from %s: ", random_device);
	fflush(stdout);

	if (fread(&random_word, 1, sizeof(random_word), fp) != sizeof(random_word))
	{
	    printf("\n");
	    perror(random_device);
	    exit(1);
	}

	fclose(fp);
	printf("%08x\n", random_word);
    }

    nwos_start_progress_bar();

    for (i = 0; i < 4096; i++)
    {
	nwos_update_progress_bar((float)i / 4096.0);

	clock_gettime(CLOCK_REALTIME, &ts);
	srandom((uint)ts.tv_sec ^ (uint)ts.tv_nsec ^ random_word);
	usleep(random() % 659);

	clock_gettime(CLOCK_REALTIME, &ts);
	num_spins = ((int)ts.tv_nsec ^ random()) % 7919;
	for (j = 0; j < num_spins; j++) random();    /* spin it a random number of times */
	usleep(random() % 347);

	clock_gettime(CLOCK_REALTIME, &ts);
	num_spins = ((int)ts.tv_nsec ^ random()) % 2111;
	for (j = 0; j < num_spins; j++) random();    /* spin it a random number of times */

	sequence[i].block[8] = 8;

	for (j = 1; j < 248; j++)
	{
	    k = random() % (j + 1);

	    sequence[i].block[j + 8] = sequence[i].block[k + 8];
	    sequence[i].block[k + 8] = j + 8;
	}

	random_word = random_word ^ ((random() << 31) | random()); 
    }

    nwos_finish_progress_bar();

    elapsed_time = (int)ts.tv_sec - start_time;       /* elapsed time to do the first pass */

    printf("First pass time: %02d:%02d:%02d\n", elapsed_time / 3600, (elapsed_time % 3600) / 60, elapsed_time % 60);

    if (min_seconds == 0 || max_seconds == 0)
    {
	min_seconds = elapsed_time * 12;
	max_seconds = elapsed_time * 13;
    }

    seconds_to_shuffle = min_seconds + ((random() ^ (uint)ts.tv_nsec) % (max_seconds - min_seconds + 1));
    printf("Seconds to shuffle: %u\n", seconds_to_shuffle);

    clock_gettime(CLOCK_REALTIME, &ts);
    start_shuffle = (int)ts.tv_sec;

    if (signal(SIGINT, control_c_handler) == SIG_ERR)
    {
	perror("setting SIGINT signal to catch control-C");
    }

    nwos_start_progress_bar();

    num_shuffles = 0;

    do
    {
	nwos_update_progress_bar((float)((int)ts.tv_sec - start_shuffle) / (float)seconds_to_shuffle);

	clock_gettime(CLOCK_REALTIME, &ts);
	srandom((uint)ts.tv_sec ^ (uint)ts.tv_nsec ^ random_word);
	usleep(random() % 307);

	clock_gettime(CLOCK_REALTIME, &ts);
	num_spins = ((int)ts.tv_nsec ^ random()) % 6661;
	for (j = 0; j < num_spins; j++) random();    /* spin it a random number of times */
	usleep(random() % 701);

	i = random() % 4096;

	clock_gettime(CLOCK_REALTIME, &ts);
	num_spins = ((int)ts.tv_nsec ^ random()) % 3433;
	for (j = 0; j < num_spins; j++) random();    /* spin it a random number of times */

	for (j = 1; j < 248; j++)
	{
	    k = random() % (j + 1);

	    swap = sequence[i].block[j + 8];
	    sequence[i].block[j + 8] = sequence[i].block[k + 8];
	    sequence[i].block[k + 8] = swap;
	}

	clock_gettime(CLOCK_REALTIME, &ts);

	random_word = random_word ^ ((random() << 31) | random()); 

	num_shuffles++;
    }
    while (((int)ts.tv_sec - start_shuffle) < seconds_to_shuffle && !control_c_received);

    nwos_finish_progress_bar();

    printf("Number of shuffles: %llu\n", num_shuffles);

    printf("Checking...\n");
    fflush(stdout);

    nwos_start_progress_bar();

    for (i = 0; i < 4096; i++)
    {
	nwos_update_progress_bar((float)i / 4096.0);

	for (j = 0; j < 256; j++)
	{
	    k = 0;

	    if (j < 8)
	    {
		if (sequence[i].block[j] != 0)
		{
		    k = 256;
		}
	    }
	    else
	    {
		for (k = 8; k < 256; k++)
		{
		    if (sequence[i].block[k] == j) break;
		}
	    }

	    if (k >= 256)
	    {
		printf("block: %d  doesn't have: %02x\n", i, j);
		for (k = 0; k < 256; k++)
		{
		    printf("%02x%c", sequence[i].block[k], k % 16 == 15 ? '\n' : ' ');
		}
		exit(1);
	    }
	}
    }

    nwos_finish_progress_bar();

    /* write file out */

    if (gethostname(hostname, sizeof(hostname)) < 0)
    {
	fprintf(stderr, "gethostname failed, using 'default' for file name\n");
	strncpy(hostname, "default", sizeof(hostname));
    }

    printf("\n");

    clock_gettime(CLOCK_REALTIME, &ts);

    elapsed_time = (int)ts.tv_sec - start_time;

    printf("Elapsed time: %02d:%02d:%02d\n", elapsed_time / 3600, (elapsed_time % 3600) / 60, elapsed_time % 60);

    snprintf(filename, sizeof(filename), "%s-%u.tbl", hostname, (uint)ts.tv_sec);

    fp = fopen(filename, "w");

    if (fp == NULL)
    {
	perror("test.out");
	exit(1);
    }

    for (i = 0; i < 4096; i++)
    {
	if (fwrite(sequence[i].block, 1, 256, fp) != 256)
	{
	    perror("test.out");
	    exit(1);
	}
    }

    if (fclose(fp) != 0)
    {
	perror("test.out");
	exit(1);
    }

    return control_c_received;
}


